/*
 * Decompiled with CFR 0.152.
 */
package me.valkeea.fishyaddons.api;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import me.valkeea.fishyaddons.cache.ApiCache;
import me.valkeea.fishyaddons.config.FishyConfig;
import me.valkeea.fishyaddons.util.FishyNotis;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_2561;

@Environment(value=EnvType.CLIENT)
public class HypixelPriceClient {
    private static final String BZ_API_URL = "https://api.hypixel.net/v2/skyblock/bazaar";
    private static final String AUCTIONS_API_URL = "https://api.hypixel.net/v2/skyblock/auctions";
    private static final int BZ_CACHE_EXPIRY_MINUTES = 60;
    private final HttpClient httpClient;
    private final Gson gson;
    private final Map<String, PriceData> bazaarCache;
    private final Map<String, PriceData> auctionCache;
    private final ScheduledExecutorService scheduler;
    private volatile long lastBazaarUpdate = 0L;
    private volatile long lastAuctionUpdate = 0L;
    private volatile long sessionStartTime = System.currentTimeMillis();
    private static String type = "sellPrice";
    private static final String SUCCESS_KEY = "success";

    public HypixelPriceClient() {
        this.httpClient = HttpClient.newBuilder().connectTimeout(Duration.ofSeconds(10L)).build();
        this.gson = new Gson();
        this.bazaarCache = new ConcurrentHashMap<String, PriceData>();
        this.auctionCache = new ConcurrentHashMap<String, PriceData>();
        this.scheduler = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = new Thread(r, "HypixelPriceClient-Scheduler");
            t.setDaemon(true);
            return t;
        });
        this.scheduler.scheduleAtFixedRate(this::refreshBazaarAsync, 0L, 60L, TimeUnit.MINUTES);
    }

    public static void setPriceType(String newType) {
        if (newType == null || newType.isEmpty()) {
            throw new IllegalArgumentException("Price type cannot be null or empty");
        }
        if (!newType.equals("sellPrice") && !newType.equals("buyPrice")) {
            throw new IllegalArgumentException("Invalid price type: " + newType);
        }
        if (!newType.equals(type)) {
            ApiCache.clearAllCaches();
        }
        type = newType;
        FishyConfig.setString("priceType", newType);
    }

    public double getBestPrice(String itemName) {
        String apiId = this.convertToApiId(itemName);
        Double cachedPrice = ApiCache.getCachedPrice(apiId, ApiCache.PriceType.BEST_PRICE);
        if (cachedPrice != null) {
            return cachedPrice;
        }
        PriceData bazaarPrice = this.bazaarCache.get(apiId);
        double bestPrice = bazaarPrice != null && !bazaarPrice.isExpired(60) ? bazaarPrice.buyPrice : this.getLowestBinPrice(itemName);
        ApiCache.cachePrice(apiId, ApiCache.PriceType.BEST_PRICE, bestPrice);
        return bestPrice;
    }

    public double getBazaarBuyPrice(String itemName) {
        String apiId = this.convertToApiId(itemName);
        Double cachedPrice = ApiCache.getCachedPrice(apiId, ApiCache.PriceType.BZ_BUY);
        if (cachedPrice != null) {
            return cachedPrice;
        }
        PriceData priceData = this.bazaarCache.get(apiId);
        double price = 0.0;
        if (priceData != null && !priceData.isExpired(60)) {
            price = priceData.getPrice(type);
        }
        ApiCache.cachePrice(apiId, ApiCache.PriceType.BZ_BUY, price);
        return price;
    }

    public double getLowestBinPrice(String itemName) {
        double price;
        String apiId = this.convertToApiId(itemName);
        Double cachedPrice = ApiCache.getCachedPrice(apiId, ApiCache.PriceType.AH_BIN);
        if (cachedPrice != null) {
            return cachedPrice;
        }
        String cleanedItemName = this.convertDisplayNameToApiId(itemName);
        PriceData priceData = this.auctionCache.get(cleanedItemName);
        if (priceData != null) {
            price = priceData.buyPrice;
        } else {
            price = this.searchAuctionForItem(cleanedItemName);
            this.auctionCache.put(cleanedItemName, new PriceData(price, price, System.currentTimeMillis()));
        }
        ApiCache.cachePrice(apiId, ApiCache.PriceType.AH_BIN, price);
        return price;
    }

    public boolean hasBazaarData(String itemName) {
        String apiId = this.convertToApiId(itemName);
        PriceData priceData = this.bazaarCache.get(apiId);
        return priceData != null && !priceData.isExpired(60);
    }

    public boolean hasAuctionData(String itemName) {
        String cleanedItemName = this.convertDisplayNameToApiId(itemName);
        PriceData priceData = this.auctionCache.get(cleanedItemName);
        return priceData != null;
    }

    public boolean hasPriceData(String itemName) {
        return this.hasBazaarData(itemName) || this.hasAuctionData(itemName);
    }

    public PriceSource getPriceSource(String itemName) {
        if (this.hasBazaarData(itemName)) {
            return PriceSource.BAZAAR;
        }
        if (this.hasAuctionData(itemName)) {
            return PriceSource.AUCTION;
        }
        return PriceSource.NONE;
    }

    public long getLastBazaarUpdate() {
        return this.lastBazaarUpdate;
    }

    public long getLastAuctionUpdate() {
        return this.lastAuctionUpdate;
    }

    public void refreshBazaarAsync() {
        CompletableFuture.runAsync(this::refreshBazaar);
    }

    public void refreshAuctionsAsync() {
        if (this.auctionCache.isEmpty()) {
            return;
        }
        CompletableFuture.runAsync(() -> {
            String[] cachedItems;
            for (String itemName : cachedItems = this.auctionCache.keySet().toArray(new String[0])) {
                try {
                    double newPrice = this.searchAuctionForItem(itemName);
                    this.auctionCache.put(itemName, new PriceData(newPrice, newPrice, System.currentTimeMillis()));
                }
                catch (Exception e) {
                    System.err.println("Error refreshing price for " + itemName + ": " + e.getMessage());
                }
            }
        });
        FishyNotis.send((class_2561)class_2561.method_43470((String)"\u00a7bAuction cache refreshed successfully!"));
    }

    public void refreshAllAsync() {
        this.refreshBazaarAsync();
        this.refreshAuctionsAsync();
    }

    private void refreshBazaar() {
        try {
            HttpRequest request = HttpRequest.newBuilder().uri(URI.create(BZ_API_URL)).timeout(Duration.ofSeconds(30L)).header("User-Agent", "FishyAddons").GET().build();
            HttpResponse<String> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofString());
            if (response.statusCode() == 200) {
                this.parseBazaarData(response.body());
                this.lastBazaarUpdate = System.currentTimeMillis();
            } else {
                System.err.println("Failed to fetch bazaar data: HTTP " + response.statusCode());
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            System.err.println("Bazaar data fetch interrupted: " + e.getMessage());
        }
        catch (Exception e) {
            System.err.println("Error fetching bazaar data: " + e.getMessage());
        }
    }

    private void parseBazaarData(String responseBody) {
        try {
            JsonObject root = (JsonObject)this.gson.fromJson(responseBody, JsonObject.class);
            if (!root.get(SUCCESS_KEY).getAsBoolean()) {
                System.err.println("Bazaar API returned success=false");
                return;
            }
            JsonObject products = root.getAsJsonObject("products");
            long now = System.currentTimeMillis();
            for (Map.Entry entry : products.entrySet()) {
                String productId = (String)entry.getKey();
                JsonObject product = ((JsonElement)entry.getValue()).getAsJsonObject();
                JsonObject quickStatus = product.getAsJsonObject("quick_status");
                if (quickStatus == null) continue;
                double buyPrice = quickStatus.has("buyPrice") ? quickStatus.get("buyPrice").getAsDouble() : 0.0;
                double sellPrice = quickStatus.has("sellPrice") ? quickStatus.get("sellPrice").getAsDouble() : 0.0;
                PriceData priceData = new PriceData(buyPrice, sellPrice, now);
                this.bazaarCache.put(productId, priceData);
            }
        }
        catch (Exception e) {
            System.err.println("Error parsing bazaar data: " + e.getMessage());
            e.printStackTrace();
        }
        FishyNotis.send((class_2561)class_2561.method_43470((String)"\u00a7bBazaar cache refreshed successfully!"));
    }

    private String convertToApiId(String itemName) {
        String directMapping;
        String cachedApiId = ApiCache.getCachedApiId(itemName);
        if (cachedApiId != null) {
            return cachedApiId;
        }
        String normalized = itemName.trim();
        String apiId = normalized.toLowerCase().contains("shard") && !normalized.toLowerCase().contains("shard of the shredded") && !normalized.toLowerCase().contains("prismarine shard") ? this.convertShardToApiId(normalized) : ((directMapping = this.getDirectMapping(normalized.toLowerCase())) != null ? directMapping : (this.isEnchantment(normalized) ? this.convertEnchantmentToApiId(normalized, false) : normalized.replaceAll("\\s+", "_").toUpperCase()));
        ApiCache.cacheApiId(itemName, apiId);
        return apiId;
    }

    private String convertShardToApiId(String shardName) {
        String mobName = shardName.toLowerCase().replace(" shard", "").replace(" shards", "").replaceAll("\\s+", "_").toUpperCase();
        return "SHARD_" + mobName;
    }

    private boolean isEnchantment(String itemName) {
        String lower = itemName.toLowerCase();
        if (lower.matches(".*\\b([1-9]|10)\\b.*") || lower.matches(".*\\b(i{1,3}|iv|v|vi{0,3}|ix|x)\\b.*")) {
            return true;
        }
        boolean hasValidLevel = lower.matches(".*\\b([1-9]|10)\\b.*") || lower.matches(".*\\b(i{1,3}|iv|v|vi{0,3}|ix|x)\\b.*");
        boolean hasEnchantWords = lower.contains("book") || lower.contains("ultimate");
        return hasValidLevel && hasEnchantWords;
    }

    private String convertEnchantmentToApiId(String enchantName, boolean isUltimate) {
        String normalized = enchantName.toLowerCase().trim();
        Object level = "";
        Pattern levelPattern = Pattern.compile("\\b(\\d+)\\b");
        Matcher matcher = levelPattern.matcher(normalized);
        if (matcher.find()) {
            level = "_" + matcher.group(1);
            normalized = normalized.replaceAll("\\b\\d+\\b", "").trim();
        } else {
            Pattern romanPattern = Pattern.compile("\\b(I{1,3}|IV|V|VI{0,3}|IX|X)\\b");
            Matcher romanMatcher = romanPattern.matcher(normalized.toUpperCase());
            if (romanMatcher.find()) {
                String roman = romanMatcher.group(1);
                level = "_" + this.convertRomanToNumber(roman);
                normalized = normalized.replaceAll("\\b(?i)" + roman + "\\b", "").trim();
            }
        }
        normalized = normalized.replace("enchanted book", "").replace("enchantment", "").replace(" book", "").replace("ultimate ", "").trim();
        String enchantBase = normalized.replaceAll("\\s+", "_").toUpperCase();
        if (isUltimate) {
            return "ENCHANTMENT_ULTIMATE_" + enchantBase + (String)level;
        }
        return "ENCHANTMENT_" + enchantBase + (String)level;
    }

    private int convertRomanToNumber(String roman) {
        switch (roman.toUpperCase()) {
            case "I": {
                return 1;
            }
            case "II": {
                return 2;
            }
            case "III": {
                return 3;
            }
            case "IV": {
                return 4;
            }
            case "V": {
                return 5;
            }
            case "VI": {
                return 6;
            }
            case "VII": {
                return 7;
            }
            case "VIII": {
                return 8;
            }
            case "IX": {
                return 9;
            }
            case "X": {
                return 10;
            }
        }
        return 1;
    }

    public double getEnchantmentPrice(String enchantmentName, boolean isUltimate) {
        String apiId = this.convertEnchantmentToApiId(enchantmentName, isUltimate);
        Double cachedPrice = ApiCache.getCachedPrice(apiId, ApiCache.PriceType.BZ_BUY);
        if (cachedPrice != null) {
            return cachedPrice;
        }
        PriceData priceData = this.bazaarCache.get(apiId);
        double price = 0.0;
        if (priceData != null && !priceData.isExpired(60)) {
            price = priceData.buyPrice;
        }
        ApiCache.cachePrice(apiId, ApiCache.PriceType.BZ_BUY, price);
        return price;
    }

    public void shutdown() {
        this.scheduler.shutdown();
        try {
            if (!this.scheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.scheduler.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.scheduler.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    private double searchAuctionForItem(String cleanedItemName) {
        try {
            int maxPagesToSearch = 25;
            double lowestPrice = 0.0;
            int foundCount = 0;
            int emptyPageCount = 0;
            for (int page = 0; page < maxPagesToSearch; ++page) {
                String url = "https://api.hypixel.net/v2/skyblock/auctions?page=" + page;
                HttpRequest request = HttpRequest.newBuilder().uri(URI.create(url)).timeout(Duration.ofSeconds(10L)).header("User-Agent", "FishyAddons/2.0.3").GET().build();
                HttpResponse<String> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofString());
                if (response.statusCode() != 200) {
                    System.err.println("Failed to fetch auction page " + page + ": HTTP " + response.statusCode());
                    continue;
                }
                double pageLowestPrice = this.parseAuctionPageForItem(response.body(), cleanedItemName);
                if (pageLowestPrice > 0.0) {
                    ++foundCount;
                    emptyPageCount = 0;
                    if (lowestPrice == 0.0 || pageLowestPrice < lowestPrice) {
                        lowestPrice = pageLowestPrice;
                    }
                    if (foundCount >= 5 && page >= 5) {
                        break;
                    }
                } else if (foundCount > 0 && ++emptyPageCount >= 3) break;
                try {
                    Thread.sleep(15L);
                    continue;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    break;
                }
            }
            return lowestPrice;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            System.err.println("Auction search interrupted for " + cleanedItemName + ": " + e.getMessage());
            return 0.0;
        }
        catch (Exception e) {
            System.err.println("Error during on-demand auction search for " + cleanedItemName + ": " + e.getMessage());
            return 0.0;
        }
    }

    private double parseAuctionPageForItem(String responseBody, String targetItemName) {
        try {
            JsonObject root = JsonParser.parseString((String)responseBody).getAsJsonObject();
            if (!root.has(SUCCESS_KEY) || !root.get(SUCCESS_KEY).getAsBoolean()) {
                return 0.0;
            }
            JsonArray auctions = root.getAsJsonArray("auctions");
            double lowestPrice = 0.0;
            for (JsonElement auctionElement : auctions) {
                String itemName;
                String cleanedName;
                JsonObject auction = auctionElement.getAsJsonObject();
                if (!auction.has("bin") || !auction.get("bin").getAsBoolean() || !(cleanedName = this.convertDisplayNameToApiId(itemName = auction.get("item_name").getAsString())).equalsIgnoreCase(targetItemName)) continue;
                double price = auction.get("starting_bid").getAsDouble();
                if (lowestPrice != 0.0 && !(price < lowestPrice)) continue;
                lowestPrice = price;
            }
            return lowestPrice;
        }
        catch (Exception e) {
            System.err.println("Error parsing auction page for " + targetItemName + ": " + e.getMessage());
            return 0.0;
        }
    }

    public void clearAuctionCache() {
        this.auctionCache.clear();
        this.sessionStartTime = System.currentTimeMillis();
    }

    public long getSessionStartTime() {
        return this.sessionStartTime;
    }

    public void setSessionStartTime(long timestamp) {
        this.sessionStartTime = timestamp;
    }

    public double getCachedAuctionPrice(String itemName) {
        String cleanedItemName = this.convertDisplayNameToApiId(itemName);
        PriceData priceData = this.auctionCache.get(cleanedItemName);
        return priceData != null ? priceData.buyPrice : 0.0;
    }

    private String convertDisplayNameToApiId(String displayName) {
        String[] reforgePrefixes;
        String cleaned = displayName.replaceAll("\u00a7[0-9a-fk-or]", "");
        if (cleaned.matches("\\[Lvl \\d+\\].*")) {
            cleaned = cleaned.replaceAll("\\[Lvl \\d+\\]\\s*", "");
        }
        for (String prefix : reforgePrefixes = new String[]{"Sharp", "Heroic", "Spicy", "Legendary", "Fabled", "Withered", "Ancient", "Necrotic", "Pleasant", "Precise", "Spiritual", "Headstrong", "Clean", "Fierce", "Heavy", "Light", "Mythic", "Pure", "Smart", "Titanic", "Wise", "Perfect", "Refined", "Blessed", "Fruitful", "Magnetic", "Fleet", "Stellar", "Heated", "Ambered", "Keen", "Strong", "Festive", "Submerged", "Mossy"}) {
            if (!cleaned.startsWith(prefix + " ")) continue;
            cleaned = cleaned.substring(prefix.length() + 1);
            break;
        }
        cleaned = cleaned.replaceAll("[\u272a\u27a4\u25c6\u269a]+", "").trim();
        cleaned = cleaned.replaceAll("\\s*\\+\\d+\\s*$", "").trim();
        cleaned = cleaned.replaceAll("\\s*\\([^)]+\\)\\s*$", "").trim();
        cleaned = cleaned.replaceAll("\\s*\\([^)]*\\d+[^)]*\\)\\s*$", "").trim();
        return cleaned;
    }

    private String getDirectMapping(String itemName) {
        switch (itemName) {
            case "experience bottle": {
                return "EXPERIENCE_BOTTLE";
            }
            case "grand experience bottle": {
                return "GRAND_EXP_BOTTLE";
            }
            case "titanic experience bottle": {
                return "TITANIC_EXP_BOTTLE";
            }
            case "colossal experience bottle": {
                return "COLOSSAL_EXP_BOTTLE";
            }
            case "lily pad": {
                return "WATER_LILY";
            }
            case "enchanted lily pad": {
                return "ENCHANTED_WATER_LILY";
            }
            case "raw salmon": {
                return "RAW_FISH:1";
            }
            case "clownfish": {
                return "RAW_FISH:2";
            }
            case "pufferfish": {
                return "RAW_FISH:3";
            }
            case "ink sac": {
                return "INK_SACK";
            }
            case "emperor's skull": {
                return "DIVER_FRAGMENT";
            }
            case "hay bale": {
                return "HAY_BLOCK";
            }
            case "enchanted hay bale": {
                return "ENCHANTED_HAY_BLOCK";
            }
            case "lucky clover core": {
                return "PET_ITEM_LUCKY_CLOVER_DROP";
            }
            case "duplex i": {
                return "ENCHANTMENT_ULTIMATE_REITERATE_1";
            }
            case "coins": {
                return null;
            }
        }
        return null;
    }

    @Environment(value=EnvType.CLIENT)
    private static class PriceData {
        final double buyPrice;
        final double sellPrice;
        final long timestamp;

        PriceData(double buyPrice, double sellPrice, long timestamp) {
            this.buyPrice = buyPrice;
            this.sellPrice = sellPrice;
            this.timestamp = timestamp;
        }

        boolean isExpired(int expiryMinutes) {
            return System.currentTimeMillis() - this.timestamp > (long)expiryMinutes * 60000L;
        }

        double getPrice(String priceType) {
            return "sellPrice".equals(priceType) ? this.sellPrice : this.buyPrice;
        }
    }

    @Environment(value=EnvType.CLIENT)
    public static enum PriceSource {
        BAZAAR,
        AUCTION,
        NONE;

    }
}

