package eva.replacer.config;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import eva.replacer.RePlacerClient;
import eva.replacer.util.BuildHolder;
import eva.replacer.util.BuildHolderDep;

import java.io.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;

import static eva.replacer.config.RePlacerConfig.getNames;
import static eva.replacer.config.RePlacerConfig.buildDefault;

public class JsonConfigHelper {
    private static final File folder = new File("config");
    private static final File buildFolder = new File(folder,"RePlacer Builds");
    private static File rePlacerConfig;
    private static final Hashtable<String, File> builds =  new Hashtable<>();
    public static Gson configGson = new GsonBuilder().setPrettyPrinting().create();

    public static void init() {
        createConfig();
        readFromConfig();
        writeToConfig();
        createBuilds();
        writeSquare();
    }

    private static void createConfig() {
        if (!folder.exists()) folder.mkdir();

        if (folder.isDirectory()) {
            rePlacerConfig = new File(folder, "replacer.json");
            boolean seemsValid;
            if (rePlacerConfig.exists()) {
                try {
                    String templateConfigJson = Files.readString(Path.of(rePlacerConfig.getPath()));
                    seemsValid = templateConfigJson.trim().startsWith("{\n  \"v\": " + RePlacerConfig.ver + ",");
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            } else {
                seemsValid = true;
            }
            if (!rePlacerConfig.exists() || !seemsValid) {
                if (!seemsValid) {
                    RePlacerClient.LOGGER.info("Found invalid config file, creating new config file at './config/replacer.json'.");
                }
                try {
                    rePlacerConfig.createNewFile();
                    String json = configGson.toJson(RePlacerConfig.getInstance());
                    FileWriter writer = new FileWriter(rePlacerConfig);
                    writer.write(json);
                    writer.close();
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }

        }
    }

    private static void readFromConfig() {
        try {
            RePlacerConfig config = configGson.fromJson(new FileReader(rePlacerConfig), RePlacerConfig.class);
            RePlacerConfig.getInstance().updateConfigs(config);
        } catch (FileNotFoundException e) {
            throw new RuntimeException(e);
        }
    }


    static void writeToConfig() {
        try {
            String json = configGson.toJson(RePlacerConfig.getInstance());
            FileWriter writer = new FileWriter(rePlacerConfig, false);
            writer.write(json);
            writer.close();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    static void createBuilds() {
        if (!buildFolder.exists()) buildFolder.mkdir();

        if (buildFolder.isDirectory()) {
            final Hashtable<String, File> tempBuilds =  new Hashtable<>();
            List<String> names =  new ArrayList<>();
            if (getNames() == null || getNames().isEmpty()) return;
            for (String name : getNames()) {
                File file = new  File(buildFolder, name + ".json");
                boolean seemsValid;
                if (file.exists()) {
                    try {
                        String buildJson = Files.readString(Path.of(file.getPath()));
                        seemsValid = buildJson.trim().startsWith("{");
                    } catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
                else {
                    seemsValid = true;
                }
                if (!seemsValid || !file.exists()) {
                    if (!seemsValid) RePlacerClient.LOGGER.info("Found invalid build file, purging.");
                    else RePlacerClient.LOGGER.info("Found invalid build name, purging.");
                    file.delete();
                }
                else {
                    tempBuilds.put(name, file);
                    names.add(name);
                }
            }
            builds.putAll(tempBuilds);
            RePlacerConfig.setNames(new ArrayList<>(builds.keySet()));
        }
    }

    static void deleteBuild(String name) {
        builds.get(name).delete();
        builds.remove(name);
    }

    static void writeBuild(String name, BuildHolder holder) {
        try {
            builds.put(name, new File(buildFolder, name + ".json"));
            String json = configGson.toJson(holder);
            FileWriter writer = new FileWriter(builds.get(name), false);
            writer.write(json);
            writer.close();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static void writeSquare() {
        try {
            configGson.fromJson(new FileReader(builds.get("square")), BuildHolder.class);
            if (getNames().isEmpty()) {
                List<String> names = getNames();
                names.add("square");
                RePlacerConfig.setNames(names);
            }
        } catch (Exception ignored) {
            try {
                RePlacerClient.LOGGER.info("Writing default build at './config/RePlacerBuilds/square.json'.");
                builds.put("square", new File(buildFolder, "square.json"));
                FileWriter writer = new FileWriter(builds.get("square"), false);
                writer.write(configGson.toJson(buildDefault()));
                writer.close();
                List<String> names = getNames();
                names.add("square");
                RePlacerConfig.setNames(names);
            } catch (Exception f) {
                throw new RuntimeException(f);
            }
        }
    }

    static BuildHolder readBuild(String name) {
        try {
            BuildHolder holder = configGson.fromJson(new FileReader(builds.get(name)), BuildHolder.class);
            if (holder.x() == null || holder.y() == null || holder.z() == null) {
                holder = configGson.fromJson(new FileReader(builds.get(name)), BuildHolderDep.class).asBuildHolder();
                writeBuild(name, holder);
            }
            if (holder.x().length != holder.y().length || holder.z().length != holder.x().length) throw new IllegalArgumentException();
            return holder;
        } catch (FileNotFoundException e) {
            throw new RuntimeException(e);
        }
    }
}