package eva.replacer.rendering;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import java.util.*;
import java.util.stream.Collectors;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_238;
import net.minecraft.class_243;

public final class BoundingBoxMerger {
    private static final Long2ObjectMap<class_2350> DIRECTION_LOOKUP = Arrays.stream(class_2350.values())
            .collect(Collectors.toMap(dir -> new class_2338(dir.method_62675()).method_10063(),
                    dir -> dir,
                    (a, b) -> {
                        throw new IllegalStateException("Duplicate direction detected.");
                    },
                    Long2ObjectOpenHashMap::new
            ));
    private final Map<class_243, class_238> positionToBox = new HashMap<>();
    private final Multimap<class_238, class_243> boxToPosition = HashMultimap.create();
    private double xCoordTracker = Double.NEGATIVE_INFINITY;
    private double yCoordTracker = Double.NEGATIVE_INFINITY;
    private class_243 currentCenter = null;
    private class_238 currentBounds = null;

    /**
     * Merges the provided block positions into a collection of bounding boxes
     * by attempting to combine adjacent positions into larger axis-aligned bounding boxes (AABBs).
     *
     * @param positions      The block positions to merge.
     * @param referencePoint The reference point for normalization.
     * @return A collection of merged bounding boxes.
     */
    public static Collection<class_238> merge(Collection<class_2338> positions, class_2338 referencePoint) {
        BoundingBoxMerger boxMerger = new BoundingBoxMerger();

        positions.stream()
                .map(pos -> pos.method_10059(referencePoint))
                .sorted()
                .map(class_238::new)
                .forEachOrdered(box -> {
                    // Reset current bounds if we encounter a new x or y coordinate
                    if (boxMerger.xCoordTracker != box.field_1323 || boxMerger.yCoordTracker != box.field_1322) {
                        boxMerger.currentBounds = null;
                    }

                    boxMerger.xCoordTracker = box.field_1323;
                    boxMerger.yCoordTracker = box.field_1322;

                    class_243 center = box.method_1005();
                    boxMerger.currentCenter = center;

                    // Attempt to combine with the current bounds or adjacent boxes
                    if (boxMerger.currentBounds != null && boxMerger.canCombine(
                            boxMerger.currentBounds,
                            box,
                            center
                    )) {
                        return;
                    }

                    if (boxMerger.tryCombineAdjacent(center, box)) {
                        return;
                    }

                    // Store as a new bounding box
                    boxMerger.currentBounds = box;
                    boxMerger.positionToBox.put(center, box);
                    boxMerger.boxToPosition.put(box, center);
                });

        return boxMerger.boxToPosition.keySet();
    }

    /**
     * Determines if two bounding boxes are aligned along a given direction.
     *
     * @param first     The first bounding box.
     * @param second    The second bounding box.
     * @param direction The direction to check for alignment.
     * @return True if the bounding boxes are aligned, false otherwise.
     */
    private static boolean isAligned(class_238 first, class_238 second, class_2350 direction) {
        return getAxisValue(first, direction) == getAxisValue(
                second,
                direction.method_10153()
        ) && Arrays.stream(class_2350.values())
                .filter(d -> d.method_10166() != direction.method_10166())
                .allMatch(d -> getAxisValue(first, d) == getAxisValue(second, d));
    }

    /**
     * Retrieves the value of a bounding box along a specified direction.
     *
     * @param box       The bounding box.
     * @param direction The direction to evaluate.
     * @return The corresponding value along the given direction.
     */
    private static double getAxisValue(class_238 box, class_2350 direction) {
        return direction.method_10171() == class_2350.class_2352.field_11056 ?
                box.method_990(direction.method_10166()) : box.method_1001(
                direction.method_10166());
    }

    /**
     * Converts a vector into a direction based on its coordinates.
     *
     * @param vector The vector to convert.
     * @return The corresponding direction, or null if none exists.
     */
    private static class_2350 directionFromVector(class_243 vector) {
        return DIRECTION_LOOKUP.get(class_2338.method_10064((int) vector.field_1352, (int) vector.field_1351, (int) vector.field_1350));
    }

    /**
     * Attempts to merge the current bounding box with its neighboring bounding box.
     *
     * @param current  The current bounding box.
     * @param neighbor The neighboring bounding box.
     * @param center   The center position of the neighboring box.
     * @return True if the boxes are neighbors and merged, false otherwise.
     */
    private boolean canCombine(class_238 current, class_238 neighbor, class_243 center) {
        class_2350 direction = directionFromVector(center.method_1020(current.method_1005()));
        return direction != null && isAligned(current, neighbor, direction) && mergeBoxes(
                current,
                neighbor,
                center
        );
    }

    /**
     * Attempts to merge the provided bounding box with any neighboring bounding boxes.
     *
     * @param center The center position of the current bounding box.
     * @param box    The bounding box to merge with.
     * @return True if a neighbor is found and merged, false otherwise.
     */
    private boolean tryCombineAdjacent(class_243 center, class_238 box) {
        for (class_2350 direction : class_2350.values()) {
            class_243 adjacentCenter = center.method_1019(class_243.method_24954(direction.method_62675()));
            class_238 adjacentBox = positionToBox.get(adjacentCenter);

            if (adjacentBox != null && isAligned(box, adjacentBox, direction)) {
                return mergeBoxes(adjacentBox, box, center);
            }
        }
        return false;
    }

    /**
     * Merges two bounding boxes and updates the necessary mappings.
     *
     * @param source The initial bounding box.
     * @param target The bounding box to merge with.
     * @param center The center position of the target bounding box.
     * @return True once the merge is successful.
     */
    private boolean mergeBoxes(class_238 source, class_238 target, class_243 center) {
        class_238 expanded = source.method_991(target);

        Set<class_243> mergedPositions = new HashSet<>(boxToPosition.removeAll(source));
        mergedPositions.forEach(v -> positionToBox.put(v, expanded));

        boxToPosition.putAll(expanded, mergedPositions);
        positionToBox.put(center, expanded);
        boxToPosition.put(expanded, center);

        currentBounds = expanded;
        return true;
    }
}