package eva.replacer.config;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import eva.replacer.RePlacerClient;
import eva.replacer.util.BuildHolder;
import eva.replacer.util.BuildHolderDep;

import java.io.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;

import static eva.replacer.config.RePlacerConfig.*;

public class JsonConfigHelper {
    private static final File folder = new File("config");
    private static final File buildFolder = new File(folder,"RePlacer Builds");
    private static List<File> files;
    private static File rePlacerConfig;
    public static final Gson configGson = new GsonBuilder().setPrettyPrinting().create();
    private static final File square = new File(buildFolder, "square.json");
    private static final Map<String, BuildHolder> buildDict = new HashMap<>();

    public static void init() {
        createConfig();
        readFromConfig();
        writeToConfig();
        createBuilds();
        writeSquare();
    }

    private static void createConfig() {
        if (!folder.exists()) folder.mkdir();

        if (folder.isDirectory()) {
            rePlacerConfig = new File(folder, "replacer.json");
            boolean seemsValid;
            if (rePlacerConfig.exists()) {
                try {
                    String templateConfigJson = Files.readString(Path.of(rePlacerConfig.getPath()));
                    seemsValid = templateConfigJson.trim().startsWith("{\n  \"v\": " + RePlacerConfig.ver + ",");
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            } else {
                seemsValid = true;
            }
            if (!rePlacerConfig.exists() || !seemsValid) {
                if (!seemsValid) {
                    RePlacerClient.LOGGER.info("Found invalid config file, creating new config file at './config/replacer.json'.");
                }
                try {
                    rePlacerConfig.createNewFile();
                    String json = configGson.toJson(RePlacerConfig.getInstance());
                    FileWriter writer = new FileWriter(rePlacerConfig);
                    writer.write(json);
                    writer.close();
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }

        }
    }

    private static void readFromConfig() {
        try {
            RePlacerConfig config = configGson.fromJson(new FileReader(rePlacerConfig), RePlacerConfig.class);
            RePlacerConfig.getInstance().updateConfigs(config);
        } catch (FileNotFoundException e) {
            throw new RuntimeException(e);
        }
    }


    static void writeToConfig() {
        try {
            String json = configGson.toJson(RePlacerConfig.getInstance());
            FileWriter writer = new FileWriter(rePlacerConfig, false);
            writer.write(json);
            writer.close();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    static void createBuilds() {
        if (!buildFolder.exists()) buildFolder.mkdir();

        if (buildFolder.isDirectory()) {
            List<String> names = new ArrayList<>();
            files = new ArrayList<>(List.of(Objects.requireNonNull(new File("config/RePlacer Builds").listFiles())));
            List<File> markedForDeletion = new ArrayList<>();
            if (files.isEmpty()) return;
            try {
                for (File file : files) {
                    if (file.isFile()) {
                        if (file.getName().endsWith(".json")) {
                            boolean seemsValid;
                            BuildHolder build = null;
                            if (file.exists()) {
                                try {
                                    build = pullBuild(file.getName().replace(".json", ""));
                                    seemsValid = true;
                                } catch (FileNotFoundException e) {
                                    seemsValid = false;
                                }
                            } else
                                seemsValid = true;
                            if (!seemsValid) {
                                RePlacerClient.LOGGER.info("Found invalid build file, purging.");
                                markedForDeletion.add(file);
                                file.delete();
                            } else {
                                String name = file.getName().replace(".json", "");
                                if (!hasTail(name) || isValidTail(tail(name))) {
                                } else {
                                    RePlacerClient.LOGGER.info("Found invalid build file tail (" + name + "), removing. Resulting file name: " + removeTail(name));
                                    file.renameTo(new File(buildFolder, removeTail(name)));
                                }
                                names.add(file.getName().replace(".json", ""));
                                buildDict.put(file.getName().replace(".json", ""), build);
                            }
                        }
                    }
                }
                RePlacerConfig.setNames(names);
            } catch (NullPointerException ignored) {
            }
            files.removeAll(markedForDeletion);
            if (getTaillessNames().contains(getDefBuild()))
                selection = getTaillessNames().indexOf(getDefBuild());
            else {
                selection = 0;
                setDefBuild("");
            }
        }
    }

    static void deleteBuild(String name) {
        File forDeletion = new File(buildFolder, name + ".json");
        files.remove(forDeletion);
        forDeletion.delete();
    }

    static void writeBuild(String name, BuildHolder holder) {
        try {
            String json = configGson.toJson(holder);
            File forCreation = new File(buildFolder, name + ".json");
            FileWriter writer = new FileWriter(forCreation, false);
            writer.write(json);
            writer.close();
            files.add(forCreation);
            buildDict.put(name, holder);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static void writeSquare() {
        if (!getTaillessNames().contains("square")) {
            try {
                configGson.fromJson(new FileReader(square), BuildHolder.class);
                if (getNames().isEmpty()) {
                    List<String> names = getNames();
                    names.add("square");
                    setNames(names);
                }
            } catch (Exception ignored) {
                try {
                    RePlacerClient.LOGGER.info("Writing default build at './config/RePlacerBuilds/square.json'.");
                    FileWriter writer = new FileWriter(square, false);
                    writer.write(configGson.toJson(buildDefault()));
                    writer.close();
                    List<String> names = getNames();
                    names.add("square");
                    setNames(names);
                } catch (Exception f) {
                    throw new RuntimeException(f);
                }
            }
            if (!files.contains(square)) {
                files.add(square);
                buildDict.put(square.getName(), buildDefault());
            }
        }
    }

    static BuildHolder readBuild(String name) {
        if (!buildDict.containsKey(name)) {
            RePlacerClient.LOGGER.info("Invalid buildname found! Purging!");
            List<String> temp = getNames();
            temp.remove(name);
            setNames(temp);
            files.remove(new File(buildFolder, name + ".json"));
            selection = 0;
            return null;
        }
        return buildDict.get(name);
//        try {
//            BuildHolder holder = pullBuild(name);
//            if (holder.x().length != holder.y().length || holder.z().length != holder.x().length) throw new IllegalArgumentException();
//            return holder;
//        } catch (FileNotFoundException e) {
//            RePlacerClient.LOGGER.info("Invalid buildname found! Purging!");
//            List<String> temp = getNames();
//            temp.remove(name);
//            setNames(temp);
//            files.remove(new File(buildFolder, name + ".json"));
//            selection = 0;
//            return null;
//        }
    }

    static BuildHolder pullBuild(String name) throws FileNotFoundException {
        File toRead = new File(buildFolder, name + ".json");
        BuildHolder holder = configGson.fromJson(new FileReader(toRead), BuildHolder.class);
        if (holder.x() == null || holder.y() == null || holder.z() == null) {
            holder = configGson.fromJson(new FileReader(toRead), BuildHolderDep.class).asBuildHolder();
            writeBuild(name, holder);
        }
        return holder;
    }

    static boolean renameBuild(String oldName, String newName) {
        File oldFile = new File(buildFolder, oldName + ".json");
        if (!oldFile.exists()) return false;
        File newFile = new File(buildFolder, newName + ".json");
        if (newFile.exists()) return false;
        if (!oldFile.renameTo(newFile)) return false;
        List<String> names = getNames();
        names.remove(oldName);
        names.add(newName);
        setNames(names);
        files.remove(oldFile);
        files.add(newFile);
        return true;
    }

    static boolean reAddBuild(String name) {
        checkUpdates();
        final Map<String, String> mappedNames = new HashMap<>();
        buildDict.keySet().forEach(build -> mappedNames.put(removeTail(build), build));
        if (mappedNames.containsKey(removeTail(name)) || mappedNames.containsValue(name)) {
            boolean bl = buildDict.containsKey(name);
            writeBuild(name, buildDict.get(bl ? name : mappedNames.get(removeTail(name))));
            return true;
        }
        return false;
    }

    static boolean hasCachedBuild(String buildName) {
        if (buildDict.containsKey(removeTail(buildName))) return true;
        final Set<String> mappedNames = new HashSet<>();
        buildDict.keySet().forEach(build -> mappedNames.add(removeTail(build)));
        return mappedNames.contains(removeTail(buildName));
    }

    static String getCachedVer(String name) {
        if (buildDict.containsKey(removeTail(name))) return removeTail(name);
        final Map<String, String> mappedNames = new HashMap<>();
        buildDict.keySet().forEach(build -> mappedNames.put(removeTail(build), build));
        return mappedNames.get(removeTail(buildName));
    }

    public static void checkUpdates() {
        try {
            try {
                String name = getNames().get(selection);
                List<File> newFiles = List.of(Objects.requireNonNull(new File("config/RePlacer Builds").listFiles()));
                if (files.equals(newFiles)) return;
                createBuilds();
                if (getNames().contains(name)) selection = getNames().indexOf(name);
            } catch (IndexOutOfBoundsException e) {
                writeSquare();
            }
        } catch (NullPointerException ignored) {}
    }
}