package eva.replacer.config;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import eva.replacer.RePlacerClient;
import eva.replacer.util.BuildHolder;
import eva.replacer.util.BuildHolderDep;

import java.io.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

import static eva.replacer.config.RePlacerConfig.*;

public class JsonConfigHelper {
    private static final File folder = new File("config");
    private static final File buildFolder = new File(folder,"RePlacer Builds");
    private static File rePlacerConfig;
    public static final Gson configGson = new GsonBuilder().setPrettyPrinting().create();
    private static final File square = new File(buildFolder, "square.json");

    public static void init() {
        createConfig();
        readFromConfig();
        writeToConfig();
        createBuilds();
        writeSquare();
    }

    private static void createConfig() {
        if (!folder.exists()) folder.mkdir();

        if (folder.isDirectory()) {
            rePlacerConfig = new File(folder, "replacer.json");
            boolean seemsValid;
            if (rePlacerConfig.exists()) {
                try {
                    String templateConfigJson = Files.readString(Path.of(rePlacerConfig.getPath()));
                    seemsValid = templateConfigJson.trim().startsWith("{\n  \"v\": " + RePlacerConfig.ver + ",");
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            } else {
                seemsValid = true;
            }
            if (!rePlacerConfig.exists() || !seemsValid) {
                if (!seemsValid) {
                    RePlacerClient.LOGGER.info("Found invalid config file, creating new config file at './config/replacer.json'.");
                }
                try {
                    rePlacerConfig.createNewFile();
                    String json = configGson.toJson(RePlacerConfig.getInstance());
                    FileWriter writer = new FileWriter(rePlacerConfig);
                    writer.write(json);
                    writer.close();
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }

        }
    }

    private static void readFromConfig() {
        try {
            RePlacerConfig config = configGson.fromJson(new FileReader(rePlacerConfig), RePlacerConfig.class);
            RePlacerConfig.getInstance().updateConfigs(config);
        } catch (FileNotFoundException e) {
            throw new RuntimeException(e);
        }
    }


    static void writeToConfig() {
        try {
            String json = configGson.toJson(RePlacerConfig.getInstance());
            FileWriter writer = new FileWriter(rePlacerConfig, false);
            writer.write(json);
            writer.close();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    static void createBuilds() {
        if (!buildFolder.exists()) buildFolder.mkdir();

        if (buildFolder.isDirectory()) {
            List<String> names = new ArrayList<>();
            File[] files = new File("config/RePlacer Builds").listFiles();
            if (files == null) return;
            try {
                for (File file : files) {
                    if (file.isFile()) {
                        if (file.getName().endsWith(".json")) {
                            boolean seemsValid;
                            if (file.exists()) {
                                try {
                                    pullBuild(file.getName().replace(".json", ""));
                                    seemsValid = true;
                                } catch (FileNotFoundException e) {
                                    seemsValid = false;
                                }
                            } else
                                seemsValid = true;
                            if (!seemsValid) {
                                RePlacerClient.LOGGER.info("Found invalid build file, purging.");
                                file.delete();
                            } else
                                names.add(file.getName().replace(".json", ""));
                        }
                    }
                }
                RePlacerConfig.setNames(names);
            } catch (NullPointerException ignored) {
            }
        }
    }

    static void deleteBuild(String name) {
        new File(buildFolder, name + ".json").delete();
    }

    static void writeBuild(String name, BuildHolder holder) {
        try {
            String json = configGson.toJson(holder);
            FileWriter writer = new FileWriter(new File(buildFolder, name + ".json"), false);
            writer.write(json);
            writer.close();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static void writeSquare() {
        try {
            configGson.fromJson(new FileReader(square), BuildHolder.class);
            if (getNames().isEmpty()) {
                getNames().add("square");
            }
        } catch (Exception ignored) {
            try {
                RePlacerClient.LOGGER.info("Writing default build at './config/RePlacerBuilds/square.json'.");
                FileWriter writer = new FileWriter(square, false);
                writer.write(configGson.toJson(buildDefault()));
                writer.close();
                getNames().add("square");
            } catch (Exception f) {
                throw new RuntimeException(f);
            }
        }
    }

    static BuildHolder readBuild(String name) {
        try {
            BuildHolder holder = pullBuild(name);
            if (holder.x().length != holder.y().length || holder.z().length != holder.x().length) throw new IllegalArgumentException();
            return holder;
        } catch (FileNotFoundException e) {
            getNames().remove(name);
            RePlacerClient.LOGGER.info("Invalid buildname found! Purging!");
            selection = 0;
            return null;
        }
    }

    static BuildHolder pullBuild(String name) throws FileNotFoundException {
        File toRead = new File(buildFolder, name + ".json");
        BuildHolder holder = configGson.fromJson(new FileReader(toRead), BuildHolder.class);
        if (holder.x() == null || holder.y() == null || holder.z() == null) {
            holder = configGson.fromJson(new FileReader(toRead), BuildHolderDep.class).asBuildHolder();
            writeBuild(name, holder);
        }
        return holder;
    }

    static boolean renameBuild(String oldName, String newName) {
        File oldFile = new File(buildFolder, oldName + ".json");
        if (!oldFile.exists()) return false;
        File newFile = new File(buildFolder, newName + ".json");
        if (newFile.exists()) return false;
        if (!oldFile.renameTo(newFile)) return false;
        getNames().remove(oldName);
        getNames().add(newName);
        return true;
    }
}