/*
 * Decompiled with CFR 0.152.
 */
package me.wethink.weGuardian.database;

import java.io.File;
import java.io.IOException;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import me.wethink.weGuardian.WeGuardian;
import me.wethink.weGuardian.database.DatabaseManager;
import me.wethink.weGuardian.models.BanwaveEntry;
import me.wethink.weGuardian.models.PlayerData;
import me.wethink.weGuardian.models.Punishment;
import me.wethink.weGuardian.models.PunishmentType;
import org.bukkit.configuration.ConfigurationSection;
import org.bukkit.configuration.file.YamlConfiguration;

public class YamlDatabaseManager
implements DatabaseManager {
    private final WeGuardian plugin;
    private final File dataDir;
    private final File globalFile;
    private final YamlConfiguration globalConfig;
    private final AtomicInteger nextId;
    private final Map<Integer, Punishment> punishmentCache;
    private final Map<UUID, PlayerData> playerDataCache;
    private final Map<Integer, BanwaveEntry> banwaveCache;
    private final Map<UUID, Map<String, Long>> playerConnectionsCache;
    private final DateTimeFormatter formatter = DateTimeFormatter.ISO_LOCAL_DATE_TIME;

    public YamlDatabaseManager(WeGuardian plugin) {
        this.plugin = plugin;
        this.dataDir = new File(plugin.getDataFolder(), "data");
        this.globalFile = new File(plugin.getDataFolder(), "global.yml");
        this.globalConfig = YamlConfiguration.loadConfiguration((File)this.globalFile);
        this.punishmentCache = new ConcurrentHashMap<Integer, Punishment>();
        this.playerDataCache = new ConcurrentHashMap<UUID, PlayerData>();
        this.banwaveCache = new ConcurrentHashMap<Integer, BanwaveEntry>();
        this.playerConnectionsCache = new ConcurrentHashMap<UUID, Map<String, Long>>();
        this.nextId = new AtomicInteger(1);
        if (!this.dataDir.exists()) {
            this.dataDir.mkdirs();
        }
        this.loadData();
        this.setupAutoBackup();
    }

    private void loadData() {
        YamlConfiguration ipConfig;
        ConfigurationSection punishmentsSection;
        File file;
        File[] playerFiles;
        if (!this.globalFile.exists()) {
            try {
                this.globalFile.createNewFile();
                this.globalConfig.set("next_id", (Object)1);
                this.globalConfig.set("banwave", new HashMap());
                this.saveGlobalConfig();
            }
            catch (IOException e) {
                this.plugin.getLogger().severe("Failed to create global YAML file: " + e.getMessage());
            }
        }
        this.nextId.set(this.globalConfig.getInt("next_id", 1));
        ConfigurationSection banwaveSection = this.globalConfig.getConfigurationSection("banwave");
        if (banwaveSection != null) {
            for (String string : banwaveSection.getKeys(false)) {
                try {
                    BanwaveEntry entry;
                    int id = Integer.parseInt(string);
                    ConfigurationSection entrySection = banwaveSection.getConfigurationSection(string);
                    if (entrySection == null || (entry = this.deserializeBanwaveEntry(entrySection)) == null) continue;
                    this.banwaveCache.put(id, entry);
                }
                catch (NumberFormatException e) {
                    this.plugin.getLogger().warning("Invalid banwave ID in global YAML: " + string);
                }
            }
        }
        if ((playerFiles = this.dataDir.listFiles((dir, name) -> name.endsWith(".yml"))) != null) {
            for (File playerFile : playerFiles) {
                try {
                    ConfigurationSection punishmentsSection2;
                    ConfigurationSection connectionsSection;
                    String fileName = playerFile.getName();
                    String uuidString = fileName.substring(0, fileName.length() - 4);
                    UUID playerUuid = UUID.fromString(uuidString);
                    YamlConfiguration playerConfig = YamlConfiguration.loadConfiguration((File)playerFile);
                    PlayerData playerData = this.deserializePlayerData(playerConfig, playerUuid);
                    if (playerData != null) {
                        this.playerDataCache.put(playerUuid, playerData);
                    }
                    if ((connectionsSection = playerConfig.getConfigurationSection("connections")) != null) {
                        ConcurrentHashMap<String, Long> ipMap = new ConcurrentHashMap<String, Long>();
                        for (String ip : connectionsSection.getKeys(false)) {
                            long lastSeenTs = connectionsSection.getLong(ip, 0L);
                            ipMap.put(ip, lastSeenTs);
                        }
                        this.playerConnectionsCache.put(playerUuid, ipMap);
                    }
                    if ((punishmentsSection2 = playerConfig.getConfigurationSection("punishments")) == null) continue;
                    for (String key : punishmentsSection2.getKeys(false)) {
                        try {
                            int id = Integer.parseInt(key);
                            Punishment punishment = this.deserializePunishment(punishmentsSection2.getConfigurationSection(key));
                            if (punishment == null) continue;
                            this.punishmentCache.put(id, punishment);
                        }
                        catch (NumberFormatException e) {
                            this.plugin.getLogger().warning("Invalid punishment ID in player file " + fileName + ": " + key);
                        }
                    }
                }
                catch (IllegalArgumentException e) {
                    this.plugin.getLogger().warning("Invalid UUID in filename: " + playerFile.getName());
                }
            }
        }
        if ((file = new File(this.plugin.getDataFolder(), "ip_punishments.yml")).exists() && (punishmentsSection = (ipConfig = YamlConfiguration.loadConfiguration((File)file)).getConfigurationSection("punishments")) != null) {
            for (String key : punishmentsSection.getKeys(false)) {
                try {
                    int id = Integer.parseInt(key);
                    Punishment punishment = this.deserializePunishment(punishmentsSection.getConfigurationSection(key));
                    if (punishment == null) continue;
                    this.punishmentCache.put(id, punishment);
                }
                catch (NumberFormatException e) {
                    this.plugin.getLogger().warning("Invalid punishment ID in IP punishments file: " + key);
                }
            }
        }
    }

    private File getPlayerFile(UUID uuid) {
        return new File(this.dataDir, uuid.toString() + ".yml");
    }

    private YamlConfiguration getPlayerConfig(UUID uuid) {
        File playerFile = this.getPlayerFile(uuid);
        return YamlConfiguration.loadConfiguration((File)playerFile);
    }

    private void savePlayerConfig(UUID uuid, YamlConfiguration config) {
        try {
            File playerFile = this.getPlayerFile(uuid);
            config.save(playerFile);
        }
        catch (IOException e) {
            this.plugin.getLogger().severe("Failed to save player YAML file for " + String.valueOf(uuid) + ": " + e.getMessage());
        }
    }

    private void saveGlobalConfig() {
        try {
            this.globalConfig.save(this.globalFile);
        }
        catch (IOException e) {
            this.plugin.getLogger().severe("Failed to save global YAML file: " + e.getMessage());
        }
    }

    private void setupAutoBackup() {
        if (this.plugin.getConfig().getBoolean("database.yaml.auto_backup", true)) {
            int interval = this.plugin.getConfig().getInt("database.yaml.backup_interval", 300) * 20;
            this.plugin.getFoliaLib().getScheduler().runTimerAsync(this::createBackup, (long)interval, (long)interval);
        }
    }

    private void createBackup() {
        try {
            File ipPunishmentsFile;
            File backupDir = new File(this.plugin.getDataFolder(), "backups");
            if (!backupDir.exists()) {
                backupDir.mkdirs();
            }
            String timestamp = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd_HH-mm-ss"));
            File backupDataDir = new File(backupDir, "data_" + timestamp);
            backupDataDir.mkdirs();
            YamlConfiguration backupGlobal = new YamlConfiguration();
            if (this.globalConfig.getDefaults() != null) {
                backupGlobal.setDefaults(this.globalConfig.getDefaults());
            }
            for (File[] key : this.globalConfig.getKeys(true)) {
                backupGlobal.set((String)key, this.globalConfig.get((String)key));
            }
            backupGlobal.save(new File(backupDataDir, "global.yml"));
            File[] playerFiles = this.dataDir.listFiles((dir, name) -> name.endsWith(".yml"));
            if (playerFiles != null) {
                for (File playerFile : playerFiles) {
                    YamlConfiguration playerConfig = YamlConfiguration.loadConfiguration((File)playerFile);
                    YamlConfiguration backupPlayer = new YamlConfiguration();
                    if (playerConfig.getDefaults() != null) {
                        backupPlayer.setDefaults(playerConfig.getDefaults());
                    }
                    for (String key : playerConfig.getKeys(true)) {
                        backupPlayer.set(key, playerConfig.get(key));
                    }
                    backupPlayer.save(new File(backupDataDir, playerFile.getName()));
                }
            }
            if ((ipPunishmentsFile = new File(this.plugin.getDataFolder(), "ip_punishments.yml")).exists()) {
                YamlConfiguration ipConfig = YamlConfiguration.loadConfiguration((File)ipPunishmentsFile);
                YamlConfiguration backupIp = new YamlConfiguration();
                if (ipConfig.getDefaults() != null) {
                    backupIp.setDefaults(ipConfig.getDefaults());
                }
                for (String key : ipConfig.getKeys(true)) {
                    backupIp.set(key, ipConfig.get(key));
                }
                backupIp.save(new File(backupDataDir, "ip_punishments.yml"));
            }
            this.cleanOldBackups(backupDir);
        }
        catch (IOException e) {
            this.plugin.getLogger().warning("Failed to create backup: " + e.getMessage());
        }
    }

    private void cleanOldBackups(File backupDir) {
        File[] backups = backupDir.listFiles((dir, name) -> name.startsWith("data_"));
        if (backups != null && backups.length > this.plugin.getConfig().getInt("database.yaml.max_backups", 5)) {
            Arrays.sort(backups, Comparator.comparingLong(File::lastModified));
            for (int i = 0; i < backups.length - this.plugin.getConfig().getInt("database.yaml.max_backups", 5); ++i) {
                this.deleteDirectory(backups[i]);
            }
        }
    }

    private void deleteDirectory(File directory) {
        File[] files = directory.listFiles();
        if (files != null) {
            for (File file : files) {
                if (file.isDirectory()) {
                    this.deleteDirectory(file);
                    continue;
                }
                file.delete();
            }
        }
        directory.delete();
    }

    @Override
    public CompletableFuture<Integer> addPunishment(Punishment punishment) {
        return CompletableFuture.supplyAsync(() -> {
            int id = this.nextId.getAndIncrement();
            punishment.setId(id);
            this.punishmentCache.put(id, punishment);
            this.plugin.debug("Added punishment to cache with ID: %d, type: %s, target: %s", new Object[]{id, punishment.getType(), punishment.getTargetName()});
            UUID targetUuid = punishment.getTargetUuid();
            if (targetUuid != null) {
                this.plugin.debug("Saving punishment to player file: %s", targetUuid);
                YamlConfiguration playerConfig = this.getPlayerConfig(targetUuid);
                ConfigurationSection punishmentsSection = playerConfig.getConfigurationSection("punishments");
                if (punishmentsSection == null) {
                    punishmentsSection = playerConfig.createSection("punishments");
                    this.plugin.debug("Created punishments section for player: %s", targetUuid);
                }
                ConfigurationSection punishmentSection = punishmentsSection.createSection(String.valueOf(id));
                this.serializePunishment(punishmentSection, punishment);
                this.plugin.debug("Serialized punishment to player config: %s", targetUuid);
                this.savePlayerConfig(targetUuid, playerConfig);
            } else {
                this.plugin.debug("Saving IP punishment to IP punishments file");
                this.saveIPPunishment(punishment);
            }
            this.globalConfig.set("next_id", (Object)this.nextId.get());
            this.saveGlobalConfig();
            this.plugin.debug("Punishment saved successfully with ID: %d", id);
            return id;
        });
    }

    private void saveIPPunishment(Punishment punishment) {
        File ipPunishmentsFile = new File(this.plugin.getDataFolder(), "ip_punishments.yml");
        YamlConfiguration ipConfig = YamlConfiguration.loadConfiguration((File)ipPunishmentsFile);
        ConfigurationSection punishmentsSection = ipConfig.getConfigurationSection("punishments");
        if (punishmentsSection == null) {
            punishmentsSection = ipConfig.createSection("punishments");
        }
        ConfigurationSection punishmentSection = punishmentsSection.createSection(String.valueOf(punishment.getId()));
        this.serializePunishment(punishmentSection, punishment);
        try {
            ipConfig.save(ipPunishmentsFile);
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Failed to save IP punishment: " + e.getMessage());
        }
    }

    @Override
    public CompletableFuture<Void> savePlayerData(PlayerData playerData) {
        return CompletableFuture.runAsync(() -> {
            this.playerDataCache.put(playerData.getUuid(), playerData);
            YamlConfiguration playerConfig = this.getPlayerConfig(playerData.getUuid());
            this.serializePlayerData(playerConfig, playerData);
            this.serializePlayerConnections(playerConfig, playerData.getUuid());
            this.savePlayerConfig(playerData.getUuid(), playerConfig);
        });
    }

    @Override
    public CompletableFuture<Void> removePunishment(int id, UUID removedBy, String removedByName, String reason) {
        return CompletableFuture.runAsync(() -> {
            Punishment punishment = this.punishmentCache.get(id);
            if (punishment != null) {
                punishment.setActive(false);
                punishment.setRemovedAt(LocalDateTime.now());
                punishment.setRemovedBy(removedByName);
                punishment.setRemovalReason(reason);
                punishment.setRemovedByUuid(removedBy);
                UUID targetUuid = punishment.getTargetUuid();
                if (targetUuid != null) {
                    YamlConfiguration playerConfig = this.getPlayerConfig(targetUuid);
                    ConfigurationSection punishmentSection = playerConfig.getConfigurationSection("punishments." + id);
                    if (punishmentSection != null) {
                        this.serializePunishment(punishmentSection, punishment);
                        this.savePlayerConfig(targetUuid, playerConfig);
                    }
                } else {
                    this.saveIPPunishment(punishment);
                }
            }
        });
    }

    private PlayerData deserializePlayerData(YamlConfiguration config, UUID uuid) {
        try {
            PlayerData playerData = new PlayerData();
            playerData.setUuid(uuid);
            playerData.setPlayerName(config.getString("player_name", "Unknown"));
            Object firstJoinRaw = config.get("first_join");
            if (firstJoinRaw instanceof String) {
                try {
                    LocalDateTime firstJoin = LocalDateTime.parse((String)firstJoinRaw, this.formatter);
                    playerData.setFirstJoin(firstJoin.atZone(ZoneId.systemDefault()).toEpochSecond() * 1000L);
                }
                catch (Exception e) {
                    playerData.setFirstJoin(System.currentTimeMillis());
                    this.plugin.getLogger().warning("Failed to parse first_join for " + String.valueOf(uuid) + ", using current time");
                }
            } else if (firstJoinRaw instanceof Number) {
                playerData.setFirstJoin(((Number)firstJoinRaw).longValue());
            } else {
                playerData.setFirstJoin(System.currentTimeMillis());
                this.plugin.getLogger().info("Migrated corrupted first_join data for " + String.valueOf(uuid));
                config.set("first_join", (Object)System.currentTimeMillis());
            }
            Object lastJoinRaw = config.get("last_join");
            if (lastJoinRaw instanceof String) {
                try {
                    LocalDateTime lastJoin = LocalDateTime.parse((String)lastJoinRaw, this.formatter);
                    playerData.setLastJoin(lastJoin.atZone(ZoneId.systemDefault()).toEpochSecond() * 1000L);
                }
                catch (Exception e) {
                    playerData.setLastJoin(System.currentTimeMillis());
                    this.plugin.getLogger().warning("Failed to parse last_join for " + String.valueOf(uuid) + ", using current time");
                }
            } else if (lastJoinRaw instanceof Number) {
                playerData.setLastJoin(((Number)lastJoinRaw).longValue());
            } else {
                playerData.setLastJoin(System.currentTimeMillis());
                this.plugin.getLogger().info("Migrated corrupted last_join data for " + String.valueOf(uuid));
                config.set("last_join", (Object)System.currentTimeMillis());
            }
            playerData.setLastIP(config.getString("last_ip", ""));
            return playerData;
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Failed to deserialize player data for " + String.valueOf(uuid) + ": " + e.getMessage());
            return null;
        }
    }

    private void serializePlayerData(YamlConfiguration config, PlayerData playerData) {
        config.set("player_name", (Object)playerData.getPlayerName());
        config.set("first_join", (Object)playerData.getFirstJoinTimestamp());
        config.set("last_join", (Object)playerData.getLastJoinTimestamp());
        config.set("last_ip", (Object)playerData.getLastIP());
    }

    private void serializePlayerConnections(YamlConfiguration config, UUID uuid) {
        Map<String, Long> ipMap = this.playerConnectionsCache.get(uuid);
        if (ipMap == null) {
            return;
        }
        ConfigurationSection section = config.getConfigurationSection("connections");
        if (section == null) {
            section = config.createSection("connections");
        }
        for (Map.Entry<String, Long> entry : ipMap.entrySet()) {
            section.set(entry.getKey(), (Object)entry.getValue());
        }
    }

    private BanwaveEntry deserializeBanwaveEntry(ConfigurationSection section) {
        try {
            BanwaveEntry entry = new BanwaveEntry();
            entry.setId(section.getInt("id"));
            entry.setPlayerName(section.getString("player_name"));
            entry.setReason(section.getString("reason"));
            entry.setStaffName(section.getString("staff_name"));
            entry.setExecuted(section.getBoolean("executed", false));
            return entry;
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Failed to deserialize banwave entry: " + e.getMessage());
            return null;
        }
    }

    private void serializePunishment(ConfigurationSection section, Punishment punishment) {
        section.set("id", (Object)punishment.getId());
        section.set("type", (Object)punishment.getType().name());
        section.set("target_uuid", punishment.getTargetUuid() != null ? punishment.getTargetUuid().toString() : null);
        section.set("target_name", (Object)punishment.getTargetName());
        section.set("target_ip", (Object)punishment.getTargetIP());
        section.set("staff_uuid", (Object)punishment.getStaffUuid().toString());
        section.set("staff_name", (Object)punishment.getStaffName());
        section.set("reason", (Object)punishment.getReason());
        section.set("server_name", (Object)punishment.getServerName());
        section.set("created_at", (Object)punishment.getCreatedAt().format(this.formatter));
        section.set("active", (Object)punishment.isActive());
        if (punishment.getExpiresAt() != null) {
            section.set("expires_at", (Object)punishment.getExpiresAt().format(this.formatter));
        }
        if (punishment.getRemovedAt() != null) {
            section.set("removed_at", (Object)punishment.getRemovedAt().format(this.formatter));
            section.set("removed_by", (Object)punishment.getRemovedBy());
            section.set("removal_reason", (Object)punishment.getRemovalReason());
        }
    }

    private Punishment deserializePunishment(ConfigurationSection section) {
        if (section == null) {
            return null;
        }
        try {
            String removedAt;
            Punishment punishment = new Punishment();
            punishment.setId(section.getInt("id"));
            punishment.setType(PunishmentType.valueOf(section.getString("type")));
            punishment.setTargetUuid(UUID.fromString(section.getString("target_uuid")));
            punishment.setTargetName(section.getString("target_name"));
            punishment.setTargetIP(section.getString("target_ip"));
            punishment.setStaffUuid(UUID.fromString(section.getString("staff_uuid")));
            punishment.setStaffName(section.getString("staff_name"));
            punishment.setReason(section.getString("reason"));
            punishment.setServerName(section.getString("server_name"));
            punishment.setCreatedAt(LocalDateTime.parse(section.getString("created_at"), this.formatter));
            punishment.setActive(section.getBoolean("active", true));
            String expiresAt = section.getString("expires_at");
            if (expiresAt != null) {
                punishment.setExpiresAt(LocalDateTime.parse(expiresAt, this.formatter));
            }
            if ((removedAt = section.getString("removed_at")) != null) {
                punishment.setRemovedAt(LocalDateTime.parse(removedAt, this.formatter));
                punishment.setRemovedBy(section.getString("removed_by"));
                punishment.setRemovalReason(section.getString("removal_reason"));
            }
            return punishment;
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Failed to deserialize punishment: " + e.getMessage());
            return null;
        }
    }

    @Override
    public CompletableFuture<Integer> addBanwaveEntry(BanwaveEntry entry) {
        return CompletableFuture.supplyAsync(() -> {
            int id = this.nextId.getAndIncrement();
            entry.setId(id);
            this.banwaveCache.put(id, entry);
            ConfigurationSection banwaveSection = this.globalConfig.getConfigurationSection("banwave");
            if (banwaveSection == null) {
                banwaveSection = this.globalConfig.createSection("banwave");
            }
            ConfigurationSection entrySection = banwaveSection.createSection(String.valueOf(id));
            this.serializeBanwaveEntry(entrySection, entry);
            this.saveGlobalConfig();
            return id;
        });
    }

    private void serializeBanwaveEntry(ConfigurationSection section, BanwaveEntry entry) {
        section.set("id", (Object)entry.getId());
        section.set("player_name", (Object)entry.getPlayerName());
        section.set("reason", (Object)entry.getReason());
        section.set("staff_name", (Object)entry.getStaffName());
        section.set("executed", (Object)entry.isExecuted());
    }

    @Override
    public CompletableFuture<Void> initialize() {
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Boolean> connect() {
        return CompletableFuture.completedFuture(true);
    }

    @Override
    public CompletableFuture<Void> disconnect() {
        this.saveGlobalConfig();
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> createTables() {
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public void close() {
        this.saveGlobalConfig();
    }

    @Override
    public CompletableFuture<PlayerData> getPlayerData(UUID uuid) {
        return CompletableFuture.completedFuture(this.playerDataCache.get(uuid));
    }

    @Override
    public CompletableFuture<PlayerData> getPlayerData(String name) {
        return CompletableFuture.supplyAsync(() -> this.playerDataCache.values().stream().filter(p -> name.equalsIgnoreCase(p.getPlayerName())).findFirst().orElse(null));
    }

    @Override
    public CompletableFuture<Void> updatePlayerName(UUID uuid, String newName) {
        return CompletableFuture.runAsync(() -> {
            PlayerData data = this.playerDataCache.get(uuid);
            if (data != null) {
                data.setPlayerName(newName);
                this.savePlayerConfig(uuid, this.getPlayerConfig(uuid));
            }
        });
    }

    @Override
    public CompletableFuture<Punishment> getPunishment(int id) {
        return CompletableFuture.supplyAsync(() -> {
            YamlConfiguration ipConfig;
            ConfigurationSection ipPunishmentsSection;
            File ipPunishmentsFile;
            Punishment cachedPunishment = this.punishmentCache.get(id);
            if (cachedPunishment != null) {
                return cachedPunishment;
            }
            File playersFolder = new File(this.plugin.getDataFolder(), "players");
            if (playersFolder.exists() && playersFolder.isDirectory()) {
                for (File playerFile : playersFolder.listFiles()) {
                    YamlConfiguration playerConfig;
                    ConfigurationSection punishmentsSection;
                    if (!playerFile.isFile() || !playerFile.getName().endsWith(".yml") || (punishmentsSection = (playerConfig = YamlConfiguration.loadConfiguration((File)playerFile)).getConfigurationSection("punishments")) == null) continue;
                    for (String key : punishmentsSection.getKeys(false)) {
                        Punishment punishment;
                        if (!key.equals(String.valueOf(id)) || (punishment = this.deserializePunishment(punishmentsSection.getConfigurationSection(key))) == null) continue;
                        this.punishmentCache.put(punishment.getId(), punishment);
                        return punishment;
                    }
                }
            }
            if ((ipPunishmentsFile = new File(this.plugin.getDataFolder(), "ip_punishments.yml")).exists() && (ipPunishmentsSection = (ipConfig = YamlConfiguration.loadConfiguration((File)ipPunishmentsFile)).getConfigurationSection("punishments")) != null) {
                for (String key : ipPunishmentsSection.getKeys(false)) {
                    Punishment punishment;
                    if (!key.equals(String.valueOf(id)) || (punishment = this.deserializePunishment(ipPunishmentsSection.getConfigurationSection(key))) == null) continue;
                    this.punishmentCache.put(punishment.getId(), punishment);
                    return punishment;
                }
            }
            return null;
        });
    }

    @Override
    public CompletableFuture<Punishment> getPunishmentById(int id) {
        return this.getPunishment(id);
    }

    @Override
    public CompletableFuture<List<Punishment>> getActivePunishments(UUID uuid) {
        return CompletableFuture.supplyAsync(() -> {
            if (uuid == null) {
                return new ArrayList();
            }
            ArrayList<Punishment> activePunishments = new ArrayList<Punishment>();
            YamlConfiguration playerConfig = this.getPlayerConfig(uuid);
            ConfigurationSection punishmentsSection = playerConfig.getConfigurationSection("punishments");
            if (punishmentsSection != null) {
                for (String key : punishmentsSection.getKeys(false)) {
                    Punishment punishment;
                    ConfigurationSection punishmentSection = punishmentsSection.getConfigurationSection(key);
                    if (punishmentSection == null || (punishment = this.deserializePunishment(punishmentSection)) == null || !punishment.isActive() || punishment.isExpired()) continue;
                    activePunishments.add(punishment);
                }
            }
            return activePunishments;
        });
    }

    @Override
    public CompletableFuture<List<Punishment>> getPunishmentHistory(UUID uuid) {
        return CompletableFuture.supplyAsync(() -> this.punishmentCache.values().stream().filter(p -> p.getTargetUuid() != null && p.getTargetUuid().equals(uuid)).sorted(Comparator.comparing(Punishment::getCreatedAt).reversed()).collect(Collectors.toList()));
    }

    @Override
    public CompletableFuture<List<Punishment>> getPunishmentHistory(UUID uuid, PunishmentType type) {
        return CompletableFuture.supplyAsync(() -> this.punishmentCache.values().stream().filter(p -> p.getTargetUuid() != null && p.getTargetUuid().equals(uuid) && p.getType() == type).sorted(Comparator.comparing(Punishment::getCreatedAt).reversed()).collect(Collectors.toList()));
    }

    @Override
    public CompletableFuture<List<Punishment>> getPunishmentsByType(UUID uuid, PunishmentType type) {
        return this.getPunishmentHistory(uuid, type);
    }

    @Override
    public CompletableFuture<Punishment> getActivePunishment(UUID uuid, PunishmentType type) {
        return CompletableFuture.supplyAsync(() -> this.punishmentCache.values().stream().filter(p -> p.getTargetUuid() != null && p.getTargetUuid().equals(uuid) && p.getType() == type && p.getRemovedAt() == null && (p.getExpiresAt() == null || p.getExpiresAt().isAfter(LocalDateTime.now()))).findFirst().orElse(null));
    }

    @Override
    public CompletableFuture<Void> expirePunishments() {
        return CompletableFuture.runAsync(() -> {
            LocalDateTime now = LocalDateTime.now();
            boolean changed = false;
            for (Punishment punishment : this.punishmentCache.values()) {
                if (punishment.getExpiresAt() == null || !punishment.getExpiresAt().isBefore(now) || punishment.getRemovedAt() != null) continue;
                punishment.setRemovedAt(now);
                punishment.setRemovedBy("System");
                punishment.setRemovalReason("Expired");
                ConfigurationSection section = this.getPlayerConfig(punishment.getTargetUuid()).getConfigurationSection("punishments." + punishment.getId());
                if (section == null) continue;
                this.serializePunishment(section, punishment);
                changed = true;
            }
            if (changed) {
                for (Punishment punishment : this.punishmentCache.values()) {
                    if (punishment.getRemovedAt() == null) continue;
                    this.savePlayerConfig(punishment.getTargetUuid(), this.getPlayerConfig(punishment.getTargetUuid()));
                }
            }
        });
    }

    @Override
    public CompletableFuture<Integer> getStaffPunishmentCount(UUID staffUuid, PunishmentType type) {
        return CompletableFuture.supplyAsync(() -> (int)this.punishmentCache.values().stream().filter(p -> p.getStaffUuid().equals(staffUuid) && p.getType() == type).count());
    }

    @Override
    public CompletableFuture<Integer> getTotalPunishmentCount(PunishmentType type) {
        return CompletableFuture.supplyAsync(() -> (int)this.punishmentCache.values().stream().filter(p -> p.getType() == type).count());
    }

    @Override
    public CompletableFuture<List<Punishment>> getRecentPunishments(int limit) {
        return CompletableFuture.supplyAsync(() -> this.punishmentCache.values().stream().sorted(Comparator.comparing(Punishment::getCreatedAt).reversed()).limit(limit).collect(Collectors.toList()));
    }

    @Override
    public CompletableFuture<List<Punishment>> getPunishmentsByStaff(String staffName) {
        return CompletableFuture.supplyAsync(() -> this.punishmentCache.values().stream().filter(p -> staffName.equals(p.getStaffName())).collect(Collectors.toList()));
    }

    @Override
    public CompletableFuture<List<Punishment>> getAllPunishments() {
        return CompletableFuture.supplyAsync(() -> new ArrayList<Punishment>(this.punishmentCache.values()));
    }

    @Override
    public CompletableFuture<List<BanwaveEntry>> getPendingBanwaveEntries() {
        return CompletableFuture.supplyAsync(() -> this.banwaveCache.values().stream().filter(e -> !e.isExecuted()).collect(Collectors.toList()));
    }

    @Override
    public CompletableFuture<Void> executeBanwaveEntry(int id) {
        return CompletableFuture.runAsync(() -> {
            BanwaveEntry entry = this.banwaveCache.get(id);
            if (entry != null) {
                ConfigurationSection entrySection;
                entry.setExecuted(true);
                ConfigurationSection banwaveSection = this.globalConfig.getConfigurationSection("banwave");
                if (banwaveSection != null && (entrySection = banwaveSection.getConfigurationSection(String.valueOf(id))) != null) {
                    this.serializeBanwaveEntry(entrySection, entry);
                    this.saveGlobalConfig();
                }
            }
        });
    }

    @Override
    public CompletableFuture<Void> removeBanwaveEntry(int id) {
        return CompletableFuture.runAsync(() -> {
            this.banwaveCache.remove(id);
            ConfigurationSection banwaveSection = this.globalConfig.getConfigurationSection("banwave");
            if (banwaveSection != null) {
                banwaveSection.set(String.valueOf(id), null);
                this.saveGlobalConfig();
            }
        });
    }

    @Override
    public CompletableFuture<Boolean> isPlayerBanned(UUID uuid) {
        return CompletableFuture.supplyAsync(() -> this.punishmentCache.values().stream().anyMatch(p -> !(p.getTargetUuid() == null || !p.getTargetUuid().equals(uuid) || p.getType() != PunishmentType.BAN && p.getType() != PunishmentType.TEMPBAN || p.getRemovedAt() != null || p.getExpiresAt() != null && !p.getExpiresAt().isAfter(LocalDateTime.now()))));
    }

    @Override
    public CompletableFuture<Boolean> isPlayerMuted(UUID uuid) {
        return CompletableFuture.supplyAsync(() -> this.punishmentCache.values().stream().anyMatch(p -> !(p.getTargetUuid() == null || !p.getTargetUuid().equals(uuid) || p.getType() != PunishmentType.MUTE && p.getType() != PunishmentType.TEMPMUTE || p.getRemovedAt() != null || p.getExpiresAt() != null && !p.getExpiresAt().isAfter(LocalDateTime.now()))));
    }

    @Override
    public CompletableFuture<Boolean> isIPBanned(String ip) {
        return CompletableFuture.supplyAsync(() -> this.punishmentCache.values().stream().anyMatch(p -> ip.equals(p.getTargetIP()) && (p.getType() == PunishmentType.IPBAN || p.getType() == PunishmentType.IPTEMPBAN) && p.isActive() && !p.isExpired()));
    }

    @Override
    public CompletableFuture<Boolean> isIPMuted(String ip) {
        return CompletableFuture.supplyAsync(() -> this.punishmentCache.values().stream().anyMatch(p -> ip.equals(p.getTargetIP()) && (p.getType() == PunishmentType.IPMUTE || p.getType() == PunishmentType.IPTEMPMUTE) && p.isActive() && !p.isExpired()));
    }

    @Override
    public CompletableFuture<List<String>> searchPlayers(String query) {
        return CompletableFuture.supplyAsync(() -> this.playerDataCache.values().stream().map(PlayerData::getPlayerName).filter(name -> name.toLowerCase().contains(query.toLowerCase())).collect(Collectors.toList()));
    }

    @Override
    public CompletableFuture<List<PlayerData>> searchPlayersByName(String name) {
        return CompletableFuture.supplyAsync(() -> this.playerDataCache.values().stream().filter(playerData -> playerData.getPlayerName().toLowerCase().contains(name.toLowerCase())).sorted((p1, p2) -> p2.getLastSeen().compareTo(p1.getLastSeen())).limit(10L).collect(Collectors.toList()));
    }

    @Override
    public CompletableFuture<Integer> getTotalPunishments() {
        return CompletableFuture.completedFuture(this.punishmentCache.size());
    }

    @Override
    public CompletableFuture<List<DatabaseManager.PlayerConnection>> getPlayerConnections(UUID uuid) {
        return CompletableFuture.supplyAsync(() -> {
            ArrayList<PlayerConnectionImpl> list = new ArrayList<PlayerConnectionImpl>();
            Map<String, Long> ipMap = this.playerConnectionsCache.get(uuid);
            if (ipMap == null || ipMap.isEmpty()) {
                return list;
            }
            PlayerData data = this.playerDataCache.get(uuid);
            String playerName = data != null ? data.getPlayerName() : "Unknown";
            for (Map.Entry<String, Long> e : ipMap.entrySet()) {
                LocalDateTime ts = LocalDateTime.ofEpochSecond(e.getValue() / 1000L, 0, ZoneOffset.UTC);
                list.add(new PlayerConnectionImpl(uuid, playerName, e.getKey(), ts));
            }
            return list;
        });
    }

    @Override
    public CompletableFuture<List<DatabaseManager.PlayerConnection>> getPlayersFromIP(String ip, long expirationTimeMillis) {
        return CompletableFuture.supplyAsync(() -> {
            ArrayList<PlayerConnectionImpl> list = new ArrayList<PlayerConnectionImpl>();
            for (Map.Entry<UUID, Map<String, Long>> entry : this.playerConnectionsCache.entrySet()) {
                Long tsMillis = entry.getValue().get(ip);
                if (tsMillis == null || tsMillis < expirationTimeMillis) continue;
                UUID uuid = entry.getKey();
                PlayerData data = this.playerDataCache.get(uuid);
                String name = data != null ? data.getPlayerName() : "Unknown";
                LocalDateTime ts = LocalDateTime.ofEpochSecond(tsMillis / 1000L, 0, ZoneOffset.UTC);
                list.add(new PlayerConnectionImpl(uuid, name, ip, ts));
            }
            return list;
        });
    }

    @Override
    public CompletableFuture<Void> recordPlayerConnection(UUID uuid, String playerName, String ip) {
        return CompletableFuture.runAsync(() -> {
            Map ipMap = this.playerConnectionsCache.computeIfAbsent(uuid, k -> new ConcurrentHashMap());
            long now = System.currentTimeMillis();
            ipMap.put(ip, now);
            PlayerData data = this.playerDataCache.computeIfAbsent(uuid, u -> new PlayerData(uuid, playerName));
            data.setPlayerName(playerName);
            data.setLastIP(ip);
            data.setLastJoin(now);
            YamlConfiguration playerConfig = this.getPlayerConfig(uuid);
            this.serializePlayerData(playerConfig, data);
            this.serializePlayerConnections(playerConfig, uuid);
            this.savePlayerConfig(uuid, playerConfig);
        });
    }

    @Override
    public CompletableFuture<List<Punishment>> getAllActivePunishments() {
        return CompletableFuture.supplyAsync(() -> this.punishmentCache.values().stream().filter(p -> p.getRemovedAt() == null && (p.getExpiresAt() == null || p.getExpiresAt().isAfter(LocalDateTime.now()))).collect(Collectors.toList()));
    }

    private static class PlayerConnectionImpl
    implements DatabaseManager.PlayerConnection {
        private final UUID uuid;
        private final String playerName;
        private final String ip;
        private final LocalDateTime timestamp;

        PlayerConnectionImpl(UUID uuid, String playerName, String ip, LocalDateTime timestamp) {
            this.uuid = uuid;
            this.playerName = playerName;
            this.ip = ip;
            this.timestamp = timestamp;
        }

        @Override
        public UUID getUuid() {
            return this.uuid;
        }

        @Override
        public String getPlayerName() {
            return this.playerName;
        }

        @Override
        public String getIp() {
            return this.ip;
        }

        @Override
        public LocalDateTime getTimestamp() {
            return this.timestamp;
        }
    }
}

