/*
 * Decompiled with CFR 0.152.
 */
package me.wethink.weGuardian.services;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import me.wethink.weGuardian.WeGuardian;
import me.wethink.weGuardian.models.BanwaveEntry;
import me.wethink.weGuardian.models.Punishment;
import org.bukkit.configuration.ConfigurationSection;

public class CrossServerSyncService {
    private final WeGuardian plugin;
    private final boolean enabled;
    private final int syncInterval;
    private final List<ServerConfig> servers;
    private final boolean syncPunishments;
    private final boolean syncPlayerData;
    private final boolean syncBanwave;
    private final boolean broadcastPunishments;
    private final boolean requireAuthentication;
    private final String authenticationKey;

    public CrossServerSyncService(WeGuardian plugin) {
        this.plugin = plugin;
        ConfigurationSection config = plugin.getConfig().getConfigurationSection("cross_server");
        this.enabled = config != null && config.getBoolean("enabled", false);
        this.syncInterval = config != null ? config.getInt("sync_interval", 30) : 30;
        this.syncPunishments = config != null && config.getBoolean("sync_punishments", true);
        this.syncPlayerData = config != null && config.getBoolean("sync_player_data", true);
        this.syncBanwave = config != null && config.getBoolean("sync_banwave", true);
        this.broadcastPunishments = config != null && config.getBoolean("broadcast_punishments", true);
        this.requireAuthentication = config != null && config.getBoolean("require_authentication", false);
        this.authenticationKey = config != null ? config.getString("authentication_key", "") : "";
        this.servers = this.loadServerConfigs(config);
        if (this.enabled) {
            this.startSyncTask();
            plugin.debug("Cross-server synchronization enabled with %d servers", this.servers.size());
        } else {
            plugin.debug("Cross-server synchronization disabled");
        }
    }

    private List<ServerConfig> loadServerConfigs(ConfigurationSection config) {
        ArrayList<ServerConfig> serverList = new ArrayList<ServerConfig>();
        if (config != null && config.contains("servers")) {
            List serverMaps = config.getMapList("servers");
            for (Map serverMap : serverMaps) {
                boolean serverEnabled;
                String name = serverMap.containsKey("name") ? (String)serverMap.get("name") : "unknown";
                String host = serverMap.containsKey("host") ? (String)serverMap.get("host") : "localhost";
                int port = serverMap.containsKey("port") ? (Integer)serverMap.get("port") : 25565;
                boolean bl = serverEnabled = serverMap.containsKey("enabled") ? (Boolean)serverMap.get("enabled") : true;
                if (serverEnabled) {
                    serverList.add(new ServerConfig(name, host, port));
                    this.plugin.debug("Loaded server config: %s (%s:%d)", name, host, port);
                    continue;
                }
                this.plugin.debug("Skipped disabled server: %s", name);
            }
        }
        return serverList;
    }

    private void startSyncTask() {
        if (this.syncInterval > 0) {
            this.plugin.debug("Starting cross-server sync task with interval: %d seconds", this.syncInterval);
            this.plugin.getFoliaLib().getScheduler().runTimerAsync(() -> {
                try {
                    this.plugin.debug("Starting cross-server synchronization cycle");
                    this.syncAllData();
                }
                catch (Exception e) {
                    this.plugin.getLogger().warning("Cross-server sync task failed: " + e.getMessage());
                    this.plugin.debug("Cross-server sync task error: %s", e.getMessage());
                }
            }, (long)this.syncInterval * 20L, (long)this.syncInterval * 20L);
        }
    }

    public void syncAllData() {
        if (!this.enabled) {
            return;
        }
        CompletableFuture.runAsync(() -> {
            try {
                if (this.syncPunishments) {
                    this.syncPunishments();
                }
                if (this.syncPlayerData) {
                    this.syncPlayerData();
                }
                if (this.syncBanwave) {
                    this.syncBanwave();
                }
                this.plugin.debug("Cross-server synchronization completed successfully");
            }
            catch (Exception e) {
                this.plugin.getLogger().warning("Cross-server synchronization failed: " + e.getMessage());
                this.plugin.debug("Cross-server sync error: %s", e.getMessage());
            }
        });
    }

    private void syncPunishments() {
        this.plugin.debug("Syncing punishments across servers");
        ((CompletableFuture)this.plugin.getDatabaseManager().getRecentPunishments(100).thenAccept(punishments -> {
            for (Punishment punishment : punishments) {
                this.broadcastPunishment(punishment);
            }
        })).exceptionally(throwable -> {
            this.plugin.getLogger().warning("Failed to sync punishments: " + throwable.getMessage());
            return null;
        });
    }

    private void syncPlayerData() {
        this.plugin.debug("Syncing player data across servers");
        for (ServerConfig server : this.servers) {
            this.syncPlayerDataFromServer(server);
        }
    }

    private void syncBanwave() {
        this.plugin.debug("Syncing banwave data across servers");
        ((CompletableFuture)this.plugin.getDatabaseManager().getPendingBanwaveEntries().thenAccept(entries -> {
            for (BanwaveEntry entry : entries) {
                this.broadcastBanwaveEntry(entry);
            }
        })).exceptionally(throwable -> {
            this.plugin.getLogger().warning("Failed to sync banwave: " + throwable.getMessage());
            return null;
        });
    }

    public void broadcastPunishment(Punishment punishment) {
        if (!this.enabled || !this.broadcastPunishments) {
            return;
        }
        this.plugin.debug("Broadcasting punishment: %s for %s", new Object[]{punishment.getType(), punishment.getTargetName()});
        for (ServerConfig server : this.servers) {
            this.sendPunishmentToServer(server, punishment);
        }
    }

    public void broadcastBanwaveEntry(BanwaveEntry entry) {
        if (!this.enabled) {
            return;
        }
        this.plugin.debug("Broadcasting banwave entry for %s", entry.getTargetName());
        for (ServerConfig server : this.servers) {
            this.sendBanwaveEntryToServer(server, entry);
        }
    }

    private void sendPunishmentToServer(ServerConfig server, Punishment punishment) {
        this.plugin.getFoliaLib().getScheduler().runNextTick(task -> CompletableFuture.runAsync(() -> {
            try {
                String json = this.createPunishmentJson(punishment);
                this.sendHttpRequest(server, "/api/punishments", "POST", json);
                this.plugin.debug("Sent punishment to server %s", server.name);
            }
            catch (Exception e) {
                this.plugin.debug("Failed to send punishment to server %s: %s", server.name, e.getMessage());
            }
        }));
    }

    private void sendBanwaveEntryToServer(ServerConfig server, BanwaveEntry entry) {
        this.plugin.getFoliaLib().getScheduler().runNextTick(task -> CompletableFuture.runAsync(() -> {
            try {
                String json = this.createBanwaveEntryJson(entry);
                this.sendHttpRequest(server, "/api/banwave", "POST", json);
                this.plugin.debug("Sent banwave entry to server %s", server.name);
            }
            catch (Exception e) {
                this.plugin.debug("Failed to send banwave entry to server %s: %s", server.name, e.getMessage());
            }
        }));
    }

    private void syncPlayerDataFromServer(ServerConfig server) {
        this.plugin.getFoliaLib().getScheduler().runNextTick(task -> CompletableFuture.runAsync(() -> {
            try {
                String response = this.sendHttpRequest(server, "/api/players", "GET", null);
                if (response != null) {
                    this.processPlayerDataResponse(response);
                    this.plugin.debug("Synced player data from server %s", server.name);
                }
            }
            catch (Exception e) {
                this.plugin.debug("Failed to sync player data from server %s: %s", server.name, e.getMessage());
            }
        }));
    }

    private String sendHttpRequest(ServerConfig server, String endpoint, String method, String jsonData) throws IOException {
        int responseCode;
        String urlString = "http://" + server.host + ":" + server.port + endpoint;
        URI uri = URI.create(urlString);
        HttpURLConnection connection = (HttpURLConnection)uri.toURL().openConnection();
        connection.setRequestMethod(method);
        connection.setConnectTimeout(5000);
        connection.setReadTimeout(10000);
        if (this.requireAuthentication && !this.authenticationKey.isEmpty()) {
            connection.setRequestProperty("Authorization", "Bearer " + this.authenticationKey);
        }
        if (jsonData != null) {
            connection.setRequestProperty("Content-Type", "application/json");
            connection.setDoOutput(true);
            try (OutputStream os = connection.getOutputStream();){
                byte[] input = jsonData.getBytes(StandardCharsets.UTF_8);
                os.write(input, 0, input.length);
            }
        }
        if ((responseCode = connection.getResponseCode()) >= 200 && responseCode < 300) {
            try (BufferedReader br = new BufferedReader(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8));){
                String responseLine;
                StringBuilder response = new StringBuilder();
                while ((responseLine = br.readLine()) != null) {
                    response.append(responseLine.trim());
                }
                String string = response.toString();
                return string;
            }
        }
        throw new IOException("HTTP error: " + responseCode);
    }

    private String createPunishmentJson(Punishment punishment) {
        return String.format("{\"id\":%d,\"target_uuid\":\"%s\",\"target_name\":\"%s\",\"staff_uuid\":\"%s\",\"staff_name\":\"%s\",\"type\":\"%s\",\"reason\":\"%s\",\"created_at\":\"%s\",\"expires_at\":%s,\"active\":%s,\"server_name\":\"%s\"}", new Object[]{punishment.getId(), punishment.getTargetUuid(), punishment.getTargetName(), punishment.getStaffUuid(), punishment.getStaffName(), punishment.getType(), punishment.getReason(), punishment.getCreatedAt(), punishment.getExpiresAt() != null ? "\"" + String.valueOf(punishment.getExpiresAt()) + "\"" : "null", punishment.isActive(), punishment.getServerName()});
    }

    private String createBanwaveEntryJson(BanwaveEntry entry) {
        return String.format("{\"id\":%d,\"target_uuid\":\"%s\",\"target_name\":\"%s\",\"staff_uuid\":\"%s\",\"staff_name\":\"%s\",\"reason\":\"%s\",\"created_at\":\"%s\",\"executed\":%s}", entry.getId(), entry.getTargetUuid(), entry.getTargetName(), entry.getStaffUuid(), entry.getStaffName(), entry.getReason(), entry.getCreatedAt(), entry.isExecuted());
    }

    private void processPlayerDataResponse(String response) {
        this.plugin.debug("Processing player data response: %s", response);
    }

    public void validateServerConnections() {
        if (!this.enabled) {
            return;
        }
        this.plugin.debug("Validating cross-server connections...");
        for (ServerConfig server : this.servers) {
            this.plugin.getFoliaLib().getScheduler().runNextTick(task -> CompletableFuture.runAsync(() -> {
                try {
                    String response = this.sendHttpRequest(server, "/api/health", "GET", null);
                    if (response != null) {
                        this.plugin.debug("Server %s is reachable", server.name);
                    }
                }
                catch (Exception e) {
                    this.plugin.debug("Server %s is not reachable: %s", server.name, e.getMessage());
                }
            }));
        }
    }

    public void shutdown() {
        this.plugin.debug("Cross-server synchronization service shutting down");
    }

    public boolean isEnabled() {
        return this.enabled;
    }

    public List<ServerConfig> getServers() {
        return this.servers;
    }

    public static class ServerConfig {
        public final String name;
        public final String host;
        public final int port;

        public ServerConfig(String name, String host, int port) {
            this.name = name;
            this.host = host;
            this.port = port;
        }
    }
}

