/*
 * Decompiled with CFR 0.152.
 */
package me.wethink.weGuardian.services;

import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import me.wethink.weGuardian.WeGuardian;
import me.wethink.weGuardian.database.DatabaseManager;
import me.wethink.weGuardian.models.Punishment;
import me.wethink.weGuardian.models.PunishmentType;
import me.wethink.weGuardian.services.DiscordWebhookService;
import me.wethink.weGuardian.utils.MessageUtils;
import me.wethink.weGuardian.utils.TimeUtils;
import net.kyori.adventure.text.Component;
import org.bukkit.Bukkit;
import org.bukkit.OfflinePlayer;
import org.bukkit.command.CommandSender;
import org.bukkit.entity.Entity;
import org.bukkit.entity.Player;

public class PunishmentService {
    private final WeGuardian plugin;
    private final DatabaseManager database;
    private final ConcurrentHashMap<UUID, Boolean> banCache;
    private final ConcurrentHashMap<UUID, Boolean> muteCache;
    private final ConcurrentHashMap<UUID, String> playerNameCache;
    private final ConcurrentHashMap<UUID, Long> cacheTimestamps;
    private final ConcurrentHashMap<String, UUID> nameToUuidCache;
    private final ScheduledExecutorService executorService;
    private final ExecutorService asyncExecutor;
    private final AtomicInteger activeOperations;
    private final int maxConcurrentOperations;
    private final int operationTimeout;
    private final boolean cacheEnabled;

    public PunishmentService(WeGuardian plugin) {
        this.plugin = plugin;
        this.database = plugin.getDatabaseManager();
        this.banCache = new ConcurrentHashMap();
        this.muteCache = new ConcurrentHashMap();
        this.playerNameCache = new ConcurrentHashMap();
        this.cacheTimestamps = new ConcurrentHashMap();
        this.nameToUuidCache = new ConcurrentHashMap();
        this.activeOperations = new AtomicInteger(0);
        this.maxConcurrentOperations = plugin.getConfig().getInt("performance.max_concurrent_operations", 50);
        this.operationTimeout = plugin.getConfig().getInt("performance.operation_timeout", 30);
        this.cacheEnabled = plugin.getConfig().getBoolean("performance.cache.enabled", true);
        int threadPoolSize = plugin.getConfig().getInt("performance.thread_pool_size", 4);
        int asyncWorkers = plugin.getConfig().getInt("performance.async.worker_threads", 2);
        this.executorService = Executors.newScheduledThreadPool(threadPoolSize, r -> {
            Thread t = new Thread(r, "WeGuardian-Scheduler");
            t.setDaemon(true);
            return t;
        });
        this.asyncExecutor = Executors.newFixedThreadPool(asyncWorkers, r -> {
            Thread t = new Thread(r, "WeGuardian-Async");
            t.setDaemon(true);
            return t;
        });
        this.startCacheCleanupTask();
        this.startPerformanceMonitoring();
    }

    public CompletableFuture<Boolean> executePunishment(PunishmentType type, String targetName, String staffName, String reason, String duration) {
        this.plugin.debug("Executing punishment: type=%s, target=%s, staff=%s, reason=%s, duration=%s", new Object[]{type, targetName, staffName, reason, duration});
        if (this.activeOperations.get() >= this.maxConcurrentOperations) {
            this.plugin.debug("Punishment execution rejected: max concurrent operations reached (%d/%d)", this.activeOperations.get(), this.maxConcurrentOperations);
            return CompletableFuture.completedFuture(false);
        }
        return CompletableFuture.supplyAsync(() -> {
            Boolean bl;
            this.activeOperations.incrementAndGet();
            this.plugin.debug("Punishment execution started: active operations now %d", this.activeOperations.get());
            try {
                boolean result = switch (type) {
                    case PunishmentType.BAN -> this.ban(targetName, staffName, reason).join();
                    case PunishmentType.TEMPBAN -> this.tempban(targetName, staffName, reason, duration).join();
                    case PunishmentType.MUTE -> this.mute(targetName, staffName, reason).join();
                    case PunishmentType.TEMPMUTE -> this.tempmute(targetName, staffName, reason, duration).join();
                    case PunishmentType.KICK -> this.kick(targetName, staffName, reason).join();
                    case PunishmentType.WARN -> this.warn(targetName, staffName, reason).join();
                    case PunishmentType.IPBAN -> this.ipban(targetName, staffName, reason).join();
                    case PunishmentType.IPMUTE -> this.ipmute(targetName, staffName, reason).join();
                    default -> false;
                };
                this.plugin.debug("Punishment execution completed: result=%s", result);
                bl = result;
                this.activeOperations.decrementAndGet();
            }
            catch (Throwable throwable) {
                this.activeOperations.decrementAndGet();
                this.plugin.debug("Punishment execution finished: active operations now %d", this.activeOperations.get());
                throw throwable;
            }
            this.plugin.debug("Punishment execution finished: active operations now %d", this.activeOperations.get());
            return bl;
        }, this.asyncExecutor).orTimeout(this.operationTimeout, TimeUnit.SECONDS);
    }

    public CompletableFuture<Boolean> ban(String targetName, String staffName, String reason) {
        this.plugin.debug("Starting ban process for player: %s", targetName);
        return this.executePlayerLookup(targetName).thenCompose(targetUuid -> {
            if (targetUuid == null) {
                this.plugin.debug("Ban failed: player not found - %s", targetName);
                return CompletableFuture.completedFuture(false);
            }
            return this.getPlayerName((UUID)targetUuid).thenCompose(finalTargetName -> {
                UUID staffUuid = this.getStaffUuid(staffName);
                this.plugin.debug("Ban process: targetUuid=%s, finalTargetName=%s, staffUuid=%s", targetUuid, finalTargetName, staffUuid);
                if (this.cacheEnabled && this.isCacheValid((UUID)targetUuid) && this.banCache.getOrDefault(targetUuid, false).booleanValue()) {
                    this.plugin.debug("Ban skipped: player already banned (cached) - %s", finalTargetName);
                    return CompletableFuture.completedFuture(false);
                }
                Punishment punishment = new Punishment((UUID)targetUuid, (String)finalTargetName, staffUuid, staffName, PunishmentType.BAN, reason, null, this.getServerName());
                return this.database.addPunishment(punishment).thenCompose(punishmentId -> {
                    if (punishmentId > 0) {
                        Player targetPlayer;
                        this.plugin.debug("Ban punishment added to database with ID: %d", punishmentId);
                        if (this.cacheEnabled) {
                            this.banCache.put((UUID)targetUuid, true);
                            this.updateCacheTimestamp((UUID)targetUuid);
                            this.nameToUuidCache.put(finalTargetName.toLowerCase(), (UUID)targetUuid);
                            this.plugin.debug("Ban cache updated for player: %s", finalTargetName);
                        }
                        if ((targetPlayer = Bukkit.getPlayer((UUID)targetUuid)) != null && targetPlayer.isOnline()) {
                            this.plugin.debug("Kicking online player: %s", finalTargetName);
                            String kickMessage = this.getKickMessage(punishment);
                            Component kickComponent = MessageUtils.toComponent(kickMessage);
                            this.plugin.getFoliaLib().getScheduler().runAtEntity((Entity)targetPlayer, task -> {
                                if (targetPlayer.isOnline()) {
                                    targetPlayer.kick(kickComponent);
                                }
                            });
                        } else {
                            this.plugin.debug("Player not online, skipping kick: %s", finalTargetName);
                        }
                        this.plugin.getNotificationService().broadcastPunishment(punishment);
                        this.plugin.debug("Ban completed successfully for player: %s", finalTargetName);
                        return CompletableFuture.completedFuture(true);
                    }
                    this.plugin.debug("Ban failed: could not add punishment to database");
                    return CompletableFuture.completedFuture(false);
                });
            });
        });
    }

    public CompletableFuture<Boolean> tempban(String targetName, String staffName, String reason, String duration) {
        return this.executePlayerLookup(targetName).thenCompose(targetUuid -> {
            if (targetUuid == null) {
                return CompletableFuture.completedFuture(false);
            }
            return this.getPlayerName((UUID)targetUuid).thenCompose(finalTargetName -> {
                UUID staffUuid = this.getStaffUuid(staffName);
                if (this.cacheEnabled && this.banCache.getOrDefault(targetUuid, false).booleanValue()) {
                    return CompletableFuture.completedFuture(false);
                }
                LocalDateTime expiresAt = TimeUtils.parseTime(duration);
                if (expiresAt == null) {
                    return CompletableFuture.completedFuture(false);
                }
                Punishment punishment = new Punishment((UUID)targetUuid, (String)finalTargetName, staffUuid, staffName, PunishmentType.TEMPBAN, reason, expiresAt, this.getServerName());
                return this.database.addPunishment(punishment).thenCompose(punishmentId -> {
                    if (punishmentId > 0) {
                        Player targetPlayer;
                        if (this.cacheEnabled) {
                            this.banCache.put((UUID)targetUuid, true);
                            this.updateCacheTimestamp((UUID)targetUuid);
                            this.nameToUuidCache.put(finalTargetName.toLowerCase(), (UUID)targetUuid);
                        }
                        if ((targetPlayer = Bukkit.getPlayer((UUID)targetUuid)) != null && targetPlayer.isOnline()) {
                            String kickMessage = this.getKickMessage(punishment);
                            Component kickComponent = MessageUtils.toComponent(kickMessage);
                            this.plugin.getFoliaLib().getScheduler().runAtEntity((Entity)targetPlayer, task -> {
                                if (targetPlayer.isOnline()) {
                                    targetPlayer.kick(kickComponent);
                                }
                            });
                        }
                        this.plugin.getNotificationService().broadcastPunishment(punishment);
                        DiscordWebhookService webhook = new DiscordWebhookService(this.plugin);
                        webhook.sendPunishmentWebhook(punishment);
                        return CompletableFuture.completedFuture(true);
                    }
                    return CompletableFuture.completedFuture(false);
                });
            });
        });
    }

    public CompletableFuture<Boolean> mute(String targetName, String staffName, String reason) {
        return this.executePlayerLookup(targetName).thenCompose(targetUuid -> {
            if (targetUuid == null) {
                return CompletableFuture.completedFuture(false);
            }
            return this.getPlayerName((UUID)targetUuid).thenCompose(finalTargetName -> {
                UUID staffUuid = this.getStaffUuid(staffName);
                if (this.cacheEnabled && this.muteCache.getOrDefault(targetUuid, false).booleanValue()) {
                    return CompletableFuture.completedFuture(false);
                }
                Punishment punishment = new Punishment((UUID)targetUuid, (String)finalTargetName, staffUuid, staffName, PunishmentType.MUTE, reason, null, this.getServerName());
                return this.database.addPunishment(punishment).thenCompose(punishmentId -> {
                    if (punishmentId > 0) {
                        String actionBarMessage;
                        Player targetPlayer;
                        if (this.cacheEnabled) {
                            this.muteCache.put((UUID)targetUuid, true);
                        }
                        if ((targetPlayer = Bukkit.getPlayer((UUID)targetUuid)) != null && targetPlayer.isOnline() && !(actionBarMessage = this.getActionBarMessage(punishment)).isEmpty()) {
                            Component component = MessageUtils.toComponent(actionBarMessage);
                            targetPlayer.sendActionBar(component);
                        }
                        this.plugin.getNotificationService().broadcastPunishment(punishment);
                        DiscordWebhookService webhook = new DiscordWebhookService(this.plugin);
                        webhook.sendPunishmentWebhook(punishment);
                        return CompletableFuture.completedFuture(true);
                    }
                    return CompletableFuture.completedFuture(false);
                });
            });
        });
    }

    public CompletableFuture<Boolean> tempmute(String targetName, String staffName, String reason, String duration) {
        return this.executePlayerLookup(targetName).thenCompose(targetUuid -> {
            if (targetUuid == null) {
                return CompletableFuture.completedFuture(false);
            }
            return this.getPlayerName((UUID)targetUuid).thenCompose(finalTargetName -> {
                UUID staffUuid = this.getStaffUuid(staffName);
                if (this.cacheEnabled && this.muteCache.getOrDefault(targetUuid, false).booleanValue()) {
                    return CompletableFuture.completedFuture(false);
                }
                LocalDateTime expiresAt = TimeUtils.parseTime(duration);
                if (expiresAt == null) {
                    return CompletableFuture.completedFuture(false);
                }
                Punishment punishment = new Punishment((UUID)targetUuid, (String)finalTargetName, staffUuid, staffName, PunishmentType.TEMPMUTE, reason, expiresAt, this.getServerName());
                return this.database.addPunishment(punishment).thenCompose(punishmentId -> {
                    if (punishmentId > 0) {
                        String actionBarMessage;
                        Player targetPlayer;
                        if (this.cacheEnabled) {
                            this.muteCache.put((UUID)targetUuid, true);
                        }
                        if ((targetPlayer = Bukkit.getPlayer((UUID)targetUuid)) != null && targetPlayer.isOnline() && !(actionBarMessage = this.getActionBarMessage(punishment)).isEmpty()) {
                            Component component = MessageUtils.toComponent(actionBarMessage);
                            targetPlayer.sendActionBar(component);
                        }
                        this.plugin.getNotificationService().broadcastPunishment(punishment);
                        DiscordWebhookService webhook = new DiscordWebhookService(this.plugin);
                        webhook.sendPunishmentWebhook(punishment);
                        return CompletableFuture.completedFuture(true);
                    }
                    return CompletableFuture.completedFuture(false);
                });
            });
        });
    }

    public CompletableFuture<Boolean> kick(String targetName, String staffName, String reason) {
        return this.executePlayerLookup(targetName).thenCompose(targetUuid -> {
            if (targetUuid == null) {
                return CompletableFuture.completedFuture(false);
            }
            return this.getPlayerName((UUID)targetUuid).thenCompose(finalTargetName -> {
                UUID staffUuid = this.getStaffUuid(staffName);
                Punishment punishment = new Punishment((UUID)targetUuid, (String)finalTargetName, staffUuid, staffName, PunishmentType.KICK, reason, null, this.getServerName());
                return this.database.addPunishment(punishment).thenCompose(punishmentId -> {
                    if (punishmentId > 0) {
                        Player targetPlayer = Bukkit.getPlayer((UUID)targetUuid);
                        if (targetPlayer != null && targetPlayer.isOnline()) {
                            String kickMessage = this.getKickMessage(punishment);
                            Component kickComponent = MessageUtils.toComponent(kickMessage);
                            this.plugin.getFoliaLib().getScheduler().runAtEntity((Entity)targetPlayer, task -> {
                                if (targetPlayer.isOnline()) {
                                    targetPlayer.kick(kickComponent);
                                }
                            });
                        }
                        this.plugin.getNotificationService().broadcastPunishment(punishment);
                        DiscordWebhookService webhook = new DiscordWebhookService(this.plugin);
                        webhook.sendPunishmentWebhook(punishment);
                        return CompletableFuture.completedFuture(true);
                    }
                    return CompletableFuture.completedFuture(false);
                });
            });
        });
    }

    public CompletableFuture<Boolean> warn(String targetName, String staffName, String reason) {
        return this.executePlayerLookup(targetName).thenCompose(targetUuid -> {
            if (targetUuid == null) {
                return CompletableFuture.completedFuture(false);
            }
            return this.getPlayerName((UUID)targetUuid).thenCompose(finalTargetName -> {
                UUID staffUuid = this.getStaffUuid(staffName);
                Punishment punishment = new Punishment((UUID)targetUuid, (String)finalTargetName, staffUuid, staffName, PunishmentType.WARN, reason, null, this.getServerName());
                return this.database.addPunishment(punishment).thenCompose(punishmentId -> {
                    if (punishmentId > 0) {
                        Player targetPlayer = Bukkit.getPlayer((UUID)targetUuid);
                        if (targetPlayer != null && targetPlayer.isOnline()) {
                            String warnMessage = this.getWarnMessage(punishment);
                            Component warnComponent = MessageUtils.toComponent(warnMessage);
                            targetPlayer.sendMessage(warnComponent);
                        }
                        this.plugin.getNotificationService().broadcastPunishment(punishment);
                        DiscordWebhookService webhook = new DiscordWebhookService(this.plugin);
                        webhook.sendPunishmentWebhook(punishment);
                        return CompletableFuture.completedFuture(true);
                    }
                    return CompletableFuture.completedFuture(false);
                });
            });
        });
    }

    public CompletableFuture<Boolean> unban(String targetName, String staffName) {
        return this.executePlayerLookup(targetName).thenCompose(targetUuid -> {
            if (targetUuid == null) {
                return CompletableFuture.completedFuture(false);
            }
            return this.database.getActivePunishments((UUID)targetUuid).thenCompose(punishments -> {
                Punishment banPunishment = punishments.stream().filter(p -> (p.getType() == PunishmentType.BAN || p.getType() == PunishmentType.TEMPBAN) && !p.isExpired()).findFirst().orElse(null);
                if (banPunishment == null) {
                    return CompletableFuture.completedFuture(false);
                }
                UUID staffUuid = this.getStaffUuid(staffName);
                return this.database.removePunishment(banPunishment.getId(), staffUuid, staffName, "Unbanned by " + staffName).thenApply(success -> {
                    if (this.cacheEnabled) {
                        this.banCache.remove(targetUuid);
                    }
                    this.plugin.getNotificationService().broadcastUnpunishment(banPunishment, staffName);
                    DiscordWebhookService webhook = new DiscordWebhookService(this.plugin);
                    webhook.sendUnpunishmentWebhook(banPunishment, staffName);
                    return true;
                });
            });
        });
    }

    public CompletableFuture<Boolean> unmute(String targetName, String staffName) {
        return this.executePlayerLookup(targetName).thenCompose(targetUuid -> {
            if (targetUuid == null) {
                return CompletableFuture.completedFuture(false);
            }
            return this.database.getActivePunishments((UUID)targetUuid).thenCompose(punishments -> {
                Punishment mutePunishment = punishments.stream().filter(p -> (p.getType() == PunishmentType.MUTE || p.getType() == PunishmentType.TEMPMUTE) && !p.isExpired()).findFirst().orElse(null);
                if (mutePunishment == null) {
                    return CompletableFuture.completedFuture(false);
                }
                UUID staffUuid = this.getStaffUuid(staffName);
                return this.database.removePunishment(mutePunishment.getId(), staffUuid, staffName, "Unmuted by " + staffName).thenApply(success -> {
                    if (this.cacheEnabled) {
                        this.muteCache.remove(targetUuid);
                    }
                    this.plugin.getNotificationService().broadcastUnpunishment(mutePunishment, staffName);
                    DiscordWebhookService webhook = new DiscordWebhookService(this.plugin);
                    webhook.sendUnpunishmentWebhook(mutePunishment, staffName);
                    return true;
                });
            });
        });
    }

    public CompletableFuture<Boolean> ipban(String ipAddress, String staffName, String reason) {
        return this.executeIPLookup(ipAddress).thenCompose(validIP -> {
            if (validIP == null) {
                return CompletableFuture.completedFuture(false);
            }
            UUID staffUuid = this.getStaffUuid(staffName);
            Punishment punishment = new Punishment((String)validIP, staffUuid, staffName, PunishmentType.IPBAN, reason, null, this.plugin.getServer().getName());
            return this.database.addPunishment(punishment).thenCompose(punishmentId -> {
                if (punishmentId > 0) {
                    this.kickPlayersWithIP(ipAddress, punishment);
                    this.plugin.getNotificationService().broadcastPunishment(punishment);
                    return CompletableFuture.completedFuture(true);
                }
                return CompletableFuture.completedFuture(false);
            });
        });
    }

    public CompletableFuture<Boolean> ipmute(String ipAddress, String staffName, String reason) {
        return this.executeIPLookup(ipAddress).thenCompose(validIP -> {
            if (validIP == null) {
                return CompletableFuture.completedFuture(false);
            }
            UUID staffUuid = this.getStaffUuid(staffName);
            Punishment punishment = new Punishment((String)validIP, staffUuid, staffName, PunishmentType.IPMUTE, reason, null, this.plugin.getServer().getName());
            return this.database.addPunishment(punishment).thenCompose(punishmentId -> {
                if (punishmentId > 0) {
                    this.notifyPlayersWithIP(ipAddress, punishment);
                    this.plugin.getNotificationService().broadcastPunishment(punishment);
                    return CompletableFuture.completedFuture(true);
                }
                return CompletableFuture.completedFuture(false);
            });
        });
    }

    public CompletableFuture<Boolean> ipMute(String ip, String staffName, String reason) {
        return this.ipmute(ip, staffName, reason);
    }

    public CompletableFuture<Boolean> ipMute(String ip, String staffName, String reason, String duration) {
        if (duration != null && !duration.isEmpty()) {
            return this.ipTempmute(ip, staffName, reason, duration);
        }
        return this.ipmute(ip, staffName, reason);
    }

    public CompletableFuture<Boolean> ipTempmute(String ip, String staffName, String reason, String duration) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                LocalDateTime expires = null;
                if (duration != null && !duration.isEmpty()) {
                    expires = TimeUtils.parseTimeToExpiration(duration);
                }
                UUID staffUuid = this.getStaffUuid(staffName);
                Punishment punishment = new Punishment(ip, staffUuid, staffName, PunishmentType.IPMUTE, reason, expires, this.plugin.getServer().getName());
                this.database.addPunishment(punishment);
                for (Player player : Bukkit.getOnlinePlayers()) {
                    if (!player.getAddress().getAddress().getHostAddress().equals(ip)) continue;
                    player.sendMessage(MessageUtils.colorize("&cYour IP has been muted for: " + reason));
                }
                return true;
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Failed to IP mute " + ip + ": " + e.getMessage());
                return false;
            }
        }, this.asyncExecutor);
    }

    public CompletableFuture<UUID> executePlayerLookup(String playerName) {
        Player onlinePlayer = Bukkit.getPlayer((String)playerName);
        if (onlinePlayer != null) {
            UUID uuid = onlinePlayer.getUniqueId();
            if (this.cacheEnabled) {
                this.playerNameCache.put(uuid, playerName);
            }
            return CompletableFuture.completedFuture(uuid);
        }
        if (this.cacheEnabled) {
            for (Map.Entry<UUID, String> entry : this.playerNameCache.entrySet()) {
                if (!entry.getValue().equalsIgnoreCase(playerName)) continue;
                return CompletableFuture.completedFuture(entry.getKey());
            }
        }
        try {
            OfflinePlayer offlinePlayer = Bukkit.getOfflinePlayer((String)playerName);
            if (offlinePlayer != null && offlinePlayer.hasPlayedBefore()) {
                UUID uuid = offlinePlayer.getUniqueId();
                if (this.cacheEnabled) {
                    this.playerNameCache.put(uuid, playerName);
                }
                return CompletableFuture.completedFuture(uuid);
            }
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Failed to get UUID for player " + playerName + ": " + e.getMessage());
        }
        return this.database.getPlayerData(playerName).thenApply(playerData -> {
            if (playerData != null) {
                if (this.cacheEnabled) {
                    this.playerNameCache.put(playerData.getUuid(), playerName);
                }
                return playerData.getUuid();
            }
            return null;
        });
    }

    public CompletableFuture<String> getPlayerName(UUID playerUuid) {
        if (this.cacheEnabled && this.playerNameCache.containsKey(playerUuid)) {
            return CompletableFuture.completedFuture(this.playerNameCache.get(playerUuid));
        }
        return this.database.getPlayerData(playerUuid).thenApply(playerData -> {
            if (playerData != null) {
                if (this.cacheEnabled) {
                    this.playerNameCache.put(playerUuid, playerData.getName());
                }
                return playerData.getName();
            }
            try {
                String name = Bukkit.getOfflinePlayer((UUID)playerUuid).getName();
                if (name != null && this.cacheEnabled) {
                    this.playerNameCache.put(playerUuid, name);
                }
                return name != null ? name : "Unknown";
            }
            catch (Exception e) {
                this.plugin.getLogger().warning("Failed to get name for UUID " + String.valueOf(playerUuid) + ": " + e.getMessage());
                return "Unknown";
            }
        });
    }

    public CompletableFuture<String> executeIPLookup(String ipAddress) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                InetAddress.getByName(ipAddress);
                return ipAddress;
            }
            catch (Exception e) {
                this.plugin.getLogger().warning("Invalid IP address: " + ipAddress);
                return null;
            }
        }, this.asyncExecutor);
    }

    public void kickPlayersWithIP(String ipAddress, Punishment punishment) {
        this.plugin.getFoliaLib().getScheduler().runNextTick(task -> {
            for (Player player : Bukkit.getOnlinePlayers()) {
                InetSocketAddress address = player.getAddress();
                if (address == null || !address.getAddress().getHostAddress().equals(ipAddress)) continue;
                String kickMessage = this.getKickMessage(punishment);
                Component kickComponent = MessageUtils.toComponent(kickMessage);
                player.kick(kickComponent);
            }
        });
    }

    public void notifyPlayersWithIP(String ipAddress, Punishment punishment) {
        this.plugin.getFoliaLib().getScheduler().runNextTick(task -> {
            for (Player player : Bukkit.getOnlinePlayers()) {
                String actionBarMessage;
                InetSocketAddress address = player.getAddress();
                if (address == null || !address.getAddress().getHostAddress().equals(ipAddress) || (actionBarMessage = this.getActionBarMessage(punishment)).isEmpty()) continue;
                Component component = MessageUtils.toComponent(actionBarMessage);
                player.sendActionBar(component);
            }
        });
    }

    private void startCacheCleanupTask() {
        if (!this.cacheEnabled) {
            return;
        }
        int cleanupInterval = this.plugin.getConfig().getInt("performance.cache.cleanup_interval", 300);
        int maxCacheSize = this.plugin.getConfig().getInt("performance.cache.max_cache_size", 10000);
        this.plugin.getFoliaLib().getScheduler().runTimerAsync(() -> {
            try {
                this.plugin.debug("Starting cache cleanup task");
                int banCacheSize = this.banCache.size();
                int muteCacheSize = this.muteCache.size();
                int nameCacheSize = this.playerNameCache.size();
                if (banCacheSize > maxCacheSize) {
                    this.plugin.debug("Clearing ban cache: size %d > max %d", banCacheSize, maxCacheSize);
                    this.banCache.clear();
                }
                if (muteCacheSize > maxCacheSize) {
                    this.plugin.debug("Clearing mute cache: size %d > max %d", muteCacheSize, maxCacheSize);
                    this.muteCache.clear();
                }
                if (nameCacheSize > maxCacheSize) {
                    this.plugin.debug("Clearing name cache: size %d > max %d", nameCacheSize, maxCacheSize);
                    this.playerNameCache.clear();
                }
                if (this.plugin.getConfig().getBoolean("debug.log_cache_statistics", false)) {
                    this.plugin.getLogger().info("Cache cleanup completed. Sizes - Ban: " + this.banCache.size() + ", Mute: " + this.muteCache.size() + ", Names: " + this.playerNameCache.size());
                }
                this.plugin.debug("Cache cleanup completed successfully");
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Error during cache cleanup: " + e.getMessage());
                this.plugin.debug("Cache cleanup error: %s", e.getMessage());
            }
        }, (long)(cleanupInterval * 20), (long)(cleanupInterval * 20));
    }

    private void startPerformanceMonitoring() {
        if (!this.plugin.getConfig().getBoolean("debug.log_performance_metrics", false)) {
            return;
        }
        this.plugin.getFoliaLib().getScheduler().runTimerAsync(() -> {
            try {
                int active = this.activeOperations.get();
                this.plugin.debug("Performance monitoring: active operations %d/%d", active, this.maxConcurrentOperations);
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Error during performance monitoring: " + e.getMessage());
                this.plugin.debug("Performance monitoring error: %s", e.getMessage());
            }
        }, 1200L, 1200L);
    }

    private boolean isCacheValid(UUID uuid) {
        if (!this.cacheEnabled) {
            return false;
        }
        Long timestamp = this.cacheTimestamps.get(uuid);
        if (timestamp == null) {
            return false;
        }
        int expireAfterAccess = this.plugin.getConfig().getInt("performance.cache.expire_after_access", 1800);
        return System.currentTimeMillis() - timestamp < (long)expireAfterAccess * 1000L;
    }

    private void updateCacheTimestamp(UUID uuid) {
        if (this.cacheEnabled) {
            this.cacheTimestamps.put(uuid, System.currentTimeMillis());
        }
    }

    private void invalidateCache(UUID uuid) {
        this.banCache.remove(uuid);
        this.muteCache.remove(uuid);
        this.cacheTimestamps.remove(uuid);
        this.plugin.debug("Cache invalidated for UUID: %s", uuid);
    }

    private void invalidateCache(String playerName) {
        UUID uuid = this.nameToUuidCache.get(playerName.toLowerCase());
        if (uuid != null) {
            this.invalidateCache(uuid);
        }
        this.nameToUuidCache.remove(playerName.toLowerCase());
    }

    private UUID getStaffUuid(String staffName) {
        Player staffPlayer = Bukkit.getPlayer((String)staffName);
        return staffPlayer != null ? staffPlayer.getUniqueId() : UUID.nameUUIDFromBytes(("Console:" + staffName).getBytes());
    }

    private String getServerName() {
        return this.plugin.getConfig().getString("server_name", "Unknown");
    }

    private String getKickMessage(Punishment punishment) {
        String template = this.plugin.getMessage("screen." + punishment.getType().name().toLowerCase(), this.getDefaultKickMessage(punishment.getType()));
        String appealUrl = this.plugin.getAppealUrl();
        return MessageUtils.formatPunishmentMessage(template, punishment, appealUrl);
    }

    private String getActionBarMessage(Punishment punishment) {
        String messageKey = "actionbar." + punishment.getType().name().toLowerCase();
        String template = this.plugin.getMessage(messageKey, this.getDefaultActionBarMessage(punishment.getType()));
        return MessageUtils.formatPunishmentMessage(template, punishment);
    }

    private String getWarnMessage(Punishment punishment) {
        String template = this.plugin.getMessage("screen.warn", "&e\u26a0 &6You have been warned!\n&7Reason: &f{reason}\n&7Staff: &f{staff}\n&7Date: &f{date}");
        return MessageUtils.formatPunishmentMessage(template, punishment);
    }

    private String getDefaultKickMessage(PunishmentType type) {
        return switch (type) {
            case PunishmentType.BAN -> "&c&l\u26a0 YOU ARE BANNED \u26a0\n\n&7Reason: &f{reason}\n&7Staff: &f{staff}\n&7Date: &f{date}\n&7Appeal: &f{appeal-url}";
            case PunishmentType.TEMPBAN -> "&e&l\u23f0 TEMPORARY BAN \u23f0\n\n&7Reason: &f{reason}\n&7Staff: &f{staff}\n&7Expires: &f{expires}\n&7Appeal: &f{appeal-url}";
            case PunishmentType.KICK -> "&c&l\u26a1 KICKED \u26a1\n\n&7Reason: &f{reason}\n&7Staff: &f{staff}\n&7Appeal: &f{appeal-url}";
            default -> "&cYou have been punished: {reason}\n&7Appeal: &f{appeal-url}";
        };
    }

    private String getDefaultActionBarMessage(PunishmentType type) {
        return switch (type) {
            case PunishmentType.MUTE -> "&6\ud83d\udd07 You are muted: &f{reason}";
            case PunishmentType.TEMPMUTE -> "&6\ud83d\udd07 Muted: &f{reason} &7| &fExpires: {time-left}";
            default -> "&7You are punished: {reason}";
        };
    }

    public void shutdown() {
        this.executorService.shutdown();
        this.asyncExecutor.shutdown();
        try {
            if (!this.executorService.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.executorService.shutdownNow();
            }
            if (!this.asyncExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.asyncExecutor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.executorService.shutdownNow();
            this.asyncExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    public CompletableFuture<Boolean> isBanned(UUID playerUuid) {
        if (this.cacheEnabled && this.banCache.containsKey(playerUuid)) {
            return CompletableFuture.completedFuture(this.banCache.get(playerUuid));
        }
        return this.database.isPlayerBanned(playerUuid);
    }

    public CompletableFuture<Boolean> isPlayerBanned(UUID playerUuid) {
        return this.isBanned(playerUuid);
    }

    public CompletableFuture<Boolean> isMuted(UUID playerUuid) {
        if (this.cacheEnabled && this.muteCache.containsKey(playerUuid)) {
            return CompletableFuture.completedFuture(this.muteCache.get(playerUuid));
        }
        return this.database.isPlayerMuted(playerUuid);
    }

    public CompletableFuture<Boolean> isPlayerMuted(UUID playerUuid) {
        return this.isMuted(playerUuid);
    }

    public CompletableFuture<List<Punishment>> getActivePunishments(UUID playerUuid) {
        return this.database.getActivePunishments(playerUuid);
    }

    public CompletableFuture<List<Punishment>> getPunishmentHistory(UUID playerUuid) {
        return this.database.getPunishmentHistory(playerUuid);
    }

    public CompletableFuture<Boolean> canPunish(UUID staffUuid, UUID targetUuid) {
        return CompletableFuture.supplyAsync(() -> {
            Player staff = Bukkit.getPlayer((UUID)staffUuid);
            Player target = Bukkit.getPlayer((UUID)targetUuid);
            if (target == null) {
                return true;
            }
            if (target.hasPermission("weguardian.bypass") && !staff.hasPermission("weguardian.bypass.override")) {
                return false;
            }
            if (target.hasPermission("weguardian.admin") && !staff.hasPermission("weguardian.owner")) {
                return false;
            }
            if (target.hasPermission("weguardian.moderator") && !staff.hasPermission("weguardian.admin")) {
                return false;
            }
            return true;
        }, this.asyncExecutor);
    }

    public CompletableFuture<Boolean> canPunish(CommandSender staff, UUID targetUuid) {
        return CompletableFuture.supplyAsync(() -> {
            Player target = Bukkit.getPlayer((UUID)targetUuid);
            if (target == null) {
                return true;
            }
            if (target.hasPermission("weguardian.bypass")) {
                return false;
            }
            if (target.hasPermission("weguardian.admin") && !staff.hasPermission("weguardian.owner")) {
                return false;
            }
            return true;
        }, this.asyncExecutor);
    }

    public CompletableFuture<Boolean> canPunish(CommandSender staff, String targetName) {
        Player target = Bukkit.getPlayer((String)targetName);
        if (target == null) {
            return CompletableFuture.completedFuture(true);
        }
        if (target.hasPermission("weguardian.bypass")) {
            return CompletableFuture.completedFuture(false);
        }
        if (target.hasPermission("weguardian.admin") && !staff.hasPermission("weguardian.owner")) {
            return CompletableFuture.completedFuture(false);
        }
        return CompletableFuture.completedFuture(true);
    }

    public CompletableFuture<UUID> getPlayerUUID(String playerName) {
        return this.executePlayerLookup(playerName);
    }

    public CompletableFuture<Boolean> banPlayer(CommandSender sender, UUID targetUuid, String reason) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                if (!this.canPunish(sender, targetUuid).join().booleanValue()) {
                    return false;
                }
                String targetName = this.getPlayerName(targetUuid).join();
                if (targetName == null) {
                    return false;
                }
                if (this.plugin.getDatabaseManager().isPlayerBanned(targetUuid).join().booleanValue()) {
                    sender.sendMessage(MessageUtils.colorize("&cPlayer " + targetName + " is already banned."));
                    return false;
                }
                return this.ban(targetName, sender.getName(), reason).join();
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Error in banPlayer: " + e.getMessage());
                return false;
            }
        }, this.asyncExecutor);
    }

    public CompletableFuture<Boolean> tempbanPlayer(CommandSender sender, UUID targetUuid, String reason, String duration) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                if (!this.canPunish(sender, targetUuid).join().booleanValue()) {
                    return false;
                }
                String targetName = this.getPlayerName(targetUuid).join();
                if (targetName == null) {
                    return false;
                }
                if (this.plugin.getDatabaseManager().isPlayerBanned(targetUuid).join().booleanValue()) {
                    sender.sendMessage(MessageUtils.colorize("&cPlayer " + targetName + " is already banned."));
                    return false;
                }
                return this.tempban(targetName, sender.getName(), reason, duration).join();
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Error in tempbanPlayer: " + e.getMessage());
                return false;
            }
        }, this.asyncExecutor);
    }

    public CompletableFuture<Boolean> mutePlayer(CommandSender sender, UUID targetUuid, String reason) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                if (!this.canPunish(sender, targetUuid).join().booleanValue()) {
                    return false;
                }
                String targetName = this.getPlayerName(targetUuid).join();
                if (targetName == null) {
                    return false;
                }
                if (this.plugin.getDatabaseManager().isPlayerMuted(targetUuid).join().booleanValue()) {
                    sender.sendMessage(MessageUtils.colorize("&cPlayer " + targetName + " is already muted."));
                    return false;
                }
                return this.mute(targetName, sender.getName(), reason).join();
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Error in mutePlayer: " + e.getMessage());
                return false;
            }
        }, this.asyncExecutor);
    }

    public CompletableFuture<Boolean> tempmutePlayer(CommandSender sender, UUID targetUuid, String reason, String duration) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                if (!this.canPunish(sender, targetUuid).join().booleanValue()) {
                    return false;
                }
                String targetName = this.getPlayerName(targetUuid).join();
                if (targetName == null) {
                    return false;
                }
                if (this.plugin.getDatabaseManager().isPlayerMuted(targetUuid).join().booleanValue()) {
                    sender.sendMessage(MessageUtils.colorize("&cPlayer " + targetName + " is already muted."));
                    return false;
                }
                return this.tempmute(targetName, sender.getName(), reason, duration).join();
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Error in tempmutePlayer: " + e.getMessage());
                return false;
            }
        }, this.asyncExecutor);
    }

    public CompletableFuture<Boolean> kickPlayer(CommandSender sender, UUID targetUuid, String reason) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                if (!this.canPunish(sender, targetUuid).join().booleanValue()) {
                    return false;
                }
                String targetName = this.getPlayerName(targetUuid).join();
                if (targetName == null) {
                    return false;
                }
                return this.kick(targetName, sender.getName(), reason).join();
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Error in kickPlayer: " + e.getMessage());
                return false;
            }
        }, this.asyncExecutor);
    }

    public CompletableFuture<Boolean> warnPlayer(CommandSender sender, UUID targetUuid, String reason) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                if (!this.canPunish(sender, targetUuid).join().booleanValue()) {
                    return false;
                }
                String targetName = this.getPlayerName(targetUuid).join();
                if (targetName == null) {
                    return false;
                }
                return this.warn(targetName, sender.getName(), reason).join();
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Error in warnPlayer: " + e.getMessage());
                return false;
            }
        }, this.asyncExecutor);
    }

    public CompletableFuture<Boolean> unbanPlayer(CommandSender sender, UUID targetUuid, String reason) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                if (!this.canPunish(sender, targetUuid).join().booleanValue()) {
                    return false;
                }
                String targetName = this.getPlayerName(targetUuid).join();
                if (targetName == null) {
                    return false;
                }
                return this.unban(targetName, sender.getName()).join();
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Error in unbanPlayer: " + e.getMessage());
                return false;
            }
        }, this.asyncExecutor);
    }

    public CompletableFuture<Boolean> unmutePlayer(CommandSender sender, UUID targetUuid, String reason) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                if (!this.canPunish(sender, targetUuid).join().booleanValue()) {
                    return false;
                }
                String targetName = this.getPlayerName(targetUuid).join();
                if (targetName == null) {
                    return false;
                }
                return this.unmute(targetName, sender.getName()).join();
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Error in unmutePlayer: " + e.getMessage());
                return false;
            }
        }, this.asyncExecutor);
    }

    public CompletableFuture<Boolean> banPlayer(UUID targetUuid, String targetName, String staffName, String reason) {
        return this.ban(targetName, staffName, reason);
    }

    public CompletableFuture<Boolean> tempbanPlayer(UUID targetUuid, String targetName, String staffName, String reason, String duration) {
        return this.tempban(targetName, staffName, reason, duration);
    }

    public CompletableFuture<Boolean> mutePlayer(UUID targetUuid, String targetName, String staffName, String reason) {
        return this.mute(targetName, staffName, reason);
    }

    public CompletableFuture<Boolean> tempmutePlayer(UUID targetUuid, String targetName, String staffName, String reason, String duration) {
        return this.tempmute(targetName, staffName, reason, duration);
    }

    public CompletableFuture<Boolean> kickPlayer(UUID targetUuid, String targetName, String staffName, String reason) {
        return this.kick(targetName, staffName, reason);
    }

    public CompletableFuture<Boolean> warnPlayer(UUID targetUuid, String targetName, String staffName, String reason) {
        return this.warn(targetName, staffName, reason);
    }

    public CompletableFuture<Boolean> unbanPlayer(UUID targetUuid, String targetName, String staffName) {
        return this.unban(targetName, staffName);
    }

    public CompletableFuture<Boolean> unmutePlayer(UUID targetUuid, String targetName, String staffName) {
        return this.unmute(targetName, staffName);
    }

    public CompletableFuture<Boolean> executeBan(CommandSender sender, String targetName, String reason, String duration, boolean silent, String templateName) {
        return CompletableFuture.supplyAsync(() -> {
            if (!sender.hasPermission("weguardian.ban")) {
                sender.sendMessage(MessageUtils.colorize("&cYou don't have permission to use this command."));
                return false;
            }
            if (templateName != null && !templateName.isEmpty()) {
                return (Boolean)((CompletableFuture)this.plugin.getTemplateService().executeTemplate(sender, targetName, templateName, duration).exceptionally(throwable -> {
                    this.plugin.getLogger().severe("Error executing template: " + throwable.getMessage());
                    sender.sendMessage(MessageUtils.colorize("&cError executing template: " + throwable.getMessage()));
                    return false;
                })).join();
            }
            UUID targetUuid = this.getPlayerUUID(targetName).join();
            if (targetUuid == null) {
                sender.sendMessage(MessageUtils.colorize("&cPlayer '" + targetName + "' not found."));
                return false;
            }
            if (this.plugin.getDatabaseManager().isPlayerBanned(targetUuid).join().booleanValue()) {
                sender.sendMessage(MessageUtils.colorize("&cPlayer " + targetName + " is already banned."));
                return false;
            }
            if (sender instanceof Player && !this.canPunish((Player)sender, targetUuid).join().booleanValue()) {
                sender.sendMessage(MessageUtils.colorize("&cYou cannot punish this player."));
                return false;
            }
            String staffName = sender instanceof Player ? sender.getName() : "Console";
            return duration != null && !duration.isEmpty() ? this.tempban(targetName, staffName, reason, duration).join() : this.ban(targetName, staffName, reason).join();
        }).exceptionally(throwable -> {
            this.plugin.getLogger().severe("Error executing ban: " + throwable.getMessage());
            sender.sendMessage(MessageUtils.colorize("&cAn error occurred while executing the ban."));
            return false;
        });
    }

    public CompletableFuture<Boolean> executeKick(CommandSender sender, String targetName, String reason, boolean silent, String templateName, boolean ipKick) {
        return CompletableFuture.supplyAsync(() -> {
            if (!sender.hasPermission("weguardian.kick")) {
                sender.sendMessage(MessageUtils.colorize("&cYou don't have permission to use this command."));
                return false;
            }
            if (templateName != null && !templateName.isEmpty()) {
                return (Boolean)((CompletableFuture)this.plugin.getTemplateService().executeTemplate(sender, targetName, templateName, null).exceptionally(throwable -> {
                    this.plugin.getLogger().severe("Error executing template: " + throwable.getMessage());
                    sender.sendMessage(MessageUtils.colorize("&cError executing template: " + throwable.getMessage()));
                    return false;
                })).join();
            }
            UUID targetUuid = this.getPlayerUUID(targetName).join();
            if (targetUuid == null) {
                sender.sendMessage(MessageUtils.colorize("&cPlayer '" + targetName + "' not found."));
                return false;
            }
            if (sender instanceof Player && !this.canPunish((Player)sender, targetUuid).join().booleanValue()) {
                sender.sendMessage(MessageUtils.colorize("&cYou cannot punish this player."));
                return false;
            }
            String staffName = sender instanceof Player ? sender.getName() : "Console";
            return this.kick(targetName, staffName, reason).join();
        }).exceptionally(throwable -> {
            this.plugin.getLogger().severe("Error executing kick: " + throwable.getMessage());
            sender.sendMessage(MessageUtils.colorize("&cAn error occurred while executing the kick."));
            return false;
        });
    }

    public CompletableFuture<Boolean> executeMute(CommandSender sender, String targetName, String reason, String duration, boolean silent, String templateName, boolean ipMute) {
        return CompletableFuture.supplyAsync(() -> {
            if (!sender.hasPermission("weguardian.mute")) {
                sender.sendMessage(MessageUtils.colorize("&cYou don't have permission to use this command."));
                return false;
            }
            if (templateName != null && !templateName.isEmpty()) {
                return (Boolean)((CompletableFuture)this.plugin.getTemplateService().executeTemplate(sender, targetName, templateName, duration).exceptionally(throwable -> {
                    this.plugin.getLogger().severe("Error executing template: " + throwable.getMessage());
                    sender.sendMessage(MessageUtils.colorize("&cError executing template: " + throwable.getMessage()));
                    return false;
                })).join();
            }
            UUID targetUuid = this.getPlayerUUID(targetName).join();
            if (targetUuid == null) {
                sender.sendMessage(MessageUtils.colorize("&cPlayer '" + targetName + "' not found."));
                return false;
            }
            if (this.plugin.getDatabaseManager().isPlayerMuted(targetUuid).join().booleanValue()) {
                sender.sendMessage(MessageUtils.colorize("&cPlayer " + targetName + " is already muted."));
                return false;
            }
            if (sender instanceof Player && !this.canPunish((Player)sender, targetUuid).join().booleanValue()) {
                sender.sendMessage(MessageUtils.colorize("&cYou cannot punish this player."));
                return false;
            }
            String staffName = sender instanceof Player ? sender.getName() : "Console";
            return duration != null && !duration.isEmpty() ? this.tempmute(targetName, staffName, reason, duration).join() : this.mute(targetName, staffName, reason).join();
        }).exceptionally(throwable -> {
            this.plugin.getLogger().severe("Error executing mute: " + throwable.getMessage());
            sender.sendMessage(MessageUtils.colorize("&cAn error occurred while executing the mute."));
            return false;
        });
    }

    public CompletableFuture<Boolean> executeWarn(CommandSender sender, String targetName, String reason, boolean silent, String templateName) {
        return CompletableFuture.supplyAsync(() -> {
            if (!sender.hasPermission("weguardian.warn")) {
                sender.sendMessage(MessageUtils.colorize("&cYou don't have permission to use this command."));
                return false;
            }
            if (templateName != null && !templateName.isEmpty()) {
                return (Boolean)((CompletableFuture)this.plugin.getTemplateService().executeTemplate(sender, targetName, templateName, null).exceptionally(throwable -> {
                    this.plugin.getLogger().severe("Error executing template: " + throwable.getMessage());
                    sender.sendMessage(MessageUtils.colorize("&cError executing template: " + throwable.getMessage()));
                    return false;
                })).join();
            }
            UUID targetUuid = this.getPlayerUUID(targetName).join();
            if (targetUuid == null) {
                sender.sendMessage(MessageUtils.colorize("&cPlayer '" + targetName + "' not found."));
                return false;
            }
            if (sender instanceof Player && !this.canPunish((Player)sender, targetUuid).join().booleanValue()) {
                sender.sendMessage(MessageUtils.colorize("&cYou cannot punish this player."));
                return false;
            }
            String staffName = sender instanceof Player ? sender.getName() : "Console";
            return this.warn(targetName, staffName, reason).join();
        }).exceptionally(throwable -> {
            this.plugin.getLogger().severe("Error executing warn: " + throwable.getMessage());
            sender.sendMessage(MessageUtils.colorize("&cAn error occurred while executing the warn."));
            return false;
        });
    }

    public CompletableFuture<Boolean> executeUnban(CommandSender sender, String targetName) {
        return CompletableFuture.supplyAsync(() -> {
            if (!sender.hasPermission("weguardian.unban")) {
                sender.sendMessage(MessageUtils.colorize("&cYou don't have permission to use this command."));
                return false;
            }
            UUID targetUuid = this.getPlayerUUID(targetName).join();
            if (targetUuid == null) {
                sender.sendMessage(MessageUtils.colorize("&cPlayer '" + targetName + "' not found."));
                return false;
            }
            if (!this.plugin.getDatabaseManager().isPlayerBanned(targetUuid).join().booleanValue()) {
                sender.sendMessage(MessageUtils.colorize("&c" + targetName + " is not currently banned."));
                return false;
            }
            String staffName = sender instanceof Player ? sender.getName() : "Console";
            return this.unban(targetName, staffName).join();
        }).exceptionally(throwable -> {
            this.plugin.getLogger().severe("Error executing unban: " + throwable.getMessage());
            sender.sendMessage(MessageUtils.colorize("&cAn error occurred while executing the unban."));
            return false;
        });
    }

    public CompletableFuture<Boolean> executeUnmute(CommandSender sender, String targetName) {
        return CompletableFuture.supplyAsync(() -> {
            if (!sender.hasPermission("weguardian.unmute")) {
                sender.sendMessage(MessageUtils.colorize("&cYou don't have permission to use this command."));
                return false;
            }
            UUID targetUuid = this.getPlayerUUID(targetName).join();
            if (targetUuid == null) {
                sender.sendMessage(MessageUtils.colorize("&cPlayer '" + targetName + "' not found."));
                return false;
            }
            if (!this.plugin.getDatabaseManager().isPlayerMuted(targetUuid).join().booleanValue()) {
                sender.sendMessage(MessageUtils.colorize("&c" + targetName + " is not currently muted."));
                return false;
            }
            String staffName = sender instanceof Player ? sender.getName() : "Console";
            return this.unmute(targetName, staffName).join();
        }).exceptionally(throwable -> {
            this.plugin.getLogger().severe("Error executing unmute: " + throwable.getMessage());
            sender.sendMessage(MessageUtils.colorize("&cAn error occurred while executing the unmute."));
            return false;
        });
    }

    public CompletableFuture<Boolean> canPunish(Player staff, UUID targetUuid) {
        return CompletableFuture.supplyAsync(() -> {
            Player target = Bukkit.getPlayer((UUID)targetUuid);
            if (target == null) {
                return true;
            }
            if (target.hasPermission("weguardian.bypass") && !staff.hasPermission("weguardian.bypass.override")) {
                return false;
            }
            if (target.hasPermission("weguardian.admin") && !staff.hasPermission("weguardian.owner")) {
                return false;
            }
            if (target.hasPermission("weguardian.moderator") && !staff.hasPermission("weguardian.admin")) {
                return false;
            }
            return true;
        }, this.asyncExecutor);
    }
}

