/*
 * Decompiled with CFR 0.152.
 */
package net.draycia.carbon.common.users;

import carbonchat.libs.com.google.inject.Inject;
import carbonchat.libs.org.checkerframework.checker.nullness.qual.NonNull;
import carbonchat.libs.org.checkerframework.checker.nullness.qual.Nullable;
import carbonchat.libs.org.checkerframework.framework.qual.DefaultQualifier;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.TypeAdapter;
import com.google.gson.reflect.TypeToken;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import net.draycia.carbon.common.users.ProfileCache;
import net.draycia.carbon.common.users.ProfileResolver;
import net.draycia.carbon.common.util.ConcurrentUtil;
import net.draycia.carbon.common.util.FastUuidSansHyphens;
import org.apache.logging.log4j.Logger;

@DefaultQualifier(value=NonNull.class)
public final class MojangProfileResolver
implements ProfileResolver {
    private final HttpClient client;
    private final Gson gson;
    private final ExecutorService executorService;
    private final Map<String, CompletableFuture<@Nullable BasicLookupResponse>> pendingUuidLookups = new HashMap<String, CompletableFuture<BasicLookupResponse>>();
    private final Map<UUID, CompletableFuture<@Nullable BasicLookupResponse>> pendingUsernameLookups = new HashMap<UUID, CompletableFuture<BasicLookupResponse>>();
    private final ProfileCache cache;
    private final RateLimiter globalRateLimit;
    private final RateLimiter uuidToProfileRateLimit;

    @Inject
    private MojangProfileResolver(Logger logger, ProfileCache cache) {
        this.client = HttpClient.newHttpClient();
        this.gson = new GsonBuilder().registerTypeAdapter(UUID.class, (Object)new UUIDTypeAdapter()).create();
        this.executorService = Executors.newFixedThreadPool(2, ConcurrentUtil.carbonThreadFactory(logger, "MojangProfileResolver"));
        this.cache = cache;
        this.globalRateLimit = new RateLimiter(600);
        this.uuidToProfileRateLimit = new RateLimiter(200);
    }

    @Override
    public synchronized CompletableFuture<@Nullable UUID> resolveUUID(String username, boolean cacheOnly) {
        if (username.length() > 25 || username.length() < 1) {
            return CompletableFuture.completedFuture(null);
        }
        if (cacheOnly || this.cache.hasCachedEntry(username)) {
            return CompletableFuture.completedFuture(this.cache.cachedId(username));
        }
        return this.pendingUuidLookups.computeIfAbsent(username, $ -> {
            if (!this.globalRateLimit.canSubmit()) {
                return CompletableFuture.completedFuture(null);
            }
            CompletableFuture<@Nullable BasicLookupResponse> mojangLookup = CompletableFuture.supplyAsync(() -> {
                try {
                    HttpRequest request = MojangProfileResolver.createRequest("https://api.mojang.com/users/profiles/minecraft/" + username);
                    return this.sendRequest(request);
                }
                catch (Exception e) {
                    throw new RuntimeException("Exception resolving UUID for name " + username, e);
                }
            }, this.executorService);
            mojangLookup.whenComplete((result, $$$) -> {
                MojangProfileResolver mojangProfileResolver = this;
                synchronized (mojangProfileResolver) {
                    this.cache.cache(result == null ? null : result.id(), username);
                    this.pendingUuidLookups.remove(username);
                }
            });
            return mojangLookup;
        }).thenApply(response -> {
            if (response == null) {
                return null;
            }
            return response.id();
        });
    }

    @Override
    public synchronized CompletableFuture<@Nullable String> resolveName(UUID uuid, boolean cacheOnly) {
        if (cacheOnly || this.cache.hasCachedEntry(uuid)) {
            return CompletableFuture.completedFuture(this.cache.cachedName(uuid));
        }
        return this.pendingUsernameLookups.computeIfAbsent(uuid, $ -> {
            boolean nameLimited;
            boolean globalLimited = !this.globalRateLimit.canSubmit();
            boolean bl = nameLimited = !this.uuidToProfileRateLimit.canSubmit();
            if (globalLimited || nameLimited) {
                if (nameLimited && !globalLimited) {
                    this.globalRateLimit.available.getAndIncrement();
                }
                return CompletableFuture.completedFuture(null);
            }
            CompletableFuture<@Nullable BasicLookupResponse> mojangLookup = CompletableFuture.supplyAsync(() -> {
                try {
                    HttpRequest request = MojangProfileResolver.createRequest("https://api.mojang.com/user/profile/" + uuid.toString().replace("-", ""));
                    return this.sendRequest(request);
                }
                catch (Exception e) {
                    throw new RuntimeException("Exception resolving name for UUID " + String.valueOf(uuid), e);
                }
            }, this.executorService);
            mojangLookup.whenComplete((result, $$$) -> {
                MojangProfileResolver mojangProfileResolver = this;
                synchronized (mojangProfileResolver) {
                    this.cache.cache(uuid, result == null ? null : result.name());
                    this.pendingUsernameLookups.remove(uuid);
                }
            });
            return mojangLookup;
        }).thenApply(response -> {
            if (response == null) {
                return null;
            }
            return response.name();
        });
    }

    private static HttpRequest createRequest(String uri) throws URISyntaxException {
        return HttpRequest.newBuilder().uri(new URI(uri)).GET().build();
    }

    private @Nullable BasicLookupResponse sendRequest(HttpRequest request) throws IOException, InterruptedException {
        HttpResponse<String> response = this.client.send(request, HttpResponse.BodyHandlers.ofString());
        if (response == null) {
            throw new RuntimeException("Null response for request " + String.valueOf(request));
        }
        if (response.statusCode() == 429) {
            throw new RuntimeException("Got rate-limited by Mojang, could not fulfill request: " + String.valueOf(request));
        }
        if (response.statusCode() == 404) {
            return null;
        }
        if (response.statusCode() == 400) {
            return null;
        }
        if (response.statusCode() != 200) {
            throw new RuntimeException("Received non-200 response code (" + response.statusCode() + ") for request " + String.valueOf(request) + ": " + response.body());
        }
        BasicLookupResponse basicLookupResponse = (BasicLookupResponse)this.gson.fromJson(response.body(), new TypeToken<BasicLookupResponse>(this){}.getType());
        if (basicLookupResponse == null) {
            throw new RuntimeException("Malformed response body for request " + String.valueOf(request) + ": '" + response.body() + "'");
        }
        return basicLookupResponse;
    }

    @Override
    public void shutdown() {
        ConcurrentUtil.shutdownExecutor(this.executorService, TimeUnit.MILLISECONDS, 500L);
        this.globalRateLimit.shutdown();
        this.uuidToProfileRateLimit.shutdown();
    }

    private static final class UUIDTypeAdapter
    extends TypeAdapter<UUID> {
        private UUIDTypeAdapter() {
        }

        public void write(JsonWriter out, UUID value) throws IOException {
            out.value(FastUuidSansHyphens.toString(value));
        }

        public UUID read(JsonReader in) throws IOException {
            String input = in.nextString();
            return FastUuidSansHyphens.parseUuid(input);
        }
    }

    private static final class RateLimiter {
        private final AtomicInteger available;
        private final Timer timer = new Timer("CarbonChat " + String.valueOf(this));

        private RateLimiter(final int perTenMinutes) {
            this.available = new AtomicInteger(perTenMinutes);
            this.timer.scheduleAtFixedRate(new TimerTask(){

                @Override
                public void run() {
                    available.set(perTenMinutes);
                }
            }, 0L, Duration.ofMinutes(10L).toMillis());
        }

        boolean canSubmit() {
            return this.available.getAndDecrement() >= 0;
        }

        void shutdown() {
            this.timer.cancel();
        }
    }

    private record BasicLookupResponse(UUID id, String name) {
    }
}

