package com.github.tartaricacid.touhoulittlemaid.entity.ai.brain.task;

import com.github.tartaricacid.touhoulittlemaid.entity.passive.EntityMaid;
import com.github.tartaricacid.touhoulittlemaid.entity.passive.MaidPathFindingBFS;
import com.github.tartaricacid.touhoulittlemaid.init.InitEntities;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import net.minecraft.Util;
import net.minecraft.core.BlockPos;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.ai.behavior.Behavior;
import net.minecraft.world.entity.ai.behavior.BehaviorUtils;
import net.minecraft.world.entity.ai.memory.MemoryModuleType;
import net.minecraft.world.entity.ai.memory.MemoryStatus;

import java.util.List;
import java.util.function.Predicate;

public class MaidPickupEntitiesTask extends Behavior<EntityMaid> {
    private static long GLOBAL_NEXT_SCHEDULE_AT = 0;

    private final Predicate<EntityMaid> predicate;
    private final float speedModifier;
    private long nextScheduledAt = 0;

    public MaidPickupEntitiesTask(float speedModifier) {
        this(Predicates.alwaysTrue(), speedModifier);
    }

    public MaidPickupEntitiesTask(Predicate<EntityMaid> predicate, float speedModifier) {
        super(ImmutableMap.of(InitEntities.VISIBLE_PICKUP_ENTITIES.get(), MemoryStatus.VALUE_PRESENT,
                MemoryModuleType.WALK_TARGET, MemoryStatus.VALUE_ABSENT));
        this.predicate = predicate;
        this.speedModifier = speedModifier;
    }

    @Override
    protected boolean checkExtraStartConditions(ServerLevel worldIn, EntityMaid owner) {
        return owner.isTame() && nextScheduledAt <= Util.getMillis() && owner.canBrainMoving() && predicate.test(owner);
    }

    @Override
    protected void start(ServerLevel worldIn, EntityMaid maid, long gameTimeIn) {
        List<Entity> items = this.getItems(maid);
        var pathFinding = new MaidPathFindingBFS(maid.getNavigation().getNodeEvaluator(), worldIn, maid);
        long millis = Util.getMillis();
        // 在不同的女仆的拾取任务之间对齐时间间隔，来尽可能发挥相邻点缓存的作用
        nextScheduledAt = GLOBAL_NEXT_SCHEDULE_AT;
        if (millis >= nextScheduledAt) {
            // 如果没有其他女仆的拾取计划，那么安排在三秒后
            GLOBAL_NEXT_SCHEDULE_AT = millis + 3000;
            nextScheduledAt = GLOBAL_NEXT_SCHEDULE_AT;
        }
        for (Entity entity : items) {
            BlockPos blockPos = entity.blockPosition();
            if (maid.isWithinRestriction(blockPos)
                && entity.isAlive()
                && !entity.isInWater()
                && pathFinding.canPathReach(blockPos)) {
                BehaviorUtils.setWalkAndLookTargetMemories(maid, entity, this.speedModifier, 0);
                // 如果成功，那么下一次计划立刻进行，方便进行连续拾取
                nextScheduledAt = 0;
                break;
            }
        }
        pathFinding.finish();
    }

    private List<Entity> getItems(EntityMaid maid) {
        return maid.getBrain().getMemory(InitEntities.VISIBLE_PICKUP_ENTITIES.get()).orElse(Lists.newArrayList());
    }
}
