/*
 * Decompiled with CFR 0.152.
 */
package com.github.tartaricacid.touhoulittlemaid.world.backups;

import com.github.tartaricacid.touhoulittlemaid.TouhouLittleMaid;
import com.github.tartaricacid.touhoulittlemaid.config.ServerConfig;
import com.github.tartaricacid.touhoulittlemaid.entity.passive.EntityMaid;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import net.minecraft.core.BlockPos;
import net.minecraft.core.HolderLookup;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.NbtAccounter;
import net.minecraft.nbt.NbtIo;
import net.minecraft.nbt.NbtUtils;
import net.minecraft.nbt.Tag;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.level.Level;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public final class MaidBackupsManager {
    private static final String BACKUPS_FOLDER_NAME = "maid_backups";
    private static final String INDEX_FILE_NAME = "index.dat";
    private static final String BACKUP_FILE_EXTENSION = ".dat";
    private static final ExecutorService EXECUTOR_SERVICE = Executors.newSingleThreadExecutor(r -> {
        Thread thread = new Thread(r, "Maid-Backups-Thread");
        thread.setDaemon(true);
        thread.setUncaughtExceptionHandler((t, e) -> TouhouLittleMaid.LOGGER.error("Uncaught exception in maid backup thread: {}", (Object)t.getName(), (Object)e));
        return thread;
    });
    private static final DateTimeFormatter BACKUP_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd-HH-mm-ss");

    private MaidBackupsManager() {
    }

    public static void save(@NotNull MinecraftServer server, @NotNull EntityMaid maid) {
        UUID ownerId = maid.getOwnerUUID();
        if (ownerId == null) {
            return;
        }
        ServerLevel overWorld = server.getLevel(Level.OVERWORLD);
        if (overWorld == null) {
            TouhouLittleMaid.LOGGER.error("Cannot access overworld for maid backup: {}", (Object)maid.getStringUUID());
            return;
        }
        try {
            BackupData backupData = MaidBackupsManager.createBackupData(maid, overWorld, ownerId);
            if (backupData != null) {
                MaidBackupsManager.executeBackupAsync(backupData);
            }
        }
        catch (Exception e) {
            TouhouLittleMaid.LOGGER.error("Failed to initiate backup for maid: {}", (Object)maid.getStringUUID(), (Object)e);
        }
    }

    public static CompoundTag getMaidIndex(ServerPlayer player) {
        UUID ownerId = player.getUUID();
        MinecraftServer server = player.getServer();
        if (server == null) {
            return new CompoundTag();
        }
        ServerLevel overWorld = server.getLevel(Level.OVERWORLD);
        if (overWorld == null) {
            TouhouLittleMaid.LOGGER.error("Cannot access overworld for maid index retrieval: {}", (Object)ownerId);
            return new CompoundTag();
        }
        Path ownerFolder = overWorld.getDataStorage().dataFolder.toPath().resolve(BACKUPS_FOLDER_NAME).resolve(ownerId.toString());
        File indexFile = ownerFolder.resolve(INDEX_FILE_NAME).toFile();
        return MaidBackupsManager.loadExistingIndexData(indexFile);
    }

    @Nullable
    private static BackupData createBackupData(@NotNull EntityMaid maid, @NotNull ServerLevel overWorld, @NotNull UUID ownerId) {
        try {
            Path saveFolder = MaidBackupsManager.buildBackupFolderPath(maid, overWorld, ownerId);
            String saveFileName = MaidBackupsManager.generateBackupFileName();
            CompoundTag entityData = new CompoundTag();
            boolean saveResult = maid.saveAsPassenger(entityData);
            if (!saveResult) {
                TouhouLittleMaid.LOGGER.warn("Failed to serialize maid data for backup: {}", (Object)maid.getStringUUID());
                return null;
            }
            CompoundTag indexData = MaidBackupsManager.createIndexData(maid);
            return new BackupData(saveFolder, saveFileName, entityData, indexData, maid.getStringUUID());
        }
        catch (Exception e) {
            TouhouLittleMaid.LOGGER.error("Error creating backup data for maid: {}", (Object)maid.getStringUUID(), (Object)e);
            return null;
        }
    }

    @NotNull
    private static CompoundTag createIndexData(@NotNull EntityMaid maid) {
        CompoundTag indexData = new CompoundTag();
        indexData.putString("Name", Component.Serializer.toJson((Component)maid.getName(), (HolderLookup.Provider)maid.level.registryAccess()));
        indexData.putString("Dimension", maid.level.dimension().location().toString());
        indexData.put("Pos", NbtUtils.writeBlockPos((BlockPos)maid.blockPosition()));
        indexData.putLong("Timestamp", System.currentTimeMillis());
        return indexData;
    }

    public static Map<UUID, IndexData> getMaidIndexMap(ServerPlayer player) {
        CompoundTag indexTag = MaidBackupsManager.getMaidIndex(player);
        HashMap map = Maps.newHashMap();
        for (String s : indexTag.getAllKeys()) {
            CompoundTag maidTag = indexTag.getCompound(s);
            UUID maidUuid = UUID.fromString(s);
            MutableComponent name = Component.Serializer.fromJson((String)maidTag.getString("Name"), (HolderLookup.Provider)player.level.registryAccess());
            BlockPos pos = NbtUtils.readBlockPos((CompoundTag)maidTag, (String)"Pos").orElse(BlockPos.ZERO);
            String dimension = maidTag.getString("Dimension");
            long timestamp = maidTag.getLong("Timestamp");
            map.put(maidUuid, new IndexData((Component)name, pos, dimension, timestamp));
        }
        return map;
    }

    public static List<String> getMaidBackupFiles(ServerPlayer player, UUID maidUuid) {
        ArrayList backupFiles = Lists.newArrayList();
        Path folderPath = player.serverLevel().getDataStorage().dataFolder.toPath().resolve(BACKUPS_FOLDER_NAME).resolve(player.getUUID().toString()).resolve(maidUuid.toString());
        if (!Files.isDirectory(folderPath, new LinkOption[0])) {
            return backupFiles;
        }
        File[] files = folderPath.toFile().listFiles((dir, name) -> name.endsWith(BACKUP_FILE_EXTENSION));
        if (files == null) {
            return backupFiles;
        }
        for (File file : files) {
            backupFiles.add(file.getName());
        }
        backupFiles.sort(Comparator.reverseOrder());
        return backupFiles;
    }

    public static CompoundTag getMaidBackFile(ServerPlayer player, UUID maidUuid, String fileName) {
        Path filePath = player.serverLevel().getDataStorage().dataFolder.toPath().resolve(BACKUPS_FOLDER_NAME).resolve(player.getUUID().toString()).resolve(maidUuid.toString()).resolve(fileName);
        File file = filePath.toFile();
        if (!file.exists() || !file.isFile()) {
            return new CompoundTag();
        }
        try {
            return NbtIo.readCompressed((Path)filePath, (NbtAccounter)NbtAccounter.create((long)0x6400000L));
        }
        catch (IOException e) {
            TouhouLittleMaid.LOGGER.error("Failed to read maid backup file: {}", (Object)filePath, (Object)e);
            return new CompoundTag();
        }
    }

    private static void executeBackupAsync(@NotNull BackupData backupData) {
        CompletableFuture.runAsync(() -> MaidBackupsManager.performBackup(backupData), EXECUTOR_SERVICE).exceptionally(error -> {
            TouhouLittleMaid.LOGGER.error("Async backup failed for maid: {}", (Object)backupData.maidUuid, error);
            return null;
        });
    }

    private static void performBackup(@NotNull BackupData backupData) {
        try {
            if (!MaidBackupsManager.ensureDirectoryExists(backupData.saveFolder)) {
                return;
            }
            if (!MaidBackupsManager.updateIndexFile(backupData)) {
                return;
            }
            MaidBackupsManager.saveEntityData(backupData);
        }
        catch (Exception e) {
            TouhouLittleMaid.LOGGER.error("Backup operation failed for maid: {}", (Object)backupData.maidUuid, (Object)e);
        }
    }

    private static boolean ensureDirectoryExists(@NotNull Path directory) {
        if (Files.isDirectory(directory, new LinkOption[0])) {
            return true;
        }
        try {
            Files.createDirectories(directory, new FileAttribute[0]);
            return true;
        }
        catch (IOException e) {
            TouhouLittleMaid.LOGGER.error("Failed to create backup directory: {}", (Object)directory, (Object)e);
            return false;
        }
    }

    private static boolean updateIndexFile(@NotNull BackupData backupData) {
        File indexFile = backupData.saveFolder.getParent().resolve(INDEX_FILE_NAME).toFile();
        try {
            CompoundTag allIndexData = MaidBackupsManager.loadExistingIndexData(indexFile);
            allIndexData.put(backupData.saveFolder.getFileName().toString(), (Tag)backupData.indexData);
            NbtIo.writeCompressed((CompoundTag)allIndexData, (Path)indexFile.toPath());
            return true;
        }
        catch (IOException e) {
            TouhouLittleMaid.LOGGER.error("Failed to update index file: {}", (Object)indexFile, (Object)e);
            return false;
        }
    }

    @NotNull
    private static CompoundTag loadExistingIndexData(@NotNull File indexFile) {
        if (!indexFile.exists()) {
            return new CompoundTag();
        }
        try {
            return NbtIo.readCompressed((Path)indexFile.toPath(), (NbtAccounter)NbtAccounter.create((long)0x6400000L));
        }
        catch (IOException e) {
            TouhouLittleMaid.LOGGER.warn("Failed to read existing index file, creating new one: {}", (Object)indexFile, (Object)e);
            return new CompoundTag();
        }
    }

    private static void saveEntityData(@NotNull BackupData backupData) {
        File backupFile = backupData.saveFolder.resolve(backupData.saveFileName).toFile();
        try {
            NbtUtils.addCurrentDataVersion((CompoundTag)backupData.entityData);
            NbtIo.writeCompressed((CompoundTag)backupData.entityData, (Path)backupFile.toPath());
            MaidBackupsManager.removeOldBackups(backupData.saveFolder, (Integer)ServerConfig.MAID_BACKUP_MAX_COUNT.get());
        }
        catch (IOException e) {
            TouhouLittleMaid.LOGGER.error("Failed to save entity data to: {}", (Object)backupFile, (Object)e);
        }
    }

    private static void removeOldBackups(@NotNull Path backupFolder, int maxBackups) {
        try {
            File[] backupFiles = backupFolder.toFile().listFiles((dir, name) -> name.endsWith(BACKUP_FILE_EXTENSION));
            if (backupFiles == null || backupFiles.length <= maxBackups) {
                return;
            }
            Arrays.sort(backupFiles, Comparator.comparing(File::getName));
            for (int i = 0; i < backupFiles.length - maxBackups; ++i) {
                if (backupFiles[i].delete()) continue;
                TouhouLittleMaid.LOGGER.warn("Failed to delete old backup file: {}", (Object)backupFiles[i].getAbsolutePath());
            }
        }
        catch (Exception e) {
            TouhouLittleMaid.LOGGER.error("Error while removing old backups in folder: {}", (Object)backupFolder, (Object)e);
        }
    }

    @NotNull
    private static Path buildBackupFolderPath(@NotNull EntityMaid maid, @NotNull ServerLevel level, @NotNull UUID ownerId) {
        return level.getDataStorage().dataFolder.toPath().resolve(BACKUPS_FOLDER_NAME).resolve(ownerId.toString()).resolve(maid.getStringUUID());
    }

    @NotNull
    public static String generateBackupFileName() {
        return LocalDateTime.now().format(BACKUP_TIME_FORMATTER) + BACKUP_FILE_EXTENSION;
    }

    private record BackupData(Path saveFolder, String saveFileName, CompoundTag entityData, CompoundTag indexData, String maidUuid) {
    }

    public record IndexData(Component name, BlockPos pos, String dimension, long timestamp) {
    }
}

