package com.github.tartaricacid.touhoulittlemaid.network.message;

import com.github.tartaricacid.touhoulittlemaid.advancements.maid.TriggerType;
import com.github.tartaricacid.touhoulittlemaid.api.event.MaidTaskEnableEvent;
import com.github.tartaricacid.touhoulittlemaid.api.task.IMaidTask;
import com.github.tartaricacid.touhoulittlemaid.entity.passive.EntityMaid;
import com.github.tartaricacid.touhoulittlemaid.entity.passive.TabIndex;
import com.github.tartaricacid.touhoulittlemaid.entity.task.TaskManager;
import com.github.tartaricacid.touhoulittlemaid.init.InitTrigger;
import com.github.tartaricacid.touhoulittlemaid.inventory.container.task.TaskConfigContainer;
import io.netty.buffer.ByteBuf;
import net.minecraft.network.codec.ByteBufCodecs;
import net.minecraft.network.codec.StreamCodec;
import net.minecraft.network.protocol.common.custom.CustomPacketPayload;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.Entity;
import net.neoforged.neoforge.common.NeoForge;
import net.neoforged.neoforge.network.handling.IPayloadContext;

import static com.github.tartaricacid.touhoulittlemaid.util.ResourceLocationUtil.getResourceLocation;

public record MaidTaskPackage(int id, ResourceLocation uid) implements CustomPacketPayload {
    public static final CustomPacketPayload.Type<MaidTaskPackage> TYPE = new CustomPacketPayload.Type<>(getResourceLocation("maid_task"));
    public static final StreamCodec<ByteBuf, MaidTaskPackage> STREAM_CODEC = StreamCodec.composite(
            ByteBufCodecs.VAR_INT,
            MaidTaskPackage::id,
            ResourceLocation.STREAM_CODEC,
            MaidTaskPackage::uid,
            MaidTaskPackage::new
    );

    @Override
    public Type<? extends CustomPacketPayload> type() {
        return TYPE;
    }

    public static void handle(MaidTaskPackage message, IPayloadContext context) {
        if (context.flow().isServerbound()) {
            context.enqueueWork(() -> {
                ServerPlayer sender = (ServerPlayer) context.player();
                Entity entity = sender.level.getEntity(message.id);
                if (entity instanceof EntityMaid maid && maid.isOwnedBy(sender)) {
                    IMaidTask task = TaskManager.findTask(message.uid).orElse(TaskManager.getIdleTask());
                    if (task != TaskManager.getIdleTask() && NeoForge.EVENT_BUS.post(new MaidTaskEnableEvent(task, maid)).isCanceled()) {
                        return;
                    }
                    if (!task.isEnable(maid)) {
                        return;
                    }
                    maid.setTask(task);
                    if (!TaskManager.getIdleTask().equals(task) && maid.getOwner() instanceof ServerPlayer serverPlayer) {
                        InitTrigger.MAID_EVENT.get().trigger(serverPlayer, TriggerType.SWITCH_TASK);
                    }
                    // 如果此时玩家打开的是配置界面
                    if (sender.containerMenu instanceof TaskConfigContainer) {
                        maid.openMaidGui(sender, TabIndex.TASK_CONFIG);
                    }
                }
            });
        }
    }
}
