package com.skrrtn.velocity.webapi;

import com.velocitypowered.api.event.Subscribe;
import com.velocitypowered.api.event.proxy.ProxyInitializeEvent;
import com.velocitypowered.api.event.proxy.ProxyShutdownEvent;
import com.velocitypowered.api.event.connection.PostLoginEvent;
import com.velocitypowered.api.event.connection.DisconnectEvent;
import com.velocitypowered.api.plugin.Plugin;
import com.velocitypowered.api.proxy.Player;
import com.velocitypowered.api.proxy.ProxyServer;
import com.velocitypowered.api.plugin.annotation.DataDirectory;
import org.slf4j.Logger;

import javax.inject.Inject;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpHandler;
import com.sun.net.httpserver.HttpServer;

@Plugin(id = "velocity-web-api", name = "Velocity Web API", version = "1.4.0-RELEASE", authors = {"SKRRTN"})
public class VelocityWebApiPlugin {
    private final ProxyServer proxyServer;
    private final Logger logger;
    private HttpServer httpServer;
    private java.util.concurrent.ExecutorService httpExecutor;
    private final Object serverLock = new Object();
    private com.velocitypowered.api.scheduler.ScheduledTask updateTask;
    private final Path dataDirectory;
    private int port = 25576;
    private boolean checkUpdates = true;
    private boolean snapshotUpdates = false;
    private boolean debugEnabled = false;
    private String allowedOrigins = "*";
    private String bindAddress = "0.0.0.0"; // default to all interfaces
    private final Map<UUID, Long> sessionStarts = new ConcurrentHashMap<>();
    private String lastAnnouncedVersion = null;
    private String latestVersionFromHangar = null;
    private String latestDownloadUrl = null;
    private boolean latestIsSnapshot = false;

    @Inject
    public VelocityWebApiPlugin(ProxyServer proxyServer, Logger logger, @DataDirectory Path dataDirectory) {
        this.proxyServer = proxyServer;
        this.logger = logger;
        this.dataDirectory = dataDirectory;

        try {
            Files.createDirectories(dataDirectory);
            Path cfg = dataDirectory.resolve("config.cfg");
            Properties props = new Properties();
            if (!Files.exists(cfg)) {
                props.setProperty("port", Integer.toString(port));
                props.setProperty("checkUpdates", "true");
                props.setProperty("snapshotUpdates", "false");
                props.setProperty("debugEnabled", Boolean.toString(debugEnabled));
                props.setProperty("allowedOrigins", allowedOrigins);
                props.setProperty("bindAddress", bindAddress);
                try {
                    StringBuilder sb = new StringBuilder();
                    sb.append("###########################################################################\n");
                    sb.append("#    /$$      /$$ /$$$$$$$$ /$$$$$$$         /$$$$$$  /$$$$$$$  /$$$$$$   #\n");
                    sb.append("#   | $$  /$ | $$| $$_____/| $$__  $$       /$$__  $$| $$__  $$|_  $$_/   #\n");
                    sb.append("#   | $$ /$$$| $$| $$      | $$  \\ $$      | $$  \\ $$| $$  \\ $$  | $$     #\n");
                    sb.append("#   | $$/$$ $$ $$| $$$$$   | $$$$$$$       | $$$$$$$$| $$$$$$$/  | $$     #\n");
                    sb.append("#   | $$$$_  $$$$| $$__/   | $$__  $$      | $$__  $$| $$____/   | $$     #\n");
                    sb.append("#   | $$$/ \\  $$$| $$      | $$  \\ $$      | $$  | $$| $$/       | $$/    #\n");
                    sb.append("#   | $$/   \\  $$| $$$$$$$$| $$$$$$$/      | $$  | $$| $$       /$$$$$$   #\n");
                    sb.append("#   |__/     \\__/|________/|_______/       |__/  |__/|__/      |______/   #\n");
                    sb.append("###########################################################################\n\n");
                    sb.append("## Port the server will run on\n");
                    sb.append("port=").append(props.getProperty("port", Integer.toString(port))).append("\n\n");
                    sb.append("## Interface/address to bind HTTP(S) server (e.g. 127.0.0.1, 0.0.0.0)\n");
                    sb.append("bindAddress=").append(props.getProperty("bindAddress", bindAddress)).append("\n\n");
                    sb.append("## Enable or disable automatic update checks\n");
                    sb.append("checkUpdates=").append(props.getProperty("checkUpdates", Boolean.toString(checkUpdates))).append("\n\n");

                    sb.append("## Enable or disable SNAPSHOT update checks\n");
                    sb.append("snapshotUpdates=").append(props.getProperty("snapshotUpdates", Boolean.toString(snapshotUpdates))).append("\n\n");
                    sb.append("## Enable debug messages (true/false)\n");
                    sb.append("debugEnabled=").append(props.getProperty("debugEnabled", Boolean.toString(debugEnabled))).append("\n\n");
                    sb.append("## CORS: Allowed origins for HTTP API access (comma-separated, or * for all)\n");
                    sb.append("## Examples: * | https://example.com | https://example.com,https://other.com\n");
                    sb.append("allowedOrigins=").append(props.getProperty("allowedOrigins", allowedOrigins)).append("\n");

                    // Append any other properties not explicitly listed above
                    for (String name : props.stringPropertyNames()) {
                        if (name.equals("port") || name.equals("bindAddress") || name.equals("checkUpdates") || name.equals("snapshotUpdates") || name.equals("debugEnabled") || name.equals("allowedOrigins")) continue;
                        sb.append(name).append("=").append(props.getProperty(name)).append("\n");
                    }

                    Files.writeString(cfg, sb.toString(), StandardCharsets.UTF_8);
                } catch (IOException e) {
                    logger.warn("Velocity Web API: failed to write default config", e);
                }
            } else {
                try (InputStream in = Files.newInputStream(cfg)) {
                    props.load(in);
                }

                boolean dirty = false;

                String pstr = props.getProperty("port");
                if (pstr != null) {
                    try {
                        int p = Integer.parseInt(pstr.trim());
                        if (p > 0) port = p;
                    } catch (NumberFormatException ignored) {
                    }
                }

                // Ensure checkUpdates exists in existing config; set default and mark dirty if missing
                String checkUpdatesStr = props.getProperty("checkUpdates");
                if (checkUpdatesStr == null) {
                    props.setProperty("checkUpdates", Boolean.toString(checkUpdates));
                    dirty = true;
                } else {
                    checkUpdates = Boolean.parseBoolean(checkUpdatesStr.trim());
                }

                // Ensure snapshotUpdates exists in existing config; set default and mark dirty if missing
                String snapshotUpdatesStr = props.getProperty("snapshotUpdates");
                if (snapshotUpdatesStr == null) {
                    props.setProperty("snapshotUpdates", Boolean.toString(snapshotUpdates));
                    dirty = true;
                } else {
                    snapshotUpdates = Boolean.parseBoolean(snapshotUpdatesStr.trim());
                }

                // Ensure debugEnabled exists in existing config; set default and mark dirty if missing
                String debugEnabledStr = props.getProperty("debugEnabled");
                if (debugEnabledStr == null) {
                    props.setProperty("debugEnabled", Boolean.toString(debugEnabled));
                    dirty = true;
                } else {
                    debugEnabled = Boolean.parseBoolean(debugEnabledStr.trim());
                }

                // Ensure allowedOrigins exists in existing config; set default and mark dirty if missing
                String allowedOriginsStr = props.getProperty("allowedOrigins");
                if (allowedOriginsStr == null) {
                    props.setProperty("allowedOrigins", allowedOrigins);
                    dirty = true;
                } else {
                    allowedOrigins = allowedOriginsStr.trim();
                }

                // Ensure bindAddress exists; set default and mark dirty if missing
                String bindAddressStr = props.getProperty("bindAddress");
                if (bindAddressStr == null) {
                    props.setProperty("bindAddress", bindAddress);
                    dirty = true;
                } else {
                    bindAddress = bindAddressStr.trim();
                }

                // If we added defaults for missing keys, persist the updated config back to disk
                if (dirty) {
                    try {
                        StringBuilder sb = new StringBuilder();
                        sb.append("###########################################################################\n");
                        sb.append("#    /$$      /$$ /$$$$$$$$ /$$$$$$$         /$$$$$$  /$$$$$$$  /$$$$$$   #\n");
                        sb.append("#   | $$  /$ | $$| $$_____/| $$__  $$       /$$__  $$| $$__  $$|_  $$_/   #\n");
                        sb.append("#   | $$ /$$$| $$| $$      | $$  \\ $$      | $$  \\ $$| $$  \\ $$  | $$     #\n");
                        sb.append("#   | $$/$$ $$ $$| $$$$$   | $$$$$$$       | $$$$$$$$| $$$$$$$/  | $$     #\n");
                        sb.append("#   | $$$$_  $$$$| $$__/   | $$__  $$      | $$__  $$| $$____/   | $$     #\n");
                        sb.append("#   | $$$/ \\  $$$| $$      | $$  \\ $$      | $$  | $$| $$/       | $$/    #\n");
                        sb.append("#   | $$/   \\  $$| $$$$$$$$| $$$$$$$/      | $$  | $$| $$       /$$$$$$   #\n");
                        sb.append("#   |__/     \\__/|________/|_______/       |__/  |__/|__/      |______/   #\n");
                        sb.append("###########################################################################\n\n");
                        sb.append("## Port the server will run on\n");
                        sb.append("port=").append(props.getProperty("port", Integer.toString(port))).append("\n\n");
                        sb.append("## Interface/address to bind HTTP(S) server (e.g. 127.0.0.1, 0.0.0.0)\n");
                        sb.append("bindAddress=").append(props.getProperty("bindAddress", bindAddress)).append("\n\n");
                        sb.append("## Enable or disable automatic update checks\n");
                        sb.append("checkUpdates=").append(props.getProperty("checkUpdates", Boolean.toString(checkUpdates))).append("\n\n");

                        sb.append("## Enable or disable SNAPSHOT update checks\n");
                        sb.append("snapshotUpdates=").append(props.getProperty("snapshotUpdates", Boolean.toString(snapshotUpdates))).append("\n\n");
                        sb.append("## Enable debug messages (true/false)\n");
                        sb.append("debugEnabled=").append(props.getProperty("debugEnabled", Boolean.toString(debugEnabled))).append("\n\n");
                        sb.append("## CORS: Allowed origins for HTTP API access (comma-separated, or * for all)\n");
                        sb.append("## Examples: * | https://example.com | https://example.com,https://other.com\n");
                        sb.append("allowedOrigins=").append(props.getProperty("allowedOrigins", allowedOrigins)).append("\n");

                        for (String name : props.stringPropertyNames()) {
                            if (name.equals("port") || name.equals("bindAddress") || name.equals("checkUpdates") || name.equals("snapshotUpdates") || name.equals("debugEnabled") || name.equals("allowedOrigins")) continue;
                            sb.append(name).append("=").append(props.getProperty(name)).append("\n");
                        }

                        Files.writeString(cfg, sb.toString(), StandardCharsets.UTF_8);
                        logger.info("Velocity Web API: updated config with new default properties");
                    } catch (IOException e) {
                        logger.warn("Velocity Web API: failed to write updated config", e);
                    }
                }
            }
        } catch (IOException e) {
            logger.warn("Failed to initialize config or data directory", e);
        }
    }

    @Subscribe
    public void onProxyInitialization(ProxyInitializeEvent ev) {
        // Initial update check (do this before starting the HTTP server so the
        // startup banner can include update information fetched from Hangar)
        if (checkUpdates) {
            checkForUpdates(true);

            // Periodic check every 2 hours
            updateTask = proxyServer.getScheduler()
                .buildTask(this, () -> checkForUpdates(false))
                .delay(2, java.util.concurrent.TimeUnit.HOURS)
                .repeat(2, java.util.concurrent.TimeUnit.HOURS)
                .schedule();
        }

        try {
            startHttpServer();
            debug("Velocity Web API HTTP server started on port {}", port);
            if (debugEnabled) {
                logger.info("Velocity Web API: debug logging is ENABLED — detailed debug messages will be shown in the console");
            }
            // Register /vwapi command (supports reload)
            try {
                com.velocitypowered.api.command.CommandManager cmd = proxyServer.getCommandManager();
                // Register a simple command: /vwapi reload using the non-deprecated method
                try {
                    com.velocitypowered.api.command.CommandMeta meta = cmd.metaBuilder("vwapi").build();
                    cmd.register(meta, (com.velocitypowered.api.command.SimpleCommand) invocation -> {
                        com.velocitypowered.api.command.CommandSource source = invocation.source();
                        String[] args = invocation.arguments();
                        // Restrict commands to console or users with vwapi.admin permission
                        boolean allowed = false;
                        try {
                            allowed = source.hasPermission("vwapi.admin");
                        } catch (Throwable ignored) {}
                        if (!allowed) {
                            try {
                                String cls = source.getClass().getSimpleName().toLowerCase();
                                if (cls.contains("console")) allowed = true;
                            } catch (Throwable ignored) {}
                        }
                        if (!allowed) {
                            try { source.sendMessage(net.kyori.adventure.text.Component.text("You do not have permission to run this command (requires vwapi.admin).")); } catch (Throwable ignored) {}
                            return;
                        }
                        try {
                            if (args.length >= 1) {
                                String sub = args[0].toLowerCase();
                                switch (sub) {
                                    case "reload" -> {
                                        try { source.sendMessage(net.kyori.adventure.text.Component.text("Velocity Web API: reloading configuration...")); } catch (Exception ignored) {}

                                        int oldPort = port;
                                        String oldBind = bindAddress;

                                        HttpServer oldServer;
                                        java.util.concurrent.ExecutorService oldExec;
                                        synchronized (serverLock) {
                                            oldServer = httpServer;
                                            oldExec = httpExecutor;
                                        }

                                        reloadConfig(false);

                                        boolean needRestart = (port != oldPort) || (bindAddress != null && !bindAddress.equals(oldBind));

                                        // Manage updateTask if checkUpdates changed
                                        synchronized (serverLock) {
                                            if (checkUpdates && updateTask == null) {
                                                updateTask = proxyServer.getScheduler()
                                                        .buildTask(this, () -> checkForUpdates(false))
                                                        .delay(2, java.util.concurrent.TimeUnit.HOURS)
                                                        .repeat(2, java.util.concurrent.TimeUnit.HOURS)
                                                        .schedule();
                                            } else if (!checkUpdates && updateTask != null) {
                                                try { updateTask.cancel(); } catch (Exception ignored) {}
                                                updateTask = null;
                                            }
                                        }

                                        if (!needRestart) {
                                            // No network change — just report success
                                            try { source.sendMessage(net.kyori.adventure.text.Component.text("Velocity Web API: configuration reloaded (no server restart required).")); } catch (Exception ignored) {}
                                        } else {
                                            try { source.sendMessage(net.kyori.adventure.text.Component.text("Velocity Web API: bind/port changed — restarting HTTP server (old server will be stopped)...")); } catch (Exception ignored) {}

                                            HttpServer candidate;
                                            java.util.concurrent.ExecutorService candidateExec;
                                            // Stop old server first (downtime acceptable)
                                            synchronized (serverLock) {
                                                if (oldServer != null) {
                                                    try { oldServer.stop(0); } catch (Exception ignored) {}
                                                }
                                                if (oldExec != null) {
                                                    try { oldExec.shutdownNow(); } catch (Exception ignored) {}
                                                }
                                                httpServer = null;
                                                httpExecutor = null;
                                            }

                                            try {
                                                candidateExec = java.util.concurrent.Executors.newCachedThreadPool();
                                                candidate = HttpServer.create(new InetSocketAddress(bindAddress, port), 0);
                                                candidate.createContext("/", new RootHandler());
                                                candidate.createContext("/playerlist", new PlayersHandler());
                                                candidate.createContext("/count", new CountHandler());
                                                candidate.createContext("/player", new PlayerHandler());
                                                candidate.createContext("/servers", new ServersHandler());
                                                candidate.createContext("/health", new HealthHandler());
                                                candidate.setExecutor(candidateExec);
                                                candidate.start();

                                                synchronized (serverLock) {
                                                    httpServer = candidate;
                                                    httpExecutor = candidateExec;
                                                }

                                                logStartupBanner();
                                                try { source.sendMessage(net.kyori.adventure.text.Component.text("Velocity Web API: HTTP server restarted and now listening on " + bindAddress + ":" + port)); } catch (Exception ignored) {}
                                                logger.info("Velocity Web API: HTTP server restarted on {}:{}", bindAddress, port);
                                            } catch (IOException e) {
                                                logger.warn("Velocity Web API: failed to start HTTP server on new bind: {}", e.getMessage());
                                                // Attempt to restore previous server (best-effort)
                                                try {
                                                    java.util.concurrent.ExecutorService restoreExec = java.util.concurrent.Executors.newCachedThreadPool();
                                                    HttpServer restored = HttpServer.create(new InetSocketAddress(oldBind, oldPort), 0);
                                                    restored.createContext("/", new RootHandler());
                                                    restored.createContext("/playerlist", new PlayersHandler());
                                                    restored.createContext("/count", new CountHandler());
                                                    restored.createContext("/player", new PlayerHandler());
                                                    restored.createContext("/servers", new ServersHandler());
                                                    restored.createContext("/health", new HealthHandler());
                                                    restored.setExecutor(restoreExec);
                                                    restored.start();
                                                    synchronized (serverLock) {
                                                        httpServer = restored;
                                                        httpExecutor = restoreExec;
                                                    }
                                                    try { source.sendMessage(net.kyori.adventure.text.Component.text("Velocity Web API: failed to restart on new bind; restored previous server on " + oldBind + ":" + oldPort)); } catch (Exception ignored) {}
                                                    logger.info("Velocity Web API: restored previous HTTP server on {}:{}", oldBind, oldPort);
                                                } catch (IOException e2) {
                                                    logger.error("Velocity Web API: failed to restore previous HTTP server after restart failure", e2);
                                                    try { source.sendMessage(net.kyori.adventure.text.Component.text("Velocity Web API: critical error - could not start HTTP server: " + e2.getMessage())); } catch (Exception ignored) {}
                                                }
                                            }
                                        }
                                    }
                                    case "version" -> {
                                        try { source.sendMessage(net.kyori.adventure.text.Component.text("Velocity Web API version: " + getCurrentPluginVersion())); } catch (NoClassDefFoundError | Exception ignored) {}
                                    }
                                    case "debug" -> {
                                        // debug [on|off] or toggle
                                        if (args.length >= 2) {
                                            String val = args[1].toLowerCase();
                                            switch (val) {
                                                case "on", "true" -> debugEnabled = true;
                                                case "off", "false" -> debugEnabled = false;
                                                default -> {
                                                    try { source.sendMessage(net.kyori.adventure.text.Component.text("Usage: /vwapi debug [on|off]")); } catch (Exception ignored) {}
                                                    return;
                                                }
                                            }
                                        } else {
                                            // toggle
                                            debugEnabled = !debugEnabled;
                                        }
                                        // persist change
                                        saveConfig();
                                        try { source.sendMessage(net.kyori.adventure.text.Component.text("Velocity Web API: debugEnabled=" + debugEnabled)); } catch (Exception ignored) {}
                                    }
                                    case "status" -> {
                                        try {
                                            if (httpServer == null) {
                                                source.sendMessage(net.kyori.adventure.text.Component.text("Velocity Web API HTTP server: NOT running"));
                                            } else {
                                                java.net.InetSocketAddress a = httpServer.getAddress();
                                                String host = (a == null) ? "unknown" : a.getHostString();
                                                int p = (a == null) ? port : a.getPort();
                                                String displayHost = host;
                                                String urlHost = host;
                                                if ("0.0.0.0".equals(host) || "::".equals(host) || "0:0:0:0:0:0:0:0".equals(host)) {
                                                    displayHost = host + " (all interfaces)";
                                                    urlHost = "localhost";
                                                }
                                                String url = "http://" + urlHost + ":" + p + "/";
                                                source.sendMessage(net.kyori.adventure.text.Component.text("Velocity Web API HTTP server: running"));
                                                source.sendMessage(net.kyori.adventure.text.Component.text("Bound: " + displayHost + ":" + p));
                                                source.sendMessage(net.kyori.adventure.text.Component.text("URL: " + url));
                                            }
                                        } catch (NoClassDefFoundError | Exception ignored) {}
                                    }
                                    case "help" -> {
                                        try {
                                            source.sendMessage(net.kyori.adventure.text.Component.text("/vwapi reload — reload config.cfg"));
                                            source.sendMessage(net.kyori.adventure.text.Component.text("/vwapi version — show plugin version"));
                                            source.sendMessage(net.kyori.adventure.text.Component.text("/vwapi debug [on|off] — toggle or set debugEnabled (persists to config.cfg)"));
                                            source.sendMessage(net.kyori.adventure.text.Component.text("/vwapi status — show HTTP server status and access URL"));
                                            source.sendMessage(net.kyori.adventure.text.Component.text("/vwapi help — show this help"));
                                        } catch (NoClassDefFoundError | Exception ignored) {}
                                    }
                                    default -> {
                                        try { source.sendMessage(net.kyori.adventure.text.Component.text("Usage: /vwapi <reload|version|debug|status|help>")); } catch (Exception ignored) {}
                                    }
                                }
                            } else {
                                try { source.sendMessage(net.kyori.adventure.text.Component.text("Usage: /vwapi <reload|version|debug|status|help>")); } catch (Exception ignored) {}
                            }
                        } catch (Exception e) {
                            try { source.sendMessage(net.kyori.adventure.text.Component.text("/vwapi: an internal error occurred")); } catch (Exception ignored) {}
                        }
                    });
                } catch (Exception e) {
                    logger.warn("Failed to register /vwapi command", e);
                }
            } catch (NoSuchMethodError | Exception e) {
                logger.warn("Failed to register /vwapi command", e);
            }
        } catch (IOException e) {
            logger.error("Failed to start HTTP server", e);
        }
    }

    private void reloadConfig(boolean log) {
        Path cfg = dataDirectory.resolve("config.cfg");
        Properties props = new Properties();
        try (InputStream in = Files.newInputStream(cfg)) {
            props.load(in);
        } catch (IOException e) {
            logger.warn("Velocity Web API: failed to reload config", e);
            return;
        }

        String pstr = props.getProperty("port");
        if (pstr != null) {
            try {
                int p = Integer.parseInt(pstr.trim());
                if (p > 0) port = p;
            } catch (NumberFormatException ignored) {}
        }

        String checkUpdatesStr = props.getProperty("checkUpdates");
        if (checkUpdatesStr != null) checkUpdates = Boolean.parseBoolean(checkUpdatesStr.trim());

        String snapshotUpdatesStr = props.getProperty("snapshotUpdates");
        if (snapshotUpdatesStr != null) snapshotUpdates = Boolean.parseBoolean(snapshotUpdatesStr.trim());

        String debugEnabledStr = props.getProperty("debugEnabled");
        if (debugEnabledStr != null) debugEnabled = Boolean.parseBoolean(debugEnabledStr.trim());

        String allowedOriginsStr = props.getProperty("allowedOrigins");
        if (allowedOriginsStr != null) allowedOrigins = allowedOriginsStr.trim();

        String bindAddressStr = props.getProperty("bindAddress");
        if (bindAddressStr != null && !bindAddressStr.trim().isEmpty()) bindAddress = bindAddressStr.trim();

        if (log) logger.info("Velocity Web API: configuration reloaded");
    }

    private void saveConfig() {
        Path cfg = dataDirectory.resolve("config.cfg");
        Properties props = new Properties();
        try {
            if (Files.exists(cfg)) {
                try (InputStream in = Files.newInputStream(cfg)) {
                    props.load(in);
                }
            }
            props.setProperty("port", Integer.toString(port));
            props.setProperty("checkUpdates", Boolean.toString(checkUpdates));
            props.setProperty("snapshotUpdates", Boolean.toString(snapshotUpdates));
            props.setProperty("debugEnabled", Boolean.toString(debugEnabled));
            props.setProperty("allowedOrigins", allowedOrigins);
            props.setProperty("bindAddress", bindAddress);

            StringBuilder sb = new StringBuilder();
            sb.append("###########################################################################\n");
            sb.append("#    /$$      /$$ /$$$$$$$$ /$$$$$$$         /$$$$$$  /$$$$$$$  /$$$$$$   #\n");
            sb.append("#   | $$  /$ | $$| $$_____/| $$__  $$       /$$__  $$| $$__  $$|_  $$_/   #\n");
            sb.append("#   | $$ /$$$| $$| $$      | $$  \\ $$      | $$  \\ $$| $$  \\ $$  | $$     #\n");
            sb.append("#   | $$/$$ $$ $$| $$$$$   | $$$$$$$       | $$$$$$$$| $$$$$$$/  | $$     #\n");
            sb.append("#   | $$$$_  $$$$| $$__/   | $$__  $$      | $$__  $$| $$____/   | $$     #\n");
            sb.append("#   | $$$/ \\  $$$| $$      | $$  \\ $$      | $$  | $$| $$/       | $$/    #\n");
            sb.append("#   | $$/   \\  $$| $$$$$$$$| $$$$$$$/      | $$  | $$| $$       /$$$$$$   #\n");
            sb.append("#   |__/     \\__/|________/|_______/       |__/  |__/|__/      |______/   #\n");
            sb.append("###########################################################################\n\n");
            sb.append("## Port the server will run on\n");
            sb.append("port=").append(props.getProperty("port", Integer.toString(port))).append("\n\n");
            sb.append("## Interface/address to bind HTTP(S) server (e.g. 127.0.0.1, 0.0.0.0)\n");
            sb.append("bindAddress=").append(props.getProperty("bindAddress", bindAddress)).append("\n\n");
            sb.append("## Enable or disable automatic update checks\n");
            sb.append("checkUpdates=").append(props.getProperty("checkUpdates", Boolean.toString(checkUpdates))).append("\n\n");
            sb.append("## Enable or disable SNAPSHOT update checks\n");
            sb.append("snapshotUpdates=").append(props.getProperty("snapshotUpdates", Boolean.toString(snapshotUpdates))).append("\n\n");
            sb.append("## Enable debug messages (true/false)\n");
            sb.append("debugEnabled=").append(props.getProperty("debugEnabled", Boolean.toString(debugEnabled))).append("\n\n");
            sb.append("## CORS: Allowed origins for HTTP API access (comma-separated, or * for all)\n");
            sb.append("## Examples: * | https://example.com | https://example.com,https://other.com\n");
            sb.append("allowedOrigins=").append(props.getProperty("allowedOrigins", allowedOrigins)).append("\n");

            for (String name : props.stringPropertyNames()) {
                if (name.equals("port") || name.equals("bindAddress") || name.equals("checkUpdates") || name.equals("snapshotUpdates") || name.equals("debugEnabled") || name.equals("allowedOrigins")) continue;
                sb.append(name).append("=").append(props.getProperty(name)).append("\n");
            }

            Files.writeString(cfg, sb.toString(), StandardCharsets.UTF_8);
            logger.info("Velocity Web API: saved configuration to disk");
        } catch (IOException e) {
            logger.warn("Velocity Web API: failed to save config", e);
        }
    }

    @Subscribe
    public void onProxyShutdown(ProxyShutdownEvent ev) {
        synchronized (serverLock) {
            if (httpServer != null) {
                try { httpServer.stop(0); } catch (Exception ignored) {}
                httpServer = null;
            }
            if (httpExecutor != null) {
                try { httpExecutor.shutdownNow(); } catch (Exception ignored) {}
                httpExecutor = null;
            }
            if (updateTask != null) {
                try { updateTask.cancel(); } catch (Exception ignored) {}
                updateTask = null;
            }
            logger.info("Velocity Web API HTTP server stopped");
        }
    }

    @Subscribe
    public void onPostLogin(PostLoginEvent ev) {
        Player p = ev.getPlayer();
        sessionStarts.put(p.getUniqueId(), System.currentTimeMillis());
    }

    @Subscribe
    public void onDisconnect(DisconnectEvent ev) {
        try {
            sessionStarts.remove(ev.getPlayer().getUniqueId());
        } catch (Exception ignored) {}
    }

    private void logStartupBanner() {
        // Figlet-style ASCII art for "VELOCITY WEB API"
        String[] figlet = new String[]{
                "   /$$      /$$ /$$$$$$$$ /$$$$$$$         /$$$$$$  /$$$$$$$  /$$$$$$",
                "  | $$  /$ | $$| $$_____/| $$__  $$       /$$__  $$| $$__  $$|_  $$_/",
                "| $$ /$$$| $$| $$      | $$  \\ $$      | $$  \\ $$| $$  \\ $$  | $$",
                "| $$/$$ $$ $$| $$$$$   | $$$$$$$       | $$$$$$$$| $$$$$$$/  | $$",
                "| $$$$_  $$$$| $$__/   | $$__  $$      | $$__  $$| $$____/   | $$",
                "  | $$$/ \\  $$$| $$      | $$  \\ $$      | $$  | $$| $$/       | $$/",
                "  | $$/   \\  $$| $$$$$$$$| $$$$$$$/      | $$  | $$| $$       /$$$$$$",
                "  |__/     \\__/|________/|_______/       |__/  |__/|__/      |______/",
                "                                                                    "
        };

        // Colors: white text on light-blue background (#06bbd9)
        // ANSI color codes removed — logger output should remain plain text for portability

        // Build content lines: title (figlet), blank line, author, version, port
        String authorLine = "Author: SKRRTN";
        String version = getCurrentPluginVersion();
        String versionLine = "Version: " + version;
        String portLine = "Port: " + port;

        // Build update notification lines if update is available. These will
        // be printed outside the boxed banner to avoid breaking box styling.
        String updateLine1 = null;
        String updateLine2 = null;
        if (latestVersionFromHangar != null && compareVersions(latestVersionFromHangar, version) > 0
                && (snapshotUpdates || !latestIsSnapshot)) {
            updateLine1 = "⚠ UPDATE AVAILABLE: " + latestVersionFromHangar + " ⚠";
            if (latestDownloadUrl != null) {
                updateLine2 = "Download: " + latestDownloadUrl;
            }
        }

        // Log the full multi-line figlet inside a solid box using the logger only.
        int figletWidth = 0;
        for (String l : figlet) figletWidth = Math.max(figletWidth, l.length());
        int contentWidth = Math.max(Math.max(Math.max(figletWidth, authorLine.length()), versionLine.length()), portLine.length());
        String horiz = "═".repeat(contentWidth + 2);

        logger.info("╔{}╗", horiz);
        for (String l : figlet) {
            logger.info("║ {} ║", padCenter(l, contentWidth));
        }
        logger.info("║ {} ║", padCenter("", contentWidth));
        logger.info("║ {} ║", padCenter(authorLine, contentWidth));
        logger.info("║ {} ║", padCenter(versionLine, contentWidth));
        logger.info("║ {} ║", padCenter(portLine, contentWidth));
        // Close the box first
        logger.info("╚{}╝", horiz);

        // Print the update notification below the boxed banner with padding
        // so long URLs don't break the box styling.
        if (updateLine1 != null) {
            logger.info("\n\n\n");
            logger.warn(updateLine1);
            if (updateLine2 != null) {
                logger.warn(updateLine2);
            } else {
                logger.warn("Download: https://hangar.papermc.io/skrrtn/Velocity-Web-API");
            }
            logger.info("\n\n\n");
        }
    }

    private String padCenter(String s, int width) {
        if (s == null) s = "";
        if (s.length() >= width) return s;
        int pad = width - s.length();
        int left = pad / 2;
        int right = pad - left;
        return " ".repeat(left) + s + " ".repeat(right);
    }

    private String getCurrentPluginVersion() {
        try {
            Plugin ann = this.getClass().getAnnotation(Plugin.class);
            if (ann != null && ann.version() != null && !ann.version().isEmpty() && !ann.version().contains("${")) {
                return ann.version();
            }
        } catch (Exception ignored) {}

        // Fallback: try reading Maven-generated pom.properties from META-INF
        try (InputStream in = this.getClass().getClassLoader().getResourceAsStream("META-INF/maven/com.skrrtn/velocity-web-api/pom.properties")) {
            if (in != null) {
                Properties p = new Properties();
                p.load(in);
                String v = p.getProperty("version");
                if (v != null && !v.isEmpty()) return v;
            }
        } catch (Exception ignored) {}

        return "unknown";
    }

    private String fetchLatestVersionFromHangar() {
        String apiUrl = "https://hangar.papermc.io/api/v1/projects/skrrtn/Velocity-Web-API/versions";
        try {
            java.net.URL url = java.net.URI.create(apiUrl).toURL();
            java.net.HttpURLConnection conn = (java.net.HttpURLConnection) url.openConnection();
            conn.setRequestMethod("GET");
            conn.setConnectTimeout(5000);
            conn.setReadTimeout(5000);
            conn.setRequestProperty("Accept", "application/json");
            conn.setRequestProperty("User-Agent", "Velocity-Web-API/" + getCurrentPluginVersion());

            // reset snapshot flag and download url before attempting fetch
            latestIsSnapshot = false;
            latestDownloadUrl = null;

            int code = conn.getResponseCode();
            if (code != 200) {
                debug("Velocity Web API: Hangar version check returned HTTP {}", code);
                return null;
            }

            try (InputStream in = conn.getInputStream()) {
                String body = new String(in.readAllBytes(), StandardCharsets.UTF_8);
                // Parse: {"result":[{"name":"1.0.0","channel":{...},"downloads":{"VELOCITY":{"downloadUrl":"..."}},...}]}
                int resultIdx = body.indexOf("\"result\"");
                if (resultIdx < 0) return null;
                
                // Find first valid version (skip SNAPSHOT if snapshotUpdates is false)
                int searchPos = resultIdx;
                while (true) {
                    int nameIdx = body.indexOf("\"name\"", searchPos);
                    if (nameIdx < 0) return null;
                    
                    int colon = body.indexOf(':', nameIdx);
                    int quote1 = body.indexOf('"', colon + 1);
                    int quote2 = body.indexOf('"', quote1 + 1);
                    if (quote1 < 0 || quote2 < 0) return null;
                    
                    String versionName = body.substring(quote1 + 1, quote2);
                    
                    // Check if this is a snapshot release. The Hangar API places
                    // the "channel" object after the version "name", so search
                    // forward from the name index for the channel and its flags.
                    boolean isSnapshot = false;
                    int channelStart = body.indexOf("\"channel\"", nameIdx);
                    if (channelStart > nameIdx) {
                        int flagsIdx = body.indexOf("\"flags\"", channelStart);
                        if (flagsIdx > 0) {
                            int unstableIdx = body.indexOf("UNSTABLE", flagsIdx);
                            if (unstableIdx > 0) {
                                isSnapshot = true;
                            }
                        }
                    }
                    
                    if (!isSnapshot || snapshotUpdates) {
                        // Find download URL for this version
                        int downloadsIdx = body.indexOf("\"downloads\"", nameIdx);
                        if (downloadsIdx > 0) {
                            int velocityIdx = body.indexOf("\"VELOCITY\"", downloadsIdx);
                            if (velocityIdx > 0) {
                                int downloadUrlIdx = body.indexOf("\"downloadUrl\"", velocityIdx);
                                if (downloadUrlIdx > 0) {
                                    int urlColon = body.indexOf(':', downloadUrlIdx);
                                    int urlQuote1 = body.indexOf('"', urlColon + 1);
                                    int urlQuote2 = body.indexOf('"', urlQuote1 + 1);
                                    if (urlQuote1 > 0 && urlQuote2 > 0) {
                                        latestDownloadUrl = body.substring(urlQuote1 + 1, urlQuote2);
                                    }
                                }
                            }
                        }
                        latestIsSnapshot = isSnapshot;
                        return versionName;
                    }
                    
                    searchPos = quote2 + 1;
                }
            }
        } catch (IOException e) {
            debug("Velocity Web API: failed to fetch latest version from Hangar", e);
            return null;
        }
    }

    private int parseIntSafe(String s) {
        try {
            return Integer.parseInt(s.replaceAll("[^0-9]", ""));
        } catch (NumberFormatException e) {
            return 0;
        }
    }

    private int compareVersions(String a, String b) {
        if (a == null || b == null) return 0;
        String[] pa = a.split("\\.");
        String[] pb = b.split("\\.");
        int len = Math.max(pa.length, pb.length);
        for (int i = 0; i < len; i++) {
            int va = (i < pa.length) ? parseIntSafe(pa[i]) : 0;
            int vb = (i < pb.length) ? parseIntSafe(pb[i]) : 0;
            if (va != vb) return Integer.compare(va, vb);
        }
        return 0;
    }

    private void checkForUpdates(boolean startup) {
        String current = getCurrentPluginVersion();
        String latest = fetchLatestVersionFromHangar();
        if (latest == null) {
            if (startup) {
                logger.info("Velocity Web API: could not check for updates on Hangar.");
            } else {
                debug("Velocity Web API: periodic update check failed (Hangar).");
            }
            return;
        }

        latestVersionFromHangar = latest;

        if (compareVersions(latest, current) > 0) {
            if (!latest.equals(lastAnnouncedVersion)) {
                lastAnnouncedVersion = latest;
                if (!startup) {
                    // Don't announce snapshot updates when snapshots are disabled
                    if (!latestIsSnapshot || snapshotUpdates) {
                        logger.warn("A new version of Velocity Web API is available: {} (current: {}).", latest, current);
                        if (latestDownloadUrl != null) {
                            logger.warn("Download it from {}", latestDownloadUrl);
                        } else {
                            logger.warn("Download it from https://hangar.papermc.io/skrrtn/Velocity-Web-API");
                        }
                    } else {
                        debug("Velocity Web API: latest available version {} is a snapshot; snapshots are disabled, skipping notification.", latest);
                    }
                }
            }
        } else if (startup) {
            logger.info("Velocity Web API is up to date (version {}).", current);
        }
    }
    

    private void startHttpServer() throws IOException {
        httpServer = HttpServer.create(new InetSocketAddress(bindAddress, port), 0);
        httpServer.createContext("/", new RootHandler());
        httpServer.createContext("/playerlist", new PlayersHandler());
        httpServer.createContext("/count", new CountHandler());
        httpServer.createContext("/player", new PlayerHandler());
        httpServer.createContext("/servers", new ServersHandler());
        httpServer.createContext("/health", new HealthHandler());
        // Track executor so we can shut it down cleanly on restart/shutdown
        httpExecutor = java.util.concurrent.Executors.newCachedThreadPool();
        httpServer.setExecutor(httpExecutor);
        httpServer.start();
        logStartupBanner();
    }

    private void logHttpAccess(HttpExchange exchange) {
        if (!debugEnabled) return;
        String origin = "unknown";
        try {
            InetSocketAddress addr = exchange.getRemoteAddress();
            if (addr != null && addr.getAddress() != null) {
                origin = addr.getAddress().getHostAddress() + ":" + addr.getPort();
            }
        } catch (Exception ignored) {}

        String method = exchange.getRequestMethod();
        String path = exchange.getRequestURI().toString();
        String ua = "-";
        try {
            String a = exchange.getRequestHeaders().getFirst("User-Agent");
            if (a != null) ua = a;
        } catch (Exception ignored) {}

        debug("HTTP access from {} - {} {} - UA: {}", origin, method, path, ua);
    }

    private void debug(String msg, Object... args) {
        if (!debugEnabled) return;
        try {
            if (logger.isDebugEnabled()) {
                logger.debug(msg, args);
            } else {
                logger.info("[DEBUG] " + msg, args);
            }
        } catch (Exception e) {
            try {
                logger.info("[DEBUG] " + msg, args);
            } catch (Exception ignored) {}
        }
    }

    private void debug(String msg, Throwable t) {
        if (!debugEnabled) return;
        try {
            if (logger.isDebugEnabled()) {
                logger.debug(msg, t);
            } else {
                logger.info("[DEBUG] " + msg, t);
            }
        } catch (Exception e) {
            try {
                logger.info("[DEBUG] " + msg, t);
            } catch (Exception ignored) {}
        }
    }

    private void applyCorsHeaders(HttpExchange exchange) {
        String origin = exchange.getRequestHeaders().getFirst("Origin");
        
        if ("*".equals(allowedOrigins)) {
            // Allow all origins
            exchange.getResponseHeaders().add("Access-Control-Allow-Origin", "*");
            exchange.getResponseHeaders().add("Access-Control-Allow-Methods", "GET, POST, OPTIONS");
            exchange.getResponseHeaders().add("Access-Control-Allow-Headers", "Content-Type");
        } else if (origin != null && !allowedOrigins.isEmpty()) {
            // Check if the origin is in the allowed list
            String[] allowed = allowedOrigins.split(",");
            for (String allowedOrigin : allowed) {
                if (origin.equals(allowedOrigin.trim())) {
                    exchange.getResponseHeaders().add("Access-Control-Allow-Origin", origin);
                    exchange.getResponseHeaders().add("Access-Control-Allow-Methods", "GET, POST, OPTIONS");
                    exchange.getResponseHeaders().add("Access-Control-Allow-Headers", "Content-Type");
                    exchange.getResponseHeaders().add("Access-Control-Allow-Credentials", "true");
                    break;
                }
            }
        }
    }

    

    private String playersToJsonArrayFromPlayers(List<Player> players) {
        return players.stream()
                .map(p -> {
                    String serverName = "";
                    try {
                        var cur = p.getCurrentServer();
                        if (cur.isPresent()) serverName = cur.get().getServer().getServerInfo().getName();
                    } catch (Exception ignored) {}
                    return "{\"username\":\"" + p.getUsername().replace("\"", "\\\"")
                            + "\",\"uuid\":\"" + p.getUniqueId().toString()
                            + "\",\"currentServer\":\"" + serverName.replace("\"", "\\\"") + "\"}";
                })
                .collect(Collectors.joining(",", "[", "]"));
    }

    private class RootHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            applyCorsHeaders(exchange);
            
            if ("OPTIONS".equalsIgnoreCase(exchange.getRequestMethod())) {
                exchange.sendResponseHeaders(204, -1);
                return;
            }
            
            if (!"GET".equalsIgnoreCase(exchange.getRequestMethod())) {
                exchange.sendResponseHeaders(405, -1);
                return;
            }

            logHttpAccess(exchange);

            InputStream in = this.getClass().getClassLoader().getResourceAsStream("index.html");
            if (in == null) {
                String err = "<html><body><h1>500</h1><p>index.html not found in resources</p></body></html>";
                byte[] bytes = err.getBytes(StandardCharsets.UTF_8);
                exchange.getResponseHeaders().add("Content-Type", "text/html; charset=utf-8");
                exchange.sendResponseHeaders(500, bytes.length);
                try (OutputStream os = exchange.getResponseBody()) { os.write(bytes); }
                return;
            }

            byte[] bytes = in.readAllBytes();
            exchange.getResponseHeaders().add("Content-Type", "text/html; charset=utf-8");
            exchange.sendResponseHeaders(200, bytes.length);
            try (OutputStream os = exchange.getResponseBody()) {
                os.write(bytes);
            } finally {
                in.close();
            }
        }
    }

    private class PlayersHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            applyCorsHeaders(exchange);
            
            if ("OPTIONS".equalsIgnoreCase(exchange.getRequestMethod())) {
                exchange.sendResponseHeaders(204, -1);
                return;
            }
            
            if (!"GET".equalsIgnoreCase(exchange.getRequestMethod())) {
                exchange.sendResponseHeaders(405, -1);
                return;
            }

            logHttpAccess(exchange);

            List<Player> players = proxyServer.getAllPlayers().stream()
                    .collect(Collectors.toList());
            int count = players.size();

            String json = "{\"count\":" + count + ",\"players\":" + playersToJsonArrayFromPlayers(players) + "}";
            byte[] bytes = json.getBytes(StandardCharsets.UTF_8);
            exchange.getResponseHeaders().add("Content-Type", "application/json; charset=utf-8");
            exchange.sendResponseHeaders(200, bytes.length);
            try (OutputStream os = exchange.getResponseBody()) {
                os.write(bytes);
            }
        }
    }

    private class CountHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            applyCorsHeaders(exchange);
            
            if ("OPTIONS".equalsIgnoreCase(exchange.getRequestMethod())) {
                exchange.sendResponseHeaders(204, -1);
                return;
            }
            
            if (!"GET".equalsIgnoreCase(exchange.getRequestMethod())) {
                exchange.sendResponseHeaders(405, -1);
                return;
            }

            logHttpAccess(exchange);
            int count = proxyServer.getAllPlayers().size();
            String json = "{\"count\": " + count + "}";
            byte[] bytes = json.getBytes(StandardCharsets.UTF_8);
            exchange.getResponseHeaders().add("Content-Type", "application/json; charset=utf-8");
            exchange.sendResponseHeaders(200, bytes.length);
            try (OutputStream os = exchange.getResponseBody()) {
                os.write(bytes);
            }
        }
    }

    private class PlayerHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            applyCorsHeaders(exchange);
            
            if ("OPTIONS".equalsIgnoreCase(exchange.getRequestMethod())) {
                exchange.sendResponseHeaders(204, -1);
                return;
            }
            
            if (!"GET".equalsIgnoreCase(exchange.getRequestMethod())) {
                exchange.sendResponseHeaders(405, -1);
                return;
            }

            logHttpAccess(exchange);

            String path = exchange.getRequestURI().getPath();
            // path could be /player/name
            String name = null;
            if (path.startsWith("/player/")) {
                name = path.substring("/player/".length());
            } else {
                // try query param
                String query = exchange.getRequestURI().getQuery();
                if (query != null) {
                    for (String part : query.split("&")) {
                        String[] kv = part.split("=", 2);
                        if (kv.length == 2 && kv[0].equalsIgnoreCase("username")) {
                            name = java.net.URLDecoder.decode(kv[1], StandardCharsets.UTF_8);
                            break;
                        }
                    }
                }
            }

                if (name == null || name.isEmpty()) {
                exchange.sendResponseHeaders(400, -1);
                return;
            }

                // find player by exact username (case-sensitive)
                final String lookupName = name;
                Player target = proxyServer.getAllPlayers().stream()
                    .filter(p -> p.getUsername().equals(lookupName))
                    .findFirst().orElse(null);

            if (target == null) {
                exchange.sendResponseHeaders(404, -1);
                return;
            }

            UUID uuid = target.getUniqueId();
            String username = target.getUsername();
            String serverName = "";
            try {
                var current = target.getCurrentServer();
                if (current.isPresent()) {
                    serverName = current.get().getServer().getServerInfo().getName();
                }
            } catch (Exception ignored) {}

            long ping = -1;
            try {
                ping = target.getPing();
            } catch (Exception ignored) {}

            long sessionDurationSeconds = 0;
            Long start = sessionStarts.get(uuid);
            if (start != null) {
                sessionDurationSeconds = (System.currentTimeMillis() - start) / 1000L;
            }

            String json = "{"
                    + "\"username\":\"" + username + "\"," 
                    + "\"uuid\":\"" + uuid.toString() + "\"," 
                    + "\"currentServer\":\"" + serverName + "\"," 
                    + "\"ping\":" + ping + ","
                    + "\"sessionDurationSeconds\":" + sessionDurationSeconds
                    + "}";

            byte[] bytes = json.getBytes(StandardCharsets.UTF_8);
            exchange.getResponseHeaders().add("Content-Type", "application/json; charset=utf-8");
            exchange.sendResponseHeaders(200, bytes.length);
            try (OutputStream os = exchange.getResponseBody()) {
                os.write(bytes);
            }
        }
    }

    private class ServersHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            applyCorsHeaders(exchange);
            
            if ("OPTIONS".equalsIgnoreCase(exchange.getRequestMethod())) {
                exchange.sendResponseHeaders(204, -1);
                return;
            }
            
            if (!"GET".equalsIgnoreCase(exchange.getRequestMethod())) {
                exchange.sendResponseHeaders(405, -1);
                return;
            }

            logHttpAccess(exchange);
            // Build a map serverName -> count for players currently on a server
            var byServer = proxyServer.getAllPlayers().stream()
                    .map(p -> {
                        try {
                            var cur = p.getCurrentServer();
                            return cur.isPresent() ? cur.get().getServer().getServerInfo().getName() : "";
                        } catch (Exception e) {
                            return "";
                        }
                    })
                    .filter(s -> s != null && !s.isEmpty())
                    .collect(Collectors.groupingBy(s -> s, Collectors.counting()));

            // Ensure we include all registered servers (with zero if no players)
            var serverNames = proxyServer.getAllServers().stream()
                    .map(rs -> rs.getServerInfo().getName())
                    .collect(Collectors.toList());

            StringBuilder sb = new StringBuilder();
            sb.append('{');
            boolean first = true;
            for (String name : serverNames) {
                if (!first) sb.append(',');
                first = false;
                long cnt = byServer.getOrDefault(name, 0L);
                sb.append('"').append(name).append('"').append(':').append(cnt);
            }
            sb.append('}');

            byte[] bytes = sb.toString().getBytes(StandardCharsets.UTF_8);
            exchange.getResponseHeaders().add("Content-Type", "application/json; charset=utf-8");
            exchange.sendResponseHeaders(200, bytes.length);
            try (OutputStream os = exchange.getResponseBody()) {
                os.write(bytes);
            }
        }
    }

    

    private class HealthHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            applyCorsHeaders(exchange);
            
            if ("OPTIONS".equalsIgnoreCase(exchange.getRequestMethod())) {
                exchange.sendResponseHeaders(204, -1);
                return;
            }
            
            if (!"GET".equalsIgnoreCase(exchange.getRequestMethod())) {
                exchange.sendResponseHeaders(405, -1);
                return;
            }

            logHttpAccess(exchange);
            String json = "{\"status\":\"ok\"}";
            byte[] bytes = json.getBytes(StandardCharsets.UTF_8);
            exchange.getResponseHeaders().add("Content-Type", "application/json; charset=utf-8");
            exchange.sendResponseHeaders(200, bytes.length);
            try (OutputStream os = exchange.getResponseBody()) {
                os.write(bytes);
            }
        }
    }
}
