/*
 * Decompiled with CFR 0.152.
 */
package org.complexityanalyzer.geoscan;

import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Stream;
import net.minecraft.commands.CommandSourceStack;
import net.minecraft.resources.ResourceKey;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.level.ChunkPos;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.biome.Biome;
import net.minecraft.world.level.biome.BiomeSource;
import net.neoforged.bus.api.SubscribeEvent;
import net.neoforged.neoforge.common.NeoForge;
import net.neoforged.neoforge.event.tick.ServerTickEvent;
import org.complexityanalyzer.ComplexityAnalyzer;
import org.complexityanalyzer.core.AnalysisEngine;
import org.complexityanalyzer.geoscan.GeoDatabase;
import org.complexityanalyzer.geoscan.data.BiomeScanData;
import org.complexityanalyzer.geoscan.data.ChunkSnapshot;
import org.complexityanalyzer.geoscan.data.ScanMetadata;
import org.complexityanalyzer.geoscan.task.ScanNotifier;
import org.complexityanalyzer.geoscan.task.ScanTask;
import org.complexityanalyzer.geoscan.task.SpiralChunkSearcher;
import org.complexityanalyzer.geoscan.task.WorldScanner;
import org.jetbrains.annotations.Nullable;

public class GeoAnalysisManager {
    private static final int COUNTDOWN_SECONDS = 60;
    private final MinecraftServer server;
    private final GeoDatabase database;
    private final AnalysisEngine analysisEngine;
    private final WorldScanner worldScanner;
    private final ScanNotifier notifier;
    private volatile ScanMetadata.ScanPhase scanPhase = ScanMetadata.ScanPhase.IDLE;
    private final AtomicBoolean stopRequested = new AtomicBoolean(false);
    private final AtomicBoolean isProcessingChunk = new AtomicBoolean(false);
    private int tickCounter = 0;
    private int countdownTicks = -1;
    private int scheduledChunksPerBiome = 0;
    private String scheduledInitiator = "";
    private ScanProfile scheduledProfile = ScanProfile.LITE;
    private ScanProfile currentProfile = ScanProfile.LITE;
    private final Queue<ScanTask> taskQueue = new ConcurrentLinkedQueue<ScanTask>();
    private final Set<Long> attemptedChunks = ConcurrentHashMap.newKeySet();
    private int totalTasks = 0;
    private int tasksCompleted = 0;
    private long lastPauseLogTime = 0L;
    private static final long LOG_THROTTLE_MS = 5000L;
    private boolean wasPaused = false;
    @Nullable
    private ScanTask currentTask;
    private SpiralChunkSearcher currentSearcher;
    private final List<ChunkSnapshot> pristineSnapshotsForCurrentTask = new ArrayList<ChunkSnapshot>();
    private int consecutiveScanFailures = 0;

    public GeoAnalysisManager(MinecraftServer server, GeoDatabase database, AnalysisEngine engine) {
        this.server = server;
        this.database = database;
        this.analysisEngine = engine;
        this.worldScanner = new WorldScanner(server);
        this.notifier = new ScanNotifier(server);
        NeoForge.EVENT_BUS.register((Object)this);
    }

    public void startInitialScanIfNeeded() {
        Executor executor = this.analysisEngine.getBackgroundExecutor();
        if (executor == null) {
            ComplexityAnalyzer.LOGGER.error("Cannot start initial scan, executor is not ready!");
            return;
        }
        executor.execute(() -> {
            ScanMetadata.ScanPhase phase = this.database.getScanPhase();
            if (phase == ScanMetadata.ScanPhase.COMPLETE) {
                this.notifier.logInfo("GeoDatabase is complete. Skipping initial scan.");
                this.database.loadAll();
                return;
            }
            if (phase == ScanMetadata.ScanPhase.REFINING) {
                this.notifier.logWarn("Server was stopped during refinement. Restarting refinement phase.");
                this.beginGlobalRefinement(null);
                return;
            }
            this.startScanImmediately(32, "Server", ScanProfile.LITE);
        });
    }

    public void scheduleScan(int chunksPerBiome, String initiatorName, ScanProfile profile) {
        if (this.isScanning() || this.isCountdownActive()) {
            this.notifier.sendFailure(null, "A scan is already running or scheduled.");
            return;
        }
        this.scheduledChunksPerBiome = chunksPerBiome;
        this.scheduledInitiator = initiatorName;
        this.scheduledProfile = profile;
        this.countdownTicks = 1200;
        this.notifier.broadcastWarning(String.format("World scan (%s mode) will start in %d seconds.", profile.name().toLowerCase(), 60));
    }

    private void runAtomicScan(int chunksPerBiome) {
        Executor executor = this.analysisEngine.getBackgroundExecutor();
        if (executor == null) {
            ComplexityAnalyzer.LOGGER.error("Cannot run ATOMIC scan, executor is not available!");
            return;
        }
        executor.execute(() -> {
            long startTime = System.currentTimeMillis();
            this.notifier.logInfo("[ATOMIC] Starting blocking scan...");
            List<ScanTask> preparedTasks = this.prepareScanTasks(chunksPerBiome);
            if (preparedTasks.isEmpty()) {
                this.server.execute(this.notifier::notifyDatabaseIsUpToDate);
                return;
            }
            this.scanPhase = ScanMetadata.ScanPhase.RECONNAISSANCE;
            this.database.setScanPhase(ScanMetadata.ScanPhase.RECONNAISSANCE);
            this.taskQueue.addAll(preparedTasks);
            this.totalTasks = this.taskQueue.size();
            this.tasksCompleted = 0;
            this.stopRequested.set(false);
            this.attemptedChunks.clear();
            this.currentTask = null;
            while ((this.currentTask = this.taskQueue.poll()) != null) {
                if (this.stopRequested.get()) {
                    this.notifier.logInfo("[ATOMIC] Stop requested, aborting remaining tasks.");
                    break;
                }
                ++this.tasksCompleted;
                this.pristineSnapshotsForCurrentTask.clear();
                this.consecutiveScanFailures = 0;
                ScanTask task = this.currentTask;
                Optional<ChunkPos> startPosOpt = this.worldScanner.findBiomeLocation(task.dimension(), task.biome(), false);
                if (startPosOpt.isEmpty()) {
                    this.notifier.logWarn(String.format("[ATOMIC] Could not find location for %s, skipping.", task.biome().location()));
                    continue;
                }
                SpiralChunkSearcher searcher = new SpiralChunkSearcher();
                searcher.startAt(startPosOpt.get().x, startPosOpt.get().z);
                int attempts = 0;
                int relocateTries = 0;
                while (this.pristineSnapshotsForCurrentTask.size() < task.chunksToFind() && attempts < 10000 && relocateTries < 5 && !this.stopRequested.get()) {
                    ChunkPos currentPos;
                    if (attempts > 0 && attempts % 250 == 0) {
                        Optional<ChunkPos> newStart = this.worldScanner.findBiomeLocation(task.dimension(), task.biome(), true);
                        if (!newStart.isPresent()) break;
                        searcher.startAt(newStart.get().x, newStart.get().z);
                        ++relocateTries;
                    }
                    if (!this.attemptedChunks.add((currentPos = searcher.next()).toLong())) continue;
                    CompletableFuture future = new CompletableFuture();
                    this.worldScanner.processChunk(task.dimension(), task.biome(), currentPos, (snapshotOpt, success) -> {
                        if (success.booleanValue() && snapshotOpt.isPresent() && this.database.analyzeSnapshotForRecon((ChunkSnapshot)snapshotOpt.get())) {
                            future.complete(snapshotOpt);
                        } else {
                            future.complete(Optional.empty());
                        }
                    });
                    try {
                        ((Optional)future.get()).ifPresent(this.pristineSnapshotsForCurrentTask::add);
                    }
                    catch (Exception e) {
                        ComplexityAnalyzer.LOGGER.warn("[ATOMIC] Error processing chunk future", (Throwable)e);
                    }
                    ++attempts;
                }
                if (this.pristineSnapshotsForCurrentTask.isEmpty()) continue;
                this.database.appendReconData(task.dimension().location(), task.biome().location(), new ArrayList<ChunkSnapshot>(this.pristineSnapshotsForCurrentTask));
                this.notifier.logInfo(String.format("[ATOMIC] Task %d/%d: %s... found %d candidates.", this.tasksCompleted, this.totalTasks, task.biome().location().getPath(), this.pristineSnapshotsForCurrentTask.size()));
            }
            this.currentTask = null;
            if (this.stopRequested.get()) {
                this.server.execute(() -> {
                    this.scanPhase = ScanMetadata.ScanPhase.IDLE;
                    this.taskQueue.clear();
                    this.stopRequested.set(false);
                    this.notifier.notifyReconnaissanceFinished(true);
                });
            } else {
                long duration = System.currentTimeMillis() - startTime;
                this.notifier.logInfo(String.format("[ATOMIC] Reconnaissance phase complete in %.2f seconds.", (double)duration / 1000.0));
                this.server.execute(() -> this.finishReconnaissance(false));
            }
        });
    }

    public void startScanImmediately(int chunksPerBiome, String initiatorName, ScanProfile profile) {
        this.server.execute(() -> {
            if (this.isScanning() || this.isCountdownActive()) {
                if (initiatorName.equals("Server")) {
                    this.notifier.logWarn("Scan was requested by the server, but another scan/countdown is already active. Skipping.");
                } else {
                    this.notifier.sendFailure(null, "A scan is already running or scheduled.");
                }
                return;
            }
            if (!initiatorName.equals("Server")) {
                if (profile == ScanProfile.EXTREME) {
                    this.notifier.broadcastSevere("!!! FORCED WORLD SCAN IN EXTREME MODE STARTED! SERVER MAY LAG SEVERELY! !!!");
                } else {
                    this.notifier.broadcastSevere("Forced world scan started! Severe lag may occur!");
                }
            }
            this.startScanInternal(chunksPerBiome, initiatorName, profile);
        });
    }

    public void stopScan(CommandSourceStack source) {
        if (this.isScanning()) {
            if (this.stopRequested.getAndSet(true)) {
                this.notifier.sendFailure(source, "A stop has already been requested.");
            } else {
                this.notifier.sendSuccess(source, "Scan stop requested. Finishing and saving current progress...");
            }
            return;
        }
        if (this.isCountdownActive()) {
            this.cancelScheduledScan();
            this.notifier.sendSuccess(source, "Scheduled scan has been cancelled.");
            return;
        }
        this.notifier.sendFailure(source, "No scan is currently running or scheduled.");
    }

    public void cancelScheduledScan() {
        if (this.isCountdownActive()) {
            this.countdownTicks = -1;
            this.scheduledChunksPerBiome = 0;
            this.scheduledInitiator = "";
            this.scheduledProfile = ScanProfile.LITE;
            this.notifier.broadcastInfo("Scheduled world scan has been cancelled.");
        }
    }

    public String getStatus() {
        if (this.isCountdownActive()) {
            return String.format("Scan scheduled in %s mode, starting in %d seconds...", this.scheduledProfile.name().toLowerCase(), this.countdownTicks / 20);
        }
        return switch (this.scanPhase) {
            default -> throw new MatchException(null, null);
            case ScanMetadata.ScanPhase.IDLE -> "Idle";
            case ScanMetadata.ScanPhase.RECONNAISSANCE -> {
                if (this.currentTask == null) {
                    yield "Phase 1: Reconnaissance (Initializing next task...)";
                }
                yield String.format("Phase 1: Reconnaissance. Task %d/%d: %s (%d/%d)", this.tasksCompleted, this.totalTasks, this.currentTask.biome().location().getPath(), this.pristineSnapshotsForCurrentTask.size(), this.currentTask.chunksToFind());
            }
            case ScanMetadata.ScanPhase.REFINING -> "Phase 2: Refining all collected data...";
            case ScanMetadata.ScanPhase.COMPLETE -> "Complete";
        };
    }

    public boolean isScanning() {
        return this.scanPhase == ScanMetadata.ScanPhase.RECONNAISSANCE || this.scanPhase == ScanMetadata.ScanPhase.REFINING;
    }

    public boolean isCountdownActive() {
        return this.countdownTicks > 0;
    }

    @SubscribeEvent
    public void onServerTick(ServerTickEvent.Post event) {
        if (this.handleCountdown()) {
            return;
        }
        if (this.currentProfile == ScanProfile.ATOMIC) {
            return;
        }
        if (this.handleStopRequest()) {
            return;
        }
        if (this.scanPhase != ScanMetadata.ScanPhase.RECONNAISSANCE || this.isProcessingChunk.get()) {
            return;
        }
        if (this.isServerUnderLoad()) {
            long now = System.currentTimeMillis();
            if (!this.wasPaused) {
                this.notifier.logInfo("Server is under heavy load (tick time > " + this.currentProfile.maxTickTimeMs + "ms). Geo-scan is paused.");
                this.wasPaused = true;
                this.lastPauseLogTime = now;
            } else if (now - this.lastPauseLogTime > 5000L) {
                this.notifier.logInfo("Geo-scan still paused due to server load.");
                this.lastPauseLogTime = now;
            }
            return;
        }
        if (this.wasPaused) {
            this.notifier.logInfo("Server load normalized. Geo-scan resumed.");
            this.wasPaused = false;
        }
        if (this.isTickScheduled()) {
            return;
        }
        if (this.currentTask == null) {
            if (!this.startNextTask()) {
                this.finishReconnaissance(false);
            }
            return;
        }
        if (this.pristineSnapshotsForCurrentTask.size() >= this.currentTask.chunksToFind()) {
            this.finishCurrentTask();
            return;
        }
        if (this.consecutiveScanFailures >= this.currentProfile.relocateFailureThreshold) {
            this.handleRelocation();
            return;
        }
        this.processNextChunk();
    }

    private void startScanInternal(int chunksPerBiome, String initiatorName, ScanProfile profile) {
        this.currentProfile = profile;
        this.countdownTicks = -1;
        this.scanPhase = ScanMetadata.ScanPhase.RECONNAISSANCE;
        this.stopRequested.set(false);
        this.currentTask = null;
        this.notifier.notifyScanStarting(chunksPerBiome, initiatorName + " (" + profile.name().toLowerCase() + " mode)");
        if (profile == ScanProfile.ATOMIC) {
            this.runAtomicScan(chunksPerBiome);
        } else {
            Executor executor = this.analysisEngine.getBackgroundExecutor();
            if (executor == null) {
                ComplexityAnalyzer.LOGGER.error("Cannot start scan, background executor is not available!");
                this.scanPhase = ScanMetadata.ScanPhase.IDLE;
                return;
            }
            executor.execute(() -> {
                List<ScanTask> tasks = this.prepareScanTasks(chunksPerBiome);
                this.server.execute(() -> {
                    if (tasks.isEmpty()) {
                        this.notifier.notifyDatabaseIsUpToDate();
                        this.scanPhase = ScanMetadata.ScanPhase.IDLE;
                        return;
                    }
                    this.database.setScanPhase(ScanMetadata.ScanPhase.RECONNAISSANCE);
                    this.taskQueue.addAll(tasks);
                    this.totalTasks = tasks.size();
                    this.tasksCompleted = 0;
                    this.attemptedChunks.clear();
                    this.notifier.notifyScanPreparationComplete(this.totalTasks);
                });
            });
        }
    }

    private List<ScanTask> prepareScanTasks(int chunksPerBiome) {
        this.database.loadAll();
        ArrayList<ScanTask> tasksToQueue = new ArrayList<ScanTask>();
        ComplexityAnalyzer.LOGGER.debug("[Prepare] Starting to build scan tasks. Chunks per biome: {}", (Object)chunksPerBiome);
        for (ServerLevel level : this.server.getAllLevels()) {
            if (this.stopRequested.get()) break;
            ResourceKey dimension = level.dimension();
            ComplexityAnalyzer.LOGGER.info("Scanning dimension: {}", (Object)dimension.location());
            Set<ResourceKey<Biome>> biomesToScan = this.getBiomesForDimension(level);
            ComplexityAnalyzer.LOGGER.debug("Found {} biomes in dimension {}", (Object)biomesToScan.size(), (Object)dimension.location());
            for (ResourceKey<Biome> biomeKey : biomesToScan) {
                int reconChunks;
                int finalChunks = this.database.getBiomeData(dimension.location(), biomeKey.location()).map(BiomeScanData::getChunksScanned).orElse(0);
                int chunksNeeded = chunksPerBiome - Math.max(finalChunks, reconChunks = this.database.countReconChunks(dimension.location(), biomeKey.location()));
                if (chunksNeeded <= 0) continue;
                tasksToQueue.add(new ScanTask((ResourceKey<Level>)dimension, biomeKey, chunksNeeded));
            }
        }
        ComplexityAnalyzer.LOGGER.debug("[Prepare] Found {} total scan tasks.", (Object)tasksToQueue.size());
        tasksToQueue.sort(Comparator.naturalOrder());
        return tasksToQueue;
    }

    private Set<ResourceKey<Biome>> getBiomesForDimension(ServerLevel level) {
        HashSet<ResourceKey<Biome>> biomes = new HashSet<ResourceKey<Biome>>();
        BiomeSource biomeSource = level.getChunkSource().getGenerator().getBiomeSource();
        biomeSource.possibleBiomes().forEach(holder -> holder.unwrapKey().ifPresent(biomes::add));
        return biomes;
    }

    private boolean startNextTask() {
        this.currentTask = this.taskQueue.poll();
        if (this.currentTask == null) {
            return false;
        }
        Optional<ChunkPos> startPos = this.worldScanner.findBiomeLocation(this.currentTask.dimension(), this.currentTask.biome(), false);
        if (startPos.isPresent()) {
            ++this.tasksCompleted;
            this.pristineSnapshotsForCurrentTask.clear();
            this.consecutiveScanFailures = 0;
            this.currentSearcher = new SpiralChunkSearcher();
            this.currentSearcher.startAt(startPos.get().x, startPos.get().z);
            this.notifier.logInfo(String.format("Starting reconnaissance for %s (%d/%d). Chunks to find: %d. Starting at %s", this.currentTask.biome().location().getPath(), this.tasksCompleted, this.totalTasks, this.currentTask.chunksToFind(), startPos.get()));
            return true;
        }
        this.notifier.logWarn("Could not find a reachable location for biome " + String.valueOf(this.currentTask.biome().location()) + ". Skipping.");
        this.currentTask = null;
        return this.startNextTask();
    }

    private void processNextChunk() {
        if (!this.isProcessingChunk.compareAndSet(false, true)) {
            return;
        }
        long timeBudgetNanos = 1000000L;
        long searchStartTime = System.nanoTime();
        ChunkPos nextPos = null;
        while (System.nanoTime() - searchStartTime < 1000000L) {
            ChunkPos candidatePos = this.currentSearcher.next();
            if (!this.attemptedChunks.add(candidatePos.toLong())) continue;
            nextPos = candidatePos;
            break;
        }
        if (nextPos == null) {
            this.isProcessingChunk.set(false);
            return;
        }
        this.worldScanner.processChunk(this.currentTask.dimension(), this.currentTask.biome(), nextPos, (snapshotOpt, success) -> {
            if (success.booleanValue()) {
                snapshotOpt.ifPresent(snapshot -> {
                    if (this.database.analyzeSnapshotForRecon((ChunkSnapshot)snapshot)) {
                        this.pristineSnapshotsForCurrentTask.add((ChunkSnapshot)snapshot);
                        this.consecutiveScanFailures = 0;
                    } else {
                        ++this.consecutiveScanFailures;
                    }
                });
            } else {
                ++this.consecutiveScanFailures;
            }
            this.isProcessingChunk.set(false);
        });
        this.notifier.logProgress(true, this.getStatus());
    }

    private void finishCurrentTask() {
        if (this.currentTask == null) {
            return;
        }
        if (!this.pristineSnapshotsForCurrentTask.isEmpty()) {
            this.notifier.logInfo(String.format("Finished reconnaissance for biome %s, found %d new candidates. Saving...", this.currentTask.biome().location(), this.pristineSnapshotsForCurrentTask.size()));
            this.database.appendReconData(this.currentTask.dimension().location(), this.currentTask.biome().location(), new ArrayList<ChunkSnapshot>(this.pristineSnapshotsForCurrentTask));
        }
        this.currentTask = null;
        this.currentSearcher = null;
        this.pristineSnapshotsForCurrentTask.clear();
    }

    private void finishReconnaissance(boolean wasStopped) {
        this.finishCurrentTask();
        this.scanPhase = ScanMetadata.ScanPhase.IDLE;
        this.taskQueue.clear();
        this.stopRequested.set(false);
        this.notifier.notifyReconnaissanceFinished(wasStopped);
        if (!wasStopped) {
            this.beginGlobalRefinement(null);
        }
    }

    public void beginGlobalRefinement(@Nullable CommandSourceStack source) {
        this.scanPhase = ScanMetadata.ScanPhase.REFINING;
        this.database.setScanPhase(ScanMetadata.ScanPhase.REFINING);
        this.notifier.sendSuccess(source, "Reconnaissance complete! Starting final data refinement in background...");
        Executor executor = this.analysisEngine.getBackgroundExecutor();
        if (executor == null) {
            ComplexityAnalyzer.LOGGER.error("Cannot begin global refinement, executor is not available!");
            this.scanPhase = ScanMetadata.ScanPhase.IDLE;
            this.database.setScanPhase(ScanMetadata.ScanPhase.IDLE);
            return;
        }
        executor.execute(() -> {
            try {
                Map<ResourceLocation, Map<ResourceLocation, Path>> allReconPaths = this.database.getAllReconFilePaths();
                if (allReconPaths.isEmpty()) {
                    this.notifier.logWarn("Refinement started, but no reconnaissance data was found.");
                    this.finishRefinement();
                    return;
                }
                this.notifier.logInfo("Building heuristics from reconnaissance data...");
                this.database.buildHeuristicFromFiles(allReconPaths);
                this.notifier.logInfo("Clearing old final data...");
                this.database.clearFinalData();
                this.notifier.logInfo("Refining data for each biome...");
                allReconPaths.forEach((dim, biomeMap) -> biomeMap.forEach((biome, path) -> {
                    try (Stream<ChunkSnapshot> snapshotStream = this.database.streamReconFile((Path)path);){
                        BiomeScanData finalData = this.database.refineRawDataFromStream(snapshotStream, (ResourceLocation)dim);
                        if (finalData.getChunksScanned() > 0) {
                            this.database.saveBiomeData((ResourceLocation)dim, (ResourceLocation)biome, finalData);
                        }
                    }
                }));
                this.finishRefinement();
            }
            catch (IOException e) {
                this.notifier.logError("A critical error occurred during the refinement phase!", e);
                this.scanPhase = ScanMetadata.ScanPhase.IDLE;
                this.database.setScanPhase(ScanMetadata.ScanPhase.IDLE);
            }
        });
    }

    private void finishRefinement() {
        this.notifier.logInfo("Finalizing refinement process...");
        this.scanPhase = ScanMetadata.ScanPhase.COMPLETE;
        this.database.setScanPhase(ScanMetadata.ScanPhase.COMPLETE);
        this.database.loadAll();
        this.analysisEngine.onGeoScanFinished();
        this.notifier.notifyRefinementFinished();
    }

    private boolean handleCountdown() {
        if (this.countdownTicks > 0) {
            if (--this.countdownTicks % 20 == 0) {
                int secondsLeft = this.countdownTicks / 20;
                this.notifier.notifyScanCountdown(secondsLeft);
                if (secondsLeft == 0) {
                    this.startScanInternal(this.scheduledChunksPerBiome, this.scheduledInitiator, this.scheduledProfile);
                }
            }
            return true;
        }
        return false;
    }

    private boolean handleStopRequest() {
        if (this.stopRequested.get() && this.scanPhase == ScanMetadata.ScanPhase.RECONNAISSANCE && !this.isProcessingChunk.get()) {
            this.finishReconnaissance(true);
            return true;
        }
        return this.stopRequested.get();
    }

    private void handleRelocation() {
        if (this.currentTask == null || this.currentSearcher == null) {
            this.notifier.logError("handleRelocation called but currentTask or currentSearcher is null. Skipping.");
            return;
        }
        this.notifier.logWarn("Too many consecutive scan failures for biome " + String.valueOf(this.currentTask.biome().location()) + ". Relocating...");
        Optional<ChunkPos> newStartPos = this.worldScanner.findBiomeLocation(this.currentTask.dimension(), this.currentTask.biome(), true);
        if (newStartPos.isPresent()) {
            this.currentSearcher.startAt(newStartPos.get().x, newStartPos.get().z);
            this.consecutiveScanFailures = 0;
        } else {
            this.notifier.logError("Could not relocate for biome " + String.valueOf(this.currentTask.biome().location()) + ". Skipping task.");
            this.finishCurrentTask();
        }
    }

    private boolean isServerUnderLoad() {
        return (float)this.server.getAverageTickTimeNanos() / 1000000.0f > this.currentProfile.maxTickTimeMs;
    }

    private boolean isTickScheduled() {
        ++this.tickCounter;
        if (this.tickCounter >= this.currentProfile.ticksBetweenScans) {
            this.tickCounter = 0;
            return false;
        }
        return true;
    }

    public void shutdown() {
        ComplexityAnalyzer.LOGGER.info("Shutting down GeoAnalysisManager...");
        this.stopRequested.set(true);
        NeoForge.EVENT_BUS.unregister((Object)this);
        ComplexityAnalyzer.LOGGER.info("GeoAnalysisManager has been shut down.");
    }

    public static enum ScanProfile {
        LITE(40.0f, 20, 250),
        FAST(50.0f, 5, 100),
        EXTREME(Float.MAX_VALUE, 1, 50),
        ATOMIC(0.0f, 0, 0);

        public final float maxTickTimeMs;
        public final int ticksBetweenScans;
        public final int relocateFailureThreshold;

        private ScanProfile(float maxTickTimeMs, int ticksBetweenScans, int relocateFailureThreshold) {
            this.maxTickTimeMs = maxTickTimeMs;
            this.ticksBetweenScans = ticksBetweenScans;
            this.relocateFailureThreshold = relocateFailureThreshold;
        }
    }
}

