/*
 * Decompiled with CFR 0.152.
 */
package org.complexityanalyzer.analyzer.resource.providers;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import net.minecraft.core.Holder;
import net.minecraft.core.Registry;
import net.minecraft.core.registries.Registries;
import net.minecraft.resources.ResourceKey;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.util.random.WeightedRandomList;
import net.minecraft.world.entity.EntityType;
import net.minecraft.world.entity.MobCategory;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.biome.Biome;
import net.minecraft.world.level.biome.BiomeSource;
import net.minecraft.world.level.biome.MobSpawnSettings;
import net.minecraft.world.level.chunk.ChunkGenerator;
import net.minecraft.world.level.levelgen.structure.Structure;
import net.minecraft.world.level.levelgen.structure.StructureSpawnOverride;
import org.complexityanalyzer.ComplexityAnalyzer;

public class DimensionRarityAnalyzer {
    private static final Map<ResourceKey<Level>, Double> DIMENSION_MULTIPLIERS = Map.of(Level.OVERWORLD, 1.0, Level.NETHER, 3.0, Level.END, 4.0);
    private static final double END_ISLANDS_MULTIPLIER = 10.0;
    private static final double CUSTOM_DIMENSION_MULTIPLIER = 4.0;
    private final Map<EntityType<?>, ResourceKey<Level>> structureSpawnCache = new HashMap();
    private final Map<ResourceKey<Biome>, ResourceKey<Level>> biomeToDimensionMap = new HashMap<ResourceKey<Biome>, ResourceKey<Level>>();
    private final Level level;

    public DimensionRarityAnalyzer(Level level) {
        this.level = level;
        this.analyzeStructureSpawns();
        this.buildBiomeToDimensionMap();
    }

    private void analyzeStructureSpawns() {
        ComplexityAnalyzer.LOGGER.info("Analyzing structure spawns for dimension detection...");
        Registry structureRegistry = this.level.registryAccess().registryOrThrow(Registries.STRUCTURE);
        int structuresAnalyzed = 0;
        for (Map.Entry entry : structureRegistry.entrySet()) {
            ResourceKey structureKey = (ResourceKey)entry.getKey();
            Structure structure = (Structure)entry.getValue();
            Map spawnOverrides = structure.spawnOverrides();
            if (spawnOverrides.isEmpty()) continue;
            ++structuresAnalyzed;
            ResourceKey<Level> dimension = this.guessDimensionFromStructure((ResourceKey<Structure>)structureKey);
            for (StructureSpawnOverride override : spawnOverrides.values()) {
                for (MobSpawnSettings.SpawnerData spawner : override.spawns().unwrap()) {
                    this.structureSpawnCache.putIfAbsent(spawner.type, dimension);
                }
            }
        }
        ComplexityAnalyzer.LOGGER.info("Structure spawn analysis complete: {} structures, {} unique mob types", (Object)structuresAnalyzed, (Object)this.structureSpawnCache.size());
    }

    private void buildBiomeToDimensionMap() {
        ComplexityAnalyzer.LOGGER.info("Building biome-to-dimension map from loaded levels...");
        Level level = this.level;
        if (!(level instanceof ServerLevel)) {
            ComplexityAnalyzer.LOGGER.warn("Level is not ServerLevel, cannot analyze dimensions");
            return;
        }
        ServerLevel serverLevel = (ServerLevel)level;
        MinecraftServer server = serverLevel.getServer();
        Registry biomeRegistry = this.level.registryAccess().registryOrThrow(Registries.BIOME);
        int dimensionsFound = 0;
        int skippedDimensions = 0;
        for (ServerLevel dimension : server.getAllLevels()) {
            ResourceKey dimensionKey = dimension.dimension();
            ++dimensionsFound;
            try {
                ChunkGenerator chunkGenerator = dimension.getChunkSource().getGenerator();
                BiomeSource biomeSource = chunkGenerator.getBiomeSource();
                HashSet biomeHolders = new HashSet(biomeSource.possibleBiomes());
                for (Holder biomeHolder : biomeHolders) {
                    biomeHolder.unwrapKey().ifPresent(biomeKey -> this.biomeToDimensionMap.putIfAbsent((ResourceKey<Biome>)biomeKey, (ResourceKey<Level>)dimensionKey));
                }
            }
            catch (Exception e) {
                ++skippedDimensions;
                ComplexityAnalyzer.LOGGER.warn("Failed to analyze dimension {} ({}), skipping: {}", new Object[]{dimensionKey.location(), e.getClass().getSimpleName(), e.getMessage()});
            }
        }
        for (Map.Entry entry : biomeRegistry.entrySet()) {
            ResourceKey biomeKey2 = (ResourceKey)entry.getKey();
            if (this.biomeToDimensionMap.containsKey(biomeKey2)) continue;
            ResourceKey<Level> dimension = DimensionRarityAnalyzer.getDimensionForBiome((ResourceKey<Biome>)biomeKey2);
            this.biomeToDimensionMap.put((ResourceKey<Biome>)biomeKey2, dimension);
        }
        int biomeMappings = this.biomeToDimensionMap.size();
        if (skippedDimensions > 0) {
            ComplexityAnalyzer.LOGGER.info("Found {} dimensions (skipped {} problematic), mapped {} biomes", new Object[]{dimensionsFound, skippedDimensions, biomeMappings});
        } else {
            ComplexityAnalyzer.LOGGER.info("Found {} dimensions, mapped {} biomes", (Object)dimensionsFound, (Object)biomeMappings);
        }
    }

    private ResourceKey<Level> guessDimensionFromStructure(ResourceKey<Structure> structureKey) {
        String structurePath = structureKey.location().getPath();
        if (structurePath.contains("nether") || structurePath.contains("fortress") || structurePath.contains("bastion")) {
            return Level.NETHER;
        }
        if (structurePath.contains("end") || structurePath.contains("city")) {
            return Level.END;
        }
        return Level.OVERWORLD;
    }

    public double getStructureMultiplier(EntityType<?> entityType) {
        ResourceKey<Level> structureDimension = this.structureSpawnCache.get(entityType);
        if (structureDimension != null) {
            return DIMENSION_MULTIPLIERS.getOrDefault(structureDimension, 4.0);
        }
        return 0.0;
    }

    public double getBiomeMultiplier(EntityType<?> entityType) {
        ResourceKey<Level> dimension;
        Registry biomeRegistry = this.level.registryAccess().registryOrThrow(Registries.BIOME);
        HashMap<ResourceKey, Set> dimensionBiomes = new HashMap<ResourceKey, Set>();
        for (Map.Entry entry : biomeRegistry.entrySet()) {
            ResourceKey biomeKey = (ResourceKey)entry.getKey();
            Biome biome = (Biome)entry.getValue();
            if (!DimensionRarityAnalyzer.mobSpawnsInBiome(entityType, biome)) continue;
            dimension = this.biomeToDimensionMap.getOrDefault(biomeKey, DimensionRarityAnalyzer.getDimensionForBiome((ResourceKey<Biome>)biomeKey));
            dimensionBiomes.computeIfAbsent(dimension, k -> new HashSet()).add(biomeKey);
        }
        if (!dimensionBiomes.isEmpty()) {
            double maxMultiplier = 1.0;
            for (Map.Entry entry : dimensionBiomes.entrySet()) {
                boolean isEndIslands;
                dimension = (ResourceKey<Level>)entry.getKey();
                Set biomes = (Set)entry.getValue();
                double multiplier = dimension.equals((Object)Level.END) ? ((isEndIslands = biomes.stream().anyMatch(DimensionRarityAnalyzer::isEndIslandsBiome)) ? 10.0 : DIMENSION_MULTIPLIERS.get(Level.END)) : DIMENSION_MULTIPLIERS.getOrDefault(dimension, 4.0);
                maxMultiplier = Math.max(maxMultiplier, multiplier);
            }
            return maxMultiplier;
        }
        return 0.0;
    }

    private static boolean mobSpawnsInBiome(EntityType<?> entityType, Biome biome) {
        MobSpawnSettings spawnSettings = biome.getMobSettings();
        for (MobCategory category : MobCategory.values()) {
            WeightedRandomList spawners = spawnSettings.getMobs(category);
            for (MobSpawnSettings.SpawnerData spawner : spawners.unwrap()) {
                if (spawner.type != entityType) continue;
                return true;
            }
        }
        return false;
    }

    private static ResourceKey<Level> getDimensionForBiome(ResourceKey<Biome> biomeKey) {
        String path = biomeKey.location().getPath();
        if (path.contains("nether") || path.contains("crimson") || path.contains("warped") || path.contains("basalt")) {
            return Level.NETHER;
        }
        if (path.contains("end") || path.contains("the_end")) {
            return Level.END;
        }
        return Level.OVERWORLD;
    }

    private static boolean isEndIslandsBiome(ResourceKey<Biome> biomeKey) {
        String path = biomeKey.location().getPath();
        return path.contains("end_highlands") || path.contains("end_midlands") || path.contains("small_end_islands");
    }
}

