/*
 * Decompiled with CFR 0.152.
 */
package org.complexityanalyzer.analyzer.resource.providers;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import javax.annotation.Nullable;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.entity.EntityType;
import net.minecraft.world.entity.MobCategory;
import net.minecraft.world.entity.ai.attributes.AttributeSupplier;
import net.minecraft.world.entity.ai.attributes.Attributes;
import net.minecraft.world.entity.ai.attributes.DefaultAttributes;
import net.minecraft.world.entity.boss.enderdragon.EnderDragon;
import net.minecraft.world.entity.boss.wither.WitherBoss;
import org.complexityanalyzer.ComplexityAnalyzer;
import org.complexityanalyzer.analyzer.resource.providers.DimensionRarityAnalyzer;
import org.complexityanalyzer.api.IBossRegistry;

public class MobRarityCalculator
implements IBossRegistry {
    private final DimensionRarityAnalyzer dimensionAnalyzer;
    private final Map<EntityType<?>, Double> rarityCache = new ConcurrentHashMap();
    private final Map<EntityType<?>, Double> healthCache = new ConcurrentHashMap();
    private final Map<EntityType<?>, BossLevel> bossCache = new ConcurrentHashMap();
    private final Map<EntityType<?>, BossLevel> registeredBosses = new ConcurrentHashMap();
    private static final double BOSS_RARITY = 50.0;
    private static final double MINI_BOSS_RARITY = 15.0;
    private static final double HIGH_HEALTH_BONUS = 5.0;
    private static final double VERY_HIGH_HEALTH_BONUS = 10.0;
    private static final double NETHER_BONUS = 3.0;
    private static final double END_BONUS = 5.0;
    private static final double MODDED_BONUS = 2.0;
    private static final double FALLBACK_RARITY = 1.0;
    private static final double BOSS_HEALTH_THRESHOLD = 200.0;
    private static final double MINI_BOSS_HEALTH_THRESHOLD = 80.0;
    private static final double HIGH_HEALTH_THRESHOLD = 50.0;
    private static final Set<String> BOSS_KEYWORDS = Set.of("boss", "dragon", "king", "queen", "lord", "ancient", "elder", "wither");
    private static final Set<String> RARE_KEYWORDS = Set.of("rare", "elite", "champion", "alpha", "mutant", "titan", "legendary", "prime");

    public MobRarityCalculator(DimensionRarityAnalyzer dimensionAnalyzer) {
        this.dimensionAnalyzer = dimensionAnalyzer;
    }

    @Override
    public void registerBoss(EntityType<?> entityType, IBossRegistry.BossType type) {
        BossLevel level = type == IBossRegistry.BossType.BOSS ? BossLevel.BOSS : BossLevel.MINI_BOSS;
        this.registeredBosses.put(entityType, level);
        this.bossCache.put(entityType, level);
        ComplexityAnalyzer.LOGGER.info("[BossRegistry] Registered {} as {}", (Object)BuiltInRegistries.ENTITY_TYPE.getKey(entityType), (Object)type);
    }

    @Override
    public void registerBoss(String entityId, IBossRegistry.BossType type) {
        ResourceLocation id = ResourceLocation.tryParse((String)entityId);
        if (id == null) {
            ComplexityAnalyzer.LOGGER.warn("[BossRegistry] Invalid entity ID: {}", (Object)entityId);
            return;
        }
        EntityType entityType = (EntityType)BuiltInRegistries.ENTITY_TYPE.get(id);
        this.registerBoss(entityType, type);
    }

    @Override
    public boolean isBoss(EntityType<?> entityType) {
        BossLevel level = this.detectBossLevel(entityType);
        return level == BossLevel.BOSS;
    }

    @Override
    public boolean isMiniBoss(EntityType<?> entityType) {
        BossLevel level = this.detectBossLevel(entityType);
        return level == BossLevel.MINI_BOSS;
    }

    public double calculateRarity(EntityType<?> entityType) {
        return this.rarityCache.computeIfAbsent(entityType, this::calculateRarityInternal);
    }

    private double calculateRarityInternal(EntityType<?> entityType) {
        double yLevelBonus;
        double dimensionBonus;
        double health;
        double biomeRarity;
        Double hardcoded;
        boolean isModded = this.isModdedEntity(entityType);
        if (!isModded && (hardcoded = this.getHardcodedRarity(entityType)) != null) {
            return hardcoded;
        }
        double rarity = 0.0;
        BossLevel bossLevel = this.detectBossLevel(entityType);
        switch (bossLevel.ordinal()) {
            case 2: {
                rarity += 50.0;
                break;
            }
            case 1: {
                rarity += 15.0;
            }
        }
        double structRarity = this.dimensionAnalyzer.getStructureMultiplier(entityType);
        if (structRarity > 0.0) {
            rarity += structRarity;
        }
        if (structRarity == 0.0 && (biomeRarity = this.dimensionAnalyzer.getBiomeMultiplier(entityType)) > 0.0) {
            rarity += biomeRarity;
        }
        if ((health = this.getEntityHealth(entityType)) >= 200.0 && bossLevel != BossLevel.BOSS) {
            rarity += 10.0;
        } else if (health >= 50.0 && bossLevel == BossLevel.NONE) {
            rarity += 5.0;
        }
        double categoryBonus = this.analyzeMobCategory(entityType);
        if (categoryBonus > 0.0) {
            rarity += categoryBonus;
        }
        if ((dimensionBonus = this.analyzeDimensionExclusivity(entityType)) > 0.0) {
            rarity += dimensionBonus;
        }
        if (isModded) {
            double nameBonus = this.analyzeEntityName(entityType);
            if (nameBonus > 0.0) {
                rarity += nameBonus;
            }
            rarity += 2.0;
        }
        if ((yLevelBonus = this.analyzeYLevelRestrictions(entityType)) > 0.0) {
            rarity += yLevelBonus;
        }
        if (rarity == 0.0) {
            rarity = 1.0;
        }
        return rarity;
    }

    private BossLevel detectBossLevel(EntityType<?> entityType) {
        BossLevel registered = this.registeredBosses.get(entityType);
        if (registered != null) {
            return registered;
        }
        BossLevel cached = this.bossCache.get(entityType);
        if (cached != null) {
            return cached;
        }
        BossLevel detected = this.detectBossLevelInternal(entityType);
        this.bossCache.put(entityType, detected);
        return detected;
    }

    private BossLevel detectBossLevelInternal(EntityType<?> entityType) {
        BossLevel nameCheck;
        BossLevel classCheck = this.detectByClass(entityType);
        if (classCheck != BossLevel.NONE) {
            ComplexityAnalyzer.LOGGER.debug("[BossDetection] {} detected as {} by class", (Object)this.getEntityName(entityType), (Object)classCheck);
            return classCheck;
        }
        BossLevel vanillaCheck = this.detectVanillaBoss(entityType);
        if (vanillaCheck != BossLevel.NONE) {
            return vanillaCheck;
        }
        double health = this.getEntityHealth(entityType);
        if (health >= 200.0) {
            ComplexityAnalyzer.LOGGER.debug("[BossDetection] {} detected as BOSS by health ({})", (Object)this.getEntityName(entityType), (Object)health);
            return BossLevel.BOSS;
        }
        if (health >= 80.0 && this.hasMiniBossIndicators(entityType)) {
            ComplexityAnalyzer.LOGGER.debug("[BossDetection] {} detected as MINI_BOSS by health + indicators", (Object)this.getEntityName(entityType));
            return BossLevel.MINI_BOSS;
        }
        if (this.isModdedEntity(entityType) && health >= 40.0 && (nameCheck = this.detectByName(entityType)) != BossLevel.NONE) {
            ComplexityAnalyzer.LOGGER.debug("[BossDetection] {} detected as {} by name pattern", (Object)this.getEntityName(entityType), (Object)nameCheck);
            return nameCheck;
        }
        return BossLevel.NONE;
    }

    private BossLevel detectByClass(EntityType<?> entityType) {
        try {
            double health;
            String packageName;
            Class entityClass = entityType.getBaseClass();
            if (WitherBoss.class.isAssignableFrom(entityClass)) {
                return BossLevel.BOSS;
            }
            if (EnderDragon.class.isAssignableFrom(entityClass)) {
                return BossLevel.BOSS;
            }
            String className = entityClass.getSimpleName();
            if (className.endsWith("Boss") || className.contains("BossEntity")) {
                return BossLevel.BOSS;
            }
            for (Class superClass = entityClass.getSuperclass(); superClass != null && superClass != Object.class; superClass = superClass.getSuperclass()) {
                String superName = superClass.getSimpleName();
                if (!superName.endsWith("Boss") && !superName.equals("BossEntity")) continue;
                return BossLevel.BOSS;
            }
            String string = packageName = entityClass.getPackage() != null ? entityClass.getPackage().getName() : "";
            if ((packageName.contains(".boss.") || packageName.endsWith(".boss")) && (health = this.getEntityHealth(entityType)) >= 80.0) {
                return health >= 200.0 ? BossLevel.BOSS : BossLevel.MINI_BOSS;
            }
        }
        catch (Exception e) {
            ComplexityAnalyzer.LOGGER.debug("Error checking boss class for {}: {}", (Object)this.getEntityName(entityType), (Object)e.getMessage());
        }
        return BossLevel.NONE;
    }

    private BossLevel detectVanillaBoss(EntityType<?> entityType) {
        if (entityType == EntityType.ENDER_DRAGON || entityType == EntityType.WITHER) {
            return BossLevel.BOSS;
        }
        if (entityType == EntityType.ELDER_GUARDIAN || entityType == EntityType.WARDEN) {
            return BossLevel.MINI_BOSS;
        }
        return BossLevel.NONE;
    }

    private BossLevel detectByName(EntityType<?> entityType) {
        String name = this.getEntityName(entityType).toLowerCase();
        ResourceLocation id = BuiltInRegistries.ENTITY_TYPE.getKey(entityType);
        String idString = id.getPath().toLowerCase();
        for (String keyword : BOSS_KEYWORDS) {
            if (!name.contains(keyword) && !idString.contains(keyword)) continue;
            return BossLevel.BOSS;
        }
        for (String keyword : RARE_KEYWORDS) {
            if (!name.contains(keyword) && !idString.contains(keyword)) continue;
            return BossLevel.MINI_BOSS;
        }
        return BossLevel.NONE;
    }

    private boolean hasMiniBossIndicators(EntityType<?> entityType) {
        if (entityType.getCategory() == MobCategory.MISC) {
            return true;
        }
        if (this.isModdedEntity(entityType)) {
            BossLevel nameCheck = this.detectByName(entityType);
            return nameCheck == BossLevel.MINI_BOSS;
        }
        return false;
    }

    private boolean isModdedEntity(EntityType<?> entityType) {
        ResourceLocation id = BuiltInRegistries.ENTITY_TYPE.getKey(entityType);
        return !id.getNamespace().equals("minecraft");
    }

    private double getEntityHealth(EntityType<?> entityType) {
        return this.healthCache.computeIfAbsent(entityType, type -> {
            Double vanillaHealth;
            try {
                EntityType livingType = type;
                AttributeSupplier attributes = DefaultAttributes.getSupplier((EntityType)livingType);
                if (attributes.hasAttribute(Attributes.MAX_HEALTH)) {
                    return attributes.getValue(Attributes.MAX_HEALTH);
                }
            }
            catch (ClassCastException livingType) {
            }
            catch (Exception e) {
                ComplexityAnalyzer.LOGGER.debug("Could not get attributes for {}: {}", (Object)this.getEntityName((EntityType<?>)type), (Object)e.getMessage());
            }
            if (!this.isModdedEntity((EntityType<?>)type) && (vanillaHealth = this.getVanillaHealth((EntityType<?>)type)) != null) {
                return vanillaHealth;
            }
            return 20.0;
        });
    }

    @Nullable
    private Double getVanillaHealth(EntityType<?> entityType) {
        if (entityType == EntityType.ENDER_DRAGON) {
            return 200.0;
        }
        if (entityType == EntityType.WITHER) {
            return 300.0;
        }
        if (entityType == EntityType.WARDEN) {
            return 500.0;
        }
        if (entityType == EntityType.ELDER_GUARDIAN) {
            return 80.0;
        }
        if (entityType == EntityType.RAVAGER) {
            return 100.0;
        }
        if (entityType == EntityType.IRON_GOLEM) {
            return 100.0;
        }
        if (entityType == EntityType.PIGLIN_BRUTE) {
            return 50.0;
        }
        if (entityType == EntityType.HOGLIN) {
            return 40.0;
        }
        if (entityType == EntityType.ZOGLIN) {
            return 40.0;
        }
        if (entityType == EntityType.ENDERMAN) {
            return 40.0;
        }
        if (entityType == EntityType.GUARDIAN) {
            return 30.0;
        }
        if (entityType == EntityType.SHULKER) {
            return 30.0;
        }
        if (entityType == EntityType.WITCH) {
            return 26.0;
        }
        return null;
    }

    private double analyzeEntityName(EntityType<?> entityType) {
        String name = this.getEntityName(entityType).toLowerCase();
        ResourceLocation id = BuiltInRegistries.ENTITY_TYPE.getKey(entityType);
        String idString = id.getPath().toLowerCase();
        double bonus = 0.0;
        if (name.contains("rare") || idString.contains("rare")) {
            bonus += 5.0;
        }
        if (name.contains("special") || name.contains("unique")) {
            bonus += 3.0;
        }
        if (name.contains("elite") || name.contains("champion")) {
            bonus += 7.0;
        }
        return bonus;
    }

    private double analyzeDimensionExclusivity(EntityType<?> entityType) {
        if (!this.isModdedEntity(entityType)) {
            if (this.isNetherExclusive(entityType)) {
                return 3.0;
            }
            if (this.isEndExclusive(entityType)) {
                return 5.0;
            }
        }
        return 0.0;
    }

    private double analyzeMobCategory(EntityType<?> entityType) {
        MobCategory category = entityType.getCategory();
        return switch (category) {
            default -> throw new MatchException(null, null);
            case MobCategory.MONSTER, MobCategory.CREATURE -> 0.0;
            case MobCategory.AMBIENT, MobCategory.WATER_CREATURE -> 2.0;
            case MobCategory.WATER_AMBIENT -> 3.0;
            case MobCategory.UNDERGROUND_WATER_CREATURE -> 4.0;
            case MobCategory.AXOLOTLS -> 6.0;
            case MobCategory.MISC -> 1.0;
        };
    }

    @Nullable
    private Double getHardcodedRarity(EntityType<?> entityType) {
        if (entityType == EntityType.ENDER_DRAGON) {
            return 100.0;
        }
        if (entityType == EntityType.WITHER) {
            return 80.0;
        }
        if (entityType == EntityType.WARDEN) {
            return 50.0;
        }
        if (entityType == EntityType.ELDER_GUARDIAN) {
            return 30.0;
        }
        if (entityType == EntityType.SHULKER) {
            return 20.0;
        }
        if (entityType == EntityType.EVOKER) {
            return 15.0;
        }
        if (entityType == EntityType.VINDICATOR) {
            return 12.0;
        }
        if (entityType == EntityType.RAVAGER) {
            return 18.0;
        }
        if (entityType == EntityType.PIGLIN_BRUTE) {
            return 12.0;
        }
        if (entityType == EntityType.BLAZE) {
            return 10.0;
        }
        if (entityType == EntityType.WITHER_SKELETON) {
            return 10.0;
        }
        if (entityType == EntityType.GUARDIAN) {
            return 8.0;
        }
        if (entityType == EntityType.GHAST) {
            return 7.0;
        }
        if (entityType == EntityType.ZOMBIE) {
            return 1.0;
        }
        if (entityType == EntityType.SKELETON) {
            return 1.0;
        }
        if (entityType == EntityType.CREEPER) {
            return 1.0;
        }
        if (entityType == EntityType.SPIDER) {
            return 1.0;
        }
        return null;
    }

    private boolean isNetherExclusive(EntityType<?> entityType) {
        return entityType == EntityType.GHAST || entityType == EntityType.BLAZE || entityType == EntityType.MAGMA_CUBE || entityType == EntityType.WITHER_SKELETON || entityType == EntityType.PIGLIN || entityType == EntityType.PIGLIN_BRUTE || entityType == EntityType.HOGLIN || entityType == EntityType.STRIDER;
    }

    private boolean isEndExclusive(EntityType<?> entityType) {
        return entityType == EntityType.ENDER_DRAGON || entityType == EntityType.SHULKER || entityType == EntityType.ENDERMITE;
    }

    private double analyzeYLevelRestrictions(EntityType<?> entityType) {
        if (entityType == EntityType.WARDEN) {
            return 10.0;
        }
        if (entityType == EntityType.PHANTOM) {
            return 2.0;
        }
        return 0.0;
    }

    private String getEntityName(EntityType<?> entityType) {
        return entityType.getDescription().getString();
    }

    public void clearCache() {
        this.rarityCache.clear();
        this.healthCache.clear();
        this.bossCache.clear();
        ComplexityAnalyzer.LOGGER.info("MobRarityCalculator cache cleared");
    }

    private static enum BossLevel {
        NONE,
        MINI_BOSS,
        BOSS;

    }
}

