/*
 * Decompiled with CFR 0.152.
 */
package org.complexityanalyzer.analyzer.resource.sources;

import it.unimi.dsi.fastutil.objects.ObjectArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import net.minecraft.core.Registry;
import net.minecraft.core.RegistryAccess;
import net.minecraft.core.registries.Registries;
import net.minecraft.resources.ResourceKey;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.ReloadableServerRegistries;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.entity.EntityType;
import net.minecraft.world.entity.monster.piglin.Piglin;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.Items;
import net.minecraft.world.level.ItemLike;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.storage.loot.LootParams;
import net.minecraft.world.level.storage.loot.LootTable;
import net.minecraft.world.level.storage.loot.parameters.LootContextParamSets;
import net.minecraft.world.level.storage.loot.parameters.LootContextParams;
import net.minecraft.world.phys.Vec3;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.Logger;
import org.apache.logging.log4j.core.filter.AbstractFilter;
import org.complexityanalyzer.ComplexityAnalyzer;
import org.complexityanalyzer.analyzer.resource.IResourceSource;
import org.complexityanalyzer.analyzer.resource.data.BaseResourceData;
import org.jetbrains.annotations.NotNull;

public class UniversalLootSource
implements IResourceSource {
    private static final int SIMULATION_COUNT = 500;
    private static final int SIMULATION_TIMEOUT_MS = 3000;
    private final Map<BaseResourceData.ResourceSourceType, Map<Item, BaseResourceData>> allLootData = new ConcurrentHashMap<BaseResourceData.ResourceSourceType, Map<Item, BaseResourceData>>();

    @Override
    public void initialize(Level level) {
        if (!(level instanceof ServerLevel)) {
            ComplexityAnalyzer.LOGGER.error("[ULS] Initialize called with non-server level. Aborting.");
            return;
        }
        ServerLevel serverLevel = (ServerLevel)level;
        CompletableFuture<Set<ResourceKey<LootTable>>> lootKeysFuture = this.getCompletableFuture(serverLevel);
        try {
            Set<ResourceKey<LootTable>> allLootTableKeys = lootKeysFuture.join();
            this.processLootTables(serverLevel, allLootTableKeys);
        }
        catch (Exception e) {
            ComplexityAnalyzer.LOGGER.error("[ULS] Failed to get loot table keys from server thread. Aborting analysis.", (Throwable)e);
        }
    }

    @NotNull
    private CompletableFuture<Set<ResourceKey<LootTable>>> getCompletableFuture(ServerLevel serverLevel) {
        MinecraftServer server = serverLevel.getServer();
        CompletableFuture<Set<ResourceKey<LootTable>>> lootKeysFuture = new CompletableFuture<Set<ResourceKey<LootTable>>>();
        server.execute(() -> {
            try {
                Set<ResourceKey<LootTable>> keys = this.getAllLootTableKeys(server);
                lootKeysFuture.complete(keys);
            }
            catch (Exception e) {
                lootKeysFuture.completeExceptionally(e);
            }
        });
        return lootKeysFuture;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processLootTables(ServerLevel serverLevel, Set<ResourceKey<LootTable>> allLootTableKeys) {
        MinecraftServer server = serverLevel.getServer();
        ComplexityAnalyzer.LOGGER.debug("[ULS] Auto-scanning ALL loot tables (including mods)...");
        long startTime = System.currentTimeMillis();
        int tablesProcessed = 0;
        int tablesSkipped = 0;
        LootFunctionFilter filter = new LootFunctionFilter();
        Logger rootLogger = (Logger)LogManager.getRootLogger();
        filter.start();
        rootLogger.addFilter((Filter)filter);
        try {
            ReloadableServerRegistries.Holder reloadableRegistries = server.reloadableRegistries();
            ComplexityAnalyzer.LOGGER.debug("[ULS] Found {} total loot tables to analyze.", (Object)allLootTableKeys.size());
            for (ResourceKey<LootTable> lootTableKey : allLootTableKeys) {
                ResourceLocation lootTableId = lootTableKey.location();
                Optional<LootContextDefinition> contextDefOpt = this.inferContextFromId(lootTableId);
                if (contextDefOpt.isEmpty()) {
                    ++tablesSkipped;
                    continue;
                }
                LootContextDefinition contextDef = contextDefOpt.get();
                try {
                    LootTable lootTable = CompletableFuture.supplyAsync(() -> reloadableRegistries.getLootTable(lootTableKey), (Executor)server).join();
                    if (lootTable == LootTable.EMPTY) {
                        ++tablesSkipped;
                        continue;
                    }
                    ++tablesProcessed;
                    LootParams lootParams = contextDef.createLootParams(serverLevel);
                    if (lootParams == null) {
                        ComplexityAnalyzer.LOGGER.debug("[ULS] Failed to create loot params for '{}', skipping.", (Object)lootTableId);
                        continue;
                    }
                    HashMap<Item, Integer> catchCounts = new HashMap<Item, Integer>();
                    long simulationStart = System.currentTimeMillis();
                    for (int i = 0; i < 500; ++i) {
                        if (System.currentTimeMillis() - simulationStart > 3000L) {
                            ComplexityAnalyzer.LOGGER.warn("[ULS] Simulation timeout for '{}' after {} iterations. Skipping.", (Object)lootTableId, (Object)i);
                            catchCounts.clear();
                            break;
                        }
                        ObjectArrayList items = lootTable.getRandomItems(lootParams);
                        if (items.isEmpty()) continue;
                        for (ItemStack stack : items) {
                            if (stack.isEmpty()) continue;
                            catchCounts.merge(stack.getItem(), stack.getCount(), Integer::sum);
                        }
                    }
                    if (catchCounts.isEmpty()) continue;
                    for (Map.Entry itemEntry : catchCounts.entrySet()) {
                        Item item = (Item)itemEntry.getKey();
                        double itemsPerAttempt = (double)((Integer)itemEntry.getValue()).intValue() / 500.0;
                        if (itemsPerAttempt <= 0.0) continue;
                        double baseFactor = contextDef.baseActionCost / itemsPerAttempt * contextDef.sourceType.getBaseMultiplier();
                        String details = String.format("From loot table '%s', Chance: %.3f%%", lootTableId, itemsPerAttempt * 100.0);
                        BaseResourceData.Builder builder = new BaseResourceData.Builder(item, this).sourceType(contextDef.sourceType).baseFactor(baseFactor).details(details);
                        if (contextDef.sourceType == BaseResourceData.ResourceSourceType.PIGLIN_BARTERING) {
                            builder.baseFactor(contextDef.baseActionCost);
                            builder.sourceItems(Map.of(Items.GOLD_INGOT, 1.0 / itemsPerAttempt));
                        }
                        BaseResourceData data = builder.build();
                        this.allLootData.computeIfAbsent(contextDef.sourceType, k -> new HashMap()).put(item, data);
                    }
                }
                catch (Exception e) {
                    ComplexityAnalyzer.LOGGER.debug("[ULS] Error processing '{}': {}", (Object)lootTableId, (Object)e.getMessage());
                }
            }
        }
        catch (Exception e) {
            ComplexityAnalyzer.LOGGER.error("[ULS] Critical error during auto-scan: ", (Throwable)e);
        }
        finally {
            try {
                rootLogger.get().removeFilter((Filter)filter);
                filter.stop();
            }
            catch (Exception e) {}
        }
        long duration = System.currentTimeMillis() - startTime;
        int totalItemsFound = this.allLootData.values().stream().mapToInt(Map::size).sum();
        ComplexityAnalyzer.LOGGER.info("[ULS] Auto-scan complete in {}ms. Processed {} loot tables ({} skipped), found {} unique items.", new Object[]{duration, tablesProcessed, tablesSkipped, totalItemsFound});
        for (Map.Entry<BaseResourceData.ResourceSourceType, Map<Item, BaseResourceData>> entry : this.allLootData.entrySet()) {
            ComplexityAnalyzer.LOGGER.info("[ULS]   {} -> {} items", (Object)entry.getKey().getDisplayName(), (Object)entry.getValue().size());
        }
    }

    private Set<ResourceKey<LootTable>> getAllLootTableKeys(MinecraftServer server) {
        try {
            RegistryAccess.Frozen registries = server.reloadableRegistries().get();
            Registry lootRegistry = (Registry)registries.registry(Registries.LOOT_TABLE).orElseThrow();
            HashSet<ResourceKey<LootTable>> keys = new HashSet<ResourceKey<LootTable>>(lootRegistry.registryKeySet());
            ComplexityAnalyzer.LOGGER.debug("[ULS] Found {} loot tables via reloadableRegistries.", (Object)keys.size());
            return keys;
        }
        catch (Exception e) {
            ComplexityAnalyzer.LOGGER.error("[ULS] Failed to access loot table registry:", (Throwable)e);
            return this.getFallbackLootTables();
        }
    }

    private Set<ResourceKey<LootTable>> getFallbackLootTables() {
        String[] knownTables;
        HashSet<ResourceKey<LootTable>> keys = new HashSet<ResourceKey<LootTable>>();
        for (String path : knownTables = new String[]{"gameplay/fishing", "gameplay/fishing/fish", "gameplay/fishing/treasure", "gameplay/fishing/junk", "gameplay/piglin_bartering", "chests/abandoned_mineshaft", "chests/ancient_city", "chests/bastion_treasure", "chests/bastion_bridge", "chests/buried_treasure", "chests/desert_pyramid", "chests/end_city_treasure", "chests/igloo_chest", "chests/jungle_temple", "chests/nether_bridge", "chests/pillager_outpost", "chests/shipwreck_treasure", "chests/simple_dungeon", "chests/stronghold_corridor", "chests/stronghold_library", "chests/village/village_armorer", "chests/village/village_weaponsmith", "chests/woodland_mansion", "archaeology/desert_pyramid", "archaeology/desert_well", "archaeology/ocean_ruin_cold", "archaeology/ocean_ruin_warm", "archaeology/trail_ruins_common", "archaeology/trail_ruins_rare", "shearing/beehive", "shearing/bee_nest"}) {
            ResourceLocation id = ResourceLocation.withDefaultNamespace((String)path);
            keys.add((ResourceKey<LootTable>)ResourceKey.create((ResourceKey)Registries.LOOT_TABLE, (ResourceLocation)id));
        }
        ComplexityAnalyzer.LOGGER.debug("[ULS] Loaded {} fallback loot tables.", (Object)keys.size());
        return keys;
    }

    @Override
    public boolean canProvide(Item item) {
        return this.allLootData.values().stream().anyMatch(map -> map.containsKey(item));
    }

    @Override
    public Optional<BaseResourceData> analyze(Item item) {
        for (BaseResourceData.ResourceSourceType type : BaseResourceData.ResourceSourceType.values()) {
            Map<Item, BaseResourceData> map = this.allLootData.get((Object)type);
            if (map == null || !map.containsKey(item)) continue;
            return Optional.of(map.get(item));
        }
        return Optional.empty();
    }

    private Optional<LootContextDefinition> inferContextFromId(ResourceLocation id) {
        String path = id.getPath();
        if (path.startsWith("shearing/") || path.contains("shearing")) {
            return Optional.of(new LootContextDefinition(BaseResourceData.ResourceSourceType.SHEARING, 5.0));
        }
        if (path.contains("fishing")) {
            return Optional.of(new LootContextDefinition(BaseResourceData.ResourceSourceType.FISHING, 25.0));
        }
        if (path.contains("piglin_bartering") || path.contains("bartering")) {
            return Optional.of(new LootContextDefinition(BaseResourceData.ResourceSourceType.PIGLIN_BARTERING, 0.1));
        }
        if (path.startsWith("chests/") || path.contains("chest")) {
            return Optional.of(new LootContextDefinition(BaseResourceData.ResourceSourceType.CHEST_LOOT, 100.0));
        }
        if (path.startsWith("archaeology/") || path.contains("archaeology")) {
            return Optional.of(new LootContextDefinition(BaseResourceData.ResourceSourceType.ARCHAEOLOGY, 15.0));
        }
        if (path.startsWith("gameplay/")) {
            return Optional.of(new LootContextDefinition(BaseResourceData.ResourceSourceType.GENERIC_LOOT, 50.0));
        }
        return Optional.empty();
    }

    @Override
    public int getPriority() {
        return 25;
    }

    @Override
    public String getName() {
        return "UniversalLootSource";
    }

    @Override
    public BaseResourceData.ResourceSourceType getSourceType() {
        return BaseResourceData.ResourceSourceType.GENERIC_LOOT;
    }

    public Map<BaseResourceData.ResourceSourceType, Map<Item, BaseResourceData>> getAllLootData() {
        return this.allLootData;
    }

    private static class LootFunctionFilter
    extends AbstractFilter {
        private LootFunctionFilter() {
        }

        public Filter.Result filter(LogEvent event) {
            String message;
            if (event == null || event.getLevel() != org.apache.logging.log4j.Level.WARN) {
                return Filter.Result.NEUTRAL;
            }
            String loggerName = event.getLoggerName();
            if (loggerName != null && loggerName.startsWith("net.minecraft.world.level.storage.loot.functions.") && (message = event.getMessage().getFormattedMessage()) != null && (message.contains("Couldn't set damage") || message.contains("Couldn't smelt") || message.contains("Couldn't find a compatible enchantment"))) {
                return Filter.Result.DENY;
            }
            return Filter.Result.NEUTRAL;
        }
    }

    private record LootContextDefinition(BaseResourceData.ResourceSourceType sourceType, double baseActionCost) {
        public LootParams createLootParams(ServerLevel level) {
            LootParams.Builder builder = new LootParams.Builder(level).withParameter(LootContextParams.ORIGIN, (Object)new Vec3(0.0, 0.0, 0.0));
            if (this.sourceType == BaseResourceData.ResourceSourceType.FISHING) {
                builder.withParameter(LootContextParams.TOOL, (Object)new ItemStack((ItemLike)Items.FISHING_ROD));
                return builder.create(LootContextParamSets.FISHING);
            }
            if (this.sourceType == BaseResourceData.ResourceSourceType.SHEARING) {
                builder.withParameter(LootContextParams.TOOL, (Object)new ItemStack((ItemLike)Items.SHEARS));
                return builder.create(LootContextParamSets.SHEARING);
            }
            if (this.sourceType == BaseResourceData.ResourceSourceType.PIGLIN_BARTERING) {
                try {
                    Piglin piglinEntity = (Piglin)EntityType.PIGLIN.create((Level)level);
                    if (piglinEntity != null) {
                        builder.withParameter(LootContextParams.THIS_ENTITY, (Object)piglinEntity);
                        return builder.create(LootContextParamSets.PIGLIN_BARTER);
                    }
                }
                catch (Exception e) {
                    ComplexityAnalyzer.LOGGER.warn("[ULS] Failed to create piglin entity: {}", (Object)e.getMessage());
                }
                return null;
            }
            return builder.create(LootContextParamSets.CHEST);
        }
    }
}

