/*
 * Decompiled with CFR 0.152.
 */
package org.complexityanalyzer.analyzer.solver;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.PriorityQueue;
import java.util.Set;
import net.minecraft.world.item.Item;
import org.complexityanalyzer.ComplexityAnalyzer;
import org.complexityanalyzer.analyzer.resource.SourceManager;
import org.complexityanalyzer.analyzer.resource.data.BaseResourceData;
import org.complexityanalyzer.analyzer.solver.SolverResult;
import org.complexityanalyzer.config.ComplexityConfig;
import org.complexityanalyzer.graph.IngredientSlot;
import org.complexityanalyzer.graph.RecipeCategory;
import org.complexityanalyzer.graph.RecipeGraph;
import org.complexityanalyzer.graph.RecipeNode;

public class IterativeSolver {
    private final RecipeGraph graph;
    private final SourceManager sourceManager;
    private static final double CONVERGENCE_THRESHOLD = 1.0E-8;
    private static final double EPSILON = 1.0E-12;
    private final Map<RecipeNode, RecipeCostCache> recipeCostCache;
    private final Map<Item, BaseResourceCache> baseResourceCache;
    private final Set<Item> itemsInQueue = new HashSet<Item>();
    private static final int MAX_CACHE_SIZE = 2500000;
    private int recipeCostCalculations = 0;
    private int cacheHits = 0;

    public IterativeSolver(RecipeGraph graph, SourceManager sourceManager) {
        this.graph = graph;
        this.sourceManager = sourceManager;
        this.recipeCostCache = new HashMap<RecipeNode, RecipeCostCache>();
        this.baseResourceCache = new HashMap<Item, BaseResourceCache>();
    }

    public SolverResult solve() {
        ComplexityAnalyzer.LOGGER.debug("Starting enhanced iterative solver...");
        long startTime = System.currentTimeMillis();
        Map<Item, Double> optimalComplexities = this.initializeComplexities();
        HashMap<Item, RecipeNode> optimalRecipes = new HashMap<Item, RecipeNode>();
        PriorityQueue<ItemUpdate> updateQueue = new PriorityQueue<ItemUpdate>(Comparator.comparingDouble(ItemUpdate::getPriority));
        Map<Item, Set<Item>> dependents = this.buildDependencyGraph();
        HashSet<Item> reusableVisitedSet = new HashSet<Item>();
        for (Item item : this.graph.getCorpus()) {
            updateQueue.offer(new ItemUpdate(item, 0, optimalComplexities.get(item)));
        }
        int iterations = 0;
        int itemsProcessed = 0;
        while (!updateQueue.isEmpty() && iterations < (Integer)ComplexityConfig.MAX_ITERATIONS.get()) {
            ++iterations;
            int batchSize = Math.min(updateQueue.size(), 1000);
            ArrayList<ItemUpdate> currentBatch = new ArrayList<ItemUpdate>(batchSize);
            for (int i = 0; i < batchSize && !updateQueue.isEmpty(); ++i) {
                ItemUpdate update = updateQueue.poll();
                this.itemsInQueue.remove(update.getItem());
                currentBatch.add(update);
            }
            boolean batchChanged = false;
            for (ItemUpdate update : currentBatch) {
                Item item = update.getItem();
                double oldComplexity = optimalComplexities.get(item);
                reusableVisitedSet.clear();
                ComplexityResult result = this.calculateComplexityEnhanced(item, optimalComplexities, reusableVisitedSet);
                double newComplexity = result.complexity;
                if (!this.hasSignificantChange(oldComplexity, newComplexity)) continue;
                optimalComplexities.put(item, newComplexity);
                if (result.recipe != null) {
                    optimalRecipes.put(item, result.recipe);
                } else {
                    optimalRecipes.remove(item);
                }
                this.invalidateCache(item);
                Set deps = dependents.getOrDefault(item, Collections.emptySet());
                for (Item dependent : deps) {
                    if (!this.itemsInQueue.add(dependent)) continue;
                    updateQueue.offer(new ItemUpdate(dependent, iterations, optimalComplexities.get(dependent)));
                }
                batchChanged = true;
                ++itemsProcessed;
            }
            if (batchChanged || !updateQueue.isEmpty()) continue;
            break;
        }
        boolean converged = updateQueue.isEmpty();
        long totalTime = System.currentTimeMillis() - startTime;
        this.logResults(iterations, totalTime, converged, itemsProcessed);
        this.clearCaches();
        return new SolverResult(optimalComplexities, optimalRecipes, iterations, totalTime, converged);
    }

    private Map<Item, Double> initializeComplexities() {
        HashMap<Item, Double> complexities = new HashMap<Item, Double>();
        for (Item item : this.graph.getCorpus()) {
            Optional<BaseResourceData> dataOpt = this.sourceManager.analyze(item);
            if (dataOpt.isPresent() && dataOpt.get().getSourceItems().isEmpty()) {
                complexities.put(item, dataOpt.get().getBaseFactor());
                this.baseResourceCache.put(item, new BaseResourceCache(dataOpt.get().getBaseFactor(), true));
                continue;
            }
            complexities.put(item, Double.POSITIVE_INFINITY);
        }
        return complexities;
    }

    private Map<Item, Set<Item>> buildDependencyGraph() {
        HashMap<Item, Set<Item>> dependents = new HashMap<Item, Set<Item>>();
        for (Item item : this.graph.getCorpus()) {
            Optional<BaseResourceData> dataOpt;
            if (this.graph.hasRecipe(item)) {
                for (RecipeNode recipe : this.graph.getRecipes(item)) {
                    for (IngredientSlot slot : recipe.getIngredients()) {
                        for (Item ingredient : slot.getVariants()) {
                            dependents.computeIfAbsent(ingredient, k -> new HashSet()).add(item);
                        }
                    }
                }
            }
            if (!(dataOpt = this.sourceManager.analyze(item)).isPresent()) continue;
            Map<Item, Double> sourceItems = dataOpt.get().getSourceItems();
            for (Item sourceItem : sourceItems.keySet()) {
                dependents.computeIfAbsent(sourceItem, k -> new HashSet()).add(item);
            }
        }
        return dependents;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ComplexityResult calculateComplexityEnhanced(Item item, Map<Item, Double> currentComplexities, Set<Item> visitedInPath) {
        if (visitedInPath.size() > (Integer)ComplexityConfig.CYCLE_DETECTION_DEPTH.get()) {
            ComplexityAnalyzer.LOGGER.warn("Cycle detection depth exceeded for item: {}", (Object)item);
            return new ComplexityResult(Double.POSITIVE_INFINITY, null);
        }
        if (visitedInPath.contains(item)) {
            ComplexityAnalyzer.LOGGER.debug("Circular dependency detected for item: {}", (Object)item);
            return new ComplexityResult(Double.POSITIVE_INFINITY, null);
        }
        visitedInPath.add(item);
        try {
            double sourceCost = this.getBaseResourceCostEnhanced(item, currentComplexities);
            ComplexityResult craftResult = this.getCraftingCostEnhanced(item, currentComplexities);
            double biasThreshold = (Double)ComplexityConfig.SOURCE_BIAS_THRESHOLD.get();
            if (sourceCost <= craftResult.complexity * biasThreshold) {
                ComplexityResult complexityResult = new ComplexityResult(sourceCost, null);
                return complexityResult;
            }
            ComplexityResult complexityResult = craftResult;
            return complexityResult;
        }
        finally {
            visitedInPath.remove(item);
        }
    }

    private ComplexityResult getCraftingCostEnhanced(Item item, Map<Item, Double> currentComplexities) {
        if (!this.graph.hasRecipe(item)) {
            return new ComplexityResult(Double.POSITIVE_INFINITY, null);
        }
        List<RecipeNode> allRecipes = this.graph.getRecipes(item);
        List<RecipeNode> recipesToConsider = this.filterRecipes(allRecipes);
        if (recipesToConsider.isEmpty()) {
            return new ComplexityResult(Double.POSITIVE_INFINITY, null);
        }
        double minCost = Double.POSITIVE_INFINITY;
        RecipeNode bestRecipe = null;
        for (RecipeNode recipe : recipesToConsider) {
            double cost = this.calculateRecipeCostEnhanced(recipe, currentComplexities);
            if (!(cost < minCost)) continue;
            minCost = cost;
            bestRecipe = recipe;
        }
        return new ComplexityResult(minCost, bestRecipe);
    }

    private List<RecipeNode> filterRecipes(List<RecipeNode> allRecipes) {
        List<RecipeNode> primaryRecipes = allRecipes.stream().filter(r -> r.getCategory() == RecipeCategory.PRIMARY).toList();
        if (!primaryRecipes.isEmpty()) {
            return primaryRecipes;
        }
        List<RecipeNode> filteredRecipes = allRecipes.stream().filter(r -> r.getCategory() != RecipeCategory.UNPROCESSABLE && r.getCategory() != RecipeCategory.RECYCLING && r.getCategory() != RecipeCategory.STORAGE_DECOMPRESSION).toList();
        if (!filteredRecipes.isEmpty()) {
            return filteredRecipes;
        }
        return allRecipes.stream().filter(r -> r.getCategory() != RecipeCategory.UNPROCESSABLE).toList();
    }

    private double calculateRecipeCostEnhanced(RecipeNode recipe, Map<Item, Double> currentComplexities) {
        ++this.recipeCostCalculations;
        RecipeCostCache cached = this.recipeCostCache.get(recipe);
        if (cached != null && cached.isValid(currentComplexities)) {
            ++this.cacheHits;
            return cached.cost;
        }
        double ingredientsCost = 0.0;
        HashMap<Item, Double> usedComplexities = new HashMap<Item, Double>();
        for (IngredientSlot slot : recipe.getIngredients()) {
            double slotCost = this.getSlotCostEnhanced(slot, currentComplexities, usedComplexities);
            if (Double.isInfinite(slotCost)) {
                return Double.POSITIVE_INFINITY;
            }
            ingredientsCost += slotCost * (double)slot.getCount();
        }
        double totalCost = ingredientsCost * recipe.getRecipeMultiplier() / (double)recipe.getResultCount();
        if (this.recipeCostCache.size() >= 2500000) {
            ComplexityAnalyzer.LOGGER.error("!!! CRITICAL: Recipe cache overflow ({} entries) !!!", (Object)this.recipeCostCache.size());
            ComplexityAnalyzer.LOGGER.error("This should NEVER happen in normal operation. Possible infinite loop or algorithmic bug!");
            ComplexityAnalyzer.LOGGER.error("Stopping further caching to prevent OOM. Results may be slower but correct.");
            return totalCost;
        }
        this.recipeCostCache.put(recipe, new RecipeCostCache(totalCost, usedComplexities));
        return totalCost;
    }

    private double getSlotCostEnhanced(IngredientSlot slot, Map<Item, Double> currentComplexities, Map<Item, Double> usedComplexities) {
        if (slot.getVariants().isEmpty()) {
            return Double.POSITIVE_INFINITY;
        }
        double minCost = Double.POSITIVE_INFINITY;
        Item bestVariant = null;
        for (Item variant : slot.getVariants()) {
            double cost = currentComplexities.getOrDefault(variant, Double.POSITIVE_INFINITY);
            if (!(cost < minCost)) continue;
            minCost = cost;
            bestVariant = variant;
        }
        if (bestVariant != null) {
            usedComplexities.put(bestVariant, minCost);
        }
        return minCost;
    }

    private double getBaseResourceCostEnhanced(Item item, Map<Item, Double> currentComplexities) {
        BaseResourceCache cached = this.baseResourceCache.get(item);
        if (cached != null && cached.isSimple) {
            return cached.cost;
        }
        List<BaseResourceData> allSources = this.sourceManager.findAllSources(item);
        if (allSources.isEmpty()) {
            return BaseResourceData.ResourceSourceType.UNOBTAINABLE.getBaseMultiplier();
        }
        double bestCost = Double.POSITIVE_INFINITY;
        boolean foundSimpleSource = false;
        for (BaseResourceData data : allSources) {
            double sourceCost;
            if (data.getSourceItems().isEmpty()) {
                sourceCost = data.getBaseFactor();
                foundSimpleSource = true;
            } else {
                boolean isDecompressionSource;
                if (this.graph.hasRecipe(item) && (isDecompressionSource = this.graph.getRecipes(item).stream().anyMatch(r -> {
                    if (r.getCategory() != RecipeCategory.STORAGE_DECOMPRESSION) {
                        return false;
                    }
                    HashSet recipeIngredients = new HashSet();
                    r.getIngredients().forEach(slot -> recipeIngredients.addAll(slot.getVariants()));
                    return recipeIngredients.equals(data.getSourceItems().keySet());
                }))) continue;
                double dependencyCost = 0.0;
                boolean hasInfiniteDependency = false;
                for (Map.Entry<Item, Double> entry : data.getSourceItems().entrySet()) {
                    Item sourceItem = entry.getKey();
                    Double amount = entry.getValue();
                    if (amount == null || amount <= 1.0E-12) continue;
                    double itemCost = currentComplexities.getOrDefault(sourceItem, Double.POSITIVE_INFINITY);
                    if (Double.isInfinite(itemCost)) {
                        hasInfiniteDependency = true;
                        break;
                    }
                    dependencyCost += itemCost * amount;
                }
                if (hasInfiniteDependency) continue;
                sourceCost = data.getBaseFactor() + dependencyCost;
            }
            if (!(sourceCost < bestCost)) continue;
            bestCost = sourceCost;
        }
        if (foundSimpleSource && bestCost < Double.POSITIVE_INFINITY) {
            this.baseResourceCache.put(item, new BaseResourceCache(bestCost, true));
        }
        return bestCost;
    }

    private boolean hasSignificantChange(double oldValue, double newValue) {
        if (Double.isInfinite(oldValue) && Double.isInfinite(newValue)) {
            return false;
        }
        if (Double.isInfinite(oldValue) || Double.isInfinite(newValue)) {
            return true;
        }
        double delta = Math.abs(oldValue - newValue);
        double relative = oldValue > 1.0E-12 ? delta / oldValue : delta;
        return delta > 1.0E-8 && relative > 1.0E-8;
    }

    private void invalidateCache(Item item) {
        this.baseResourceCache.remove(item);
        this.recipeCostCache.entrySet().removeIf(entry -> ((RecipeCostCache)entry.getValue()).dependsOn(item));
    }

    private void clearCaches() {
        this.recipeCostCache.clear();
        this.baseResourceCache.clear();
    }

    private void logResults(int iterations, long totalTime, boolean converged, int itemsProcessed) {
        if (!converged) {
            ComplexityAnalyzer.LOGGER.warn("Enhanced solver did NOT converge after {} iterations. Results may be approximate.", ComplexityConfig.MAX_ITERATIONS.get());
        }
        ComplexityAnalyzer.LOGGER.info("Enhanced solver finished: {}ms, {} iterations, {} items processed", new Object[]{totalTime, iterations, itemsProcessed});
        if (this.recipeCostCalculations > 0) {
            double hitRate = 100.0 * (double)this.cacheHits / (double)this.recipeCostCalculations;
            ComplexityAnalyzer.LOGGER.debug("Cache stats: {} recipe calculations, {} cache hits ({} hit rate)", new Object[]{this.recipeCostCalculations, this.cacheHits, String.format("%.2f%%", hitRate)});
        }
    }

    private static class ItemUpdate {
        private final Item item;
        private final int iteration;
        private final double currentComplexity;

        public ItemUpdate(Item item, int iteration, double currentComplexity) {
            this.item = item;
            this.iteration = iteration;
            this.currentComplexity = currentComplexity;
        }

        public Item getItem() {
            return this.item;
        }

        public double getPriority() {
            return Double.isInfinite(this.currentComplexity) ? Double.MAX_VALUE : this.currentComplexity * 0.001 + (double)this.iteration;
        }
    }

    private static class ComplexityResult {
        final double complexity;
        final RecipeNode recipe;

        ComplexityResult(double complexity, RecipeNode recipe) {
            this.complexity = complexity;
            this.recipe = recipe;
        }
    }

    private static class BaseResourceCache {
        final double cost;
        final boolean isSimple;

        BaseResourceCache(double cost, boolean isSimple) {
            this.cost = cost;
            this.isSimple = isSimple;
        }
    }

    private static class RecipeCostCache {
        final double cost;
        final Map<Item, Double> dependencies;

        RecipeCostCache(double cost, Map<Item, Double> dependencies) {
            this.cost = cost;
            this.dependencies = new HashMap<Item, Double>(dependencies);
        }

        boolean isValid(Map<Item, Double> currentComplexities) {
            for (Map.Entry<Item, Double> entry : this.dependencies.entrySet()) {
                Double currentCost = currentComplexities.get(entry.getKey());
                if (currentCost != null && !(Math.abs(currentCost - entry.getValue()) > (Double)ComplexityConfig.CONVERGENCE_THRESHOLD.get())) continue;
                return false;
            }
            return true;
        }

        boolean dependsOn(Item item) {
            return this.dependencies.containsKey(item);
        }
    }
}

