/*
 * Decompiled with CFR 0.152.
 */
package org.complexityanalyzer.export;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonSyntaxException;
import com.google.gson.TypeAdapter;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonToken;
import com.google.gson.stream.JsonWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.MinecraftServer;
import net.minecraft.world.entity.EntityType;
import net.minecraft.world.entity.MobCategory;
import net.minecraft.world.item.Item;
import net.minecraft.world.level.storage.LevelResource;
import org.complexityanalyzer.ComplexityAnalyzer;
import org.complexityanalyzer.analyzer.resource.data.BaseResourceData;
import org.complexityanalyzer.analyzer.resource.providers.MobPropertyProvider;
import org.complexityanalyzer.analyzer.resource.sources.HardcodedSourcesProvider;
import org.complexityanalyzer.api.IHardcodedSourceRegistry;
import org.complexityanalyzer.core.AnalysisEngine;
import org.complexityanalyzer.data.ItemComplexity;
import org.complexityanalyzer.export.ExportData;

public class ComplexityExporter {
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().disableHtmlEscaping().registerTypeAdapter(Double.class, (Object)new DoubleSerializer()).registerTypeAdapter(Double.TYPE, (Object)new DoubleSerializer()).create();
    private static final DateTimeFormatter TIMESTAMP_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd_HH-mm-ss");

    public static Path exportAllItems(MinecraftServer server, AnalysisEngine engine) throws IOException {
        Path exportDir = ComplexityExporter.getExportDirectory(server);
        String timestamp = LocalDateTime.now().format(TIMESTAMP_FORMAT);
        Path exportFile = exportDir.resolve("items_all_" + timestamp + ".json");
        ArrayList<ExportData.ItemData> allItems = new ArrayList<ExportData.ItemData>();
        for (Item item : BuiltInRegistries.ITEM) {
            engine.getComplexityResult(item).ifPresent(c -> allItems.add(ComplexityExporter.buildItemData(item, BuiltInRegistries.ITEM.getKey((Object)item), c, engine)));
        }
        allItems.sort(Comparator.comparingDouble(ExportData.ItemData::complexity).reversed());
        String json = GSON.toJson((Object)new ExportData(timestamp, allItems.size(), allItems));
        Files.writeString(exportFile, (CharSequence)json, new OpenOption[0]);
        return exportFile;
    }

    public static Path exportItemsByCategory(MinecraftServer server, AnalysisEngine engine, String categoryName) throws IOException {
        Path exportDir = ComplexityExporter.getExportDirectory(server);
        String timestamp = LocalDateTime.now().format(TIMESTAMP_FORMAT);
        Path exportFile = exportDir.resolve("items_category_" + categoryName.toLowerCase() + "_" + timestamp + ".json");
        ArrayList<ExportData.ItemData> filteredItems = new ArrayList<ExportData.ItemData>();
        for (Item item : BuiltInRegistries.ITEM) {
            engine.getComplexityResult(item).ifPresent(c -> {
                if (c.getCategory().getDisplayName().equalsIgnoreCase(categoryName)) {
                    filteredItems.add(ComplexityExporter.buildItemData(item, BuiltInRegistries.ITEM.getKey((Object)item), c, engine));
                }
            });
        }
        filteredItems.sort(Comparator.comparingDouble(ExportData.ItemData::complexity).reversed());
        String json = GSON.toJson((Object)new ExportData(timestamp, filteredItems.size(), filteredItems));
        Files.writeString(exportFile, (CharSequence)json, new OpenOption[0]);
        return exportFile;
    }

    public static Path exportTopItems(MinecraftServer server, AnalysisEngine engine, int count) throws IOException {
        Path exportDir = ComplexityExporter.getExportDirectory(server);
        String timestamp = LocalDateTime.now().format(TIMESTAMP_FORMAT);
        Path exportFile = exportDir.resolve("items_top" + count + "_" + timestamp + ".json");
        ArrayList<ExportData.ItemData> allItems = new ArrayList<ExportData.ItemData>();
        for (Item item : BuiltInRegistries.ITEM) {
            engine.getComplexityResult(item).ifPresent(c -> allItems.add(ComplexityExporter.buildItemData(item, BuiltInRegistries.ITEM.getKey((Object)item), c, engine)));
        }
        allItems.sort(Comparator.comparingDouble(ExportData.ItemData::complexity).reversed());
        List<ExportData.ItemData> topItems = allItems.stream().limit(count).collect(Collectors.toList());
        String json = GSON.toJson((Object)new ExportData(timestamp, topItems.size(), topItems));
        Files.writeString(exportFile, (CharSequence)json, new OpenOption[0]);
        return exportFile;
    }

    public static Path exportItemsCSV(MinecraftServer server, AnalysisEngine engine) throws IOException {
        Path exportDir = ComplexityExporter.getExportDirectory(server);
        String timestamp = LocalDateTime.now().format(TIMESTAMP_FORMAT);
        Path exportFile = exportDir.resolve("items_all_" + timestamp + ".csv");
        ArrayList<CsvRow> rows = new ArrayList<CsvRow>();
        for (Item item : BuiltInRegistries.ITEM) {
            engine.getComplexityResult(item).ifPresent(c -> {
                boolean isHardcoded = ComplexityExporter.checkIfHardcoded(item);
                rows.add(new CsvRow(BuiltInRegistries.ITEM.getKey((Object)item).toString(), item.getDescription().getString(), c.getComplexity(), c.getCategory().getDisplayName(), c.hasRecipe(), c.getDepth(), engine.getUsageCount(item), c.isValid(), c.hasCycle(), isHardcoded));
            });
        }
        rows.sort(Comparator.comparingDouble(CsvRow::complexity).reversed());
        try (PrintWriter writer = new PrintWriter(exportFile.toFile(), StandardCharsets.UTF_8);){
            writer.println("Item ID,Display Name,Complexity,Category,Has Recipe,Crafting Depth,Used In Recipes,Is Valid,Has Cycle,Is Hardcoded");
            for (CsvRow row : rows) {
                writer.println(String.format(Locale.US, "%s,\"%s\",%.2f,%s,%s,%d,%d,%s,%s,%s", row.itemId, row.displayName.replace("\"", "\"\""), row.complexity, row.category, row.hasRecipe, row.craftingDepth, row.usedInRecipes, row.isValid, row.hasCycle, row.isHardcoded));
            }
        }
        return exportFile;
    }

    public static Path exportSingleItem(MinecraftServer server, AnalysisEngine engine, String itemIdString) throws IOException {
        Path exportDir = ComplexityExporter.getExportDirectory(server).resolve("items");
        Files.createDirectories(exportDir, new FileAttribute[0]);
        ResourceLocation itemId = ResourceLocation.parse((String)itemIdString);
        Item item = (Item)BuiltInRegistries.ITEM.getOptional(itemId).orElseThrow(() -> new IllegalArgumentException("Item not found: " + itemIdString));
        ItemComplexity complexity = engine.getComplexityResult(item).orElseThrow(() -> new IllegalStateException("Failed to analyze item: " + itemIdString));
        ExportData.ItemData itemData = ComplexityExporter.buildItemData(item, itemId, complexity, engine);
        Path exportFile = exportDir.resolve(itemId.getNamespace() + "_" + itemId.getPath() + ".json");
        Files.writeString(exportFile, (CharSequence)GSON.toJson((Object)itemData), new OpenOption[0]);
        return exportFile;
    }

    public static Path exportAllMobs(MinecraftServer server, AnalysisEngine engine, String format) throws IOException {
        return ComplexityExporter.exportMobsAs(server, engine, format, null, -1, "all");
    }

    public static Path exportMobsByCategory(MinecraftServer server, AnalysisEngine engine, String categoryName) throws IOException {
        return ComplexityExporter.exportMobsAs(server, engine, "json", categoryName, -1, "category_" + categoryName);
    }

    public static Path exportTopMobs(MinecraftServer server, AnalysisEngine engine, int count) throws IOException {
        return ComplexityExporter.exportMobsAs(server, engine, "json", null, count, "top" + count);
    }

    public static Path exportSingleMob(MinecraftServer server, AnalysisEngine engine, String mobIdString) throws IOException {
        Path exportDir = ComplexityExporter.getExportDirectory(server).resolve("mobs");
        Files.createDirectories(exportDir, new FileAttribute[0]);
        ResourceLocation mobId = ResourceLocation.parse((String)mobIdString);
        EntityType mobType = Optional.of((EntityType)BuiltInRegistries.ENTITY_TYPE.get(mobId)).orElseThrow(() -> new IllegalArgumentException("Mob not found: " + mobIdString));
        ExportData.MobData mobData = ComplexityExporter.buildMobData(mobType, engine);
        if (mobData == null) {
            throw new IllegalStateException("Failed to analyze mob: " + mobIdString);
        }
        Path exportFile = exportDir.resolve(mobId.getNamespace() + "_" + mobId.getPath() + ".json");
        Files.writeString(exportFile, (CharSequence)GSON.toJson((Object)mobData), new OpenOption[0]);
        return exportFile;
    }

    private static Path getExportDirectory(MinecraftServer server) throws IOException {
        Path dir = server.getWorldPath(LevelResource.ROOT).resolve("data").resolve("complexityanalyzer").resolve("export");
        if (!Files.exists(dir = dir.toAbsolutePath().normalize(), new LinkOption[0])) {
            Files.createDirectories(dir, new FileAttribute[0]);
        }
        ComplexityAnalyzer.LOGGER.info("[Exporter] Resolved export directory to absolute path: {}", (Object)dir);
        return dir;
    }

    private static boolean checkIfHardcoded(Item item) {
        try {
            IHardcodedSourceRegistry registry = HardcodedSourcesProvider.getRegistry();
            return registry.isRegistered(item);
        }
        catch (IllegalStateException e) {
            return false;
        }
    }

    private static ExportData.ItemData buildItemData(Item item, ResourceLocation itemId, ItemComplexity complexity, AnalysisEngine engine) {
        List<ExportData.SourceData> sources = engine.findAllSourcesForItem(item).stream().map(data -> ComplexityExporter.buildSourceData(data, engine)).sorted(Comparator.comparingDouble(ExportData.SourceData::estimatedCost)).collect(Collectors.toList());
        boolean isHardcoded = ComplexityExporter.checkIfHardcoded(item);
        return new ExportData.ItemData(itemId.toString(), item.getDescription().getString(), complexity.getComplexity(), complexity.getCategory().getDisplayName(), complexity.hasRecipe(), complexity.getDepth(), engine.getUsageCount(item), complexity.isValid(), complexity.hasCycle(), isHardcoded, sources);
    }

    private static ExportData.SourceData buildSourceData(BaseResourceData data, AnalysisEngine engine) {
        double fullCost = data.getBaseFactor();
        HashMap<String, Double> ingredients = new HashMap<String, Double>();
        if (!data.getSourceItems().isEmpty()) {
            for (Map.Entry<Item, Double> entry : data.getSourceItems().entrySet()) {
                Item sourceItem = entry.getKey();
                double amount = entry.getValue();
                ingredients.put(BuiltInRegistries.ITEM.getKey((Object)sourceItem).toString(), amount);
                Optional<ItemComplexity> sourceComplexity = engine.getComplexityResult(sourceItem);
                if (sourceComplexity.isPresent() && sourceComplexity.get().isValid()) {
                    fullCost += sourceComplexity.get().getComplexity() * amount;
                    continue;
                }
                fullCost = Double.POSITIVE_INFINITY;
                break;
            }
        }
        return new ExportData.SourceData(data.getSourceType().getDisplayName(), data.getBaseFactor(), fullCost, data.getDetails(), ingredients);
    }

    private static Path exportMobsAs(MinecraftServer server, AnalysisEngine engine, String format, String categoryFilter, int topN, String fileSuffix) throws IOException {
        List<Object> mobDataList = new ArrayList<ExportData.MobData>();
        for (EntityType type : BuiltInRegistries.ENTITY_TYPE) {
            ExportData.MobData data;
            if (type.getCategory() == MobCategory.MISC || categoryFilter != null && !type.getCategory().getName().equalsIgnoreCase(categoryFilter) || (data = ComplexityExporter.buildMobData(type, engine)) == null) continue;
            mobDataList.add(data);
        }
        mobDataList.sort(Comparator.comparingDouble(ExportData.MobData::combatPower).reversed());
        if (topN > 0) {
            mobDataList = mobDataList.stream().limit(topN).collect(Collectors.toList());
        }
        Path exportDir = ComplexityExporter.getExportDirectory(server);
        String timestamp = LocalDateTime.now().format(TIMESTAMP_FORMAT);
        Path exportPath = exportDir.resolve("mobs_" + fileSuffix + "_" + timestamp + "." + format);
        if ("csv".equalsIgnoreCase(format)) {
            try (PrintWriter writer = new PrintWriter(exportPath.toFile(), StandardCharsets.UTF_8);){
                writer.println("Name,ID,Category,Health,Damage,Armor,Survivability,Threat,Combat Power,Rarity,Is Boss,Is MiniBoss,Notable Drops");
                for (ExportData.MobData mobData : mobDataList) {
                    String drops = mobData.drops().stream().map(d -> String.format("%s (%.2f)", d.itemName(), d.yieldPerKill())).collect(Collectors.joining("; "));
                    writer.println(String.format(Locale.US, "\"%s\",\"%s\",\"%s\",%.2f,%.2f,%.2f,%.2f,%.2f,%.2f,%.2f,%b,%b,\"%s\"", mobData.name(), mobData.id(), mobData.category(), mobData.health(), mobData.damage(), mobData.armor(), mobData.survivability(), mobData.threat(), mobData.combatPower(), mobData.rarity(), mobData.isBoss(), mobData.isMiniBoss(), drops.isEmpty() ? "None" : drops));
                }
            }
        } else {
            Files.writeString(exportPath, (CharSequence)GSON.toJson(mobDataList), new OpenOption[0]);
        }
        return exportPath;
    }

    private static ExportData.MobData buildMobData(EntityType<?> type, AnalysisEngine engine) {
        MobPropertyProvider mobProvider = engine.getMobPropertyProvider().orElse(null);
        if (mobProvider == null) {
            return null;
        }
        MobPropertyProvider.MobProperties props = mobProvider.getProperties(type).orElse(null);
        if (props == null) {
            return null;
        }
        ArrayList<ExportData.MobDropData> drops = new ArrayList<ExportData.MobDropData>();
        engine.getMobDropSource().ifPresent(source -> drops.addAll(source.getDropsForEntity(type).stream().map(d -> new ExportData.MobDropData(BuiltInRegistries.ITEM.getKey((Object)d.item()).toString(), d.item().getDescription().getString(), d.averageYield())).toList()));
        return new ExportData.MobData(type.getDescription().getString(), BuiltInRegistries.ENTITY_TYPE.getKey(type).toString(), type.getCategory().getName(), props.maxHealth(), props.attackDamage(), props.armor(), props.calculateSurvivability(), props.calculateThreat(), props.calculateCombatPower(), mobProvider.getRarity(type), mobProvider.isBoss(type), mobProvider.isMiniBoss(type), drops);
    }

    private record CsvRow(String itemId, String displayName, double complexity, String category, boolean hasRecipe, int craftingDepth, int usedInRecipes, boolean isValid, boolean hasCycle, boolean isHardcoded) {
    }

    private static class DoubleSerializer
    extends TypeAdapter<Double> {
        private DoubleSerializer() {
        }

        public void write(JsonWriter out, Double value) throws IOException {
            if (value == null) {
                out.nullValue();
            } else if (Double.isInfinite(value)) {
                out.value(value > 0.0 ? "Infinity" : "-Infinity");
            } else if (Double.isNaN(value)) {
                out.value("NaN");
            } else {
                out.value((Number)value);
            }
        }

        public Double read(JsonReader in) throws IOException {
            return switch (in.peek()) {
                case JsonToken.STRING -> {
                    String str;
                    switch (str = in.nextString()) {
                        case "Infinity": {
                            yield Double.POSITIVE_INFINITY;
                        }
                        case "-Infinity": {
                            yield Double.NEGATIVE_INFINITY;
                        }
                        case "NaN": {
                            yield Double.NaN;
                        }
                    }
                    yield Double.parseDouble(str);
                }
                case JsonToken.NUMBER -> in.nextDouble();
                case JsonToken.NULL -> {
                    in.nextNull();
                    yield null;
                }
                default -> throw new JsonSyntaxException("Expected number or string");
            };
        }
    }
}

