/*
 * Decompiled with CFR 0.152.
 */
package org.complexityanalyzer.geoscan;

import java.io.IOException;
import java.nio.file.Path;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Stream;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.MinecraftServer;
import net.minecraft.world.level.block.Block;
import org.complexityanalyzer.ComplexityAnalyzer;
import org.complexityanalyzer.geoscan.analysis.HeuristicAnalyzer;
import org.complexityanalyzer.geoscan.data.BiomeDataMapper;
import org.complexityanalyzer.geoscan.data.BiomeScanData;
import org.complexityanalyzer.geoscan.data.ChunkSnapshot;
import org.complexityanalyzer.geoscan.data.ScanMetadata;
import org.complexityanalyzer.geoscan.storage.GeoDataStorage;

public class GeoDatabase {
    private final GeoDataStorage storage;
    private final HeuristicAnalyzer analyzer;
    private final BiomeDataMapper mapper;
    private final Map<ResourceLocation, Map<ResourceLocation, BiomeScanData>> inMemoryData;
    private final Map<Block, AtomicLong> globalBlockCountsCache = new ConcurrentHashMap<Block, AtomicLong>();
    private final AtomicLong totalBlocksInCache = new AtomicLong(0L);

    public GeoDatabase(MinecraftServer server) {
        this.storage = new GeoDataStorage(server);
        this.analyzer = new HeuristicAnalyzer();
        this.mapper = new BiomeDataMapper();
        this.storage.ensureDirectoriesExist();
        this.inMemoryData = new ConcurrentHashMap<ResourceLocation, Map<ResourceLocation, BiomeScanData>>();
    }

    public ScanMetadata.ScanPhase getScanPhase() {
        return this.storage.loadMetadata().scanPhase();
    }

    public void setScanPhase(ScanMetadata.ScanPhase phase) {
        this.storage.saveMetadata(new ScanMetadata(phase));
    }

    public boolean analyzeSnapshotForRecon(ChunkSnapshot snapshot) {
        return this.analyzer.analyzeSnapshotForRecon(snapshot);
    }

    public void appendReconData(ResourceLocation dimension, ResourceLocation biome, List<ChunkSnapshot> newSnapshots) {
        this.storage.appendReconData(dimension, biome, newSnapshots);
    }

    public int countReconChunks(ResourceLocation dimension, ResourceLocation biome) {
        return this.storage.countReconChunks(dimension, biome);
    }

    public Map<ResourceLocation, Map<ResourceLocation, Path>> getAllReconFilePaths() {
        return this.storage.getAllReconFilePaths();
    }

    public void buildHeuristicFromFiles(Map<ResourceLocation, Map<ResourceLocation, Path>> reconFilePaths) {
        this.analyzer.buildHeuristics(reconFilePaths, this.storage);
    }

    public BiomeScanData refineRawDataFromStream(Stream<ChunkSnapshot> snapshotStream, ResourceLocation dimensionId) {
        return this.analyzer.refineRawData(snapshotStream, dimensionId);
    }

    public Stream<ChunkSnapshot> streamReconFile(Path path) {
        return this.storage.streamReconFile(path);
    }

    public void loadAll() {
        if (this.isLoaded()) {
            return;
        }
        ComplexityAnalyzer.LOGGER.info("Loading all final geo-data from disk...");
        this.inMemoryData.clear();
        Map<ResourceLocation, Map<ResourceLocation, BiomeScanData>> loadedData = this.storage.loadAllFinalData(this.mapper);
        this.inMemoryData.putAll(loadedData);
        this.rebuildGlobalCache();
        ComplexityAnalyzer.LOGGER.info("Finished loading geo-data. Found data for {} dimensions.", (Object)this.inMemoryData.size());
    }

    public void saveBiomeData(ResourceLocation dimension, ResourceLocation biome, BiomeScanData data) {
        if (data == null || data.getChunksScanned() == 0) {
            return;
        }
        this.storage.saveFinalBiomeData(dimension, biome, data, this.mapper);
        Map dimData = this.inMemoryData.computeIfAbsent(dimension, k -> new ConcurrentHashMap());
        BiomeScanData oldData = dimData.put(biome, data);
        this.updateGlobalCache(oldData, data);
    }

    public void clearFinalData() throws IOException {
        this.storage.deleteFinalData();
        this.inMemoryData.clear();
        this.rebuildGlobalCache();
    }

    public void clearAllData() {
        this.storage.deleteAllData();
        this.inMemoryData.clear();
        this.analyzer.clear();
        this.rebuildGlobalCache();
        ComplexityAnalyzer.LOGGER.info("Cleared all geo-data files and reset in-memory state.");
    }

    public void clear() {
        this.clearAllData();
    }

    public boolean isLoaded() {
        return !this.inMemoryData.isEmpty();
    }

    public Optional<BiomeScanData> getBiomeData(ResourceLocation dim, ResourceLocation biome) {
        return Optional.ofNullable(this.inMemoryData.get(dim)).map(dimData -> (BiomeScanData)dimData.get(biome));
    }

    public double getGlobalRarity(Block block) {
        if (!this.isLoaded() && this.analyzer.hasHeuristics()) {
            return 0.0;
        }
        long total = this.totalBlocksInCache.get();
        if (total == 0L) {
            return 0.0;
        }
        long targetCount = this.globalBlockCountsCache.getOrDefault(block, new AtomicLong(0L)).get();
        return (double)targetCount / (double)total;
    }

    private void updateGlobalCache(BiomeScanData oldData, BiomeScanData newData) {
        if (oldData != null) {
            oldData.getInternalBlockCounts().forEach((block, count) -> {
                long value = count.get();
                this.globalBlockCountsCache.computeIfPresent((Block)block, (k, v) -> {
                    long newCount = v.addAndGet(-value);
                    return newCount > 0L ? v : null;
                });
                this.totalBlocksInCache.addAndGet(-value);
            });
        }
        if (newData != null) {
            newData.getInternalBlockCounts().forEach((block, count) -> {
                long value = count.get();
                this.globalBlockCountsCache.computeIfAbsent((Block)block, k -> new AtomicLong(0L)).addAndGet(value);
                this.totalBlocksInCache.addAndGet(value);
            });
        }
    }

    private void rebuildGlobalCache() {
        this.globalBlockCountsCache.clear();
        this.totalBlocksInCache.set(0L);
        long totalItems = this.inMemoryData.values().stream().mapToLong(Map::size).sum();
        if (totalItems == 0L) {
            ComplexityAnalyzer.LOGGER.info("Global block rarity cache is empty (no data loaded).");
            return;
        }
        ComplexityAnalyzer.LOGGER.debug("Rebuilding global cache from {} biome data entries...", (Object)totalItems);
        long processedItems = 0L;
        int nextLogPercentage = 10;
        for (Map<ResourceLocation, BiomeScanData> dimMap : this.inMemoryData.values()) {
            for (BiomeScanData data : dimMap.values()) {
                double currentPercentage;
                this.updateGlobalCache(null, data);
                if (!((currentPercentage = (double)(++processedItems) / (double)totalItems * 100.0) >= (double)nextLogPercentage)) continue;
                ComplexityAnalyzer.LOGGER.debug("Cache rebuild: {}% complete", (Object)nextLogPercentage);
                nextLogPercentage += 10;
            }
        }
        ComplexityAnalyzer.LOGGER.info("Global block rarity cache rebuilt. Total blocks counted: {}", (Object)this.totalBlocksInCache.get());
    }
}

