/*
 * Decompiled with CFR 0.152.
 */
package org.complexityanalyzer.geoscan.analysis;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.Blocks;
import org.complexityanalyzer.ComplexityAnalyzer;
import org.complexityanalyzer.geoscan.data.BiomeScanData;
import org.complexityanalyzer.geoscan.data.ChunkSnapshot;
import org.complexityanalyzer.geoscan.storage.GeoDataStorage;

public class HeuristicAnalyzer {
    private static final int RECON_DIVERSITY_THRESHOLD = 50;
    private static final int REFINE_UNNATURAL_THRESHOLD = 64;
    private static final double NATURAL_BLOCK_RARITY_THRESHOLD = 5.0E-5;
    private final Map<ResourceLocation, Set<Block>> dimensionalHeuristics = new ConcurrentHashMap<ResourceLocation, Set<Block>>();

    public boolean analyzeSnapshotForRecon(ChunkSnapshot snapshot) {
        int uniqueBlockTypes = snapshot.blockCounts().size();
        if (uniqueBlockTypes > 50) {
            ComplexityAnalyzer.LOGGER.debug("Recon Pass: Skipping chunk [{}, {}]. Found {} unique block types (Threshold: {})", new Object[]{snapshot.chunkX(), snapshot.chunkZ(), uniqueBlockTypes, 50});
            return false;
        }
        return true;
    }

    public void buildHeuristics(Map<ResourceLocation, Map<ResourceLocation, Path>> reconFilePaths, GeoDataStorage storage) {
        this.dimensionalHeuristics.clear();
        ConcurrentHashMap<ResourceLocation, List> pathsByDimension = new ConcurrentHashMap<ResourceLocation, List>();
        reconFilePaths.forEach((dimId, biomeMap) -> pathsByDimension.computeIfAbsent((ResourceLocation)dimId, k -> new ArrayList()).addAll(biomeMap.values()));
        pathsByDimension.forEach((dimId, paths) -> {
            ComplexityAnalyzer.LOGGER.debug("Building heuristic for dimension: {}", dimId);
            try (Stream allSnapshotsInDim = paths.parallelStream().flatMap(storage::streamReconFile);){
                Map<Block, Long> totalCounts = allSnapshotsInDim.flatMap(snapshot -> snapshot.blockCounts().entrySet().stream()).collect(Collectors.groupingBy(entry -> (Block)BuiltInRegistries.BLOCK.get(ResourceLocation.parse((String)((String)entry.getKey()))), Collectors.summingLong(Map.Entry::getValue)));
                long totalBlocksInDim = totalCounts.values().stream().mapToLong(Long::longValue).sum();
                ConcurrentHashMap.KeySetView dimensionHeuristic = ConcurrentHashMap.newKeySet();
                if (totalBlocksInDim > 0L) {
                    totalCounts.forEach((block, count) -> {
                        if (block != null && block != Blocks.AIR && (double)count.longValue() / (double)totalBlocksInDim > 5.0E-5) {
                            dimensionHeuristic.add(block);
                        }
                    });
                }
                this.dimensionalHeuristics.put((ResourceLocation)dimId, dimensionHeuristic);
                ComplexityAnalyzer.LOGGER.info("Heuristic for {} built. Found {} common 'natural' blocks.", dimId, (Object)dimensionHeuristic.size());
            }
        });
    }

    public BiomeScanData refineRawData(Stream<ChunkSnapshot> snapshotStream, ResourceLocation dimensionId) {
        BiomeScanData finalCleanData = new BiomeScanData();
        snapshotStream.filter(snapshot -> this.isChunkCleanByHeuristic((ChunkSnapshot)snapshot, dimensionId)).forEach(snapshot -> {
            finalCleanData.addScannedChunk(snapshot.chunkX(), snapshot.chunkZ());
            snapshot.blockCounts().forEach((blockId, count) -> {
                Block block = (Block)BuiltInRegistries.BLOCK.get(ResourceLocation.parse((String)blockId));
                if (block != Blocks.AIR && block != Blocks.BEDROCK) {
                    finalCleanData.addBlock(block, count.longValue());
                }
            });
        });
        return finalCleanData;
    }

    public void clear() {
        this.dimensionalHeuristics.clear();
    }

    public boolean hasHeuristics() {
        return this.dimensionalHeuristics.isEmpty();
    }

    private boolean isChunkCleanByHeuristic(ChunkSnapshot snapshot, ResourceLocation dimensionId) {
        if (this.hasHeuristics()) {
            return true;
        }
        Set<Block> heuristic = this.dimensionalHeuristics.get(dimensionId);
        if (heuristic == null) {
            ComplexityAnalyzer.LOGGER.warn("No heuristic found for dimension {}. Accepting chunk at [{}, {}] without filtering.", new Object[]{dimensionId, snapshot.chunkX(), snapshot.chunkZ()});
            return true;
        }
        int unnaturalBlockCount = 0;
        for (Map.Entry<String, Integer> entry : snapshot.blockCounts().entrySet()) {
            Block block = (Block)BuiltInRegistries.BLOCK.get(ResourceLocation.parse((String)entry.getKey()));
            if (block != Blocks.AIR && !heuristic.contains(block)) {
                unnaturalBlockCount += entry.getValue().intValue();
            }
            if (unnaturalBlockCount <= 64) continue;
            return false;
        }
        return true;
    }
}

