/*
 * Decompiled with CFR 0.152.
 */
package org.complexityanalyzer.geoscan.storage;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonSyntaxException;
import java.io.BufferedWriter;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Stream;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.MinecraftServer;
import net.minecraft.world.level.storage.LevelResource;
import org.complexityanalyzer.ComplexityAnalyzer;
import org.complexityanalyzer.geoscan.data.BiomeDataMapper;
import org.complexityanalyzer.geoscan.data.BiomeScanData;
import org.complexityanalyzer.geoscan.data.ChunkSnapshot;
import org.complexityanalyzer.geoscan.data.ScanMetadata;

public class GeoDataStorage {
    private static final Gson GSON = new GsonBuilder().create();
    private static final Gson PRETTY_GSON = new GsonBuilder().setPrettyPrinting().create();
    private final Path dataDir;
    private final Path reconDir;
    private final Path finalDir;
    private final Path metadataFile;

    public GeoDataStorage(MinecraftServer server) {
        this.dataDir = server.getWorldPath(LevelResource.ROOT).resolve("data").resolve("complexityanalyzer");
        this.reconDir = this.dataDir.resolve("recon");
        this.finalDir = this.dataDir.resolve("final");
        this.metadataFile = this.dataDir.resolve("metadata.json");
    }

    public void ensureDirectoriesExist() {
        try {
            Files.createDirectories(this.reconDir, new FileAttribute[0]);
            Files.createDirectories(this.finalDir, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new RuntimeException("Could not create world-specific geo-data directories!", e);
        }
    }

    public ScanMetadata loadMetadata() {
        ScanMetadata scanMetadata;
        if (!Files.exists(this.metadataFile, new LinkOption[0])) {
            return new ScanMetadata(ScanMetadata.ScanPhase.IDLE);
        }
        FileReader reader = new FileReader(this.metadataFile.toFile());
        try {
            ScanMetadata meta = (ScanMetadata)PRETTY_GSON.fromJson((Reader)reader, ScanMetadata.class);
            scanMetadata = meta != null ? meta : new ScanMetadata(ScanMetadata.ScanPhase.IDLE);
        }
        catch (Throwable throwable) {
            try {
                try {
                    reader.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                ComplexityAnalyzer.LOGGER.error("Failed to read metadata file! Assuming IDLE state.", (Throwable)e);
                return new ScanMetadata(ScanMetadata.ScanPhase.IDLE);
            }
        }
        reader.close();
        return scanMetadata;
    }

    public void saveMetadata(ScanMetadata metadata) {
        try (FileWriter writer = new FileWriter(this.metadataFile.toFile());){
            PRETTY_GSON.toJson((Object)metadata, (Appendable)writer);
        }
        catch (IOException e) {
            ComplexityAnalyzer.LOGGER.error("Failed to write metadata file!", (Throwable)e);
        }
    }

    public void appendReconData(ResourceLocation dimension, ResourceLocation biome, List<ChunkSnapshot> newSnapshots) {
        if (newSnapshots.isEmpty()) {
            return;
        }
        Path file = this.getReconFilePath(dimension, biome);
        try {
            Files.createDirectories(file.getParent(), new FileAttribute[0]);
            try (BufferedWriter writer = Files.newBufferedWriter(file, StandardCharsets.UTF_8, StandardOpenOption.CREATE, StandardOpenOption.APPEND);){
                for (ChunkSnapshot snapshot : newSnapshots) {
                    writer.write(GSON.toJson((Object)snapshot));
                    writer.newLine();
                }
            }
        }
        catch (IOException e) {
            ComplexityAnalyzer.LOGGER.error("Failed to append recon data for biome {}", (Object)biome, (Object)e);
        }
    }

    public Map<ResourceLocation, Map<ResourceLocation, Path>> getAllReconFilePaths() {
        HashMap<ResourceLocation, Map<ResourceLocation, Path>> allPaths = new HashMap<ResourceLocation, Map<ResourceLocation, Path>>();
        if (!Files.exists(this.reconDir, new LinkOption[0])) {
            return allPaths;
        }
        try (Stream<Path> dimNamespaces = Files.list(this.reconDir);){
            dimNamespaces.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).forEach(dimNamespaceDir -> {
                try (Stream<Path> dimPaths = Files.list(dimNamespaceDir);){
                    dimPaths.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).forEach(dimPathDir -> {
                        ResourceLocation dimensionId = ResourceLocation.fromNamespaceAndPath((String)dimNamespaceDir.getFileName().toString(), (String)dimPathDir.getFileName().toString());
                        HashMap biomeFiles = new HashMap();
                        try (Stream<Path> files = Files.list(dimPathDir);){
                            files.filter(f -> f.toString().endsWith(".jsonl")).forEach(filePath -> {
                                String fileName = filePath.getFileName().toString();
                                String encodedName = fileName.substring(0, fileName.length() - 6);
                                ResourceLocation biomeId = this.decodeLocation(encodedName);
                                biomeFiles.put(biomeId, filePath);
                            });
                        }
                        catch (IOException e) {
                            ComplexityAnalyzer.LOGGER.error("Failed to list biome files in {}", dimPathDir, (Object)e);
                        }
                        if (!biomeFiles.isEmpty()) {
                            allPaths.put(dimensionId, biomeFiles);
                        }
                    });
                }
                catch (IOException e) {
                    ComplexityAnalyzer.LOGGER.error("Failed to list dimension paths in {}", dimNamespaceDir, (Object)e);
                }
            });
        }
        catch (IOException e) {
            ComplexityAnalyzer.LOGGER.error("Failed to list dimension namespaces in recon directory", (Throwable)e);
        }
        return allPaths;
    }

    public Stream<ChunkSnapshot> streamReconFile(Path path) {
        Stream<ChunkSnapshot> stream;
        block9: {
            if (!Files.exists(path, new LinkOption[0])) {
                return Stream.empty();
            }
            Stream<String> lines = Files.lines(path, StandardCharsets.UTF_8);
            try {
                List<ChunkSnapshot> snapshots = lines.map(line -> {
                    try {
                        return (ChunkSnapshot)GSON.fromJson(line, ChunkSnapshot.class);
                    }
                    catch (JsonSyntaxException e) {
                        ComplexityAnalyzer.LOGGER.error("Failed to parse line in recon file {}: {}", new Object[]{path, line, e});
                        return null;
                    }
                }).filter(Objects::nonNull).toList();
                stream = snapshots.stream();
                if (lines == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (lines != null) {
                        try {
                            lines.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    ComplexityAnalyzer.LOGGER.error("Failed to stream recon file {}", (Object)path, (Object)e);
                    return Stream.empty();
                }
            }
            lines.close();
        }
        return stream;
    }

    public Map<ResourceLocation, Map<ResourceLocation, BiomeScanData>> loadAllFinalData(BiomeDataMapper mapper) {
        Map<ResourceLocation, Map<ResourceLocation, BiomeScanData>> loadedData = this.loadDataFromDirectory(this.finalDir, reader -> {
            BiomeScanData data = (BiomeScanData)PRETTY_GSON.fromJson((Reader)reader, BiomeScanData.class);
            if (data != null) {
                mapper.afterLoad(data);
            }
            return data;
        });
        ConcurrentHashMap<ResourceLocation, Map<ResourceLocation, BiomeScanData>> concurrentData = new ConcurrentHashMap<ResourceLocation, Map<ResourceLocation, BiomeScanData>>();
        loadedData.forEach((dim, biomeMap) -> concurrentData.put((ResourceLocation)dim, new ConcurrentHashMap(biomeMap)));
        return concurrentData;
    }

    public void saveFinalBiomeData(ResourceLocation dimension, ResourceLocation biome, BiomeScanData data, BiomeDataMapper mapper) {
        mapper.prepareForSave(data);
        Path file = this.getFinalFilePath(dimension, biome);
        this.saveJson(file, data);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteAllData() {
        block11: {
            try {
                if (!Files.exists(this.dataDir, new LinkOption[0])) break block11;
                try (Stream<Path> walk = Files.walk(this.dataDir, new FileVisitOption[0]);){
                    walk.sorted(Comparator.reverseOrder()).forEach(this::deletePath);
                }
            }
            catch (IOException e) {
                ComplexityAnalyzer.LOGGER.error("Failed to clear geo-data directory.", (Throwable)e);
            }
            finally {
                this.ensureDirectoriesExist();
            }
        }
    }

    public void deleteFinalData() throws IOException {
        this.deleteDirectory(this.finalDir);
    }

    private void deleteDirectory(Path dir) throws IOException {
        if (Files.exists(dir, new LinkOption[0])) {
            try (Stream<Path> walk = Files.walk(dir, new FileVisitOption[0]);){
                walk.sorted(Comparator.reverseOrder()).forEach(this::deletePath);
            }
        }
        Files.createDirectories(dir, new FileAttribute[0]);
    }

    private void deletePath(Path path) {
        try {
            Files.delete(path);
        }
        catch (IOException e) {
            ComplexityAnalyzer.LOGGER.error("Failed to delete path: {}", (Object)path, (Object)e);
        }
    }

    private void saveJson(Path file, Object data) {
        try {
            Files.createDirectories(file.getParent(), new FileAttribute[0]);
            try (FileWriter writer = new FileWriter(file.toFile());){
                PRETTY_GSON.toJson(data, (Appendable)writer);
            }
        }
        catch (IOException e) {
            ComplexityAnalyzer.LOGGER.error("Failed to save JSON to file {}", (Object)file, (Object)e);
        }
    }

    private <T> Map<ResourceLocation, Map<ResourceLocation, T>> loadDataFromDirectory(Path rootDir, ThrowingFunction<FileReader, T> fromJson) {
        HashMap allData = new HashMap();
        if (!Files.exists(rootDir, new LinkOption[0])) {
            return allData;
        }
        try (Stream<Path> dimNamespaces = Files.list(rootDir);){
            dimNamespaces.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).forEach(dimNamespaceDir -> {
                try (Stream<Path> dimPaths = Files.list(dimNamespaceDir);){
                    dimPaths.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).forEach(dimPathDir -> {
                        ResourceLocation dimensionId = ResourceLocation.fromNamespaceAndPath((String)dimNamespaceDir.getFileName().toString(), (String)dimPathDir.getFileName().toString());
                        HashMap biomeData = new HashMap();
                        try (Stream<Path> biomeFiles = Files.list(dimPathDir);){
                            biomeFiles.filter(f -> f.toString().endsWith(".json")).forEach(biomeFile -> {
                                try (FileReader reader = new FileReader(biomeFile.toFile());){
                                    Object data = fromJson.apply(reader);
                                    if (data != null) {
                                        String fileName = biomeFile.getFileName().toString();
                                        String encodedName = fileName.substring(0, fileName.length() - 5);
                                        ResourceLocation biomeId = this.decodeLocation(encodedName);
                                        biomeData.put(biomeId, data);
                                    }
                                }
                                catch (Exception e) {
                                    ComplexityAnalyzer.LOGGER.error("Failed to load/parse data from file {}", biomeFile, (Object)e);
                                }
                            });
                        }
                        catch (IOException e) {
                            ComplexityAnalyzer.LOGGER.error("Failed to list biome files in {}", dimPathDir, (Object)e);
                        }
                        if (!biomeData.isEmpty()) {
                            allData.put(dimensionId, biomeData);
                        }
                    });
                }
                catch (IOException e) {
                    ComplexityAnalyzer.LOGGER.error("Failed to list dimension paths in {}", dimNamespaceDir, (Object)e);
                }
            });
        }
        catch (IOException e) {
            ComplexityAnalyzer.LOGGER.error("Failed to list dimension namespaces in directory {}", (Object)rootDir, (Object)e);
        }
        return allData;
    }

    private String encodeLocation(ResourceLocation location) {
        return location.getNamespace() + "~" + location.getPath().replace('/', '_');
    }

    private ResourceLocation decodeLocation(String encoded) {
        String[] parts = encoded.split("~", 2);
        if (parts.length != 2) {
            return ResourceLocation.fromNamespaceAndPath((String)"minecraft", (String)encoded.replace('_', '/'));
        }
        return ResourceLocation.fromNamespaceAndPath((String)parts[0], (String)parts[1].replace('_', '/'));
    }

    private Path getReconFilePath(ResourceLocation dimension, ResourceLocation biome) {
        String biomeFileName = this.encodeLocation(biome) + ".jsonl";
        return this.reconDir.resolve(dimension.getNamespace()).resolve(dimension.getPath()).resolve(biomeFileName);
    }

    public int countReconChunks(ResourceLocation dimension, ResourceLocation biome) {
        int n;
        block9: {
            Path path = this.getReconFilePath(dimension, biome);
            if (!Files.exists(path, new LinkOption[0])) {
                return 0;
            }
            Stream<String> lines = Files.lines(path, StandardCharsets.UTF_8);
            try {
                n = (int)lines.count();
                if (lines == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (lines != null) {
                        try {
                            lines.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    return 0;
                }
            }
            lines.close();
        }
        return n;
    }

    private Path getFinalFilePath(ResourceLocation dimension, ResourceLocation biome) {
        String biomeFileName = this.encodeLocation(biome) + ".json";
        return this.finalDir.resolve(dimension.getNamespace()).resolve(dimension.getPath()).resolve(biomeFileName);
    }

    @FunctionalInterface
    private static interface ThrowingFunction<T, R> {
        public R apply(T var1) throws Exception;
    }
}

