/*
 * Decompiled with CFR 0.152.
 */
package org.complexityanalyzer.geoscan.task;

import com.mojang.datafixers.util.Pair;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.minecraft.core.BlockPos;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.core.registries.Registries;
import net.minecraft.resources.ResourceKey;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerChunkCache;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.level.ChunkPos;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.biome.Biome;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.chunk.ChunkAccess;
import net.minecraft.world.level.chunk.LevelChunk;
import net.minecraft.world.level.chunk.LevelChunkSection;
import net.minecraft.world.level.chunk.status.ChunkStatus;
import org.complexityanalyzer.ComplexityAnalyzer;
import org.complexityanalyzer.geoscan.data.ChunkSnapshot;

public class WorldScanner {
    private final MinecraftServer server;
    private final Random random = new Random();
    private static final List<BlockPos> SEARCH_ORIGINS = List.of(BlockPos.ZERO, new BlockPos(5000, 64, 5000), new BlockPos(-5000, 64, 5000), new BlockPos(5000, 64, -5000), new BlockPos(-5000, 64, -5000));
    private static final int SEARCH_RADIUS = 10000;
    private static final int RELOCATION_ATTEMPTS = 10;

    public WorldScanner(MinecraftServer server) {
        this.server = server;
    }

    public Optional<ChunkPos> findBiomeLocation(ResourceKey<Level> dimension, ResourceKey<Biome> biomeKey, boolean isRelocation) {
        if (!this.server.isSameThread()) {
            CompletableFuture future = new CompletableFuture();
            this.server.execute(() -> {
                try {
                    Optional<ChunkPos> result = this.findBiomeLocationInternal(dimension, biomeKey, isRelocation);
                    future.complete(result);
                }
                catch (Exception e) {
                    ComplexityAnalyzer.LOGGER.error("Error finding biome location for {} in {}", new Object[]{biomeKey.location(), dimension.location(), e});
                    future.complete(Optional.empty());
                }
            });
            try {
                return (Optional)future.join();
            }
            catch (Exception e) {
                ComplexityAnalyzer.LOGGER.error("Failed to get biome location from main thread", (Throwable)e);
                return Optional.empty();
            }
        }
        return this.findBiomeLocationInternal(dimension, biomeKey, isRelocation);
    }

    private Optional<ChunkPos> findBiomeLocationInternal(ResourceKey<Level> dimension, ResourceKey<Biome> biomeKey, boolean isRelocation) {
        ServerLevel level = this.server.getLevel(dimension);
        if (level == null) {
            ComplexityAnalyzer.LOGGER.error("Cannot find biome location, level {} is not loaded.", (Object)dimension.location());
            return Optional.empty();
        }
        if (!level.registryAccess().registryOrThrow(Registries.BIOME).containsKey(biomeKey)) {
            ComplexityAnalyzer.LOGGER.error("Cannot find biome location, biome {} is not registered.", (Object)biomeKey.location());
            return Optional.empty();
        }
        List<BlockPos> originsToTry = isRelocation ? this.generateRandomOrigins() : SEARCH_ORIGINS;
        for (BlockPos origin : originsToTry) {
            Pair foundResult = level.findClosestBiome3d(holder -> holder.is(biomeKey), origin, 10000, 32, 64);
            if (foundResult == null) continue;
            BlockPos blockPos = (BlockPos)foundResult.getFirst();
            return Optional.of(new ChunkPos(blockPos));
        }
        return Optional.empty();
    }

    public void processChunk(ResourceKey<Level> dimension, ResourceKey<Biome> targetBiomeKey, ChunkPos pos, BiConsumer<Optional<ChunkSnapshot>, Boolean> onComplete) {
        ServerLevel level = this.server.getLevel(dimension);
        if (level == null) {
            this.server.execute(() -> onComplete.accept(Optional.empty(), false));
            return;
        }
        ServerChunkCache chunkCache = level.getChunkSource();
        if (!level.registryAccess().registryOrThrow(Registries.BIOME).containsKey(targetBiomeKey)) {
            this.server.execute(() -> onComplete.accept(Optional.empty(), false));
            return;
        }
        ((CompletableFuture)chunkCache.getChunkFuture(pos.x, pos.z, ChunkStatus.BIOMES, true).thenCompose(either -> {
            ChunkAccess chunk = (ChunkAccess)either.orElse(null);
            if (chunk == null || !this.isBiomePresentInChunk(chunk, targetBiomeKey)) {
                return CompletableFuture.completedFuture(null);
            }
            return chunkCache.getChunkFuture(pos.x, pos.z, ChunkStatus.FULL, true);
        })).thenAccept(either -> {
            if (either == null) {
                this.server.execute(() -> onComplete.accept(Optional.empty(), false));
                return;
            }
            ChunkAccess chunk = (ChunkAccess)either.orElse(null);
            if (chunk instanceof LevelChunk) {
                LevelChunk levelChunk = (LevelChunk)chunk;
                ChunkSnapshot snapshot = this.createSnapshot(levelChunk);
                this.server.execute(() -> onComplete.accept(Optional.of(snapshot), true));
            } else {
                this.server.execute(() -> onComplete.accept(Optional.empty(), false));
            }
        });
    }

    private boolean isBiomePresentInChunk(ChunkAccess chunk, ResourceKey<Biome> targetBiomeKey) {
        for (int qy = chunk.getMinSection() * 4; qy < chunk.getMaxSection() * 4; ++qy) {
            for (int qx = 0; qx < 4; ++qx) {
                for (int qz = 0; qz < 4; ++qz) {
                    if (!chunk.getNoiseBiome(qx, qy, qz).is(targetBiomeKey)) continue;
                    return true;
                }
            }
        }
        return false;
    }

    private ChunkSnapshot createSnapshot(LevelChunk chunk) {
        HashMap<String, Integer> counts = new HashMap<String, Integer>();
        for (LevelChunkSection section : chunk.getSections()) {
            if (section == null || section.hasOnlyAir()) continue;
            for (int y = 0; y < 16; ++y) {
                for (int x = 0; x < 16; ++x) {
                    for (int z = 0; z < 16; ++z) {
                        BlockState state = section.getBlockState(x, y, z);
                        if (state.isAir()) continue;
                        String blockId = BuiltInRegistries.BLOCK.getKey((Object)state.getBlock()).toString();
                        counts.merge(blockId, 1, Integer::sum);
                    }
                }
            }
        }
        return new ChunkSnapshot(chunk.getPos().x, chunk.getPos().z, counts);
    }

    private List<BlockPos> generateRandomOrigins() {
        int searchDiameter = 20000;
        return Stream.generate(() -> {
            int x = this.random.nextInt(searchDiameter * 2) - searchDiameter;
            int z = this.random.nextInt(searchDiameter * 2) - searchDiameter;
            return new BlockPos(x, 64, z);
        }).limit(10L).collect(Collectors.toList());
    }
}

