package games.enchanted.eg_bedrock_books.common.config;

import com.google.gson.*;
import com.google.gson.stream.JsonReader;
import games.enchanted.eg_bedrock_books.common.Logging;
import games.enchanted.eg_bedrock_books.common.ModConstants;
import games.enchanted.eg_bedrock_books.common.config.option.BoolOption;
import games.enchanted.eg_bedrock_books.common.config.option.ConfigOption;
import games.enchanted.eg_bedrock_books.common.config.option.IntOption;
import games.enchanted.eg_bedrock_books.common.config.option.KeyOption;
import games.enchanted.eg_bedrock_books.common.util.InputUtil;
import games.enchanted.eg_bedrock_books.common.util.McUtil;
import games.enchanted.eg_bedrock_books.platform.PlatformHelper;

import java.io.*;
import java.util.ArrayList;
import java.util.List;
import net.minecraft.class_3675;

public class ConfigOptions {
    // general
    public static final ConfigOption<Boolean> CLOSE_BOOK_WHEN_RUNNING_COMMAND;
    public static final ConfigOption<class_3675.class_306> MOVE_FORWARD_PAGE_KEY;
    public static final ConfigOption<class_3675.class_306> MOVE_BACKWARD_PAGE_KEY;
    public static final ConfigOption<Boolean> VANILLA_BOOK_KEY_ENABLED;
    public static final ConfigOption<class_3675.class_306> VANILLA_BOOK_KEY;

    // visual
    public static final ConfigOption<Boolean> SHOW_X_BUTTON;
    public static final ConfigOption<Integer> RIBBON_HEIGHT;

    // debug
    public static final ConfigOption<Boolean> DEBUG_TEXT_BOUNDS;
    public static final ConfigOption<Boolean> DEBUG_WIDGET_BOUNDS;
    public static final ConfigOption<Boolean> DEBUG_CONTAINER_DATA;
    public static final ConfigOption<Boolean> DEBUG_VARIABLES;

    private static final List<ConfigOption<?>> OPTIONS = new ArrayList<>();

    static {
        CLOSE_BOOK_WHEN_RUNNING_COMMAND = new BoolOption(
            true,
            true,
            "close_book_when_running_command"
        );
        registerOption(CLOSE_BOOK_WHEN_RUNNING_COMMAND);

        MOVE_FORWARD_PAGE_KEY = new KeyOption(
            InputUtil.getKey(class_3675.field_31991),
            InputUtil.getKey(class_3675.field_31991),
            "move_forward_page_key"
        );
        registerOption(MOVE_FORWARD_PAGE_KEY);

        MOVE_BACKWARD_PAGE_KEY = new KeyOption(
            InputUtil.getKey(class_3675.field_31992),
            InputUtil.getKey(class_3675.field_31992),
            "move_backward_page_key"
        );
        registerOption(MOVE_BACKWARD_PAGE_KEY);

        VANILLA_BOOK_KEY_ENABLED = new BoolOption(
            false,
            false,
            "vanilla_book_key_enabled"
        );
        registerOption(VANILLA_BOOK_KEY_ENABLED);

        VANILLA_BOOK_KEY = new KeyOption(
            InputUtil.getKey(class_3675.field_31949),
            InputUtil.getKey(class_3675.field_31949),
            "vanilla_book_key"
        );
        registerOption(VANILLA_BOOK_KEY);


        SHOW_X_BUTTON = new BoolOption(
            true,
            true,
            "show_x_button"
        );
        registerOption(SHOW_X_BUTTON);

        RIBBON_HEIGHT = new IntOption(
            76,
            76,
            "ribbon_height"
        );
        registerOption(RIBBON_HEIGHT);


        DEBUG_TEXT_BOUNDS = new BoolOption(
            false,
            false,
            "debug_text_bounds"
        );
        registerOption(DEBUG_TEXT_BOUNDS);

        DEBUG_WIDGET_BOUNDS = new BoolOption(
            false,
            false,
            "debug_widget_bounds"
        );
        registerOption(DEBUG_WIDGET_BOUNDS);

        DEBUG_CONTAINER_DATA = new BoolOption(
            false,
            false,
            "debug_container_data"
        );
        registerOption(DEBUG_CONTAINER_DATA);

        DEBUG_VARIABLES = new BoolOption(
            false,
            false,
            "debug_variables"
        );
        registerOption(DEBUG_VARIABLES);
    }

    private static void registerOption(ConfigOption<?> option) {
        OPTIONS.add(option);
    }

    private static final String FILE_NAME = ModConstants.MOD_ID + ".json";

    private static File getConfigFile() {
        return PlatformHelper.getConfigPath().resolve(FILE_NAME).toFile();
    }

    public static void saveIfAnyDirtyOptions() {
        if(OPTIONS.stream().noneMatch(ConfigOption::isDirty)) return;
        for (ConfigOption<?> option : OPTIONS) {
            if(option.isDirty()) option.applyPendingValue();
        }
        saveConfig();
    }

    public static void saveConfig() {
        JsonObject root = new JsonObject();

        for (ConfigOption<?> option : OPTIONS) {
            root.add(option.getJsonKey(), option.toJson());
        }

        Gson gson = new GsonBuilder().setPrettyPrinting().create();
        String encodedJson = gson.toJson(root);

        try (FileWriter writer = new FileWriter(getConfigFile())) {
            writer.write(encodedJson);
        } catch (IOException e) {
            Logging.error("Failed to write config file '{}', {}", FILE_NAME, e);
        }
    }

    public static void readConfig() {
        Gson gson = new Gson();
        JsonObject decodedConfig = new JsonObject();

        try {
            JsonReader jsonReader = gson.newJsonReader(new FileReader(getConfigFile()));
            jsonReader.setStrictness(Strictness.LENIENT);
            decodedConfig = JsonParser.parseReader(jsonReader).getAsJsonObject();
        } catch (JsonParseException e) {
            Logging.error("Failed to parse config file '{}', {}", FILE_NAME, e);
        } catch (FileNotFoundException e) {
            Logging.info("Config file '{}' not found", FILE_NAME);
            saveConfig();
        }

        for (ConfigOption<?> option : OPTIONS) {
            option.fromJson(decodedConfig);
        }
    }

    public static void resetAndSaveAllOptions() {
        for (ConfigOption<?> option : OPTIONS) {
            option.resetToDefault(true);
        }
        saveConfig();
    }

    public static void clearAllPendingValues() {
        for (ConfigOption<?> option : OPTIONS) {
            option.clearPendingValue();
        }
    }
}
