/*
 * Decompiled with CFR 0.152.
 */
package me.aov.sellgui.managers;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import me.aov.sellgui.SellGUIMain;
import org.bukkit.Bukkit;
import org.bukkit.entity.Player;
import org.bukkit.inventory.ItemStack;
import org.bukkit.plugin.Plugin;

public class AsyncPriceCalculator {
    private final SellGUIMain plugin;
    private final ExecutorService executor;
    private final boolean asyncEnabled;
    private final int maxConcurrentOps;
    private final Semaphore semaphore;

    public AsyncPriceCalculator(SellGUIMain plugin) {
        this.plugin = plugin;
        this.asyncEnabled = plugin.getConfig().getBoolean("performance.async-calculations", true);
        this.maxConcurrentOps = plugin.getConfig().getInt("performance.max-concurrent-operations", 10);
        if (this.asyncEnabled) {
            this.executor = Executors.newFixedThreadPool(this.maxConcurrentOps);
            this.semaphore = new Semaphore(this.maxConcurrentOps);
        } else {
            this.executor = null;
            this.semaphore = null;
        }
    }

    public void calculatePriceAsync(ItemStack item, Player player, Consumer<Double> callback) {
        if (!this.asyncEnabled) {
            double price2 = this.calculatePriceSync(item, player);
            callback.accept(price2);
            return;
        }
        CompletableFuture.supplyAsync(() -> {
            try {
                this.semaphore.acquire();
                Double d = this.calculatePriceSync(item, player);
                return d;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                Double d = 0.0;
                return d;
            }
            finally {
                this.semaphore.release();
            }
        }, this.executor).thenAcceptAsync(price -> Bukkit.getScheduler().runTask((Plugin)this.plugin, () -> callback.accept((Double)price)));
    }

    public void calculatePricesAsync(ItemStack[] items, Player player, Consumer<Double[]> callback) {
        if (!this.asyncEnabled) {
            Double[] prices2 = new Double[items.length];
            for (int i = 0; i < items.length; ++i) {
                prices2[i] = this.calculatePriceSync(items[i], player);
            }
            callback.accept(prices2);
            return;
        }
        CompletableFuture<Double[]> future = CompletableFuture.supplyAsync(() -> {
            Double[] prices = new Double[items.length];
            CountDownLatch latch = new CountDownLatch(items.length);
            for (int i = 0; i < items.length; ++i) {
                int index = i;
                ItemStack item = items[i];
                this.executor.submit(() -> {
                    try {
                        this.semaphore.acquire();
                        prices[index] = this.calculatePriceSync(item, player);
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        prices[index] = 0.0;
                    }
                    finally {
                        this.semaphore.release();
                        latch.countDown();
                    }
                });
            }
            try {
                latch.await(5L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            return prices;
        }, this.executor);
        future.thenAcceptAsync(prices -> Bukkit.getScheduler().runTask((Plugin)this.plugin, () -> callback.accept((Double[])prices)));
    }

    public void calculateInventoryValueAsync(Player player, Consumer<Double> callback) {
        ItemStack[] contents = player.getInventory().getContents();
        this.calculatePricesAsync(contents, player, prices -> {
            double total = 0.0;
            for (int i = 0; i < contents.length; ++i) {
                if (contents[i] == null || prices[i] == null) continue;
                total += prices[i] * (double)contents[i].getAmount();
            }
            callback.accept(total);
        });
    }

    private double calculatePriceSync(ItemStack item, Player player) {
        if (item == null) {
            return 0.0;
        }
        try {
            Double cachedPrice;
            if (this.plugin.getPriceCache() != null && (cachedPrice = this.plugin.getPriceCache().getCachedPrice(item, player)) != null) {
                return cachedPrice;
            }
            double price = 0.0;
            if (this.plugin.getPriceManager() != null) {
                price = this.plugin.getPriceManager().getItemPriceWithPlayer(item, player);
            }
            if (this.plugin.getPriceCache() != null && price > 0.0) {
                this.plugin.getPriceCache().cachePrice(item, player, price);
            }
            return price;
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Error calculating price for " + String.valueOf(item.getType()) + ": " + e.getMessage());
            return 0.0;
        }
    }

    public AsyncStats getStats() {
        int activeThreads = 0;
        int queuedTasks = 0;
        if (this.executor instanceof ThreadPoolExecutor) {
            ThreadPoolExecutor tpe = (ThreadPoolExecutor)this.executor;
            activeThreads = tpe.getActiveCount();
            queuedTasks = tpe.getQueue().size();
        }
        return new AsyncStats(this.asyncEnabled, this.maxConcurrentOps, activeThreads, queuedTasks);
    }

    public void shutdown() {
        if (this.executor != null) {
            this.executor.shutdown();
            try {
                if (!this.executor.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.executor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.executor.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
    }

    public static class AsyncStats {
        private final boolean enabled;
        private final int maxConcurrent;
        private final int activeThreads;
        private final int queuedTasks;

        public AsyncStats(boolean enabled, int maxConcurrent, int activeThreads, int queuedTasks) {
            this.enabled = enabled;
            this.maxConcurrent = maxConcurrent;
            this.activeThreads = activeThreads;
            this.queuedTasks = queuedTasks;
        }

        public boolean isEnabled() {
            return this.enabled;
        }

        public int getMaxConcurrent() {
            return this.maxConcurrent;
        }

        public int getActiveThreads() {
            return this.activeThreads;
        }

        public int getQueuedTasks() {
            return this.queuedTasks;
        }
    }
}

