package dev.gegy.roles.command;

import com.mojang.brigadier.Command;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.brigadier.exceptions.DynamicCommandExceptionType;
import com.mojang.brigadier.exceptions.SimpleCommandExceptionType;
import com.mojang.brigadier.suggestion.SuggestionProvider;
import dev.gegy.roles.SimpleRole;
import dev.gegy.roles.api.PlayerRolesApi;
import dev.gegy.roles.api.Role;
import dev.gegy.roles.config.PlayerRolesConfig;
import dev.gegy.roles.override.command.CommandOverride;
import dev.gegy.roles.store.PlayerRoleManager;
import dev.gegy.roles.store.PlayerRoleSet;
import net.minecraft.class_11560;
import net.minecraft.class_124;
import net.minecraft.class_2168;
import net.minecraft.class_2172;
import net.minecraft.class_2191;
import net.minecraft.class_2561;
import net.minecraft.class_2564;
import net.minecraft.class_2583;
import net.minecraft.class_5250;
import net.minecraft.server.MinecraftServer;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Comparator;
import java.util.function.BiPredicate;

import static net.minecraft.class_2170.method_9244;
import static net.minecraft.class_2170.method_9247;

public final class RoleCommand {
    public static final DynamicCommandExceptionType ROLE_NOT_FOUND = new DynamicCommandExceptionType(arg ->
            class_2561.method_54159("Role with name '%s' was not found!", arg)
    );

    public static final SimpleCommandExceptionType ROLE_POWER_TOO_LOW = new SimpleCommandExceptionType(
            class_2561.method_43470("You do not have sufficient power to manage this role")
    );

    public static final SimpleCommandExceptionType TOO_MANY_SELECTED = new SimpleCommandExceptionType(
            class_2561.method_43470("Too many players selected!")
    );

    // @formatter:off
    public static void register(CommandDispatcher<class_2168> dispatcher) {
        dispatcher.register(method_9247("role")
                .requires(s -> s.method_9259(4))
                .then(method_9247("assign")
                    .then(method_9244("targets", class_2191.method_9329())
                    .then(method_9244("role", StringArgumentType.word()).suggests(roleSuggestions())
                    .executes(ctx -> {
                        var source = ctx.getSource();
                        var targets = class_2191.method_9330(ctx, "targets");
                        var roleName = StringArgumentType.getString(ctx, "role");
                        return updateRoles(source, targets, roleName, PlayerRoleSet::add, "'%s' assigned to %s players");
                    })
                )))
                .then(method_9247("remove")
                    .then(method_9244("targets", class_2191.method_9329())
                    .then(method_9244("role", StringArgumentType.word()).suggests(roleSuggestions())
                    .executes(ctx -> {
                        var source = ctx.getSource();
                        var targets = class_2191.method_9330(ctx, "targets");
                        var roleName = StringArgumentType.getString(ctx, "role");
                        return updateRoles(source, targets, roleName, PlayerRoleSet::remove, "'%s' removed from %s players");
                    })
                )))
                .then(method_9247("list")
                    .then(method_9244("target", class_2191.method_9329()).executes(ctx -> {
                        var source = ctx.getSource();
                        var gameProfiles = class_2191.method_9330(ctx, "target");
                        if (gameProfiles.size() != 1) {
                            throw TOO_MANY_SELECTED.create();
                        }
                        return listRoles(source, gameProfiles.iterator().next());
                    }))
                )
                .then(method_9247("reload").executes(ctx -> reloadRoles(ctx.getSource())))
        );
    }
    // @formatter:on

    private static int updateRoles(class_2168 source, Collection<class_11560> players, String roleName, BiPredicate<PlayerRoleSet, SimpleRole> apply, String success) throws CommandSyntaxException {
        var role = getRole(roleName);
        requireHasPower(source, role);

        var roleManager = PlayerRoleManager.get();
        MinecraftServer server = source.method_9211();

        int count = 0;
        for (var player : players) {
            boolean applied = roleManager.updateRoles(server, player.comp_4422(), roles -> apply.test(roles, role));
            if (applied) {
                count++;
            }
        }

        int finalCount = count;
        source.method_9226(() -> class_2561.method_43469(success, roleName, finalCount), true);

        return Command.SINGLE_SUCCESS;
    }

    private static int listRoles(class_2168 source, class_11560 player) {
        var roleManager = PlayerRoleManager.get();
        var server = source.method_9211();

        var roles = roleManager.peekRoles(server, player.comp_4422()).stream().toList();
        source.method_9226(() -> {
            var rolesComponent = class_2564.method_10884(roles, role -> class_2561.method_43470(role.getId()).method_10862(class_2583.field_24360.method_10977(class_124.field_1080)));
            return class_2561.method_43469("Found %s roles on player: %s", roles.size(), rolesComponent);
        }, false);

        return Command.SINGLE_SUCCESS;
    }

    private static int reloadRoles(class_2168 source) {
        var server = source.method_9211();

        server.execute(() -> {
            var errors = PlayerRolesConfig.setup();

            var roleManager = PlayerRoleManager.get();
            roleManager.onRoleReload(server, PlayerRolesConfig.get());

            if (errors.isEmpty()) {
                source.method_9226(() -> class_2561.method_43470("Role configuration successfully reloaded"), false);
            } else {
                class_5250 errorFeedback = class_2561.method_43470("Failed to reload roles configuration!");
                for (String error : errors) {
                    errorFeedback = errorFeedback.method_27693("\n - " + error);
                }
                source.method_9213(errorFeedback);
            }
        });

        return Command.SINGLE_SUCCESS;
    }

    private static void requireHasPower(class_2168 source, SimpleRole role) throws CommandSyntaxException {
        if (hasAdminPower(source)) {
            return;
        }

        var highestRole = getHighestRole(source);
        if (highestRole == null || role.compareTo(highestRole) <= 0) {
            throw ROLE_POWER_TOO_LOW.create();
        }
    }

    private static SimpleRole getRole(String roleName) throws CommandSyntaxException {
        var role = PlayerRolesConfig.get().get(roleName);
        if (role == null) throw ROLE_NOT_FOUND.create(roleName);
        return role;
    }

    private static SuggestionProvider<class_2168> roleSuggestions() {
        return (ctx, builder) -> {
            var source = ctx.getSource();

            boolean admin = hasAdminPower(source);
            var highestRole = getHighestRole(source);
            Comparator<Role> comparator = Comparator.nullsLast(Comparator.naturalOrder());

            return class_2172.method_9264(
                    PlayerRolesConfig.get().stream()
                            .filter(role -> admin || comparator.compare(role, highestRole) > 0)
                            .map(Role::getId),
                    builder
            );
        };
    }

    @Nullable
    private static Role getHighestRole(class_2168 source) {
        return PlayerRolesApi.lookup().bySource(source).stream()
                .min(Comparator.naturalOrder())
                .orElse(null);
    }

    private static boolean hasAdminPower(class_2168 source) {
        return source.method_9228() == null || CommandOverride.doesBypassPermissions(source);
    }
}
