package io.github.zhengzhengyiyi.gui;

import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.google.gson.JsonSyntaxException;
import io.github.zhengzhengyiyi.util.BackupHelper;
import io.github.zhengzhengyiyi.*;
import io.github.zhengzhengyiyi.config.ModConfigData;
import io.github.zhengzhengyiyi.gui.theme.ThemeManager;
import io.github.zhengzhengyiyi.gui.widget.*;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_11908;
import net.minecraft.class_11909;
import net.minecraft.class_156;
import net.minecraft.class_2561;
import net.minecraft.class_332;
import net.minecraft.class_339;
import net.minecraft.class_342;
import net.minecraft.class_410;
import net.minecraft.class_4185;
import net.minecraft.class_437;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.List;
import java.util.ArrayList;
import java.util.stream.Collectors;
import java.io.IOException;
import org.lwjgl.glfw.GLFW;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EditorScreen extends class_437 {
    private static final Logger LOGGER = LoggerFactory.getLogger(EditorScreen.class);
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private List<Path> configFiles;
    private int selectedIndex = 0;
    private MultilineEditor multilineEditor;
    private GeneralMultilineEditor universalEditor;
    private AbstractEditor currentEditor;
    private boolean modified = false;
    private class_4185 saveButton;
    private class_4185 openFolderButton;
    private class_4185 backupButton;
    private String buffer = "";
    private class_342 searchField;
    private class_4185 searchNextButton;
    private class_4185 searchPrevButton;
    private class_4185 exitButton;
    private boolean searchVisible = false;
    private ThemeManager themeManager;
    private class_4185 themeToggleButton;
    private int fileListScrollOffset = 0;
    private class_4185 scrollUpButton;
    private class_4185 scrollDownButton;
    private List<class_4185> fileButtonList;
    private boolean isJsonFile = true;

    public EditorScreen() {
        super(class_2561.method_43471("configeditor.title"));
    }

    @Override
    protected void method_25426() {
        super.method_25426();
        
        themeManager = ThemeManager.getInstance();
        themeToggleButton = class_4185.method_46430(
                class_2561.method_43471(getThemeButtonText()),
                button -> toggleTheme())
                .method_46434(this.field_22789 - 50, 5, 45, 16)
                .method_46431();
        this.method_37063(themeToggleButton);
        
        try {
            Path configDir = FabricLoader.getInstance().getConfigDir();
            configFiles = Files.list(configDir)
                    .filter(Files::isRegularFile)
                    .collect(Collectors.toList());
//            LOGGER.info("Found {} config files", configFiles.size());
        } catch (Exception e) {
            configFiles = new ArrayList<>();
            LOGGER.error("Failed to list config files", e);
        }

        fileButtonList = new ArrayList<>();
        
        scrollUpButton = class_4185.method_46430(class_2561.method_43470("↑"), button -> scrollUp())
                .method_46434(140, 25, 20, 20)
                .method_46431();
        scrollDownButton = class_4185.method_46430(class_2561.method_43470("↓"), button -> scrollDown())
                .method_46434(140, this.field_22790 - 45, 20, 20)
                .method_46431();
        
        this.method_37063(scrollUpButton);
        this.method_37063(scrollDownButton);
        
        renderFileList();
        
        multilineEditor = new MultilineEditor(
                170, 20, 
                this.field_22789 - 180, this.field_22790 - 60,
                class_2561.method_43471("configeditor.editor"));
        multilineEditor.setChangedListener(text -> {
            if (!buffer.equals(text)) {
                modified = true;
                updateButtonStates();
            } else {
                modified = false;
            }
        });

        universalEditor = new GeneralMultilineEditor(
                170, 20,
                this.field_22789 - 180, this.field_22790 - 60,
                class_2561.method_43471("configeditor.editor"));
        universalEditor.setChangedListener(text -> {
            if (!buffer.equals(text)) {
                modified = true;
                updateButtonStates();
            } else {
                modified = false;
            }
        });

        saveButton = class_4185.method_46430(
                class_2561.method_43471("configeditor.button.save"),
                button -> saveFile())
                .method_46434(this.field_22789 - 170, this.field_22790 - 30, 80, 20)
                .method_46431();
        
        backupButton = class_4185.method_46430(
                class_2561.method_43471("configeditor.button.backup"), 
                button -> BackupHelper.backupEntireConfigDirectory())
                .method_46434(this.field_22789 - 260, this.field_22790 - 30, 80, 20)
                .method_46431();

        openFolderButton = class_4185.method_46430(
                class_2561.method_43471("configeditor.button.openfolder"),
                button -> openConfigFolder())
                .method_46434(this.field_22789 - 80, this.field_22790 - 30, 70, 20)
                .method_46431();
        
        exitButton = class_4185.method_46430(
                class_2561.method_43471("configeditor.button.close"),
                button -> this.method_25419())
                .method_46434(0, 0, 80, 20)
                .method_46431();
        
        searchField = new class_342(field_22793, this.field_22789 - 300, 5, 150, 16, class_2561.method_43471("configeditor.search.placeholder"));
        searchField.method_1863(text -> {
            if (!text.trim().isEmpty()) {
                startSearch(text.trim());
            }
        });
        searchField.method_1862(true);

        searchNextButton = class_4185.method_46430(class_2561.method_43470("↓"), button -> {
            findNext();
        }).method_46434(this.field_22789 - 145, 5, 20, 16).method_46431();
        searchNextButton.field_22764 = true;
        
        searchPrevButton = class_4185.method_46430(class_2561.method_43470("↑"), button -> {
            findPrevious();
        }).method_46434(this.field_22789 - 165, 5, 20, 16).method_46431();
        searchPrevButton.field_22764 = true;
        
        class_4185 closeSearchButton = class_4185.method_46430(class_2561.method_43470("✕"), button -> {
            searchField.method_1852("");
            endSearch();
        }).method_46434(this.field_22789 - 120, 5, 20, 16).method_46431();
        
        this.method_37063(saveButton);
        this.method_37063(backupButton);
        this.method_37063(openFolderButton);
        this.method_37063(searchField);
        this.method_37063(searchNextButton);
        this.method_37063(searchPrevButton);
        this.method_37063(closeSearchButton);
        this.method_37063(exitButton);
        
        this.method_48265(multilineEditor);

        if (!configFiles.isEmpty()) {
            loadFile(selectedIndex);
        } else {
            multilineEditor.setText("{}");
            multilineEditor.setEditable(false);
            LOGGER.warn("No config files found in config directory");
        }
        
        for (io.github.zhengzhengyiyi.api.ApiEntrypoint entrypoint : ConfigEditorClient.ENTRYPOINTS) {
            entrypoint.onEditerOpen(this);
        }
        
        updateButtonStates();
        updateScrollButtons();
    }

    private void renderFileList() {
        for (class_4185 button : fileButtonList) {
            this.method_37066(button);
        }
        fileButtonList.clear();
        
        int buttonY = 25;
        for (int i = fileListScrollOffset; i < configFiles.size() && i < fileListScrollOffset + 15; i++) {
            int index = i;
            Path file = configFiles.get(i);
            String fileName = file.getFileName().toString();
            
            class_4185 button = class_4185.method_46430(
                    class_2561.method_43470(fileName.length() > 20 ? fileName.substring(0, 17) + "..." : fileName),
                    _button -> switchFile(index))
                    .method_46434(10, buttonY, 130, 20)
                    .method_46431();
            this.method_37063(button);
            fileButtonList.add(button);
            buttonY += 23;
        }
    }

    private void scrollUp() {
        if (fileListScrollOffset > 0) {
            fileListScrollOffset--;
            renderFileList();
            updateScrollButtons();
        }
    }

    private void scrollDown() {
        if (fileListScrollOffset < configFiles.size() - 15) {
            fileListScrollOffset++;
            renderFileList();
            updateScrollButtons();
        }
    }

    private void updateScrollButtons() {
        scrollUpButton.field_22763 = fileListScrollOffset > 0;
        scrollDownButton.field_22763 = fileListScrollOffset < configFiles.size() - 15;
    }

    private void updateButtonStates() {
        if (saveButton != null) {
            saveButton.field_22763 = true;
        }
    }

    private void switchFile(int index) {
        loadFile(index);
    }
    
    private void loadFile(int index) {
        if (index < 0 || index >= configFiles.size()) {
            LOGGER.error("Invalid file index: {}", index);
            return;
        }
        
        selectedIndex = index;
        modified = false;
        Path file = configFiles.get(index);
        
        try {
            String content = Files.readString(file);
            buffer = content;
            
            boolean isJson = checkIfJson(content);
            isJsonFile = isJson;
            
            switchEditor(isJson);
            
            if (isJson) {
                JsonElement json = JsonParser.parseString(content);
                String formattedContent = GSON.toJson(json);
                currentEditor.setText(formattedContent);
            } else {
                currentEditor.setText(content);
            }
        } catch (Exception e) {
            String text = null;
            try {
                text = Files.readString(file);
            } catch (IOException ioexception) {
                LOGGER.error("tried to read file except IOException: ", ioexception.toString());
            }
            if (text == null) {
                LOGGER.error("Failed to load config file: {}", file.getFileName(), e);
                switchEditor(true);
                currentEditor.setText("{}");
                currentEditor.setEditable(false);
                showErrorPopup(class_2561.method_43471("configeditor.error.loadfailed"));
            } else {
                boolean isJson = checkIfJson(text);
                isJsonFile = isJson;
                switchEditor(isJson);
                currentEditor.setText(text);
            }
        }
        
        updateButtonStates();
    }

//    private void loadFile(int index) {
//        if (index < 0 || index >= configFiles.size()) {
//            LOGGER.error("Invalid file index: {}", index);
//            return;
//        }
//        
//        selectedIndex = index;
//        modified = false;
//        Path file = configFiles.get(index);
//        
//        try {
//            String content = Files.readString(file);
//            buffer = content;
//            
//            boolean isJson = checkIfJson(content);
//            isJsonFile = isJson;
//            
//            switchEditor(isJson);
//            
//            if (isJson) {
//                JsonElement json = JsonParser.parseString(content);
//                String formattedContent = GSON.toJson(json);
//                multilineEditor.setText(formattedContent);
////                LOGGER.info("Successfully loaded JSON config file: {}", file.getFileName());
//            } else {
//                universalEditor.setText(content);
////                LOGGER.info("Successfully loaded text file: {}", file.getFileName());
//            }
//        } catch (Exception e) {
//            String text = null;
//            try {
//                text = Files.readString(file);
//            } catch (IOException ioexception) {
//                LOGGER.error("tried to read file except IOException: ", ioexception.toString());
//            }
//            if (text == null) {
//                LOGGER.error("Failed to load config file: {}", file.getFileName(), e);
//                switchEditor(true);
//                multilineEditor.setText("{}");
//                multilineEditor.setEditable(false);
//                showErrorPopup(Text.translatable("configeditor.error.loadfailed"));
//            } else {
//                boolean isJson = checkIfJson(text);
//                isJsonFile = isJson;
//                switchEditor(isJson);
//                if (isJson) {
//                    multilineEditor.setText(text);
//                } else {
//                    universalEditor.setText(text);
//                }
//            }
//        }
//        
//        updateButtonStates();
//    }

    private boolean checkIfJson(String content) {
        if (content == null || content.trim().isEmpty()) {
            return false;
        }
        
        String trimmed = content.trim();
        if (!trimmed.startsWith("{") && !trimmed.startsWith("[")) {
            return false;
        }
        
        try {
            JsonParser.parseString(content);
            return true;
        } catch (JsonSyntaxException e) {
            return false;
        }
    }

    private void switchEditor(boolean useJsonEditor) {
        if (currentEditor != null) {
            this.method_37066(currentEditor);
        }
        
        if (useJsonEditor) {
            currentEditor = multilineEditor;
            this.method_37063(multilineEditor);
        } else {
            currentEditor = universalEditor;
            this.method_37063(universalEditor);
        }
        
        this.method_48265(currentEditor);
    }

    private void startSearch(String query) {
        if (currentEditor instanceof MultilineEditor) {
            ((MultilineEditor) currentEditor).startSearch(query);
        }
        if (currentEditor instanceof GeneralMultilineEditor) {
            ((GeneralMultilineEditor) currentEditor).startSearch(query);
        }
    }

    private void findNext() {
        if (currentEditor instanceof MultilineEditor) {
            ((MultilineEditor) currentEditor).findNext();
        }
    }

    private void findPrevious() {
        if (currentEditor instanceof MultilineEditor) {
            ((MultilineEditor) currentEditor).findPrevious();
        }
    }

    private void endSearch() {
        if (currentEditor instanceof MultilineEditor) {
            ((MultilineEditor) currentEditor).endSearch();
        }
    }

    private void saveFile() {
        saveFileAsync(null);
    }
    
    private void saveFileAsync(Runnable callback) {
        if (configFiles.isEmpty()) return;
        
        Path file = configFiles.get(selectedIndex);
        String content = getCurrentEditorText();
        
        if (isJsonFile) {
            try {
                JsonParser.parseString(content);
            } catch (JsonSyntaxException e) {
                LOGGER.warn("Invalid JSON syntax in file: {}", file.getFileName());
                showErrorPopup(class_2561.method_43471("configeditor.error.invalidjson"));
                return;
            }
        }
        
        new Thread(() -> {
            int retryCount = 0;
            final int maxRetries = 3;
            
            while (retryCount <= maxRetries) {
                try {
                    try (FileChannel channel = FileChannel.open(file, 
                         StandardOpenOption.WRITE, StandardOpenOption.CREATE)) {
                        FileLock lock = channel.tryLock(0, Long.MAX_VALUE, false);
                        if (lock != null) {
                            try {
                                Files.writeString(file, content, StandardOpenOption.TRUNCATE_EXISTING);
                                modified = false;
                                LOGGER.info("Successfully saved file: {}", file.getFileName());
                                
                                if (callback != null) {
                                    field_22787.execute(callback);
                                } else {
                                    field_22787.execute(this::updateButtonStates);
                                }
                                return;
                            } catch (Exception e) {
                                LOGGER.error(e.toString());
                            } finally {
                                lock.release();
                            }
                        }
                    }
                    
                    retryCount++;
                    if (retryCount <= maxRetries) {
                        Thread.sleep(300);
                    }
                    
                } catch (Exception e) {
                    retryCount++;
                    if (retryCount > maxRetries) {
                        LOGGER.error("Failed to save file after {} attempts: {}", maxRetries, file.getFileName(), e);
                        field_22787.execute(() -> {
                            if (this.field_22787.field_1755 != null && this.field_22787.field_1755.equals(this)) {
                                showErrorPopup(class_2561.method_43471("configeditor.error.savefailed"));
                            }
                        });
                    }
                }
            }
            
            field_22787.execute(() -> {
                if (this.field_22787.field_1755 != null && this.field_22787.field_1755.equals(this)) {
                    showErrorPopup(class_2561.method_43471("configeditor.error.fileretryfailed"));
                }
            });
            
        }).start();
    }
    
    private String getCurrentEditorText() {
        if (currentEditor instanceof MultilineEditor) {
            return ((MultilineEditor) currentEditor).getText();
        } else if (currentEditor instanceof GeneralMultilineEditor) {
            return ((GeneralMultilineEditor) currentEditor).getText();
        }
        return "";
    }
    
    private void openConfigFolder() {
        try {
            Path configDir = FabricLoader.getInstance().getConfigDir();
            LOGGER.info("Config folder location: {}", configDir);
            class_156.method_668().method_673(configDir.toUri());
        } catch (Exception e) {
            LOGGER.error("Failed to get config folder", e);
        }
    }

    private void showErrorPopup(class_2561 message) {
        field_22787.method_1507(new class_410(
            result -> {
                this.field_22787.method_1507(this);
            },
            class_2561.method_43471("configeditor.confirm.title"),
            message
        ));
    }

    public void showMessagePopup(class_2561 message) {
        field_22787.method_1507(new class_410(
            result -> {
                this.method_25419();
                this.field_22787.method_1507(null);
            },
            class_2561.method_43471("configeditor.confirm.title"),
            message
        ));
    }

    @Override
    public void method_25394(class_332 context, int mouseX, int mouseY, float delta) {
        if (ConfigEditorClient.configManager.getConfig().doRenderBackground) context.method_25294(0, 0, this.field_22789, this.field_22790, themeManager.getBackgroundColor());
        super.method_25394(context, mouseX, mouseY, delta);
        
        if (!configFiles.isEmpty()) {
            String status = modified ? "* " + configFiles.get(selectedIndex).getFileName().toString() : 
                configFiles.get(selectedIndex).getFileName().toString();
            String editorType = isJsonFile ? " [JSON]" : " [Text]";
            context.method_51433(this.field_22793, status + editorType, 170, 5, modified ? 0xFFFF00 : 0xFFFFFF, false);
        }
    }

    @Override
    public boolean method_25422() {
        if (modified) {
            class_410 confirmScreen = new class_410(
                result -> {
                    if (result) {
                        saveFileAsync(() -> {this.method_25419(); this.field_22787.method_1507(null);});
                    } else {
                        this.method_25419();
                        this.field_22787.method_1507(null);
                    }
                },
                class_2561.method_43471("configeditor.confirm.title"),
                class_2561.method_43471("configeditor.confirm.unsavedclose")
            );
            this.field_22787.method_1507(confirmScreen);
            return false;
        }
        return true;
    }
    
    public class_339 getTextWidget() {
        return this.currentEditor;
    }
    
    private void toggleSearch() {
        searchVisible = !searchVisible;
        searchField.method_1862(searchVisible);
        searchNextButton.field_22764 = searchVisible;
        searchPrevButton.field_22764 = searchVisible;
        
        if (searchVisible) {
            method_25395(searchField);
            String searchText = searchField.method_1882();
            if (searchText != null && !searchText.trim().isEmpty()) {
                startSearch(searchText);
            }
        } else {
            endSearch();
        }
    }

    @Override
    public void method_25419() {
        super.method_25419();
        for (io.github.zhengzhengyiyi.api.ApiEntrypoint entrypoint : ConfigEditorClient.ENTRYPOINTS) {
            entrypoint.onEditerClose(this);
        }
        LOGGER.info("Config editor closed");
    }

    @Override
    public void method_25432() {
        super.method_25432();
        LOGGER.info("Config editor screen removed");
    }
    
    @Override
    public boolean method_25404(class_11908 input) {
        if (input.method_74228() == GLFW.GLFW_KEY_Q && input.method_74240() && input.method_74238()) {
            this.field_22787.method_1507(null);
            LOGGER.info("Config editor force closed by user shortcut");
            return true;
        }
        
        if (currentEditor == null) return super.method_25404(input);
        
        if (searchVisible) {
            if (input.method_74228() == GLFW.GLFW_KEY_ENTER) {
                startSearch(searchField.method_1882());
                return true;
            }
            if (input.method_74228() == GLFW.GLFW_KEY_ESCAPE) {
                toggleSearch();
                return true;
            }
            if (input.method_74228() == GLFW.GLFW_KEY_F3) {
                findNext();
                return true;
            }
        }
        
        if (input.method_74228() == GLFW.GLFW_KEY_F && input.method_74240()) {
            toggleSearch();
            return true;
        }
        
        return super.method_25404(input);
    }
    
    private void toggleTheme() {
        ModConfigData config = ConfigEditorClient.configManager.getConfig();
        switch (config.theme) {
            case DARK -> config.theme = ModConfigData.ThemeMode.LIGHT;
            case LIGHT -> config.theme = ModConfigData.ThemeMode.AUTO;
            case AUTO -> config.theme = ModConfigData.ThemeMode.DARK;
        }
        ConfigEditorClient.configManager.save();
        themeToggleButton.method_25355(class_2561.method_43471(getThemeButtonText()));
    }
    
    private String getThemeButtonText() {
        return switch (ConfigEditorClient.configManager.getConfig().theme) {
            case DARK -> "configeditor.theme.dark";
            case LIGHT -> "configeditor.theme.light";
            case AUTO -> "configeditor.theme.auto";
        };
    }

    @Override
    public boolean method_25402(class_11909 click, boolean doubled) {
        return super.method_25402(click, doubled);
    }
}
