/*
 * Decompiled with CFR 0.152.
 */
package io.github.zhengzhengyiyi.util;

import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import net.fabricmc.loader.api.FabricLoader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BackupHelper {
    private static final Logger LOGGER = LoggerFactory.getLogger(BackupHelper.class);
    private static final String BACKUP_DIR_NAME = "backups";
    private static final DateTimeFormatter TIMESTAMP_FORMATTER = DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss");

    public static Path backupConfigFile(Path configFile) {
        try {
            Path backupDir = BackupHelper.getBackupDirectory(configFile.getParent());
            String timestamp = LocalDateTime.now().format(TIMESTAMP_FORMATTER);
            String backupFileName = String.valueOf(configFile.getFileName()) + "." + timestamp + ".bak";
            Path backupFile = backupDir.resolve(backupFileName);
            Files.copy(configFile, backupFile, StandardCopyOption.REPLACE_EXISTING);
            LOGGER.info("Backup created: {}", (Object)backupFile);
            return backupFile;
        }
        catch (IOException e) {
            LOGGER.error("Failed to create backup for file: {}", (Object)configFile, (Object)e);
            return null;
        }
    }

    public static Path backupEntireConfigDirectory() {
        Path configDir = FabricLoader.getInstance().getGameDir();
        return BackupHelper.backupDirectoryToZip(configDir, "config_backup");
    }

    public static Path backupDirectoryToZip(Path directory, String backupName) {
        try {
            Path backupDir = BackupHelper.getBackupDirectory(directory);
            String timestamp = LocalDateTime.now().format(TIMESTAMP_FORMATTER);
            String zipFileName = backupName + "_" + timestamp + ".zip";
            Path zipFile = backupDir.resolve(zipFileName);
            BackupHelper.createZipFile(directory, zipFile);
            LOGGER.info("Directory backup created: {}", (Object)zipFile);
            return zipFile;
        }
        catch (IOException e) {
            LOGGER.error("Failed to create directory backup: {}", (Object)directory, (Object)e);
            return null;
        }
    }

    private static Path getBackupDirectory(Path parentDir) throws IOException {
        Path backupDir = parentDir.resolve(BACKUP_DIR_NAME);
        if (!Files.exists(backupDir, new LinkOption[0])) {
            Files.createDirectories(backupDir, new FileAttribute[0]);
        }
        return backupDir;
    }

    private static void createZipFile(final Path sourceDir, final Path zipFile) throws IOException {
        try (final ZipOutputStream zos = new ZipOutputStream(Files.newOutputStream(zipFile, new OpenOption[0]));){
            Files.walkFileTree(sourceDir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    if (!Files.isSameFile(file, zipFile)) {
                        BackupHelper.addToZipFile(file, sourceDir.relativize(file), zos);
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    if (!Files.isSameFile(dir, sourceDir)) {
                        ZipEntry entry = new ZipEntry(String.valueOf(sourceDir.relativize(dir)) + "/");
                        zos.putNextEntry(entry);
                        zos.closeEntry();
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        }
    }

    private static void addToZipFile(Path file, Path relativePath, ZipOutputStream zos) throws IOException {
        ZipEntry zipEntry = new ZipEntry(relativePath.toString());
        zos.putNextEntry(zipEntry);
        Files.copy(file, zos);
        zos.closeEntry();
    }

    public static Path getConfigBackupDirectory() {
        Path configDir = FabricLoader.getInstance().getConfigDir();
        return configDir.resolve(BACKUP_DIR_NAME);
    }

    public static Path[] listBackups() throws IOException {
        Path backupDir = BackupHelper.getConfigBackupDirectory();
        if (!Files.exists(backupDir, new LinkOption[0])) {
            return new Path[0];
        }
        return (Path[])Files.list(backupDir).sorted((p1, p2) -> {
            try {
                return Files.getLastModifiedTime(p2, new LinkOption[0]).compareTo(Files.getLastModifiedTime(p1, new LinkOption[0]));
            }
            catch (IOException e) {
                return 0;
            }
        }).toArray(Path[]::new);
    }

    public static int cleanupOldBackups(int maxBackups) throws IOException {
        Path[] backups = BackupHelper.listBackups();
        int deletedCount = 0;
        for (int i = maxBackups; i < backups.length; ++i) {
            Files.delete(backups[i]);
            ++deletedCount;
            LOGGER.info("Deleted old backup: {}", (Object)backups[i].getFileName());
        }
        return deletedCount;
    }
}

