package io.github.zhengzhengyiyi.gui;

import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.google.gson.JsonSyntaxException;
import io.github.zhengzhengyiyi.util.BackupHelper;
import io.github.zhengzhengyiyi.*;
import io.github.zhengzhengyiyi.config.ConfigData;
import io.github.zhengzhengyiyi.config.ConfigManager;
import io.github.zhengzhengyiyi.gui.theme.ThemeManager;
import io.github.zhengzhengyiyi.gui.widget.*;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_156;
import net.minecraft.class_2561;
import net.minecraft.class_332;
import net.minecraft.class_342;
import net.minecraft.class_410;
import net.minecraft.class_4185;
import net.minecraft.class_437;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.List;
import java.util.ArrayList;
import java.util.stream.Collectors;
import java.io.IOException;
import org.lwjgl.glfw.GLFW;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EditorScreen extends class_437 {
    private static final Logger LOGGER = LoggerFactory.getLogger(EditorScreen.class);
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private List<Path> configFiles;
    private int selectedIndex = 0;
    private boolean isJsonFile = true;
    private MultilineEditor multilineEditor;
    private boolean modified = false;
    private class_4185 saveButton;
    private class_4185 openFolderButton;
    private class_4185 backupButton;
    private String buffer = "";
    private class_342 searchField;
    private class_4185 searchNextButton;
    private class_4185 searchPrevButton;
    private class_4185 exitButton;
    private boolean searchVisible = false;
    private ThemeManager themeManager;
    private class_4185 themeToggleButton;
    private int fileListScrollOffset = 0;
    private class_4185 scrollUpButton;
    private class_4185 scrollDownButton;
    private List<class_4185> fileButtonList;

    public EditorScreen() {
        super(class_2561.method_43471("configeditor.title"));
    }

    @Override
    protected void method_25426() {
        super.method_25426();
        
        themeManager = ThemeManager.getInstance();
        themeToggleButton = class_4185.method_46430(
                class_2561.method_43471(getThemeButtonText()),
                button -> toggleTheme())
                .method_46434(this.field_22789 - 50, 5, 45, 16)
                .method_46431();
        this.method_37063(themeToggleButton);
        
        try {
            Path configDir = FabricLoader.getInstance().getConfigDir();
            configFiles = new ArrayList<>();
            loadConfigFilesRecursively(configDir, configFiles);
            LOGGER.info("Found {} config files", configFiles.size());
        } catch (Exception e) {
            configFiles = new ArrayList<>();
            LOGGER.error("Failed to list config files", e);
        }

        fileButtonList = new ArrayList<>();
        
        scrollUpButton = class_4185.method_46430(class_2561.method_43470("↑"), button -> scrollUp())
                .method_46434(140, 25, 20, 20)
                .method_46431();
        scrollDownButton = class_4185.method_46430(class_2561.method_43470("↓"), button -> scrollDown())
                .method_46434(140, this.field_22790 - 45, 20, 20)
                .method_46431();
        
        this.method_37063(scrollUpButton);
        this.method_37063(scrollDownButton);
        
        renderFileList();
        
        multilineEditor = new MultilineEditor(
                170, 20, 
                this.field_22789 - 180, this.field_22790 - 60,
                class_2561.method_43471("configeditor.editor"));
        multilineEditor.setChangedListener(text -> {
            if (!buffer.equals(text)) {
                modified = true;
                updateButtonStates();
            } else {
                modified = false;
            }
        });

        saveButton = class_4185.method_46430(
                class_2561.method_43471("configeditor.button.save"),
                button -> saveFile())
                .method_46434(this.field_22789 - 170, this.field_22790 - 30, 80, 20)
                .method_46431();
        
        backupButton = class_4185.method_46430(
                class_2561.method_43471("configeditor.button.backup"), 
                button -> BackupHelper.backupEntireConfigDirectory())
                .method_46434(this.field_22789 - 260, this.field_22790 - 30, 80, 20)
                .method_46431();

        openFolderButton = class_4185.method_46430(
                class_2561.method_43471("configeditor.button.openfolder"),
                button -> openConfigFolder())
                .method_46434(this.field_22789 - 80, this.field_22790 - 30, 70, 20)
                .method_46431();
        
        exitButton = class_4185.method_46430(
                class_2561.method_43471("configeditor.button.close"),
                button -> this.method_25419())
                .method_46434(0, 0, 80, 20)
                .method_46431();
        
        searchField = new class_342(field_22793, this.field_22789 - 300, 5, 150, 16, class_2561.method_43471("configeditor.search.placeholder"));
        searchField.method_1863(text -> {
            if (!text.trim().isEmpty()) {
                multilineEditor.startSearch(text.trim());
            }
        });
        searchField.method_1862(true);

        searchNextButton = class_4185.method_46430(class_2561.method_43470("↓"), button -> {
            if (multilineEditor != null) {
                multilineEditor.findNext();
            }
        }).method_46434(this.field_22789 - 145, 5, 20, 16).method_46431();
        searchNextButton.field_22764 = true;
        
        searchPrevButton = class_4185.method_46430(class_2561.method_43470("↑"), button -> {
            if (multilineEditor != null) {
                multilineEditor.findPrevious();
            }
        }).method_46434(this.field_22789 - 165, 5, 20, 16).method_46431();
        searchPrevButton.field_22764 = true;
        
        class_4185 closeSearchButton = class_4185.method_46430(class_2561.method_43470("✕"), button -> {
            searchField.method_1852("");
            multilineEditor.endSearch();
        }).method_46434(this.field_22789 - 120, 5, 20, 16).method_46431();
        
        this.method_37063(saveButton);
        this.method_37063(backupButton);
        this.method_37063(openFolderButton);
        this.method_37063(searchField);
        this.method_37063(searchNextButton);
        this.method_37063(searchPrevButton);
        this.method_37063(closeSearchButton);
        this.method_37063(exitButton);
        this.method_37063(multilineEditor);
        
        this.method_48265(multilineEditor);

        if (!configFiles.isEmpty()) {
            loadFile(selectedIndex);
        } else {
            multilineEditor.setText("{}");
            multilineEditor.setEditable(false);
            LOGGER.warn("No config files found in config directory");
        }
        
        for (io.github.zhengzhengyiyi.api.ApiEntrypoint entrypoint : ConfigEditorClient.ENTRYPOINTS) {
            entrypoint.onEditerOpen(this);
        }
        
        updateButtonStates();
        updateScrollButtons();
    }
    
    /**
     * === 关键修改 2: 从1.21.10移植的递归文件扫描方法 ===
     * 递归加载配置目录下的所有配置文件
     */
    private void loadConfigFilesRecursively(Path directory, List<Path> fileList) throws IOException {
        if (!Files.exists(directory) || !Files.isDirectory(directory)) {
            return;
        }
        
        try (var stream = Files.list(directory)) {
            List<Path> entries = stream.collect(Collectors.toList());
            
            for (Path entry : entries) {
                if (Files.isDirectory(entry)) {
                    loadConfigFilesRecursively(entry, fileList);
                } else if (Files.isRegularFile(entry) && isConfigFile(entry)) {
                    fileList.add(entry);
                }
            }
        }
    }
    
    public boolean isConfigFile(Path file) {
        String fileName = file.getFileName().toString();
        if (fileName.equals(".DS_Store") || 
            fileName.startsWith(".") || 
            fileName.equals("Thumbs.db")) {
            return false;
        }
        
        String lowerName = fileName.toLowerCase();
        return lowerName.endsWith(".json") || 
               lowerName.endsWith(".txt") || 
               lowerName.endsWith(".yml") || 
               lowerName.endsWith(".yaml") || 
               lowerName.endsWith(".properties") || 
               lowerName.endsWith(".toml") || 
               lowerName.endsWith(".conf") || 
               lowerName.endsWith(".cfg") || 
               lowerName.endsWith(".ini");
    }
    
    private void renderFileList() {
        for (class_4185 button : fileButtonList) {
            this.method_37066(button);
        }
        fileButtonList.clear();
        
        int buttonY = 25;
        for (int i = fileListScrollOffset; i < configFiles.size() && i < fileListScrollOffset + 15; i++) {
            int index = i;
            Path file = configFiles.get(i);
            Path configDir = FabricLoader.getInstance().getConfigDir();
            String relativePath = configDir.relativize(file).toString();
            
            class_4185 button = class_4185.method_46430(
                    class_2561.method_43470(formatFileName(relativePath)),
                    _button -> switchFile(index))
                    .method_46434(10, buttonY, 130, 20)
                    .method_46431();
            this.method_37063(button);
            fileButtonList.add(button);
            buttonY += 23;
        }
    }

    private void scrollUp() {
        if (fileListScrollOffset > 0) {
            fileListScrollOffset--;
            renderFileList();
            updateScrollButtons();
        }
    }

    private void scrollDown() {
        if (fileListScrollOffset < configFiles.size() - 15) {
            fileListScrollOffset++;
            renderFileList();
            updateScrollButtons();
        }
    }

    private void updateScrollButtons() {
        scrollUpButton.field_22763 = fileListScrollOffset > 0;
        scrollDownButton.field_22763 = fileListScrollOffset < configFiles.size() - 15;
    }

    private void updateButtonStates() {
        if (saveButton != null) {
            saveButton.field_22763 = true;
        }
    }
    
    private String formatFileName(String filePath) {
    if (filePath == null || filePath.isEmpty()) {
        return "";
    }
   	
    if (filePath.length() <= 22) {
        return filePath;
    }
    
    int lastSeparator = filePath.lastIndexOf('/');
    if (lastSeparator != -1) {
        String folder = filePath.substring(0, lastSeparator);
        String fileName = filePath.substring(lastSeparator + 1);
        
        if (folder.length() > 8) {
            folder = folder.substring(0, Math.min(12, folder.length())) + "..";
        }
        if (fileName.length() > 12) {
            fileName = fileName.substring(0, Math.min(12, fileName.length())) + "..";
        }
        return folder + "/" + fileName;
    } else {
        return filePath.substring(0, Math.min(17, filePath.length())) + 
               (filePath.length() > 17 ? "..." : "");
    }
}

    private void switchFile(int index) {
        loadFile(index);
    }
    
    public static String getFileName(java.nio.file.Path path) {
        if (path == null) {
            return "";
        }
        
        return path.getFileName().toString();
    }

    /**
     */
    private void loadFile(int index) {
        if (index < 0 || index >= configFiles.size()) {
            LOGGER.error("Invalid file index: {}", index);
            return;
        }
        
        selectedIndex = index;
        modified = false;
        Path file = configFiles.get(index);
        
        try {
            String content = readFileWithFallbackEncoding(file);
            buffer = content;
            
            boolean isJson = checkIfJson(content);
            isJsonFile = isJson;
            
            String formattedContent = content;
            if (isJsonFile) {
                try {
                    JsonElement json = JsonParser.parseString(content);
                    formattedContent = GSON.toJson(json);
                } catch (JsonSyntaxException e) {
                    isJsonFile = false;
                    LOGGER.warn("File {} looks like JSON but has syntax errors, treating as text", file.getFileName());
                }
            }
            
            multilineEditor.setText(formattedContent);
            multilineEditor.setFileName(getFileName(file));
            
            LOGGER.info("Successfully loaded {} file: {}", isJsonFile ? "JSON" : "text", file.getFileName());
        } catch (Exception e) {
            LOGGER.error("Failed to load config file: {}", file.getFileName(), e);
            multilineEditor.setText("{}");
            multilineEditor.setEditable(false);
            showErrorPopup(class_2561.method_43471("configeditor.error.loadfailed"));
        }
        
        updateButtonStates();
    }
    
    /**
     */
    private String readFileWithFallbackEncoding(Path file) throws IOException {
        try {
            return Files.readString(file, java.nio.charset.StandardCharsets.UTF_8);
        } catch (java.nio.charset.MalformedInputException e) {
            try {
                return Files.readString(file, java.nio.charset.Charset.defaultCharset());
            } catch (java.nio.charset.MalformedInputException e2) {
                return Files.readString(file, java.nio.charset.StandardCharsets.ISO_8859_1);
            }
        }
    }

    private void saveFile() {
        saveFileAsync(null);
    }
    
    /**
     */
    private boolean checkIfJson(String content) {
        if (content == null || content.trim().isEmpty()) {
            return false;
        }
        
        String trimmed = content.trim();
        if (!trimmed.startsWith("{") && !trimmed.startsWith("[")) {
            return false;
        }
        
        try {
            JsonParser.parseString(content);
            return true;
        } catch (JsonSyntaxException e) {
            return false;
        }
    }
    
    private void saveFileAsync(Runnable callback) {
        if (configFiles.isEmpty()) return;
        
        Path file = configFiles.get(selectedIndex);
        String content = multilineEditor.getText();
        
        if (isJsonFile) {
            try {
                JsonParser.parseString(content);
            } catch (JsonSyntaxException e) {
                LOGGER.warn("Invalid JSON syntax in file: {}", file.getFileName());
                showErrorPopup(class_2561.method_43471("configeditor.error.invalidjson"));
                return;
            }
        }
        
        new Thread(() -> {
            int retryCount = 0;
            final int maxRetries = 3;
            
            while (retryCount <= maxRetries) {
                try {
                    try (FileChannel channel = FileChannel.open(file, 
                         StandardOpenOption.WRITE, StandardOpenOption.CREATE)) {
                        FileLock lock = channel.tryLock(0, Long.MAX_VALUE, false);
                        if (lock != null) {
                            try {
                                Files.writeString(file, content, StandardOpenOption.TRUNCATE_EXISTING);
                                modified = false;
                                buffer = content;
                                LOGGER.info("Successfully saved {} file: {}", isJsonFile ? "JSON" : "text", file.getFileName());
                                
                                if (callback != null) {
                                    field_22787.execute(callback);
                                } else {
                                    field_22787.execute(this::updateButtonStates);
                                }
                                return;
                            } finally {
                                lock.release();
                            }
                        }
                    }
                    
                    retryCount++;
                    if (retryCount <= maxRetries) {
                        Thread.sleep(300);
                    }
                    
                } catch (Exception e) {
                    retryCount++;
                    if (retryCount > maxRetries) {
                        LOGGER.error("Failed to save config file after {} attempts: {}", maxRetries, file.getFileName(), e);
                        field_22787.execute(() -> {
                            if (this.field_22787.field_1755 != null && this.field_22787.field_1755.equals(this)) {
                                showErrorPopup(class_2561.method_43471("configeditor.error.savefailed"));
                            }
                        });
                    }
                }
            }
            
            field_22787.execute(() -> {
                if (this.field_22787.field_1755 != null && this.field_22787.field_1755.equals(this)) {
                    showErrorPopup(class_2561.method_43471("configeditor.error.fileretryfailed"));
                }
            });
            
        }).start();
    }
    
    private void openConfigFolder() {
        try {
            Path configDir = FabricLoader.getInstance().getConfigDir();
            LOGGER.info("Config folder location: {}", configDir);
            class_156.method_668().method_673(configDir.toUri());
        } catch (Exception e) {
            LOGGER.error("Failed to get config folder", e);
        }
    }

    private void showErrorPopup(class_2561 message) {
        field_22787.method_1507(new class_410(
            result -> {
                this.field_22787.method_1507(this);
            },
            class_2561.method_43471("configeditor.confirm.title"),
            message
        ));
    }

    public void showMessagePopup(class_2561 message) {
        field_22787.method_1507(new class_410(
            result -> {
                this.method_25419();
                this.field_22787.method_1507(null);
            },
            class_2561.method_43471("configeditor.confirm.title"),
            message
        ));
    }

    @Override
    public void method_25394(class_332 context, int mouseX, int mouseY, float delta) {
        if (ConfigManager.getConfig().doRenderBackground) context.method_25294(0, 0, this.field_22789, this.field_22790, themeManager.getBackgroundColor());
        super.method_25394(context, mouseX, mouseY, delta);
        
        if (!configFiles.isEmpty()) {
            String status = modified ? "* " + configFiles.get(selectedIndex).getFileName().toString() : 
                configFiles.get(selectedIndex).getFileName().toString();
            String editorType = isJsonFile ? " [JSON]" : " [Text]";
            context.method_51433(this.field_22793, status + editorType, 170, 5, modified ? 0xFFFF00 : 0xFFFFFF, false);
        }
    }

    @Override
    public boolean method_25422() {
        if (modified) {
            class_410 confirmScreen = new class_410(
                result -> {
                    if (result) {
                        saveFileAsync(() -> {this.method_25419(); this.field_22787.method_1507(null);});
                    } else {
                        this.method_25419();
                        this.field_22787.method_1507(null);
                    }
                },
                class_2561.method_43471("configeditor.confirm.title"),
                class_2561.method_43471("configeditor.confirm.unsavedclose")
            );
            this.field_22787.method_1507(confirmScreen);
            return false;
        }
        return true;
    }
    
    public MultilineEditor getTextWidget() {
        return this.multilineEditor;
    }
    
    private void toggleSearch() {
        if (multilineEditor == null) return;
        
        searchVisible = !searchVisible;
        searchField.method_1862(searchVisible);
        searchNextButton.field_22764 = searchVisible;
        searchPrevButton.field_22764 = searchVisible;
        
        if (searchVisible) {
            method_25395(searchField);
            String searchText = searchField.method_1882();
            if (searchText != null && !searchText.trim().isEmpty()) {
                multilineEditor.startSearch(searchText);
            }
        } else {
            multilineEditor.endSearch();
        }
    }

    @Override
    public void method_25419() {
        super.method_25419();
        for (io.github.zhengzhengyiyi.api.ApiEntrypoint entrypoint : ConfigEditorClient.ENTRYPOINTS) {
            entrypoint.onEditerClose(this);
        }
        LOGGER.info("Config editor closed");
    }

    @Override
    public void method_25432() {
        super.method_25432();
        LOGGER.info("Config editor screen removed");
    }
    
    @Override
    public boolean method_25404(int keyCode, int scanCode, int modifiers) {
        if (keyCode == GLFW.GLFW_KEY_Q && method_25441() && method_25443()) {
            this.field_22787.method_1507(null);
            LOGGER.info("Config editor force closed by user shortcut");
            return true;
        }
        
        if (multilineEditor == null) return super.method_25404(keyCode, scanCode, modifiers);
        
        if (searchVisible) {
            if (keyCode == GLFW.GLFW_KEY_ENTER) {
                multilineEditor.startSearch(searchField.method_1882());
                return true;
            }
            if (keyCode == GLFW.GLFW_KEY_ESCAPE) {
                toggleSearch();
                return true;
            }
            if (keyCode == GLFW.GLFW_KEY_F3) {
                multilineEditor.findNext();
                return true;
            }
        }
        
        if (keyCode == GLFW.GLFW_KEY_F && method_25441()) {
            toggleSearch();
            return true;
        }
        
        return super.method_25404(keyCode, scanCode, modifiers);
    }
    
    private void toggleTheme() {
        ConfigData config = ConfigManager.getConfig();
        switch (config.theme) {
            case DARK -> config.theme = ConfigData.ThemeMode.LIGHT;
            case LIGHT -> config.theme = ConfigData.ThemeMode.AUTO;
            case AUTO -> config.theme = ConfigData.ThemeMode.DARK;
        }
        ConfigManager.save();
        themeToggleButton.method_25355(class_2561.method_43471(getThemeButtonText()));
    }
    
    private String getThemeButtonText() {
        return switch (ConfigManager.getConfig().theme) {
            case DARK -> "configeditor.theme.dark";
            case LIGHT -> "configeditor.theme.light";
            case AUTO -> "configeditor.theme.auto";
        };
    }
}
