package io.github.zhengzhengyiyi.gui;

import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.google.gson.JsonSyntaxException;
import io.github.zhengzhengyiyi.util.BackupHelper;
import io.github.zhengzhengyiyi.util.ConfigDiffEngine;
import io.github.zhengzhengyiyi.*;
import io.github.zhengzhengyiyi.config.ConfigData;
import io.github.zhengzhengyiyi.config.ConfigManager;
import io.github.zhengzhengyiyi.gui.theme.ThemeManager;
import io.github.zhengzhengyiyi.gui.widget.*;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_156;
import net.minecraft.class_2561;
import net.minecraft.class_332;
import net.minecraft.class_342;
import net.minecraft.class_410;
import net.minecraft.class_4185;
import net.minecraft.class_437;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.List;
import java.util.ArrayList;
import java.util.stream.Collectors;
import java.io.IOException;
import org.lwjgl.glfw.GLFW;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EditorScreen extends class_437 {
    private static final Logger LOGGER = LoggerFactory.getLogger(EditorScreen.class);
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private List<Path> configFiles;
    private int selectedIndex = 0;
    private MultilineEditor multilineEditor;
    private boolean modified = false;
    private class_4185 saveButton;
    private class_4185 openFolderButton;
    private class_4185 backupButton;
    private String buffer = "";
    private class_342 searchField;
    private class_4185 searchNextButton;
    private class_4185 searchPrevButton;
    private class_4185 exitButton;
    private class_4185 diffButton;
    private boolean searchVisible = false;
    private ThemeManager themeManager;
    private class_4185 themeToggleButton;
    private int fileListScrollOffset = 0;
    private class_4185 scrollUpButton;
    private class_4185 scrollDownButton;
    private List<class_4185> fileButtonList;
    private boolean showDiffView = false;
    private List<ConfigDiffEngine.DiffLine> currentDiff = new ArrayList<>();
    private String originalText = "";

    public EditorScreen() {
        super(class_2561.method_43471("configeditor.title"));
    }

    @Override
    protected void method_25426() {
        super.method_25426();
        
        themeManager = ThemeManager.getInstance();
        themeToggleButton = class_4185.method_46430(
                class_2561.method_43471(getThemeButtonText()),
                button -> toggleTheme())
                .method_46434(this.field_22789 - 350, 5, 45, 16)
                .method_46431();
        this.method_37063(themeToggleButton);
        
        try {
            Path configDir = FabricLoader.getInstance().getConfigDir();
            configFiles = Files.list(configDir)
                    .filter(Files::isRegularFile)
                    .filter(path -> path.toString().endsWith(".json"))
                    .collect(Collectors.toList());
            LOGGER.info("Found {} config files", configFiles.size());
        } catch (Exception e) {
            configFiles = new ArrayList<>();
            LOGGER.error("Failed to list config files", e);
        }

        fileButtonList = new ArrayList<>();
        
        scrollUpButton = class_4185.method_46430(class_2561.method_43470("↑"), button -> scrollUp())
                .method_46434(140, 25, 20, 20)
                .method_46431();
        scrollDownButton = class_4185.method_46430(class_2561.method_43470("↓"), button -> scrollDown())
                .method_46434(140, this.field_22790 - 45, 20, 20)
                .method_46431();
        
        this.method_37063(scrollUpButton);
        this.method_37063(scrollDownButton);
        
        renderFileList();
        
        multilineEditor = new MultilineEditor(
                170, 20, 
                this.field_22789 - 180, this.field_22790 - 60,
                class_2561.method_43471("configeditor.editor"));
        multilineEditor.setChangedListener(text -> {
            if (!buffer.equals(text)) {
                modified = true;
                updateButtonStates();
            } else {
                modified = false;
            }
        });

        saveButton = class_4185.method_46430(
                class_2561.method_43471("configeditor.button.save"),
                button -> saveFile())
                .method_46434(this.field_22789 - 170, this.field_22790 - 30, 80, 20)
                .method_46431();
        
        backupButton = class_4185.method_46430(
                class_2561.method_43471("configeditor.button.backup"), 
                button -> BackupHelper.backupEntireConfigDirectory())
                .method_46434(0, 0, 70, 20)
                .method_46431();

        openFolderButton = class_4185.method_46430(
                class_2561.method_43471("configeditor.button.openfolder"),
                button -> openConfigFolder())
                .method_46434(this.field_22789 - 80, this.field_22790 - 30, 70, 20)
                .method_46431();
        
        exitButton = class_4185.method_46430(
                class_2561.method_43471("configeditor.button.close"),
                button -> this.method_25419())
                .method_46434(0, this.field_22790 - 25, 80, 20)
                .method_46431();
        
        diffButton = class_4185.method_46430(
            class_2561.method_43470("Different"),
            button -> {
                if (!showDiffView) {
                    originalText = multilineEditor.getText();
                    currentDiff = ConfigDiffEngine.computeDiff(originalText, multilineEditor.getText());
                } else {
                    currentDiff.clear();
                }
                showDiffView = !showDiffView;
            }
        ).method_46434(this.field_22789 - 40, 0, 40, 16).method_46431();
        this.method_37063(diffButton);
        
        searchField = new class_342(field_22793, this.field_22789 - 250, 5, 120, 16, class_2561.method_43471("configeditor.search.placeholder"));
        searchField.method_1862(false);
        
        searchNextButton = class_4185.method_46430(class_2561.method_43471("configeditor.search.next"), button -> {
            if (multilineEditor != null) {
                multilineEditor.findNext();
            }
        }).method_46434(this.field_22789 - 125, 5, 50, 16).method_46431();
        searchNextButton.field_22764 = false;
        
        searchPrevButton = class_4185.method_46430(class_2561.method_43471("configeditor.search.prev"), button -> {
            if (multilineEditor != null) {
                multilineEditor.findPrevious();
            }
        }).method_46434(this.field_22789 - 70, 5, 50, 16).method_46431();
        searchPrevButton.field_22764 = false;
        
        class_4185 searchButton = class_4185.method_46430(class_2561.method_43471("configeditor.button.search"), button -> {
            toggleSearch();
        }).method_46434(this.field_22789 - 300, 5, 45, 16).method_46431();
        
        this.method_37063(saveButton);
        this.method_37063(backupButton);
        this.method_37063(openFolderButton);
        this.method_37063(searchField);
        this.method_37063(searchNextButton);
        this.method_37063(searchPrevButton);
        this.method_37063(exitButton);
        this.method_37063(searchButton);
        this.method_37063(multilineEditor);
        
        this.method_48265(multilineEditor);

        if (!configFiles.isEmpty()) {
            loadFile(selectedIndex);
        } else {
            multilineEditor.setText("{}");
            multilineEditor.setEditable(false);
            LOGGER.warn("No config files found in config directory");
        }
        
        for (io.github.zhengzhengyiyi.api.ApiEntrypoint entrypoint : ConfigEditorClient.ENTRYPOINTS) {
            entrypoint.onEditerOpen(this);
        }
        
        updateButtonStates();
        updateScrollButtons();
    }

    private void renderFileList() {
        for (class_4185 button : fileButtonList) {
            this.method_37066(button);
        }
        fileButtonList.clear();
        
        int buttonY = 25;
        for (int i = fileListScrollOffset; i < configFiles.size() && i < fileListScrollOffset + 15; i++) {
            int index = i;
            Path file = configFiles.get(i);
            String fileName = file.getFileName().toString();
            
            class_4185 button = class_4185.method_46430(
                    class_2561.method_43470(fileName.length() > 20 ? fileName.substring(0, 17) + "..." : fileName),
                    _button -> switchFile(index))
                    .method_46434(10, buttonY, 130, 20)
                    .method_46431();
            this.method_37063(button);
            fileButtonList.add(button);
            buttonY += 23;
        }
    }

    private void scrollUp() {
        if (fileListScrollOffset > 0) {
            fileListScrollOffset--;
            renderFileList();
            updateScrollButtons();
        }
    }

    private void scrollDown() {
        if (fileListScrollOffset < configFiles.size() - 15) {
            fileListScrollOffset++;
            renderFileList();
            updateScrollButtons();
        }
    }

    private void updateScrollButtons() {
        scrollUpButton.field_22763 = fileListScrollOffset > 0;
        scrollDownButton.field_22763 = fileListScrollOffset < configFiles.size() - 15;
    }

    private void updateButtonStates() {
        if (saveButton != null) {
            saveButton.field_22763 = modified && !configFiles.isEmpty();
        }
    }

    private void switchFile(int index) {
    	loadFile(index);
//        if (modified) {
//            ConfirmScreen confirmScreen = new ConfirmScreen(
//                result -> {
//                    if (result) {
//                        saveFileAsync(() -> loadFile(index));
//                    } else {
//                        loadFile(index);
//                    }
//                },
//                Text.translatable("configeditor.confirm.title"),
//                Text.translatable("configeditor.confirm.unsaved")
//            );
//            this.client.setScreen(confirmScreen);
//        } else {
//            loadFile(index);
//        }
    }

    private void loadFile(int index) {
        if (index < 0 || index >= configFiles.size()) {
            LOGGER.error("Invalid file index: {}", index);
            return;
        }
        
        selectedIndex = index;
        modified = false;
        Path file = configFiles.get(index);
        
        try {
            String content = Files.readString(file);
            buffer = content;
            JsonElement json = JsonParser.parseString(content);
            String formattedContent = GSON.toJson(json);
            multilineEditor.setText(formattedContent);
            LOGGER.info("Successfully loaded config file: {}", file.getFileName());
        } catch (Exception e) {
            String text = null;
            try {
                text = Files.readString(file);
            } catch (IOException ioexception) {
                LOGGER.error("tried to read file except IOException: ", ioexception.toString());
            }
            if (text == null) {
                LOGGER.error("Failed to load config file: {}", file.getFileName(), e);
                multilineEditor.setText("{}");
                multilineEditor.setEditable(false);
                showErrorPopup(class_2561.method_43471("configeditor.error.loadfailed"));
            }
            multilineEditor.setText(text);
        }
        
        updateButtonStates();
    }

    private void saveFile() {
        saveFileAsync(null);
    }

//    private void saveFileAsync(Runnable callback) {
//        if (configFiles.isEmpty()) return;
//        
//        Path file = configFiles.get(selectedIndex);
//        String content = multilineEditor.getText();
//        
//        try {
//            JsonParser.parseString(content);
//        } catch (JsonSyntaxException e) {
//            LOGGER.warn("Invalid JSON syntax in file: {}", file.getFileName());
//            showErrorPopup(Text.translatable("configeditor.error.invalidjson"));
//            return;
//        }
//        
//        new Thread(() -> {
//            try {
//                Files.writeString(file, content);
//                modified = false;
//                LOGGER.info("Successfully saved config file: {}", file.getFileName());
//                
//                if (callback != null) {
//                    client.execute(callback);
//                } else {
//                    client.execute(() -> {
//                        updateButtonStates();
//                    });
//                }
//            } catch (Exception e) {
//                LOGGER.error("Failed to save config file: {}", file.getFileName(), e);
//                client.execute(() -> 
//                    showErrorPopup(Text.translatable("configeditor.error.savefailed")));
//            }
//        }).start();
//    }
    
    private void saveFileAsync(Runnable callback) {
        if (configFiles.isEmpty()) return;
        
        Path file = configFiles.get(selectedIndex);
        String content = multilineEditor.getText();
        
        try {
            JsonParser.parseString(content);
        } catch (JsonSyntaxException e) {
            LOGGER.warn("Invalid JSON syntax in file: {}", file.getFileName());
            showErrorPopup(class_2561.method_43471("configeditor.error.invalidjson"));
            return;
        }
        
        new Thread(() -> {
            int retryCount = 0;
            final int maxRetries = 3;
            
            while (retryCount <= maxRetries) {
                try {
                    try (FileChannel channel = FileChannel.open(file, 
                         StandardOpenOption.WRITE, StandardOpenOption.CREATE)) {
                        FileLock lock = channel.tryLock(0, Long.MAX_VALUE, false);
                        if (lock != null) {
                            try {
                                Files.writeString(file, content, StandardOpenOption.TRUNCATE_EXISTING);
                                modified = false;
                                LOGGER.info("Successfully saved config file: {}", file.getFileName());
                                
                                if (callback != null) {
                                    field_22787.execute(callback);
                                } else {
                                    field_22787.execute(this::updateButtonStates);
                                }
                                return;
                            } finally {
                                lock.release();
                            }
                        }
                    }
                    
                    retryCount++;
                    if (retryCount <= maxRetries) {
                        Thread.sleep(300);
                    }
                    
                } catch (Exception e) {
                    retryCount++;
                    if (retryCount > maxRetries) {
                        LOGGER.error("Failed to save config file after {} attempts: {}", maxRetries, file.getFileName(), e);
                        field_22787.execute(() -> {
                            if (this.field_22787.field_1755 != null && this.field_22787.field_1755.equals(this)) {
                                showErrorPopup(class_2561.method_43471("configeditor.error.savefailed"));
                            }
                        });
                    }
                }
            }
            
            field_22787.execute(() -> {
                if (this.field_22787.field_1755 != null && this.field_22787.field_1755.equals(this)) {
                    showErrorPopup(class_2561.method_43471("configeditor.error.fileretryfailed"));
                }
            });
            
        }).start();
    }
    
    private void openConfigFolder() {
        try {
            Path configDir = FabricLoader.getInstance().getConfigDir();
            LOGGER.info("Config folder location: {}", configDir);
            class_156.method_668().method_673(configDir.toUri());
        } catch (Exception e) {
            LOGGER.error("Failed to get config folder", e);
        }
    }

    private void showErrorPopup(class_2561 message) {
        field_22787.method_1507(new class_410(
            result -> {
                this.field_22787.method_1507(this);
            },
            class_2561.method_43471("configeditor.confirm.title"),
            message
        ));
    }

    public void showMessagePopup(class_2561 message) {
        field_22787.method_1507(new class_410(
            result -> {
                this.method_25419();
                this.field_22787.method_1507(null);
            },
            class_2561.method_43471("configeditor.confirm.title"),
            message
        ));
    }

    @Override
    public void method_25394(class_332 context, int mouseX, int mouseY, float delta) {
        if (ConfigManager.getConfig().doRenderBackground) context.method_25294(0, 0, this.field_22789, this.field_22790, themeManager.getBackgroundColor());
        super.method_25394(context, mouseX, mouseY, delta);
        
        if (!configFiles.isEmpty()) {
            String status = modified ? "* " + configFiles.get(selectedIndex).getFileName().toString() : 
                configFiles.get(selectedIndex).getFileName().toString();
            context.method_51433(this.field_22793, status, 170, 5, modified ? 0xFFFF00 : 0xFFFFFF, false);
        }
        
        renderDiffPanel(context, mouseX, mouseY);
    }

    @Override
    public boolean method_25422() {
        if (modified) {
            class_410 confirmScreen = new class_410(
                result -> {
                    if (result) {
                        saveFileAsync(() -> {this.method_25419(); this.field_22787.method_1507(null);});
                    } else {
                        this.method_25419();
                        this.field_22787.method_1507(null);
                    }
                },
                class_2561.method_43471("configeditor.confirm.title"),
                class_2561.method_43471("configeditor.confirm.unsavedclose")
            );
            this.field_22787.method_1507(confirmScreen);
            return false;
        }
        return true;
    }
    
    public MultilineEditor getTextWidget() {
        return this.multilineEditor;
    }
    
    private void toggleSearch() {
        if (multilineEditor == null) return;
        
        searchVisible = !searchVisible;
        searchField.method_1862(searchVisible);
        searchNextButton.field_22764 = searchVisible;
        searchPrevButton.field_22764 = searchVisible;
        
        if (searchVisible) {
            method_25395(searchField);
            String searchText = searchField.method_1882();
            if (searchText != null && !searchText.trim().isEmpty()) {
                multilineEditor.startSearch(searchText);
            }
        } else {
            multilineEditor.endSearch();
        }
    }

    @Override
    public void method_25419() {
        super.method_25419();
        for (io.github.zhengzhengyiyi.api.ApiEntrypoint entrypoint : ConfigEditorClient.ENTRYPOINTS) {
            entrypoint.onEditerClose(this);
        }
        LOGGER.info("Config editor closed");
    }

    @Override
    public void method_25432() {
        super.method_25432();
        LOGGER.info("Config editor screen removed");
    }
    
    @Override
    public boolean method_25404(int keyCode, int scanCode, int modifiers) {
    	if (keyCode == GLFW.GLFW_KEY_Q && method_25441() && method_25443()) {
            this.field_22787.method_1507(null);
            LOGGER.info("Config editor force closed by user shortcut");
            return true;
        }
    	
        if (multilineEditor == null) return super.method_25404(keyCode, scanCode, modifiers);
        
        if (searchVisible) {
            if (keyCode == GLFW.GLFW_KEY_ENTER) {
                multilineEditor.startSearch(searchField.method_1882());
                return true;
            }
            if (keyCode == GLFW.GLFW_KEY_ESCAPE) {
                toggleSearch();
                return true;
            }
            if (keyCode == GLFW.GLFW_KEY_F3) {
                multilineEditor.findNext();
                return true;
            }
        }
        
        if (keyCode == GLFW.GLFW_KEY_F && method_25441()) {
            toggleSearch();
            return true;
        }
        
        return super.method_25404(keyCode, scanCode, modifiers);
    }
    
    private void toggleTheme() {
        ConfigData config = ConfigManager.getConfig();
        switch (config.theme) {
            case DARK -> config.theme = ConfigData.ThemeMode.LIGHT;
            case LIGHT -> config.theme = ConfigData.ThemeMode.AUTO;
            case AUTO -> config.theme = ConfigData.ThemeMode.DARK;
        }
        ConfigManager.save();
        themeToggleButton.method_25355(class_2561.method_43471(getThemeButtonText()));
    }
    
    private String getThemeButtonText() {
        return switch (ConfigManager.getConfig().theme) {
            case DARK -> "configeditor.theme.dark";
            case LIGHT -> "configeditor.theme.light";
            case AUTO -> "configeditor.theme.auto";
        };
    }
    
    @SuppressWarnings("incomplete-switch")
	private void renderDiffPanel(class_332 context, int mouseX, int mouseY) {
        if (!showDiffView || currentDiff == null || currentDiff.isEmpty()) return;
        
        int panelWidth = 180;
        int panelHeight = 60;
        int x = this.field_22789 - panelWidth - 10;
        int y = this.field_22790 - panelHeight - 10;
        
        context.method_25294(x, y, x + panelWidth, y + panelHeight, 0xCC000000);
        context.method_49601(x, y, panelWidth, panelHeight, 0xFFFFFFFF);
        
        int startY = y + 5;
        context.method_51433(field_22793, "Line Changes", x + 5, startY, 0xFFFFFF00, false);
        
        int added = 0, deleted = 0;
        for (ConfigDiffEngine.DiffLine line : currentDiff) {
            switch (line.type) {
                case ADDED -> added++;
                case DELETED -> deleted++;
            }
        }
        
        startY += 12;
        context.method_51433(field_22793, "Added: +" + added, x + 10, startY, 0xFF00FF00, false);
        startY += 10;
        context.method_51433(field_22793, "Deleted: -" + deleted, x + 10, startY, 0xFFFF0000, false);
        startY += 10;
        context.method_51433(field_22793, "Total: " + currentDiff.size(), x + 10, startY, 0xFFFFFFFF, false);
        
        if (isMouseOverPanel(mouseX, mouseY, x, y, panelWidth, panelHeight)) {
            context.method_51433(field_22793, "Click to close", x + panelWidth - 60, y + panelHeight - 12, 0xFF8888FF, false);
        }
    }

    private boolean isMouseOverPanel(int mouseX, int mouseY, int panelX, int panelY, int width, int height) {
        return mouseX >= panelX && mouseX <= panelX + width && 
               mouseY >= panelY && mouseY <= panelY + height;
    }

    @Override
    public boolean method_25402(double mouseX, double mouseY, int button) {
        if (showDiffView && currentDiff != null) {
            int panelWidth = 200;
            int panelHeight = 80;
            int x = this.field_22789 - panelWidth - 10;
            int y = this.field_22790 - panelHeight - 10;
            
            if (isMouseOverPanel((int)mouseX, (int)mouseY, x, y, panelWidth, panelHeight)) {
                showDiffView = false;
                return true;
            }
        }
        
        return super.method_25402(mouseX, mouseY, button);
    }
}
