package io.github.zhengzhengyiyi.util;

import java.util.ArrayList;
import java.util.List;

public class ConfigDiffEngine {
    public static class DiffLine {
        public enum ChangeType { UNCHANGED, ADDED, DELETED, MODIFIED }
        
        public final int oldLineNumber;
        public final int newLineNumber;
        public final String oldContent;
        public final String newContent;
        public final ChangeType type;
        
        public DiffLine(int oldLineNumber, int newLineNumber, String oldContent, String newContent, ChangeType type) {
            this.oldLineNumber = oldLineNumber;
            this.newLineNumber = newLineNumber;
            this.oldContent = oldContent;
            this.newContent = newContent;
            this.type = type;
        }
        
        public String getDisplayContent() {
            return type == ChangeType.DELETED ? oldContent : newContent;
        }
    }

    public static List<DiffLine> computeDiff(String oldText, String newText) {
        String[] oldLines = oldText.split("\n", -1);
        String[] newLines = newText.split("\n", -1);
        
        int[][] dp = new int[oldLines.length + 1][newLines.length + 1];
        
        for (int i = 0; i <= oldLines.length; i++) {
            for (int j = 0; j <= newLines.length; j++) {
                if (i == 0 || j == 0) {
                    dp[i][j] = 0;
                } else if (oldLines[i - 1].equals(newLines[j - 1])) {
                    dp[i][j] = dp[i - 1][j - 1] + 1;
                } else {
                    dp[i][j] = Math.max(dp[i - 1][j], dp[i][j - 1]);
                }
            }
        }
        
        List<DiffLine> diffLines = new ArrayList<>();
        int i = oldLines.length, j = newLines.length;
        
        while (i > 0 || j > 0) {
            if (i > 0 && j > 0 && oldLines[i - 1].equals(newLines[j - 1])) {
                diffLines.add(0, new DiffLine(i, j, oldLines[i - 1], newLines[j - 1], DiffLine.ChangeType.UNCHANGED));
                i--;
                j--;
            } else if (j > 0 && (i == 0 || dp[i][j - 1] >= dp[i - 1][j])) {
                diffLines.add(0, new DiffLine(0, j, "", newLines[j - 1], DiffLine.ChangeType.ADDED));
                j--;
            } else if (i > 0 && (j == 0 || dp[i][j - 1] < dp[i - 1][j])) {
                diffLines.add(0, new DiffLine(i, 0, oldLines[i - 1], "", DiffLine.ChangeType.DELETED));
                i--;
            }
        }
        
        return mergeSimilarLines(diffLines);
    }
    
    private static List<DiffLine> mergeSimilarLines(List<DiffLine> diffLines) {
        List<DiffLine> merged = new ArrayList<>();
        DiffLine lastLine = null;
        
        for (DiffLine currentLine : diffLines) {
            if (lastLine != null && 
                ((lastLine.type == DiffLine.ChangeType.DELETED && currentLine.type == DiffLine.ChangeType.ADDED) ||
                 (lastLine.type == DiffLine.ChangeType.ADDED && currentLine.type == DiffLine.ChangeType.DELETED))) {
                
                double similarity = computeLineSimilarity(lastLine.getDisplayContent(), currentLine.getDisplayContent());
                if (similarity > 0.6) {
                    merged.remove(merged.size() - 1);
                    merged.add(new DiffLine(
                        lastLine.oldLineNumber, 
                        currentLine.newLineNumber,
                        lastLine.oldContent,
                        currentLine.newContent,
                        DiffLine.ChangeType.MODIFIED
                    ));
                    lastLine = null;
                    continue;
                }
            }
            
            merged.add(currentLine);
            lastLine = currentLine;
        }
        
        return merged;
    }
    
    private static double computeLineSimilarity(String line1, String line2) {
        if (line1.isEmpty() || line2.isEmpty()) {
            return 0.0;
        }
        
        String trimmed1 = line1.trim();
        String trimmed2 = line2.trim();
        
        if (trimmed1.equals(trimmed2)) {
            return 1.0;
        }
        
        int maxLength = Math.max(trimmed1.length(), trimmed2.length());
        int editDistance = computeEditDistance(trimmed1, trimmed2);
        
        return 1.0 - (double) editDistance / maxLength;
    }
    
    private static int computeEditDistance(String s1, String s2) {
        int[][] dp = new int[s1.length() + 1][s2.length() + 1];
        
        for (int i = 0; i <= s1.length(); i++) {
            for (int j = 0; j <= s2.length(); j++) {
                if (i == 0) {
                    dp[i][j] = j;
                } else if (j == 0) {
                    dp[i][j] = i;
                } else {
                    int cost = s1.charAt(i - 1) == s2.charAt(j - 1) ? 0 : 1;
                    dp[i][j] = Math.min(Math.min(
                        dp[i - 1][j] + 1,
                        dp[i][j - 1] + 1),
                        dp[i - 1][j - 1] + cost
                    );
                }
            }
        }
        
        return dp[s1.length()][s2.length()];
    }
    
    public static String generateDiffSummary(List<DiffLine> diffLines) {
        int added = 0, deleted = 0, modified = 0, unchanged = 0;
        
        for (DiffLine line : diffLines) {
            switch (line.type) {
                case ADDED -> added++;
                case DELETED -> deleted++;
                case MODIFIED -> modified++;
                case UNCHANGED -> unchanged++;
            }
        }
        
        return String.format("changed: +%d -%d ~%d =%d", added, deleted, modified, unchanged);
    }
    
    public static String applyDiff(String originalText, List<DiffLine> diffLines, boolean acceptAll) {
        List<String> resultLines = new ArrayList<>();
        
        for (DiffLine line : diffLines) {
            if (acceptAll || line.type != DiffLine.ChangeType.DELETED) {
                resultLines.add(line.newContent.isEmpty() ? line.oldContent : line.newContent);
            }
        }
        
        return String.join("\n", resultLines);
    }
}
