/*
 * Decompiled with CFR 0.152.
 */
package de.maxhenkel.audioplayer.audioloader;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import de.maxhenkel.audioplayer.AudioPlayerMod;
import de.maxhenkel.audioplayer.audioloader.Metadata;
import de.maxhenkel.audioplayer.utils.FileUtils;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Consumer;
import javax.annotation.Nullable;

public class FileMetadataManager {
    public static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final int META_VERSION = 1;
    private static final ExecutorService SAVE_EXECUTOR_SERVICE = Executors.newSingleThreadExecutor(runnable -> {
        Thread thread = new Thread(runnable);
        thread.setName("MetadataSaver");
        thread.setDaemon(true);
        return thread;
    });
    private final Path file;
    private Map<UUID, Metadata> metadata;

    public FileMetadataManager(Path file) throws Exception {
        this.file = file;
        this.metadata = new ConcurrentHashMap<UUID, Metadata>();
        this.load();
    }

    private void load() throws Exception {
        if (!Files.exists(this.file, new LinkOption[0])) {
            return;
        }
        ConcurrentHashMap<UUID, Metadata> meta = new ConcurrentHashMap<UUID, Metadata>();
        String content = Files.readString(this.file);
        JsonObject root = JsonParser.parseString((String)content).getAsJsonObject();
        JsonElement metaVersionElement = root.get("version");
        int metaVersion = metaVersionElement == null ? -1 : metaVersionElement.getAsInt();
        JsonObject files = root.getAsJsonObject("files");
        if (files == null) {
            files = new JsonObject();
        }
        for (String key : files.keySet()) {
            UUID uuid;
            try {
                uuid = UUID.fromString(key);
            }
            catch (IllegalArgumentException e) {
                AudioPlayerMod.LOGGER.warn("Invalid UUID in metadata: {}", (Object)key);
                continue;
            }
            meta.put(uuid, Metadata.fromJson(uuid, files.getAsJsonObject(key)));
        }
        this.metadata = meta;
        this.saveSync();
    }

    private void saveSync() {
        try {
            Files.createDirectories(this.file.getParent(), new FileAttribute[0]);
            JsonObject root = new JsonObject();
            root.addProperty("version", (Number)1);
            JsonObject files = new JsonObject();
            for (Map.Entry<UUID, Metadata> entry : this.metadata.entrySet()) {
                JsonObject metaJson = entry.getValue().toJson();
                if (metaJson.isEmpty()) continue;
                files.add(entry.getKey().toString(), (JsonElement)metaJson);
            }
            root.add("files", (JsonElement)files);
            Files.writeString(this.file, (CharSequence)GSON.toJson((JsonElement)root), new OpenOption[0]);
        }
        catch (Exception e) {
            AudioPlayerMod.LOGGER.error("Failed to save metadata", (Throwable)e);
        }
    }

    public void saveAsync() {
        SAVE_EXECUTOR_SERVICE.execute(this::saveSync);
    }

    public Optional<Metadata> getMetadata(UUID uuid) {
        return Optional.ofNullable(this.metadata.get(uuid));
    }

    public Metadata getOrCreateMetadata(UUID uuid) {
        return this.metadata.computeIfAbsent(uuid, Metadata::new);
    }

    public void modifyMetadata(UUID uuid, Consumer<Metadata> metadataConsumer) {
        Metadata metadata = this.metadata.computeIfAbsent(uuid, Metadata::new);
        metadataConsumer.accept(metadata);
        this.saveAsync();
    }

    public void setVolumeOverride(UUID uuid, @Nullable Float volume) {
        this.modifyMetadata(uuid, metadata -> metadata.setVolume(volume));
    }

    public Optional<Float> getVolumeOverride(UUID uuid) {
        return this.getMetadata(uuid).map(Metadata::getVolume);
    }

    @Nullable
    public String getFileName(UUID uuid) {
        return this.getMetadata(uuid).map(Metadata::getFileName).orElse(null);
    }

    public List<Metadata> getByFileName(String fileName, boolean exact) {
        return this.metadata.values().stream().filter(metadata -> FileMetadataManager.matchesName(metadata, fileName, exact)).sorted(Comparator.comparingLong(o -> o.getCreated() == null ? 0L : o.getCreated())).toList();
    }

    private static boolean matchesName(Metadata metadata, String name, boolean exact) {
        String fileName = metadata.getFileName();
        if (fileName == null) {
            return false;
        }
        if (!exact) {
            return fileName.toLowerCase().contains(name.toLowerCase());
        }
        String withoutExt = FileUtils.fileNameWithoutExtension(fileName);
        if (withoutExt.equals(name)) {
            return true;
        }
        return fileName.equals(name);
    }
}

