/*
 * Decompiled with CFR 0.152.
 */
package xyz.harfull.nexuseco.economy;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.bukkit.Bukkit;
import org.bukkit.plugin.java.JavaPlugin;
import xyz.harfull.nexuseco.NexusEco;
import xyz.harfull.nexuseco.data.DataMethods;
import xyz.harfull.nexuseco.data.DatabaseManager;
import xyz.harfull.nexuseco.model.PlayerBalance;
import xyz.harfull.nexuseco.model.TransactionID;
import xyz.harfull.nexuseco.model.TransactionType;
import xyz.harfull.nexuseco.util.NumberFormatting;
import xyz.harfull.nexuseco.util.TaskScheduler;

public class EconomyManager {
    private final JavaPlugin plugin;
    private final NexusEco nexusEco;
    private final DataMethods dataMethods;
    private final DatabaseManager databaseManager;
    private final TaskScheduler taskScheduler;
    private final Cache<UUID, PlayerBalance> balanceCache;
    private final Cache<UUID, Boolean> payToggleCache;
    private final Cache<UUID, List<TransactionID>> transactionCache;
    private final Map<UUID, Double> pendingBalanceWrites = new ConcurrentHashMap<UUID, Double>();
    private final Map<UUID, Boolean> pendingToggleWrites = new ConcurrentHashMap<UUID, Boolean>();
    private Map<UUID, Double> allBalancesCache;
    private long lastAllBalancesCacheUpdate = 0L;
    private static final long ALL_BALANCES_CACHE_EXPIRY = 60000L;
    private ScheduledFuture<?> flushTask;
    private static final long FLUSH_INTERVAL = 10L;

    public EconomyManager(JavaPlugin plugin) {
        this.plugin = plugin;
        this.nexusEco = (NexusEco)plugin;
        this.databaseManager = new DatabaseManager(plugin);
        this.databaseManager.initializeDatabase();
        this.taskScheduler = this.nexusEco.getTaskScheduler();
        this.dataMethods = DatabaseManager.getEconomyMethods();
        this.balanceCache = Caffeine.newBuilder().maximumSize(5000L).expireAfterAccess(5L, TimeUnit.MINUTES).build();
        this.payToggleCache = Caffeine.newBuilder().maximumSize(5000L).expireAfterAccess(10L, TimeUnit.MINUTES).build();
        this.transactionCache = Caffeine.newBuilder().maximumSize(2000L).expireAfterWrite(5L, TimeUnit.MINUTES).build();
        this.preloadOnlinePlayersData();
        this.startFlushTask();
    }

    private void preloadOnlinePlayersData() {
        CompletableFuture.runAsync(() -> {
            for (UUID playerId : Bukkit.getOnlinePlayers().stream().map(p -> p.getUniqueId()).toList()) {
                PlayerBalance balance = this.dataMethods.getBalance(playerId);
                if (balance != null) {
                    this.balanceCache.put(playerId, balance);
                }
                Boolean payToggle = this.dataMethods.getPayToggle(playerId);
                this.payToggleCache.put(playerId, payToggle);
            }
        });
    }

    private void startFlushTask() {
        this.taskScheduler.runTaskTimer(this::flushPendingWrites, 200L, 200L);
    }

    private void flushPendingWrites() {
        if (!this.pendingBalanceWrites.isEmpty()) {
            HashMap<UUID, Double> writesToProcess = new HashMap<UUID, Double>(this.pendingBalanceWrites);
            this.pendingBalanceWrites.clear();
            for (Map.Entry entry : writesToProcess.entrySet()) {
                this.dataMethods.setBalance((UUID)entry.getKey(), (Double)entry.getValue());
            }
        }
        if (!this.pendingToggleWrites.isEmpty()) {
            HashMap<UUID, Boolean> togglesToProcess = new HashMap<UUID, Boolean>(this.pendingToggleWrites);
            this.pendingToggleWrites.clear();
            for (Map.Entry entry : togglesToProcess.entrySet()) {
                this.dataMethods.setPayToggle((UUID)entry.getKey(), (Boolean)entry.getValue());
            }
        }
    }

    public double getBalance(UUID playerId) {
        if (this.pendingBalanceWrites.containsKey(playerId)) {
            return this.pendingBalanceWrites.get(playerId);
        }
        PlayerBalance playerBalance = this.balanceCache.getIfPresent(playerId);
        if (playerBalance == null && (playerBalance = this.dataMethods.getBalance(playerId)) != null) {
            this.balanceCache.put(playerId, playerBalance);
        }
        return playerBalance != null ? playerBalance.getBalance() : 0.0;
    }

    public String getBalanceFormatted(UUID playerId) {
        return NumberFormatting.formatNum(this.getBalance(playerId));
    }

    public void setBalance(UUID playerId, double amount) {
        this.balanceCache.put(playerId, new PlayerBalance(playerId, amount));
        this.pendingBalanceWrites.put(playerId, amount);
        this.invalidateAllBalancesCache();
    }

    public void addBalance(UUID playerId, double amount) {
        PlayerBalance cachedBalance;
        double currentBalance = this.pendingBalanceWrites.containsKey(playerId) ? this.pendingBalanceWrites.get(playerId) : ((cachedBalance = this.balanceCache.getIfPresent(playerId)) != null ? cachedBalance.getBalance() : this.getBalance(playerId));
        double newAmount = currentBalance + amount;
        this.balanceCache.put(playerId, new PlayerBalance(playerId, newAmount));
        this.pendingBalanceWrites.put(playerId, newAmount);
        this.invalidateAllBalancesCache();
    }

    public void subtractBalance(UUID playerId, double amount) {
        PlayerBalance cachedBalance;
        double currentBalance = this.pendingBalanceWrites.containsKey(playerId) ? this.pendingBalanceWrites.get(playerId) : ((cachedBalance = this.balanceCache.getIfPresent(playerId)) != null ? cachedBalance.getBalance() : this.getBalance(playerId));
        double newAmount = currentBalance - amount;
        this.balanceCache.put(playerId, new PlayerBalance(playerId, newAmount));
        this.pendingBalanceWrites.put(playerId, newAmount);
        this.invalidateAllBalancesCache();
    }

    public void resetBalance(UUID playerId) {
        this.balanceCache.put(playerId, new PlayerBalance(playerId, 0.0));
        this.pendingBalanceWrites.put(playerId, 0.0);
        this.invalidateAllBalancesCache();
    }

    public Map<UUID, Double> getAllBalances() {
        long currentTime = System.currentTimeMillis();
        if (this.allBalancesCache == null || currentTime - this.lastAllBalancesCacheUpdate > 60000L) {
            this.allBalancesCache = new HashMap<UUID, Double>(this.dataMethods.getAllBalances());
            for (Map.Entry<UUID, Double> entry : this.pendingBalanceWrites.entrySet()) {
                this.allBalancesCache.put(entry.getKey(), entry.getValue());
            }
            this.lastAllBalancesCacheUpdate = currentTime;
        }
        return this.allBalancesCache;
    }

    private void invalidateAllBalancesCache() {
        this.allBalancesCache = null;
    }

    public void createTransactionID(UUID senderId, UUID receiverId, double amount, TransactionType transactionType) {
        String transactionID = UUID.randomUUID().toString().replace("-", "").substring(0, 10).toUpperCase();
        CompletableFuture.runAsync(() -> {
            this.dataMethods.createTransactionID(senderId, receiverId, amount, transactionType, transactionID);
            this.transactionCache.invalidate(senderId);
            this.transactionCache.invalidate(receiverId);
            this.nexusEco.getHistoryGUI().refreshTransactions(senderId);
            this.nexusEco.getHistoryGUI().refreshTransactions(receiverId);
        });
    }

    public TransactionID getTransactionID(String transactionID) {
        return this.dataMethods.getTransactionID(transactionID);
    }

    public List<TransactionID> getTransactionHistory(UUID playerId) {
        List<TransactionID> transactions = this.transactionCache.getIfPresent(playerId);
        if (transactions == null && (transactions = this.dataMethods.getTransactionHistory(playerId)) != null && !transactions.isEmpty()) {
            this.transactionCache.put(playerId, transactions);
        }
        return transactions;
    }

    public boolean hasBalance(UUID playerId, double amount) {
        return this.getBalance(playerId) >= amount;
    }

    public boolean hasAccount(UUID playerId) {
        return this.getBalance(playerId) >= 0.0 || this.balanceCache.getIfPresent(playerId) != null;
    }

    public void createAccount(UUID playerId) {
        if (!this.hasAccount(playerId)) {
            this.setBalance(playerId, 0.0);
            this.setPayToggle(playerId, true);
        }
    }

    public void setPayToggle(UUID playerId, boolean toggle) {
        this.payToggleCache.put(playerId, toggle);
        this.pendingToggleWrites.put(playerId, toggle);
    }

    public boolean getPayToggle(UUID playerId) {
        if (this.pendingToggleWrites.containsKey(playerId)) {
            return this.pendingToggleWrites.get(playerId);
        }
        Boolean toggle = this.payToggleCache.getIfPresent(playerId);
        if (toggle == null) {
            toggle = this.dataMethods.getPayToggle(playerId);
            this.payToggleCache.put(playerId, toggle);
        }
        return toggle;
    }

    public void close() {
        if (this.flushTask != null) {
            this.flushTask.cancel(false);
        }
        this.flushPendingWrites();
        this.dataMethods.close();
    }

    public void clearCaches() {
        this.flushPendingWrites();
        this.balanceCache.invalidateAll();
        this.payToggleCache.invalidateAll();
        this.transactionCache.invalidateAll();
        this.allBalancesCache = null;
    }

    public void forceDatabaseSync() {
        this.flushPendingWrites();
    }
}

