/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb.internal.connection;

import com.mongodb.MongoConnectionPoolClearedException;
import com.mongodb.MongoException;
import com.mongodb.MongoInterruptedException;
import com.mongodb.MongoTimeoutException;
import com.mongodb.annotations.NotThreadSafe;
import com.mongodb.annotations.ThreadSafe;
import com.mongodb.assertions.Assertions;
import com.mongodb.connection.ClusterId;
import com.mongodb.connection.ConnectionDescription;
import com.mongodb.connection.ConnectionId;
import com.mongodb.connection.ConnectionPoolSettings;
import com.mongodb.connection.ServerDescription;
import com.mongodb.connection.ServerId;
import com.mongodb.event.ConnectionCheckOutFailedEvent;
import com.mongodb.event.ConnectionCheckOutStartedEvent;
import com.mongodb.event.ConnectionCheckedInEvent;
import com.mongodb.event.ConnectionCheckedOutEvent;
import com.mongodb.event.ConnectionClosedEvent;
import com.mongodb.event.ConnectionCreatedEvent;
import com.mongodb.event.ConnectionPoolClearedEvent;
import com.mongodb.event.ConnectionPoolClosedEvent;
import com.mongodb.event.ConnectionPoolCreatedEvent;
import com.mongodb.event.ConnectionPoolListener;
import com.mongodb.event.ConnectionPoolReadyEvent;
import com.mongodb.event.ConnectionReadyEvent;
import com.mongodb.internal.Locks;
import com.mongodb.internal.async.ErrorHandlingResultCallback;
import com.mongodb.internal.async.SingleResultCallback;
import com.mongodb.internal.connection.CommandMessage;
import com.mongodb.internal.connection.ConcurrentPool;
import com.mongodb.internal.connection.Connection;
import com.mongodb.internal.connection.ConnectionGenerationSupplier;
import com.mongodb.internal.connection.ConnectionPool;
import com.mongodb.internal.connection.InternalConnection;
import com.mongodb.internal.connection.InternalConnectionFactory;
import com.mongodb.internal.connection.InternalConnectionPoolSettings;
import com.mongodb.internal.connection.InternalOperationContextFactory;
import com.mongodb.internal.connection.OperationContext;
import com.mongodb.internal.connection.ResponseBuffers;
import com.mongodb.internal.connection.SdamServerDescriptionManager;
import com.mongodb.internal.connection.UsageTrackingInternalConnection;
import com.mongodb.internal.diagnostics.logging.Logger;
import com.mongodb.internal.diagnostics.logging.Loggers;
import com.mongodb.internal.event.EventListenerHelper;
import com.mongodb.internal.event.EventReasonMessageResolver;
import com.mongodb.internal.inject.OptionalProvider;
import com.mongodb.internal.logging.LogMessage;
import com.mongodb.internal.logging.StructuredLogger;
import com.mongodb.internal.thread.DaemonThreadFactory;
import com.mongodb.internal.time.StartTime;
import com.mongodb.internal.time.Timeout;
import com.mongodb.lang.NonNull;
import com.mongodb.lang.Nullable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Deque;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.StampedLock;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.Supplier;
import org.bson.ByteBuf;
import org.bson.codecs.Decoder;
import org.bson.types.ObjectId;

@ThreadSafe
final class DefaultConnectionPool
implements ConnectionPool {
    private static final Logger LOGGER = Loggers.getLogger("connection");
    private static final StructuredLogger STRUCTURED_LOGGER = new StructuredLogger("connection");
    private final ConcurrentPool<UsageTrackingInternalConnection> pool;
    private final ConnectionPoolSettings settings;
    private final InternalOperationContextFactory operationContextFactory;
    private final BackgroundMaintenanceManager backgroundMaintenance;
    private final AsyncWorkManager asyncWorkManager;
    private final ConnectionPoolListener connectionPoolListener;
    private final ServerId serverId;
    private final PinnedStatsManager pinnedStatsManager = new PinnedStatsManager();
    private final ServiceStateManager serviceStateManager = new ServiceStateManager();
    private final ConnectionGenerationSupplier connectionGenerationSupplier;
    private final OpenConcurrencyLimiter openConcurrencyLimiter;
    private final StateAndGeneration stateAndGeneration;
    private final OptionalProvider<SdamServerDescriptionManager> sdamProvider;

    DefaultConnectionPool(ServerId serverId, InternalConnectionFactory internalConnectionFactory, ConnectionPoolSettings connectionPoolSettings, OptionalProvider<SdamServerDescriptionManager> optionalProvider, InternalOperationContextFactory internalOperationContextFactory) {
        this(serverId, internalConnectionFactory, connectionPoolSettings, InternalConnectionPoolSettings.builder().build(), optionalProvider, internalOperationContextFactory);
    }

    DefaultConnectionPool(ServerId serverId, InternalConnectionFactory internalConnectionFactory, ConnectionPoolSettings connectionPoolSettings, InternalConnectionPoolSettings internalConnectionPoolSettings, OptionalProvider<SdamServerDescriptionManager> optionalProvider, InternalOperationContextFactory internalOperationContextFactory) {
        this.serverId = Assertions.notNull("serverId", serverId);
        this.settings = Assertions.notNull("settings", connectionPoolSettings);
        UsageTrackingInternalConnectionItemFactory usageTrackingInternalConnectionItemFactory = new UsageTrackingInternalConnectionItemFactory(internalConnectionFactory);
        this.pool = new ConcurrentPool<UsageTrackingInternalConnection>(DefaultConnectionPool.maxSize(connectionPoolSettings), usageTrackingInternalConnectionItemFactory, String.format("The server at %s is no longer available", serverId.getAddress()));
        this.operationContextFactory = Assertions.assertNotNull(internalOperationContextFactory);
        this.sdamProvider = Assertions.assertNotNull(optionalProvider);
        this.connectionPoolListener = EventListenerHelper.getConnectionPoolListener(connectionPoolSettings);
        this.backgroundMaintenance = new BackgroundMaintenanceManager();
        this.connectionPoolCreated(this.connectionPoolListener, serverId, connectionPoolSettings);
        this.openConcurrencyLimiter = new OpenConcurrencyLimiter(connectionPoolSettings.getMaxConnecting());
        this.asyncWorkManager = new AsyncWorkManager(internalConnectionPoolSettings.isPrestartAsyncWorkManager());
        this.stateAndGeneration = new StateAndGeneration();
        this.connectionGenerationSupplier = new ConnectionGenerationSupplier(){

            @Override
            public int getGeneration() {
                return DefaultConnectionPool.this.stateAndGeneration.generation();
            }

            @Override
            public int getGeneration(@NonNull ObjectId objectId) {
                return DefaultConnectionPool.this.serviceStateManager.getGeneration(objectId);
            }
        };
    }

    @Override
    public InternalConnection get(OperationContext operationContext) {
        StartTime startTime = this.connectionCheckoutStarted(operationContext);
        Timeout timeout = operationContext.getTimeoutContext().startWaitQueueTimeout(startTime);
        try {
            this.stateAndGeneration.throwIfClosedOrPaused();
            PooledConnection pooledConnection = this.getPooledConnection(timeout, startTime);
            if (!pooledConnection.opened()) {
                pooledConnection = this.openConcurrencyLimiter.openOrGetAvailable(operationContext, pooledConnection, timeout, startTime);
            }
            pooledConnection.checkedOutForOperation(operationContext);
            this.connectionCheckedOut(operationContext, pooledConnection, startTime);
            return pooledConnection;
        }
        catch (Exception exception) {
            throw (RuntimeException)this.checkOutFailed(exception, operationContext, startTime);
        }
    }

    @Override
    public void getAsync(OperationContext operationContext, SingleResultCallback<InternalConnection> singleResultCallback) {
        StartTime startTime = this.connectionCheckoutStarted(operationContext);
        Timeout timeout = startTime.timeoutAfterOrInfiniteIfNegative(this.settings.getMaxWaitTime(TimeUnit.NANOSECONDS), TimeUnit.NANOSECONDS);
        SingleResultCallback<PooledConnection> singleResultCallback2 = (pooledConnection, throwable) -> {
            SingleResultCallback<PooledConnection> singleResultCallback2 = ErrorHandlingResultCallback.errorHandlingCallback(singleResultCallback, LOGGER);
            if (throwable == null) {
                Assertions.assertNotNull(pooledConnection).checkedOutForOperation(operationContext);
                this.connectionCheckedOut(operationContext, (PooledConnection)pooledConnection, startTime);
                singleResultCallback2.onResult((PooledConnection)pooledConnection, null);
            } else {
                singleResultCallback2.onResult(null, this.checkOutFailed(throwable, operationContext, startTime));
            }
        };
        try {
            this.stateAndGeneration.throwIfClosedOrPaused();
        }
        catch (Exception exception) {
            singleResultCallback2.onResult(null, exception);
            return;
        }
        this.asyncWorkManager.enqueue(new Task(timeout, startTime, runtimeException -> {
            if (runtimeException != null) {
                singleResultCallback2.onResult((PooledConnection)null, (Throwable)runtimeException);
            } else {
                PooledConnection pooledConnection;
                try {
                    pooledConnection = this.getPooledConnection(timeout, startTime);
                }
                catch (Exception exception) {
                    singleResultCallback2.onResult(null, exception);
                    return;
                }
                if (pooledConnection.opened()) {
                    singleResultCallback2.onResult(pooledConnection, null);
                } else {
                    this.openConcurrencyLimiter.openWithConcurrencyLimitAsync(operationContext, pooledConnection, timeout, startTime, singleResultCallback2);
                }
            }
        }));
    }

    private Throwable checkOutFailed(Throwable throwable, OperationContext operationContext, StartTime startTime) {
        ConnectionCheckOutFailedEvent.Reason reason;
        Throwable throwable2 = throwable;
        if (throwable instanceof MongoTimeoutException) {
            reason = ConnectionCheckOutFailedEvent.Reason.TIMEOUT;
        } else if (throwable instanceof MongoOpenConnectionInternalException) {
            reason = ConnectionCheckOutFailedEvent.Reason.CONNECTION_ERROR;
            throwable2 = throwable.getCause();
        } else {
            reason = throwable instanceof MongoConnectionPoolClearedException ? ConnectionCheckOutFailedEvent.Reason.CONNECTION_ERROR : (ConcurrentPool.isPoolClosedException(throwable) ? ConnectionCheckOutFailedEvent.Reason.POOL_CLOSED : ConnectionCheckOutFailedEvent.Reason.UNKNOWN);
        }
        Duration duration = startTime.elapsed();
        ClusterId clusterId = this.serverId.getClusterId();
        if (DefaultConnectionPool.requiresLogging(clusterId)) {
            String string = "Checkout failed for connection to {}:{}. Reason: {}.[ Error: {}.] Duration: {} ms";
            List<LogMessage.Entry> list = this.createBasicEntries();
            list.add(new LogMessage.Entry(LogMessage.Entry.Name.REASON_DESCRIPTION, EventReasonMessageResolver.getMessage(reason)));
            list.add(new LogMessage.Entry(LogMessage.Entry.Name.ERROR_DESCRIPTION, reason == ConnectionCheckOutFailedEvent.Reason.CONNECTION_ERROR ? throwable2.toString() : null));
            list.add(new LogMessage.Entry(LogMessage.Entry.Name.DURATION_MS, duration.toMillis()));
            DefaultConnectionPool.logMessage("Connection checkout failed", clusterId, string, list);
        }
        this.connectionPoolListener.connectionCheckOutFailed(new ConnectionCheckOutFailedEvent(this.serverId, operationContext.getId(), reason, duration.toNanos()));
        return throwable2;
    }

    @Override
    public void invalidate(@Nullable Throwable throwable) {
        Assertions.assertFalse(this.isLoadBalanced());
        if (this.stateAndGeneration.pauseAndIncrementGeneration(throwable)) {
            this.openConcurrencyLimiter.signalClosedOrPaused();
        }
    }

    @Override
    public void ready() {
        this.stateAndGeneration.ready();
    }

    @Override
    public void invalidate(ObjectId objectId, int n) {
        Assertions.assertTrue(this.isLoadBalanced());
        if (n == -1) {
            return;
        }
        if (this.serviceStateManager.incrementGeneration(objectId, n)) {
            ClusterId clusterId = this.serverId.getClusterId();
            if (DefaultConnectionPool.requiresLogging(clusterId)) {
                String string = "Connection pool for {}:{} cleared for serviceId {}";
                List<LogMessage.Entry> list = this.createBasicEntries();
                list.add(new LogMessage.Entry(LogMessage.Entry.Name.SERVICE_ID, objectId.toHexString()));
                DefaultConnectionPool.logMessage("Connection pool cleared", clusterId, string, list);
            }
            this.connectionPoolListener.connectionPoolCleared(new ConnectionPoolClearedEvent(this.serverId, objectId));
        }
    }

    @Override
    public void close() {
        if (this.stateAndGeneration.close()) {
            this.pool.close();
            this.backgroundMaintenance.close();
            this.asyncWorkManager.close();
            this.openConcurrencyLimiter.signalClosedOrPaused();
            this.logEventMessage("Connection pool closed", "Connection pool closed for {}:{}");
            this.connectionPoolListener.connectionPoolClosed(new ConnectionPoolClosedEvent(this.serverId));
        }
    }

    @Override
    public int getGeneration() {
        return this.stateAndGeneration.generation();
    }

    private PooledConnection getPooledConnection(Timeout timeout, StartTime startTime) throws MongoTimeoutException {
        try {
            UsageTrackingInternalConnection usageTrackingInternalConnection = timeout.call(TimeUnit.NANOSECONDS, () -> this.pool.get(-1L, TimeUnit.NANOSECONDS), l -> this.pool.get(l, TimeUnit.NANOSECONDS), () -> this.pool.get(0L, TimeUnit.NANOSECONDS));
            while (this.shouldPrune(usageTrackingInternalConnection)) {
                this.pool.release(usageTrackingInternalConnection, true);
                usageTrackingInternalConnection = timeout.call(TimeUnit.NANOSECONDS, () -> this.pool.get(-1L, TimeUnit.NANOSECONDS), l -> this.pool.get(l, TimeUnit.NANOSECONDS), () -> this.pool.get(0L, TimeUnit.NANOSECONDS));
            }
            return new PooledConnection(usageTrackingInternalConnection);
        }
        catch (MongoTimeoutException mongoTimeoutException) {
            throw this.createTimeoutException(startTime);
        }
    }

    @Nullable
    private PooledConnection getPooledConnectionImmediate() {
        UsageTrackingInternalConnection usageTrackingInternalConnection = this.pool.getImmediate();
        while (usageTrackingInternalConnection != null && this.shouldPrune(usageTrackingInternalConnection)) {
            this.pool.release(usageTrackingInternalConnection, true);
            usageTrackingInternalConnection = this.pool.getImmediate();
        }
        return usageTrackingInternalConnection == null ? null : new PooledConnection(usageTrackingInternalConnection);
    }

    private MongoTimeoutException createTimeoutException(StartTime startTime) {
        int n;
        long l = startTime.elapsed().toMillis();
        int n2 = this.pinnedStatsManager.getNumPinnedToCursor();
        int n3 = this.pinnedStatsManager.getNumPinnedToTransaction();
        if (n2 == 0 && n3 == 0) {
            return new MongoTimeoutException(String.format("Timed out after %d ms while waiting for a connection to server %s.", l, this.serverId.getAddress()));
        }
        int n4 = this.pool.getMaxSize();
        int n5 = this.pool.getInUseCount();
        if (n5 == 0) {
            n5 = Math.min(n2 + n3, n4);
        }
        Assertions.assertTrue((n = n5 - (n2 = Math.min(n2, n5)) - (n3 = Math.min(n3, n5 - n2))) >= 0);
        Assertions.assertTrue(n2 + n3 + n <= n4);
        return new MongoTimeoutException(String.format("Timed out after %d ms while waiting for a connection to server %s. Details: maxPoolSize: %s, connections in use by cursors: %d, connections in use by transactions: %d, connections in use by other operations: %d", l, this.serverId.getAddress(), ConcurrentPool.sizeToString(n4), n2, n3, n));
    }

    ConcurrentPool<UsageTrackingInternalConnection> getPool() {
        return this.pool;
    }

    void doMaintenance() {
        block3: {
            Predicate<Exception> predicate = exception -> exception instanceof MongoInterruptedException || exception instanceof MongoTimeoutException || exception instanceof MongoConnectionPoolClearedException || ConcurrentPool.isPoolClosedException(exception);
            try {
                this.pool.prune();
                if (this.shouldEnsureMinSize()) {
                    this.pool.ensureMinSize(this.settings.getMinSize(), usageTrackingInternalConnection -> {
                        try {
                            OperationContext operationContext = this.operationContextFactory.createMaintenanceContext();
                            this.openConcurrencyLimiter.openImmediatelyAndTryHandOverOrRelease(operationContext, new PooledConnection((UsageTrackingInternalConnection)usageTrackingInternalConnection));
                        }
                        catch (MongoException | MongoOpenConnectionInternalException runtimeException) {
                            RuntimeException runtimeException2 = runtimeException instanceof MongoOpenConnectionInternalException ? (RuntimeException)runtimeException.getCause() : runtimeException;
                            try {
                                this.sdamProvider.optional().ifPresent(sdamServerDescriptionManager -> {
                                    if (!predicate.test(runtimeException2)) {
                                        sdamServerDescriptionManager.handleExceptionBeforeHandshake(SdamServerDescriptionManager.SdamIssue.specific(runtimeException2, sdamServerDescriptionManager.context((InternalConnection)usageTrackingInternalConnection)));
                                    }
                                });
                            }
                            catch (Exception exception) {
                                runtimeException2.addSuppressed(exception);
                            }
                            throw runtimeException2;
                        }
                    });
                }
            }
            catch (Exception exception2) {
                if (predicate.test(exception2)) break block3;
                LOGGER.warn("Exception thrown during connection pool background maintenance task", exception2);
                throw exception2;
            }
        }
    }

    private boolean shouldEnsureMinSize() {
        return this.settings.getMinSize() > 0;
    }

    private boolean shouldPrune(UsageTrackingInternalConnection usageTrackingInternalConnection) {
        return this.fromPreviousGeneration(usageTrackingInternalConnection) || this.pastMaxLifeTime(usageTrackingInternalConnection) || this.pastMaxIdleTime(usageTrackingInternalConnection);
    }

    private boolean pastMaxIdleTime(UsageTrackingInternalConnection usageTrackingInternalConnection) {
        return this.expired(usageTrackingInternalConnection.getLastUsedAt(), System.currentTimeMillis(), this.settings.getMaxConnectionIdleTime(TimeUnit.MILLISECONDS));
    }

    private boolean pastMaxLifeTime(UsageTrackingInternalConnection usageTrackingInternalConnection) {
        return this.expired(usageTrackingInternalConnection.getOpenedAt(), System.currentTimeMillis(), this.settings.getMaxConnectionLifeTime(TimeUnit.MILLISECONDS));
    }

    private boolean fromPreviousGeneration(UsageTrackingInternalConnection usageTrackingInternalConnection) {
        int n = usageTrackingInternalConnection.getGeneration();
        if (n == -1) {
            return false;
        }
        ObjectId objectId = usageTrackingInternalConnection.getDescription().getServiceId();
        if (objectId != null) {
            return this.serviceStateManager.getGeneration(objectId) > n;
        }
        return this.stateAndGeneration.generation() > n;
    }

    private boolean expired(long l, long l2, long l3) {
        return l3 != 0L && l2 - l > l3;
    }

    private void connectionPoolCreated(ConnectionPoolListener connectionPoolListener, ServerId serverId, ConnectionPoolSettings connectionPoolSettings) {
        ClusterId clusterId = serverId.getClusterId();
        if (DefaultConnectionPool.requiresLogging(clusterId)) {
            String string = "Connection pool created for {}:{} using options maxIdleTimeMS={}, minPoolSize={}, maxPoolSize={}, maxConnecting={}, waitQueueTimeoutMS={}";
            List<LogMessage.Entry> list = this.createBasicEntries();
            list.add(new LogMessage.Entry(LogMessage.Entry.Name.MAX_IDLE_TIME_MS, connectionPoolSettings.getMaxConnectionIdleTime(TimeUnit.MILLISECONDS)));
            list.add(new LogMessage.Entry(LogMessage.Entry.Name.MIN_POOL_SIZE, connectionPoolSettings.getMinSize()));
            list.add(new LogMessage.Entry(LogMessage.Entry.Name.MAX_POOL_SIZE, connectionPoolSettings.getMaxSize()));
            list.add(new LogMessage.Entry(LogMessage.Entry.Name.MAX_CONNECTING, connectionPoolSettings.getMaxConnecting()));
            list.add(new LogMessage.Entry(LogMessage.Entry.Name.WAIT_QUEUE_TIMEOUT_MS, connectionPoolSettings.getMaxWaitTime(TimeUnit.MILLISECONDS)));
            DefaultConnectionPool.logMessage("Connection pool created", clusterId, string, list);
        }
        connectionPoolListener.connectionPoolCreated(new ConnectionPoolCreatedEvent(serverId, connectionPoolSettings));
    }

    private StartTime connectionCreated(ConnectionPoolListener connectionPoolListener, ConnectionId connectionId) {
        StartTime startTime = StartTime.now();
        this.logEventMessage("Connection created", "Connection created: address={}:{}, driver-generated ID={}", connectionId.getLocalValue());
        connectionPoolListener.connectionCreated(new ConnectionCreatedEvent(connectionId));
        return startTime;
    }

    private void connectionClosed(ConnectionPoolListener connectionPoolListener, ConnectionId connectionId, ConnectionClosedEvent.Reason reason) {
        ClusterId clusterId = this.serverId.getClusterId();
        if (DefaultConnectionPool.requiresLogging(clusterId)) {
            String string = "There was a socket exception raised by this connection";
            List<LogMessage.Entry> list = this.createBasicEntries();
            list.add(new LogMessage.Entry(LogMessage.Entry.Name.DRIVER_CONNECTION_ID, connectionId.getLocalValue()));
            list.add(new LogMessage.Entry(LogMessage.Entry.Name.REASON_DESCRIPTION, EventReasonMessageResolver.getMessage(reason)));
            list.add(new LogMessage.Entry(LogMessage.Entry.Name.ERROR_DESCRIPTION, reason == ConnectionClosedEvent.Reason.ERROR ? string : null));
            DefaultConnectionPool.logMessage("Connection closed", clusterId, "Connection closed: address={}:{}, driver-generated ID={}. Reason: {}.[ Error: {}]", list);
        }
        connectionPoolListener.connectionClosed(new ConnectionClosedEvent(connectionId, reason));
    }

    private void connectionCheckedOut(OperationContext operationContext, PooledConnection pooledConnection, StartTime startTime) {
        Duration duration = startTime.elapsed();
        ConnectionId connectionId = this.getId(pooledConnection);
        ClusterId clusterId = this.serverId.getClusterId();
        if (DefaultConnectionPool.requiresLogging(clusterId)) {
            List<LogMessage.Entry> list = this.createBasicEntries();
            list.add(new LogMessage.Entry(LogMessage.Entry.Name.DRIVER_CONNECTION_ID, connectionId.getLocalValue()));
            list.add(new LogMessage.Entry(LogMessage.Entry.Name.DURATION_MS, duration.toMillis()));
            DefaultConnectionPool.logMessage("Connection checked out", clusterId, "Connection checked out: address={}:{}, driver-generated ID={}, duration={} ms", list);
        }
        this.connectionPoolListener.connectionCheckedOut(new ConnectionCheckedOutEvent(connectionId, operationContext.getId(), duration.toNanos()));
    }

    private StartTime connectionCheckoutStarted(OperationContext operationContext) {
        StartTime startTime = StartTime.now();
        this.logEventMessage("Connection checkout started", "Checkout started for connection to {}:{}");
        this.connectionPoolListener.connectionCheckOutStarted(new ConnectionCheckOutStartedEvent(this.serverId, operationContext.getId()));
        return startTime;
    }

    private ConnectionId getId(InternalConnection internalConnection) {
        return internalConnection.getDescription().getConnectionId();
    }

    private boolean isLoadBalanced() {
        return !this.sdamProvider.optional().isPresent();
    }

    private static int maxSize(ConnectionPoolSettings connectionPoolSettings) {
        return connectionPoolSettings.getMaxSize() == 0 ? Integer.MAX_VALUE : connectionPoolSettings.getMaxSize();
    }

    private void logEventMessage(String string, String string2, long l) {
        ClusterId clusterId = this.serverId.getClusterId();
        if (DefaultConnectionPool.requiresLogging(clusterId)) {
            List<LogMessage.Entry> list = this.createBasicEntries();
            list.add(new LogMessage.Entry(LogMessage.Entry.Name.DRIVER_CONNECTION_ID, l));
            DefaultConnectionPool.logMessage(string, clusterId, string2, list);
        }
    }

    private void logEventMessage(String string, String string2) {
        ClusterId clusterId = this.serverId.getClusterId();
        if (DefaultConnectionPool.requiresLogging(clusterId)) {
            List<LogMessage.Entry> list = this.createBasicEntries();
            DefaultConnectionPool.logMessage(string, clusterId, string2, list);
        }
    }

    private List<LogMessage.Entry> createBasicEntries() {
        ArrayList<LogMessage.Entry> arrayList = new ArrayList<LogMessage.Entry>();
        arrayList.add(new LogMessage.Entry(LogMessage.Entry.Name.SERVER_HOST, this.serverId.getAddress().getHost()));
        arrayList.add(new LogMessage.Entry(LogMessage.Entry.Name.SERVER_PORT, this.serverId.getAddress().getPort()));
        return arrayList;
    }

    private static void logMessage(String string, ClusterId clusterId, String string2, List<LogMessage.Entry> list) {
        STRUCTURED_LOGGER.log(new LogMessage(LogMessage.Component.CONNECTION, LogMessage.Level.DEBUG, string, clusterId, list, string2));
    }

    private static boolean requiresLogging(ClusterId clusterId) {
        return STRUCTURED_LOGGER.isRequired(LogMessage.Level.DEBUG, clusterId);
    }

    @NotThreadSafe
    private final class BackgroundMaintenanceManager
    implements AutoCloseable {
        @Nullable
        private final ScheduledExecutorService maintainer;
        @Nullable
        private Future<?> cancellationHandle;
        private boolean initialStart;

        private BackgroundMaintenanceManager() {
            this.maintainer = DefaultConnectionPool.this.settings.getMaintenanceInitialDelay(TimeUnit.NANOSECONDS) < Long.MAX_VALUE ? Executors.newSingleThreadScheduledExecutor(new DaemonThreadFactory("MaintenanceTimer")) : null;
            this.cancellationHandle = null;
            this.initialStart = true;
        }

        void start() {
            if (this.maintainer != null) {
                Assertions.assertTrue(this.cancellationHandle == null);
                this.cancellationHandle = this.ignoreRejectedExectution(() -> this.maintainer.scheduleAtFixedRate(DefaultConnectionPool.this::doMaintenance, this.initialStart ? DefaultConnectionPool.this.settings.getMaintenanceInitialDelay(TimeUnit.MILLISECONDS) : 0L, DefaultConnectionPool.this.settings.getMaintenanceFrequency(TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS));
                this.initialStart = false;
            }
        }

        void runOnceAndStop() {
            if (this.maintainer != null) {
                if (this.cancellationHandle != null) {
                    this.cancellationHandle.cancel(false);
                    this.cancellationHandle = null;
                }
                this.ignoreRejectedExectution(() -> this.maintainer.execute(DefaultConnectionPool.this::doMaintenance));
            }
        }

        @Override
        public void close() {
            if (this.maintainer != null) {
                this.maintainer.shutdownNow();
            }
        }

        private void ignoreRejectedExectution(Runnable runnable) {
            this.ignoreRejectedExectution(() -> {
                runnable.run();
                return null;
            });
        }

        @Nullable
        private <T> T ignoreRejectedExectution(Supplier<T> supplier) {
            try {
                return supplier.get();
            }
            catch (RejectedExecutionException rejectedExecutionException) {
                return null;
            }
        }
    }

    private class PooledConnection
    implements InternalConnection {
        private final UsageTrackingInternalConnection wrapped;
        private final AtomicBoolean isClosed = new AtomicBoolean();
        private Connection.PinningMode pinningMode;
        private long operationId;

        PooledConnection(UsageTrackingInternalConnection usageTrackingInternalConnection) {
            this.wrapped = Assertions.notNull("wrapped", usageTrackingInternalConnection);
        }

        @Override
        public int getGeneration() {
            return this.wrapped.getGeneration();
        }

        public void checkedOutForOperation(OperationContext operationContext) {
            this.operationId = operationContext.getId();
        }

        @Override
        public void open(OperationContext operationContext) {
            StartTime startTime;
            Assertions.assertFalse(this.isClosed.get());
            try {
                startTime = DefaultConnectionPool.this.connectionCreated(DefaultConnectionPool.this.connectionPoolListener, this.wrapped.getDescription().getConnectionId());
                this.wrapped.open(operationContext);
            }
            catch (Exception exception) {
                this.closeAndHandleOpenFailure();
                throw new MongoOpenConnectionInternalException(exception);
            }
            this.handleOpenSuccess(startTime);
        }

        @Override
        public void openAsync(OperationContext operationContext, SingleResultCallback<Void> singleResultCallback) {
            Assertions.assertFalse(this.isClosed.get());
            StartTime startTime = DefaultConnectionPool.this.connectionCreated(DefaultConnectionPool.this.connectionPoolListener, this.wrapped.getDescription().getConnectionId());
            this.wrapped.openAsync(operationContext, (void_, throwable) -> {
                if (throwable != null) {
                    this.closeAndHandleOpenFailure();
                    singleResultCallback.onResult(null, new MongoOpenConnectionInternalException(throwable));
                } else {
                    this.handleOpenSuccess(startTime);
                    singleResultCallback.onResult((Void)void_, null);
                }
            });
        }

        @Override
        public void close() {
            if (!this.isClosed.getAndSet(true)) {
                this.unmarkAsPinned();
                this.connectionCheckedIn();
                if (this.wrapped.isClosed() || DefaultConnectionPool.this.shouldPrune(this.wrapped)) {
                    DefaultConnectionPool.this.pool.release(this.wrapped, true);
                } else {
                    DefaultConnectionPool.this.openConcurrencyLimiter.tryHandOverOrRelease(this.wrapped);
                }
            }
        }

        private void connectionCheckedIn() {
            ConnectionId connectionId = DefaultConnectionPool.this.getId(this.wrapped);
            DefaultConnectionPool.this.logEventMessage("Connection checked in", "Connection checked in: address={}:{}, driver-generated ID={}", connectionId.getLocalValue());
            DefaultConnectionPool.this.connectionPoolListener.connectionCheckedIn(new ConnectionCheckedInEvent(connectionId, this.operationId));
        }

        void release() {
            if (!this.isClosed.getAndSet(true)) {
                DefaultConnectionPool.this.pool.release(this.wrapped);
            }
        }

        void closeSilently() {
            if (!this.isClosed.getAndSet(true)) {
                this.wrapped.setCloseSilently();
                DefaultConnectionPool.this.pool.release(this.wrapped, true);
            }
        }

        private void closeAndHandleOpenFailure() {
            if (!this.isClosed.getAndSet(true)) {
                if (this.wrapped.getDescription().getServiceId() != null) {
                    DefaultConnectionPool.this.invalidate(Assertions.assertNotNull(this.wrapped.getDescription().getServiceId()), this.wrapped.getGeneration());
                }
                DefaultConnectionPool.this.pool.release(this.wrapped, true);
            }
        }

        private void handleOpenSuccess(StartTime startTime) {
            Duration duration = startTime.elapsed();
            ConnectionId connectionId = DefaultConnectionPool.this.getId(this);
            ClusterId clusterId = DefaultConnectionPool.this.serverId.getClusterId();
            if (DefaultConnectionPool.requiresLogging(clusterId)) {
                List list = DefaultConnectionPool.this.createBasicEntries();
                list.add(new LogMessage.Entry(LogMessage.Entry.Name.DRIVER_CONNECTION_ID, connectionId.getLocalValue()));
                list.add(new LogMessage.Entry(LogMessage.Entry.Name.DURATION_MS, duration.toMillis()));
                DefaultConnectionPool.logMessage("Connection ready", clusterId, "Connection ready: address={}:{}, driver-generated ID={}, established in={} ms", list);
            }
            DefaultConnectionPool.this.connectionPoolListener.connectionReady(new ConnectionReadyEvent(connectionId, duration.toNanos()));
        }

        @Override
        public boolean opened() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.opened();
        }

        @Override
        public boolean isClosed() {
            return this.isClosed.get() || this.wrapped.isClosed();
        }

        @Override
        public ByteBuf getBuffer(int n) {
            return this.wrapped.getBuffer(n);
        }

        @Override
        public void sendMessage(List<ByteBuf> list, int n, OperationContext operationContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.sendMessage(list, n, operationContext);
        }

        @Override
        public <T> T sendAndReceive(CommandMessage commandMessage, Decoder<T> decoder, OperationContext operationContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.sendAndReceive(commandMessage, decoder, operationContext);
        }

        @Override
        public <T> void send(CommandMessage commandMessage, Decoder<T> decoder, OperationContext operationContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.send(commandMessage, decoder, operationContext);
        }

        @Override
        public <T> T receive(Decoder<T> decoder, OperationContext operationContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.receive(decoder, operationContext);
        }

        @Override
        public boolean hasMoreToCome() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.hasMoreToCome();
        }

        @Override
        public <T> void sendAndReceiveAsync(CommandMessage commandMessage, Decoder<T> decoder, OperationContext operationContext, SingleResultCallback<T> singleResultCallback) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.sendAndReceiveAsync(commandMessage, decoder, operationContext, singleResultCallback);
        }

        @Override
        public ResponseBuffers receiveMessage(int n, OperationContext operationContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.receiveMessage(n, operationContext);
        }

        @Override
        public void sendMessageAsync(List<ByteBuf> list, int n, OperationContext operationContext, SingleResultCallback<Void> singleResultCallback) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.sendMessageAsync(list, n, operationContext, (void_, throwable) -> singleResultCallback.onResult(null, throwable));
        }

        @Override
        public void receiveMessageAsync(int n, OperationContext operationContext, SingleResultCallback<ResponseBuffers> singleResultCallback) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.receiveMessageAsync(n, operationContext, singleResultCallback);
        }

        @Override
        public void markAsPinned(Connection.PinningMode pinningMode) {
            Assertions.assertNotNull(pinningMode);
            if (this.pinningMode == null) {
                this.pinningMode = pinningMode;
                DefaultConnectionPool.this.pinnedStatsManager.increment(pinningMode);
            }
        }

        void unmarkAsPinned() {
            if (this.pinningMode != null) {
                DefaultConnectionPool.this.pinnedStatsManager.decrement(this.pinningMode);
            }
        }

        @Override
        public ConnectionDescription getDescription() {
            return this.wrapped.getDescription();
        }

        @Override
        public ServerDescription getInitialServerDescription() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.getInitialServerDescription();
        }
    }

    private static final class PinnedStatsManager {
        private final LongAdder numPinnedToCursor = new LongAdder();
        private final LongAdder numPinnedToTransaction = new LongAdder();

        private PinnedStatsManager() {
        }

        void increment(Connection.PinningMode pinningMode) {
            switch (pinningMode) {
                case CURSOR: {
                    this.numPinnedToCursor.increment();
                    break;
                }
                case TRANSACTION: {
                    this.numPinnedToTransaction.increment();
                    break;
                }
                default: {
                    Assertions.fail();
                }
            }
        }

        void decrement(Connection.PinningMode pinningMode) {
            switch (pinningMode) {
                case CURSOR: {
                    this.numPinnedToCursor.decrement();
                    break;
                }
                case TRANSACTION: {
                    this.numPinnedToTransaction.decrement();
                    break;
                }
                default: {
                    Assertions.fail();
                }
            }
        }

        int getNumPinnedToCursor() {
            return this.numPinnedToCursor.intValue();
        }

        int getNumPinnedToTransaction() {
            return this.numPinnedToTransaction.intValue();
        }
    }

    @ThreadSafe
    private final class OpenConcurrencyLimiter {
        private final ReentrantLock lock = new ReentrantLock(false);
        private final Condition permitAvailableOrHandedOverOrClosedOrPausedCondition = this.lock.newCondition();
        private final int maxPermits;
        private int permits;
        private final Deque<MutableReference<PooledConnection>> desiredConnectionSlots;

        OpenConcurrencyLimiter(int n) {
            this.permits = this.maxPermits = n;
            this.desiredConnectionSlots = new LinkedList<MutableReference<PooledConnection>>();
        }

        PooledConnection openOrGetAvailable(OperationContext operationContext, PooledConnection pooledConnection, Timeout timeout, StartTime startTime) throws MongoTimeoutException {
            PooledConnection pooledConnection2 = this.openWithConcurrencyLimit(operationContext, pooledConnection, OpenWithConcurrencyLimitMode.TRY_GET_AVAILABLE, timeout, startTime);
            return Assertions.assertNotNull(pooledConnection2);
        }

        void openImmediatelyAndTryHandOverOrRelease(OperationContext operationContext, PooledConnection pooledConnection) throws MongoTimeoutException {
            StartTime startTime = StartTime.now();
            Timeout timeout = startTime.asTimeout();
            Assertions.assertNull(this.openWithConcurrencyLimit(operationContext, pooledConnection, OpenWithConcurrencyLimitMode.TRY_HAND_OVER_OR_RELEASE, timeout, startTime));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Nullable
        private PooledConnection openWithConcurrencyLimit(OperationContext operationContext, PooledConnection pooledConnection, OpenWithConcurrencyLimitMode openWithConcurrencyLimitMode, Timeout timeout, StartTime startTime) throws MongoTimeoutException {
            PooledConnection pooledConnection2;
            try {
                pooledConnection2 = this.acquirePermitOrGetAvailableOpenedConnection(openWithConcurrencyLimitMode == OpenWithConcurrencyLimitMode.TRY_GET_AVAILABLE, timeout, startTime);
            }
            catch (Exception exception) {
                pooledConnection.closeSilently();
                throw exception;
            }
            if (pooledConnection2 != null) {
                pooledConnection.closeSilently();
                return pooledConnection2;
            }
            try {
                pooledConnection.open(operationContext);
                if (openWithConcurrencyLimitMode == OpenWithConcurrencyLimitMode.TRY_HAND_OVER_OR_RELEASE) {
                    this.tryHandOverOrRelease(pooledConnection.wrapped);
                    PooledConnection pooledConnection3 = null;
                    return pooledConnection3;
                }
                PooledConnection pooledConnection4 = pooledConnection;
                return pooledConnection4;
            }
            finally {
                this.releasePermit();
            }
        }

        void openWithConcurrencyLimitAsync(OperationContext operationContext, PooledConnection pooledConnection, Timeout timeout, StartTime startTime, SingleResultCallback<PooledConnection> singleResultCallback) {
            PooledConnection pooledConnection2;
            try {
                pooledConnection2 = this.acquirePermitOrGetAvailableOpenedConnection(true, timeout, startTime);
            }
            catch (Exception exception) {
                pooledConnection.closeSilently();
                singleResultCallback.onResult(null, exception);
                return;
            }
            if (pooledConnection2 != null) {
                pooledConnection.closeSilently();
                singleResultCallback.onResult(pooledConnection2, null);
            } else {
                pooledConnection.openAsync(operationContext, (void_, throwable) -> {
                    this.releasePermit();
                    if (throwable != null) {
                        singleResultCallback.onResult(null, throwable);
                    } else {
                        singleResultCallback.onResult(pooledConnection, null);
                    }
                });
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Nullable
        private PooledConnection acquirePermitOrGetAvailableOpenedConnection(boolean bl, Timeout timeout, StartTime startTime) throws MongoTimeoutException, MongoInterruptedException {
            PooledConnection pooledConnection = null;
            boolean bl2 = false;
            Locks.lockInterruptibly(this.lock);
            try {
                if (bl) {
                    pooledConnection = DefaultConnectionPool.this.getPooledConnectionImmediate();
                    if (pooledConnection != null) {
                        PooledConnection pooledConnection2 = pooledConnection;
                        return pooledConnection2;
                    }
                    this.expressDesireToGetAvailableConnection();
                    bl2 = true;
                }
                while (true) {
                    pooledConnection = bl ? this.tryGetAvailableConnection() : null;
                    if (!(this.permits == 0 & !DefaultConnectionPool.this.stateAndGeneration.throwIfClosedOrPaused() & pooledConnection == null)) break;
                    Timeout.onExistsAndExpired(timeout, () -> {
                        throw DefaultConnectionPool.this.createTimeoutException(startTime);
                    });
                    timeout.awaitOn(this.permitAvailableOrHandedOverOrClosedOrPausedCondition, () -> "acquiring permit or getting available opened connection");
                }
                if (pooledConnection == null) {
                    Assertions.assertTrue(this.permits > 0);
                    --this.permits;
                }
                PooledConnection pooledConnection3 = pooledConnection;
                return pooledConnection3;
            }
            finally {
                try {
                    if (bl2 && pooledConnection == null) {
                        this.giveUpOnTryingToGetAvailableConnection();
                    }
                }
                finally {
                    this.lock.unlock();
                }
            }
        }

        private void releasePermit() {
            Locks.withLock((Lock)this.lock, () -> {
                Assertions.assertTrue(this.permits < this.maxPermits);
                ++this.permits;
                this.permitAvailableOrHandedOverOrClosedOrPausedCondition.signal();
            });
        }

        private void expressDesireToGetAvailableConnection() {
            this.desiredConnectionSlots.addLast(new MutableReference());
        }

        @Nullable
        private PooledConnection tryGetAvailableConnection() {
            Assertions.assertFalse(this.desiredConnectionSlots.isEmpty());
            PooledConnection pooledConnection = (PooledConnection)((MutableReference)this.desiredConnectionSlots.peekFirst()).reference;
            if (pooledConnection != null) {
                this.desiredConnectionSlots.removeFirst();
                Assertions.assertTrue(pooledConnection.opened());
            }
            return pooledConnection;
        }

        private void giveUpOnTryingToGetAvailableConnection() {
            Assertions.assertFalse(this.desiredConnectionSlots.isEmpty());
            PooledConnection pooledConnection = (PooledConnection)((MutableReference)this.desiredConnectionSlots.removeLast()).reference;
            if (pooledConnection != null) {
                pooledConnection.release();
            }
        }

        void tryHandOverOrRelease(UsageTrackingInternalConnection usageTrackingInternalConnection) {
            boolean bl = Locks.withLock((Lock)this.lock, () -> {
                for (MutableReference<PooledConnection> mutableReference : this.desiredConnectionSlots) {
                    if (((MutableReference)mutableReference).reference != null) continue;
                    ((MutableReference)mutableReference).reference = new PooledConnection(usageTrackingInternalConnection);
                    this.permitAvailableOrHandedOverOrClosedOrPausedCondition.signal();
                    return true;
                }
                return false;
            });
            if (!bl) {
                DefaultConnectionPool.this.pool.release(usageTrackingInternalConnection);
            }
        }

        void signalClosedOrPaused() {
            Locks.withLock((Lock)this.lock, this.permitAvailableOrHandedOverOrClosedOrPausedCondition::signalAll);
        }
    }

    @ThreadSafe
    static final class ServiceStateManager {
        private final ConcurrentHashMap<ObjectId, ServiceState> stateByServiceId = new ConcurrentHashMap();

        ServiceStateManager() {
        }

        void addConnection(ObjectId objectId2) {
            this.stateByServiceId.compute(objectId2, (objectId, serviceState) -> {
                if (serviceState == null) {
                    serviceState = new ServiceState();
                }
                serviceState.incrementConnectionCount();
                return serviceState;
            });
        }

        void removeConnection(ObjectId objectId2) {
            this.stateByServiceId.compute(objectId2, (objectId, serviceState) -> {
                Assertions.assertNotNull(serviceState);
                return serviceState.decrementAndGetConnectionCount() == 0 ? null : serviceState;
            });
        }

        boolean incrementGeneration(ObjectId objectId, int n) {
            ServiceState serviceState = this.stateByServiceId.get(objectId);
            return serviceState == null || serviceState.incrementGeneration(n);
        }

        int getGeneration(ObjectId objectId) {
            ServiceState serviceState = this.stateByServiceId.get(objectId);
            return serviceState == null ? 0 : serviceState.getGeneration();
        }

        private static final class ServiceState {
            private final AtomicInteger generation = new AtomicInteger();
            private final AtomicInteger connectionCount = new AtomicInteger();

            private ServiceState() {
            }

            void incrementConnectionCount() {
                this.connectionCount.incrementAndGet();
            }

            int decrementAndGetConnectionCount() {
                return this.connectionCount.decrementAndGet();
            }

            boolean incrementGeneration(int n) {
                return this.generation.compareAndSet(n, n + 1);
            }

            public int getGeneration() {
                return this.generation.get();
            }
        }
    }

    @ThreadSafe
    private final class StateAndGeneration {
        private final ReadWriteLock lock = new StampedLock().asReadWriteLock();
        private volatile boolean paused = true;
        private final AtomicBoolean closed = new AtomicBoolean();
        private volatile int generation = 0;
        @Nullable
        private Throwable cause = null;

        StateAndGeneration() {
        }

        int generation() {
            return this.generation;
        }

        boolean pauseAndIncrementGeneration(@Nullable Throwable throwable) {
            return Locks.withLock(this.lock.writeLock(), () -> {
                boolean bl = false;
                if (!this.paused) {
                    this.paused = true;
                    DefaultConnectionPool.this.pool.pause(() -> new MongoConnectionPoolClearedException(DefaultConnectionPool.this.serverId, throwable));
                    bl = true;
                }
                this.cause = throwable;
                ++this.generation;
                if (bl) {
                    DefaultConnectionPool.this.logEventMessage("Connection pool cleared", "Connection pool for {}:{} cleared");
                    DefaultConnectionPool.this.connectionPoolListener.connectionPoolCleared(new ConnectionPoolClearedEvent(DefaultConnectionPool.this.serverId));
                    DefaultConnectionPool.this.backgroundMaintenance.runOnceAndStop();
                }
                return bl;
            });
        }

        boolean ready() {
            boolean bl = false;
            if (this.paused) {
                bl = Locks.withLock(this.lock.writeLock(), () -> {
                    if (this.paused) {
                        this.paused = false;
                        this.cause = null;
                        DefaultConnectionPool.this.pool.ready();
                        DefaultConnectionPool.this.logEventMessage("Connection pool ready", "Connection pool ready for {}:{}");
                        DefaultConnectionPool.this.connectionPoolListener.connectionPoolReady(new ConnectionPoolReadyEvent(DefaultConnectionPool.this.serverId));
                        DefaultConnectionPool.this.backgroundMaintenance.start();
                        return true;
                    }
                    return false;
                });
            }
            return bl;
        }

        boolean close() {
            return this.closed.compareAndSet(false, true);
        }

        boolean throwIfClosedOrPaused() {
            if (this.closed.get()) {
                throw DefaultConnectionPool.this.pool.poolClosedException();
            }
            if (this.paused) {
                Locks.withLock(this.lock.readLock(), () -> {
                    if (this.paused) {
                        throw new MongoConnectionPoolClearedException(DefaultConnectionPool.this.serverId, this.cause);
                    }
                });
            }
            return false;
        }
    }

    private class UsageTrackingInternalConnectionItemFactory
    implements ConcurrentPool.ItemFactory<UsageTrackingInternalConnection> {
        private final InternalConnectionFactory internalConnectionFactory;

        UsageTrackingInternalConnectionItemFactory(InternalConnectionFactory internalConnectionFactory) {
            this.internalConnectionFactory = internalConnectionFactory;
        }

        @Override
        public UsageTrackingInternalConnection create() {
            return new UsageTrackingInternalConnection(this.internalConnectionFactory.create(DefaultConnectionPool.this.serverId, DefaultConnectionPool.this.connectionGenerationSupplier), DefaultConnectionPool.this.serviceStateManager);
        }

        @Override
        public void close(UsageTrackingInternalConnection usageTrackingInternalConnection) {
            if (!usageTrackingInternalConnection.isCloseSilently()) {
                DefaultConnectionPool.this.connectionClosed(DefaultConnectionPool.this.connectionPoolListener, DefaultConnectionPool.this.getId(usageTrackingInternalConnection), this.getReasonForClosing(usageTrackingInternalConnection));
            }
            usageTrackingInternalConnection.close();
        }

        private ConnectionClosedEvent.Reason getReasonForClosing(UsageTrackingInternalConnection usageTrackingInternalConnection) {
            ConnectionClosedEvent.Reason reason = usageTrackingInternalConnection.isClosed() ? ConnectionClosedEvent.Reason.ERROR : (DefaultConnectionPool.this.fromPreviousGeneration(usageTrackingInternalConnection) ? ConnectionClosedEvent.Reason.STALE : (DefaultConnectionPool.this.pastMaxIdleTime(usageTrackingInternalConnection) ? ConnectionClosedEvent.Reason.IDLE : ConnectionClosedEvent.Reason.POOL_CLOSED));
            return reason;
        }

        @Override
        public boolean shouldPrune(UsageTrackingInternalConnection usageTrackingInternalConnection) {
            return DefaultConnectionPool.this.shouldPrune(usageTrackingInternalConnection);
        }
    }

    @ThreadSafe
    private static class AsyncWorkManager
    implements AutoCloseable {
        private volatile State state = State.NEW;
        private volatile BlockingQueue<Task> tasks = new LinkedBlockingQueue<Task>();
        private final Lock lock = new StampedLock().asWriteLock();
        @Nullable
        private ExecutorService worker;

        AsyncWorkManager(boolean bl) {
            if (bl) {
                Assertions.assertTrue(this.initUnlessClosed());
            }
        }

        void enqueue(Task task) {
            boolean bl = Locks.withLock(this.lock, () -> {
                if (this.initUnlessClosed()) {
                    this.tasks.add(task);
                    return false;
                }
                return true;
            });
            if (bl) {
                task.failAsClosed();
            }
        }

        private boolean initUnlessClosed() {
            boolean bl = true;
            if (this.state == State.NEW) {
                this.worker = Executors.newSingleThreadExecutor(new DaemonThreadFactory("AsyncGetter"));
                this.worker.submit(() -> this.runAndLogUncaught(this::workerRun));
                this.state = State.INITIALIZED;
            } else if (this.state == State.CLOSED) {
                bl = false;
            }
            return bl;
        }

        @Override
        public void close() {
            Locks.withLock(this.lock, () -> {
                if (this.state != State.CLOSED) {
                    this.state = State.CLOSED;
                    if (this.worker != null) {
                        this.worker.shutdownNow();
                    }
                }
            });
        }

        private void workerRun() {
            while (this.state != State.CLOSED) {
                try {
                    Task task = this.tasks.take();
                    task.timeout().run(TimeUnit.NANOSECONDS, () -> task.execute(), l -> task.execute(), () -> task.failAsTimedOut());
                }
                catch (InterruptedException interruptedException) {
                }
                catch (Exception exception) {
                    LOGGER.error(null, exception);
                }
            }
            this.failAllTasksAfterClosing();
        }

        private void failAllTasksAfterClosing() {
            Queue queue = Locks.withLock(this.lock, () -> {
                Assertions.assertTrue(this.state == State.CLOSED);
                BlockingQueue<Task> blockingQueue = this.tasks;
                if (!this.tasks.isEmpty()) {
                    this.tasks = new LinkedBlockingQueue<Task>();
                }
                return blockingQueue;
            });
            queue.forEach(Task::failAsClosed);
            queue.clear();
        }

        private void runAndLogUncaught(Runnable runnable) {
            try {
                runnable.run();
            }
            catch (Throwable throwable) {
                LOGGER.error("The pool is not going to work correctly from now on. You may want to recreate the MongoClient", throwable);
                throw throwable;
            }
        }

        private static enum State {
            NEW,
            INITIALIZED,
            CLOSED;

        }
    }

    @NotThreadSafe
    final class Task {
        private final Timeout timeout;
        private final StartTime startTime;
        private final Consumer<RuntimeException> action;
        private boolean completed;

        Task(Timeout timeout, StartTime startTime, Consumer<RuntimeException> consumer) {
            this.timeout = timeout;
            this.startTime = startTime;
            this.action = consumer;
        }

        void execute() {
            this.doComplete(() -> null);
        }

        void failAsClosed() {
            this.doComplete(DefaultConnectionPool.this.pool::poolClosedException);
        }

        void failAsTimedOut() {
            this.doComplete(() -> DefaultConnectionPool.this.createTimeoutException(this.startTime));
        }

        private void doComplete(Supplier<RuntimeException> supplier) {
            Assertions.assertFalse(this.completed);
            this.completed = true;
            this.action.accept(supplier.get());
        }

        Timeout timeout() {
            return this.timeout;
        }
    }

    private static final class MongoOpenConnectionInternalException
    extends RuntimeException {
        private static final long serialVersionUID = 1L;

        MongoOpenConnectionInternalException(@NonNull Throwable throwable) {
            super(throwable);
        }

        @Override
        @NonNull
        public Throwable getCause() {
            return Assertions.assertNotNull(super.getCause());
        }
    }

    @NotThreadSafe
    private static final class MutableReference<T> {
        @Nullable
        private T reference;

        private MutableReference() {
        }
    }

    private static enum OpenWithConcurrencyLimitMode {
        TRY_GET_AVAILABLE,
        TRY_HAND_OVER_OR_RELEASE;

    }
}

