package io.github.openbagtwo.lighterend.utils.math;

import com.google.common.collect.Lists;
import io.github.openbagtwo.lighterend.utils.Flags;
import io.github.openbagtwo.lighterend.utils.math.sdf.SDF;
import io.github.openbagtwo.lighterend.utils.math.sdf.operators.SDFUnion;
import io.github.openbagtwo.lighterend.utils.math.sdf.primitives.SDFLine;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;
import net.minecraft.class_2338;
import net.minecraft.class_2338.class_2339;
import net.minecraft.class_2680;
import net.minecraft.class_3532;
import net.minecraft.class_5281;
import net.minecraft.class_5819;
import org.joml.Vector3f;

public class MathUtils {

  public static final float PI2 = (float) Math.PI * 2.0f;

  public static List<Vector3f> makeSpline(float x1, float y1, float z1, float x2, float y2,
      float z2, int points) {
    List<Vector3f> spline = Lists.newArrayList();
    spline.add(new Vector3f(x1, y1, z1));
    int count = points - 1;
    for (int i = 1; i < count; i++) {
      float delta = (float) i / (float) count;
      float x = class_3532.method_16439(delta, x1, x2);
      float y = class_3532.method_16439(delta, y1, y2);
      float z = class_3532.method_16439(delta, z1, z2);
      spline.add(new Vector3f(x, y, z));
    }
    spline.add(new Vector3f(x2, y2, z2));
    return spline;
  }

  public static List<Vector3f> copySpline(List<Vector3f> spline) {
    List<Vector3f> result = new ArrayList<>(spline.size());
    for (Vector3f v : spline) {
      result.add(new Vector3f(v.x(), v.y(), v.z()));
    }
    return result;
  }

  public static void scale(List<Vector3f> spline, float scale) {
    scale(spline, scale, scale, scale);
  }

  public static void scale(List<Vector3f> spline, float x, float y, float z) {
    for (Vector3f v : spline) {
      v.set(v.x() * x, v.y() * y, v.z() * z);
    }
  }

  public static void offsetParts(List<Vector3f> spline, class_5819 random, float dx, float dy,
      float dz) {
    int count = spline.size();
    for (int i = 1; i < count; i++) {
      Vector3f pos = spline.get(i);
      float x = pos.x() + (float) random.method_43059() * dx;
      float y = pos.y() + (float) random.method_43059() * dy;
      float z = pos.z() + (float) random.method_43059() * dz;
      pos.set(x, y, z);
    }
  }

  public static boolean fillLine(
      Vector3f start,
      Vector3f end,
      class_5281 world,
      class_2680 state,
      class_2338 pos,
      Function<class_2680, Boolean> replace
  ) {
    float dx = end.x() - start.x();
    float dy = end.y() - start.y();
    float dz = end.z() - start.z();
    float max = Math.max(Math.max(Math.abs(dx), Math.abs(dy)), Math.abs(dz));
    int count = class_3532.method_15375(max + 1);
    dx /= max;
    dy /= max;
    dz /= max;
    float x = start.x();
    float y = start.y();
    float z = start.z();
    boolean down = Math.abs(dy) > 0.2;

    class_2680 bState;
    class_2339 bPos = new class_2339();
    for (int i = 0; i < count; i++) {
      bPos.method_10102(x + pos.method_10263(), y + pos.method_10264(), z + pos.method_10260());
      bState = world.method_8320(bPos);
      if (bState.equals(state) || replace.apply(bState)) {
        world.method_8652(bPos, state, Flags.SILENT);
        bPos.method_33098(bPos.method_10264() - 1);
        bState = world.method_8320(bPos);
        if (down && bState.equals(state) || replace.apply(bState)) {
          world.method_8652(bPos, state, Flags.SILENT);
        }
      } else {
        return false;
      }
      x += dx;
      y += dy;
      z += dz;
    }
    bPos.method_10102(end.x() + pos.method_10263(), end.y() + pos.method_10264(), end.z() + pos.method_10260());
    bState = world.method_8320(bPos);
    if (bState.equals(state) || replace.apply(bState)) {
      world.method_8652(bPos, state, Flags.SILENT);
      bPos.method_33098(bPos.method_10264() - 1);
      bState = world.method_8320(bPos);
      if (down && bState.equals(state) || replace.apply(bState)) {
        world.method_8652(bPos, state, Flags.SILENT);
      }
      return true;
    } else {
      return false;
    }
  }

  public static void fillLineForce(
      Vector3f start,
      Vector3f end,
      class_5281 world,
      class_2680 state,
      class_2338 pos,
      Function<class_2680, Boolean> replace
  ) {
    float dx = end.x() - start.x();
    float dy = end.y() - start.y();
    float dz = end.z() - start.z();
    float max = Math.max(Math.max(Math.abs(dx), Math.abs(dy)), Math.abs(dz));
    int count = class_3532.method_15375(max + 1);
    dx /= max;
    dy /= max;
    dz /= max;
    float x = start.x();
    float y = start.y();
    float z = start.z();
    boolean down = Math.abs(dy) > 0.2;

    class_2680 bState;
    class_2339 bPos = new class_2339();
    for (int i = 0; i < count; i++) {
      bPos.method_10102(x + pos.method_10263(), y + pos.method_10264(), z + pos.method_10260());
      bState = world.method_8320(bPos);
      if (replace.apply(bState)) {
        world.method_8652(bPos, state, Flags.SILENT);
        bPos.method_33098(bPos.method_10264() - 1);
        bState = world.method_8320(bPos);
        if (down && replace.apply(bState)) {
          world.method_8652(bPos, state, Flags.SILENT);
        }
      }
      x += dx;
      y += dy;
      z += dz;
    }
    bPos.method_10102(end.x() + pos.method_10263(), end.y() + pos.method_10264(), end.z() + pos.method_10260());
    bState = world.method_8320(bPos);
    if (replace.apply(bState)) {
      world.method_8652(bPos, state, Flags.SILENT);
      bPos.method_33098(bPos.method_10264() - 1);
      bState = world.method_8320(bPos);
      if (down && replace.apply(bState)) {
        world.method_8652(bPos, state, Flags.SILENT);
      }
    }
  }

  public static boolean fillSpline(
      List<Vector3f> spline,
      class_5281 world,
      class_2680 state,
      class_2338 pos,
      Function<class_2680, Boolean> replace
  ) {
    Vector3f startPos = spline.getFirst();
    for (int i = 1; i < spline.size(); i++) {
      Vector3f endPos = spline.get(i);
      if (!(fillLine(startPos, endPos, world, state, pos, replace))) {
        return false;
      }
      startPos = endPos;
    }

    return true;
  }

  public static void fillSplineForce(
      List<Vector3f> spline,
      class_5281 world,
      class_2680 state,
      class_2338 pos,
      Function<class_2680, Boolean> replace
  ) {
    Vector3f startPos = spline.get(0);
    for (int i = 1; i < spline.size(); i++) {
      Vector3f endPos = spline.get(i);
      fillLineForce(startPos, endPos, world, state, pos, replace);
      startPos = endPos;
    }
  }

  public static void rotateSpline(List<Vector3f> spline, float angle) {
    for (Vector3f v : spline) {
      float sin = (float) Math.sin(angle);
      float cos = (float) Math.cos(angle);
      float x = v.x() * cos + v.z() * sin;
      float z = v.x() * sin + v.z() * cos;
      v.set(x, v.y(), z);
    }
  }

  public static boolean canGenerate(
      List<Vector3f> spline,
      float scale,
      class_2338 start,
      class_5281 world,
      Function<class_2680, Boolean> canReplace
  ) {
    int count = spline.size();
    Vector3f vec = spline.get(0);
    class_2339 mut = new class_2339();
    float x1 = start.method_10263() + vec.x() * scale;
    float y1 = start.method_10264() + vec.y() * scale;
    float z1 = start.method_10260() + vec.z() * scale;
    for (int i = 1; i < count; i++) {
      vec = spline.get(i);
      float x2 = start.method_10263() + vec.x() * scale;
      float y2 = start.method_10264() + vec.y() * scale;
      float z2 = start.method_10260() + vec.z() * scale;

      for (float py = y1; py < y2; py += 3) {
        if (py - start.method_10264() < 10) {
          continue;
        }
        float lerp = (py - y1) / (y2 - y1);
        float x = class_3532.method_16439(lerp, x1, x2);
        float z = class_3532.method_16439(lerp, z1, z2);
        mut.method_10102(x, py, z);
        if (!canReplace.apply(world.method_8320(mut))) {
          return false;
        }
      }

      x1 = x2;
      y1 = y2;
      z1 = z2;
    }
    return true;
  }

  public static boolean canGenerate(
      List<Vector3f> spline,
      class_2338 start,
      class_5281 world,
      Function<class_2680, Boolean> canReplace
  ) {
    int count = spline.size();
    Vector3f vec = spline.get(0);
    class_2339 mut = new class_2339();
    float x1 = start.method_10263() + vec.x();
    float y1 = start.method_10264() + vec.y();
    float z1 = start.method_10260() + vec.z();
    for (int i = 1; i < count; i++) {
      vec = spline.get(i);
      float x2 = start.method_10263() + vec.x();
      float y2 = start.method_10264() + vec.y();
      float z2 = start.method_10260() + vec.z();

      for (float py = y1; py < y2; py += 3) {
        if (py - start.method_10264() < 10) {
          continue;
        }
        float lerp = (py - y1) / (y2 - y1);
        float x = class_3532.method_16439(lerp, x1, x2);
        float z = class_3532.method_16439(lerp, z1, z2);
        mut.method_10102(x, py, z);
        if (!canReplace.apply(world.method_8320(mut))) {
          return false;
        }
      }

      x1 = x2;
      y1 = y2;
      z1 = z2;
    }
    return true;
  }

  public static SDF buildSDF(
      List<Vector3f> spline,
      float radius1,
      float radius2,
      Function<class_2338, class_2680> placerFunction
  ) {
    int count = spline.size();
    float max = count - 2;
    SDF result = null;
    Vector3f start = spline.get(0);
    for (int i = 1; i < count; i++) {
      Vector3f pos = spline.get(i);
      float delta = (float) (i - 1) / max;
      SDF line = new SDFLine().setRadius(class_3532.method_16439(delta, radius1, radius2))
          .setStart(start.x(), start.y(), start.z())
          .setEnd(pos.x(), pos.y(), pos.z())
          .setBlock(placerFunction);
      result = result == null ? line : new SDFUnion().setSourceA(result).setSourceB(line);
      start = pos;
    }
    return result;
  }

  public static void offset(List<Vector3f> spline, Vector3f offset) {
    for (Vector3f v : spline) {
      v.set(offset.x() + v.x(), offset.y() + v.y(), offset.z() + v.z());
    }
  }

  public static Vector3f randomHorizontal(net.minecraft.class_5819 random) {
    float angleY = class_3532.method_15344(random, 0, PI2);
    float vx = (float) Math.sin(angleY);
    float vz = (float) Math.cos(angleY);
    return new Vector3f(vx, 0, vz);
  }

  public static int getSeed(int seed, int x, int y) {
    int h = seed + x * 374761393 + y * 668265263;
    h = (h ^ (h >> 13)) * 1274126177;
    return h ^ (h >> 16);
  }

  public static Vector3f getPos(List<Vector3f> spline, float index) {
    int i = (int) index;
    int last = spline.size() - 1;
    if (i >= last) {
      return spline.get(last);
    }
    float delta = index - i;
    Vector3f p1 = spline.get(i);
    Vector3f p2 = spline.get(i + 1);
    float x = class_3532.method_16439(delta, p1.x(), p2.x());
    float y = class_3532.method_16439(delta, p1.y(), p2.y());
    float z = class_3532.method_16439(delta, p1.z(), p2.z());
    return new Vector3f(x, y, z);
  }
}
