package net.bichal.bplb.client;

import com.mojang.blaze3d.systems.RenderSystem;
import net.bichal.bplb.client.render.RenderAddons;
import net.bichal.bplb.client.render.RenderUtils;
import net.bichal.bplb.config.Config;
import net.bichal.bplb.network.PositionUpdatePayload;
import net.bichal.bplb.util.Constants;
import net.bichal.bplb.util.DistanceUtils;
import net.bichal.bplb.util.MathUtils;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.fabricmc.fabric.api.client.networking.v1.ClientPlayConnectionEvents;
import net.fabricmc.fabric.api.client.networking.v1.ClientPlayNetworking;
import net.minecraft.class_1304;
import net.minecraft.class_1657;
import net.minecraft.class_1738;
import net.minecraft.class_1799;
import net.minecraft.class_243;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import net.minecraft.class_4184;
import org.lwjgl.opengl.GL11;

import java.util.*;
import java.util.stream.Collectors;

import static java.lang.Math.round;
import static net.bichal.bplb.util.ColorUtils.generateColorFromUUID;
import static net.bichal.bplb.util.Constants.CONFIG;

@Environment(EnvType.CLIENT)
public class Hud {
    public record PlayerPosition(UUID uuid, String name, double x, double y, double z) {
    }
    private static final Map<UUID, PlayerPosition> playerPositions = new HashMap<>();
    private static final Map<Object, Float> currentIconPositions = new HashMap<>();
    public static final List<class_243> deathMarkers = new ArrayList<>();
    private static final int MAX_DEATH_MARKERS = 4;
    private static List<PlayerPosition> positionsToRenderCache = new ArrayList<>();
    private static final Map<Object, class_243> lastKnownPositions = new HashMap<>();
    private static final Map<Object, Long> lastPositionUpdateTime = new HashMap<>();
    private static boolean shouldApplyHudOffset = false;
    private static final float MIN_Z_DEPTH = 100f;

    private static void updateRenderCache(class_310 client) {
        if (client.field_1724 == null) {
            positionsToRenderCache.clear();
            return;
        }

        List<PlayerPosition> positions = getPositionsToRender(client);
        positions.removeIf(pos -> pos.uuid().equals(client.field_1724.method_5667()));

        class_243 playerPos = client.field_1724.method_19538();
        final int maxIcons = CONFIG.getMaxVisibleIcons();

        positions.sort(Comparator.comparingDouble(pos -> {
            double dx = playerPos.field_1352 - pos.x;
            double dz = playerPos.field_1350 - pos.z;
            return dx * dx + dz * dz;
        }));

        for (PlayerPosition pos : positions) {
            if (!currentIconPositions.containsKey(pos.uuid())) {
                float initialPos = calculateRelativePosition(client.field_1724, pos);
                if (initialPos >= 0) currentIconPositions.put(pos.uuid(), initialPos * Constants.BAR_WIDTH);
            } else {
                float currentPos = calculateRelativePosition(client.field_1724, pos);
                if (currentPos < 0) currentIconPositions.remove(pos.uuid());
            }
        }

        for (class_243 marker : deathMarkers) {
            if (!currentIconPositions.containsKey(marker)) {
                PlayerPosition markerPos = new PlayerPosition(new UUID(marker.hashCode(), marker.hashCode()), "Death", marker.field_1352, marker.field_1351, marker.field_1350);
                float initialPos = calculateRelativePosition(client.field_1724, markerPos);
                if (initialPos >= 0) currentIconPositions.put(marker, initialPos * Constants.BAR_WIDTH);
            }
        }

        positionsToRenderCache = positions.size() > maxIcons ? new ArrayList<>(positions.subList(0, maxIcons)) : new ArrayList<>(positions);
    }

    public static void tick(class_310 client) {
        if (client.field_1687 == null) return;
        updateRenderCache(client);
        if (client.field_1724 != null) {
            deathMarkers.removeIf(marker -> {
                if (client.field_1724.method_19538().method_1022(marker) < 10) {
                    currentIconPositions.remove(marker);
                    return true;
                }
                return false;
            });
        }
    }

    public static void addDeathLocation(class_243 location) {
        if (deathMarkers.stream().anyMatch(marker -> marker.method_1022(location) < 5)) {
            return;
        }
        deathMarkers.add(location);
        while (deathMarkers.size() > MAX_DEATH_MARKERS) {
            class_243 oldest = deathMarkers.removeFirst();
            currentIconPositions.remove(oldest);
        }
    }

    public static void registerEvents() {
        ClientPlayConnectionEvents.JOIN.register((handler, sender, client) -> {
            playerPositions.clear();
            currentIconPositions.clear();
            deathMarkers.clear();
            lastKnownPositions.clear();
            lastPositionUpdateTime.clear();
            Constants.LOGGER.info("[{}] Cleared all caches on join", Constants.MOD_NAME_SHORT);

            ClientPlayNetworking.registerReceiver(PositionUpdatePayload.ID, (payload, context) -> {
                Client.updateLastServerUpdateTime();
                client.execute(() -> {
                    if (payload == null) return;

                    List<UUID> disconnectedList = payload.disconnectedPlayers();
                    if (disconnectedList != null) {
                        for (UUID disconnectedId : disconnectedList) {
                            if (disconnectedId == null) continue;
                            playerPositions.remove(disconnectedId);
                            currentIconPositions.remove(disconnectedId);
                        }
                    }

                    List<PositionUpdatePayload.PlayerInfo> newPlayersList = payload.newPlayers();
                    if (newPlayersList != null) {
                        for (PositionUpdatePayload.PlayerInfo newPlayer : newPlayersList) {
                            if (newPlayer == null || newPlayer.uuid() == null) continue;
                            PlayerPosition existingData = playerPositions.get(newPlayer.uuid());
                            if (existingData != null) {
                                playerPositions.put(newPlayer.uuid(), new PlayerPosition(newPlayer.uuid(), newPlayer.name(), existingData.x(), existingData.y(), existingData.z()));
                            } else {
                                playerPositions.put(newPlayer.uuid(), new PlayerPosition(newPlayer.uuid(), newPlayer.name(), 0, 0, 0));
                            }
                        }
                    }

                    List<PositionUpdatePayload.PositionData> positionsList = payload.positions();
                    if (positionsList != null) {
                        for (PositionUpdatePayload.PositionData posUpdate : positionsList) {
                            if (posUpdate == null || posUpdate.uuid() == null) continue;
                            PlayerPosition existingPosData = playerPositions.get(posUpdate.uuid());
                            String name;
                            if (existingPosData != null) {
                                name = existingPosData.name();
                            } else {
                                class_1657 localPlayer = client.field_1687 != null ? client.field_1687.method_18470(posUpdate.uuid()) : null;
                                name = localPlayer != null ? localPlayer.method_5477().getString() : "Player";
                            }
                            playerPositions.put(posUpdate.uuid(), new PlayerPosition(posUpdate.uuid(), name, posUpdate.x(), posUpdate.y(), posUpdate.z()));
                        }
                    }
                });
            });
        });
    }

    public static void render(class_332 context) {
        final class_310 client = class_310.method_1551();
        if (client.field_1724 == null || client.field_1687 == null || !CONFIG.isModEnabled()) {
            shouldApplyHudOffset = false;
            return;
        }

        if (positionsToRenderCache.isEmpty() && deathMarkers.isEmpty()) {
            shouldApplyHudOffset = false;
            return;
        }

        final int barX = (context.method_51421() - Constants.BAR_WIDTH) / 2;
        final int barY = context.method_51443() + Constants.BAR_Y_OFFSET;
        final boolean showDetails = Keybinds.shouldShowPlayerNames() || CONFIG.isAlwaysShowPlayerNames();

        List<RenderEntry> allEntries = new ArrayList<>();
        for (PlayerPosition pos : positionsToRenderCache) {
            final class_1657 targetPlayer = client.field_1687.method_18470(pos.uuid());
            if (targetPlayer != null && shouldHideTarget(targetPlayer)) continue;
            double distance = DistanceUtils.calculateDistance(client.field_1724.method_23317(), client.field_1724.method_23318(), client.field_1724.method_23321(), pos.x, pos.y, pos.z);
            float alpha = getDistanceAlpha(distance);
            allEntries.add(new RenderEntry(pos, pos.uuid(), distance, alpha, false));
        }

        for (class_243 marker : deathMarkers) {
            double distance = DistanceUtils.calculateDistance(client.field_1724.method_23317(), client.field_1724.method_23318(), client.field_1724.method_23321(), marker.field_1352, marker.field_1351, marker.field_1350);
            PlayerPosition markerPos = new PlayerPosition(new UUID(marker.hashCode(), marker.hashCode()), "Death", marker.field_1352, marker.field_1351, marker.field_1350);
            allEntries.add(new RenderEntry(markerPos, marker, distance, 1.0f, true));
        }

        allEntries.sort(Comparator.comparingDouble(RenderEntry::distance));

        RenderSystem.enableBlend();
        RenderSystem.blendFunc(GL11.GL_SRC_ALPHA, GL11.GL_ONE_MINUS_SRC_ALPHA);

        int totalVisibleIcons = allEntries.size();
        for (int i = 0; i < allEntries.size(); i++) {
            RenderEntry entry = allEntries.get(i);
            float baseZ = calculateBaseZ(i, totalVisibleIcons);
            renderBarIcon(context, entry.pos, entry.key, barX, barY, baseZ, showDetails, entry.alpha);
        }

        shouldApplyHudOffset = hasVisibleIconsInVisibleRange(client);
        RenderSystem.disableBlend();
    }

    private record RenderEntry(PlayerPosition pos, Object key, double distance, float alpha, boolean isDeathMarker) {}

    public static boolean hasVisibleIconsInVisibleRange(class_310 client) {
        if (client.field_1724 == null) return false;

        final int barLeft = (client.method_22683().method_4486() - Constants.BAR_WIDTH) / 2;
        final int barRight = barLeft + Constants.BAR_WIDTH;

        for (PlayerPosition pos : positionsToRenderCache) {
            float targetPos = calculateRelativePosition(client.field_1724, pos);
            if (targetPos < 0) continue;

            Float currentPos = currentIconPositions.get(pos.uuid());
            if (currentIconPos(barLeft, barRight, currentPos)) return true;
        }

        for (class_243 marker : deathMarkers) {
            PlayerPosition markerPos = new PlayerPosition(new UUID(marker.hashCode(), marker.hashCode()), "Death", marker.field_1352, marker.field_1351, marker.field_1350);
            float targetPos = calculateRelativePosition(client.field_1724, markerPos);
            if (targetPos < 0) continue;

            Float currentPos = currentIconPositions.get(marker);
            if (currentIconPos(barLeft, barRight, currentPos)) return true;
        }

        return false;
    }

    private static boolean currentIconPos(int barLeft, int barRight, Float currentPos) {
        if (currentPos != null) {
            int iconCenterX = barLeft + Math.round(currentPos);
            int iconLeft = iconCenterX - 5;
            int iconRight = iconCenterX + 5;
            return iconRight >= barLeft && iconLeft <= barRight;
        }
        return false;
    }

    private static void renderBarIcon(class_332 context, PlayerPosition pos, Object key, int barX, int barY, float baseZ, boolean showDetails, float alpha) {
        class_310 client = class_310.method_1551();
        if (client.field_1724 == null) return;

        Float targetPos = calculateRelativePosition(client.field_1724, pos);
        if (targetPos < 0) return;
        targetPos *= Constants.BAR_WIDTH;

        Float currentPos = currentIconPositions.getOrDefault(key, targetPos);
        float distance = Math.abs(currentPos - targetPos);
        if (distance > Constants.BAR_WIDTH * 0.75f) {
            currentPos = targetPos;
            currentIconPositions.put(key, currentPos);
        } else {
            float delta = targetPos - currentPos;
            float t = Math.min(CONFIG.getLerpSpeed(), 1.0f);
            currentPos = currentPos + delta * MathUtils.easeInOutQuad(t);
            currentIconPositions.put(key, currentPos);
        }

        int iconCenterX = barX + round(currentPos);
        float edgeAlpha = calculateEdgeAlpha(iconCenterX, barX);
        alpha *= edgeAlpha;

        if (alpha <= 0.01f && Math.abs(currentPos - targetPos) < 1.0f) return;

        float topLeftX = iconCenterX - Constants.ICON_BASE_SIZE / 2f;
        float topLeftY = barY - Constants.ICON_BASE_SIZE / 2f;

        float finalAlpha = alpha;
        RenderUtils.withMatrixPush(context, 0, 0, () -> {
            boolean isDeathMarker = key instanceof class_243;
            boolean showHead = !isDeathMarker && (CONFIG.isAlwaysShowPlayerHeads() || Keybinds.shouldShowPlayerNames());
            Config.PlayerAppearance appearance = isDeathMarker ? null : CONFIG.getPlayerConfig(pos.name());
            String borderStyle;
            int color;
            if (isDeathMarker) {
                color = CONFIG.getDeathMarkerColor();
                borderStyle = CONFIG.getDeathMarkerBorderStyle();
            } else {
                color = appearance != null && appearance.color != null ? appearance.color : generateColorFromUUID(pos.uuid());
                borderStyle = appearance != null && appearance.iconBorderStyle != null ? appearance.iconBorderStyle : CONFIG.getNameBorderStyle();
            }
            float nameplateAlpha = showDetails ? finalAlpha : 0f;
            String text = isDeathMarker ? (int) pos.x + " " + (int) pos.y + " " + (int) pos.z : pos.name();

            context.method_51448().method_46416(0, 0, baseZ);

            if (isDeathMarker) {
                RenderAddons.renderDeathMarker(context, topLeftX, topLeftY, Constants.ICON_BASE_SIZE, finalAlpha, CONFIG);
            } else {
                double iconDistance = DistanceUtils.calculateDistance(client.field_1724.method_23317(), client.field_1724.method_23318(), client.field_1724.method_23321(), pos.x, pos.y, pos.z);
                RenderAddons.renderPlayerIcon(context, pos.name(), pos.uuid(), iconDistance, topLeftX, topLeftY, Constants.ICON_BASE_SIZE, showHead, CONFIG, finalAlpha);
            }

            renderHeightIndicator(context, pos, iconCenterX, (int) (topLeftY + Constants.ICON_BASE_SIZE / 2f), finalAlpha, appearance);

            context.method_51448().method_46416(0, 0, 1);
            RenderAddons.renderNameplate(context, text, borderStyle, color, iconCenterX - (client.field_1772.method_1727(text) * CONFIG.getNameplateScale() + 4) / 2, topLeftY - (12 * CONFIG.getNameplateScale()) - 4, nameplateAlpha, CONFIG.getNameplateScale());
        });
    }

    public static boolean shouldApplyHudOffset() {
        return shouldApplyHudOffset;
    }

    private static boolean showUp(class_310 client, PlayerPosition pos) {
        return isArrowUp(client, pos);
    }

    private static boolean showDown(class_310 client, PlayerPosition pos) {
        return isArrowDown(client, pos);
    }

    private static void renderHeightIndicator(class_332 context, PlayerPosition pos, int centerX, int centerY, float alpha, Config.PlayerAppearance appearance) {
        class_310 client = class_310.method_1551();
        if (client.field_1724 == null) return;

        if (showUp(client, pos) || showDown(client, pos)) {
            String arrowId = (appearance != null && appearance.arrowType != null) ? appearance.arrowType : CONFIG.getArrowType();
            renderHeightArrow(context, centerX, centerY, alpha, showUp(client, pos), arrowId);
        }
    }

    private static void renderHeightArrow(class_332 context, int centerX, int centerY, float alpha, boolean isUp, String arrowId) {
        int arrowX = centerX - 5;
        int arrowY = isUp ? (centerY - Constants.ICON_BASE_SIZE - 2 - CONFIG.getVerticalPadding()) : (centerY + 2 + CONFIG.getVerticalPadding());
        RenderAddons.renderArrow(context, arrowId, isUp, arrowX, arrowY, Constants.ICON_BASE_SIZE, alpha);
    }

    private static List<PlayerPosition> getPositionsToRender(class_310 client) {
        if (client.field_1687 == null || client.field_1724 == null) return Collections.emptyList();
        boolean useServerData = !Client.isLocalMode();
        if (useServerData && !playerPositions.isEmpty()) {
            return new ArrayList<>(playerPositions.values());
        }
        return client.field_1687.method_18456().stream().filter(p -> !p.method_5667().equals(client.field_1724.method_5667())).map(p -> new PlayerPosition(p.method_5667(), p.method_5477().getString(), p.method_23317(), p.method_23318(), p.method_23321())).collect(Collectors.toList());
    }

    private static float getDistanceAlpha(double distance) {
        if (distance > CONFIG.getFadeEndDistance()) return CONFIG.getFadeAlphaMin();
        if (distance < CONFIG.getFadeStartDistance()) return CONFIG.getFadeAlphaMax();
        float progress = (float) ((distance - CONFIG.getFadeStartDistance()) / (CONFIG.getFadeEndDistance() - CONFIG.getFadeStartDistance()));
        return class_3532.method_16439(MathUtils.easeInOutQuad(progress), CONFIG.getFadeAlphaMax(), CONFIG.getFadeAlphaMin());
    }

    private static boolean isArrowUp(class_310 client, PlayerPosition pos) {
        return shouldShowArrow(client, pos, true);
    }

    private static boolean isArrowDown(class_310 client, PlayerPosition pos) {
        return shouldShowArrow(client, pos, false);
    }

    private static double getDynamicAngleThreshold(class_310 client, PlayerPosition pos) {
        if (client.field_1724 == null) return 0.0;
        double dx = pos.x - client.field_1724.method_23317();
        double dz = pos.z - client.field_1724.method_23321();
        double deltaH = Math.sqrt(dx * dx + dz * dz);
        double deltaY = Math.abs(pos.y - (client.field_1724.method_23318() + 1.0));
        double minAngle = 8.0;
        double maxAngle = 45.0;
        double maxEffectiveDeltaY = 30.0;
        double maxEffectiveDeltaH = 64.0;
        double yFactor = 1.0 - Math.min(deltaY, maxEffectiveDeltaY) / maxEffectiveDeltaY;
        double hFactor = 1.0 - Math.min(deltaH, maxEffectiveDeltaH) / maxEffectiveDeltaH;
        return class_3532.method_16436(yFactor * hFactor, minAngle, maxAngle);
    }

    private static double getCameraAngleDifference(class_4184 camera, PlayerPosition pos) {
        class_243 cameraPos = camera.method_19326();
        double dx = pos.x - cameraPos.field_1352;
        double dz = pos.z - cameraPos.field_1350;
        double horizontalDist = Math.sqrt(dx * dx + dz * dz);
        double targetY = pos.y + 1.62;
        return -(Math.toDegrees(Math.atan2(targetY - cameraPos.field_1351, horizontalDist)) + camera.method_19329());
    }

    private static float calculateRelativePosition(class_1657 viewer, PlayerPosition target) {
        if (viewer == null) return -1f;

        class_243 currentPos = new class_243(target.x, target.y, target.z);
        class_243 smoothedPos = currentPos;

        Long currentTime = System.currentTimeMillis();
        class_243 lastPos = lastKnownPositions.get(target.uuid());
        Long lastTime = lastPositionUpdateTime.get(target.uuid());

        if (lastPos != null && lastTime != null) {
            long timeDelta = currentTime - lastTime;
            if (timeDelta < 1000) {
                float alpha = Math.min(timeDelta / 100f * CONFIG.getLerpSpeed(), 1f);
                smoothedPos = lastPos.method_35590(currentPos, MathUtils.easeInOutQuad(alpha));
            }
        }

        lastKnownPositions.put(target.uuid(), smoothedPos);
        lastPositionUpdateTime.put(target.uuid(), currentTime);

        double relativeAngle = getRelativeAngle(viewer, smoothedPos);
        if (Math.abs(relativeAngle) > 90) return -1f;

        return (float) (relativeAngle + 90) / 180.0f;
    }

    private static float calculateBaseZ(int index, int totalVisibleIcons) {
        float minZ = MIN_Z_DEPTH;

        if (totalVisibleIcons <= 1) {
            return minZ;
        }

        float maxAvailableZ = 500f;
        float availableRange = maxAvailableZ - minZ;
        float spacingPerIcon = availableRange / (totalVisibleIcons - 1);

        return minZ + (index * spacingPerIcon);
    }

    private static double getRelativeAngle(class_1657 viewer, class_243 smoothedPos) {
        double relativeAngle = class_3532.method_15338(Math.toDegrees(Math.atan2(smoothedPos.field_1350 - viewer.method_23321(), smoothedPos.field_1352 - viewer.method_23317())) - 90 - class_3532.method_15393(viewer.method_36454()));
        if (CONFIG.isAdjustToFov()) {
            class_310 client = RenderUtils.getClient();
            float fov = (float) client.field_1690.method_41808().method_41753();
            float fovFactor = (90.0f / fov) * CONFIG.getFovMultiplier();
            relativeAngle *= fovFactor;
        }
        return relativeAngle;
    }

    private static float calculateEdgeAlpha(int iconX, int barX) {
        int edgeDistance = Math.min(iconX - barX, Constants.BAR_WIDTH - (iconX - barX));
        if (edgeDistance >= Constants.EDGE_ALPHA_FADE_MARGIN) return 1.0f;
        return class_3532.method_15363(edgeDistance / (float) Constants.EDGE_ALPHA_FADE_MARGIN, 0.0f, 1.0f);
    }

    private static boolean shouldHideTarget(class_1657 target) {
        final class_1799 headStack = target.method_6118(class_1304.field_6169);
        return target.method_5715() || target.method_5767() || (!headStack.method_7960() && !(headStack.method_7909() instanceof class_1738));
    }

    private static boolean shouldShowArrow(class_310 client, PlayerPosition pos, boolean up) {
        if (client.field_1724 == null) return false;

        if (CONFIG.getHeightDifferenceMode().equals("PLAYER")) {
            double diff = pos.y - (client.field_1724.method_23318() + 1.0);
            return up ? diff > 5.5 : diff < -5.5;
        }

        double angleDiff = getCameraAngleDifference(client.field_1773.method_19418(), pos);
        double threshold = getDynamicAngleThreshold(client, pos);
        return up ? angleDiff < -threshold : angleDiff > threshold;
    }
}
