package net.bichal.bplb.server;

import net.bichal.bplb.network.HandshakePayload;
import net.bichal.bplb.network.PositionUpdatePayload;
import net.bichal.bplb.util.Constants;
import net.bichal.bplb.util.DistanceUtils;
import net.fabricmc.api.DedicatedServerModInitializer;
import net.fabricmc.fabric.api.event.lifecycle.v1.ServerTickEvents;
import net.fabricmc.fabric.api.networking.v1.ServerPlayConnectionEvents;
import net.fabricmc.fabric.api.networking.v1.ServerPlayNetworking;
import net.minecraft.class_1304;
import net.minecraft.class_1738;
import net.minecraft.class_1799;
import net.minecraft.class_3222;
import net.minecraft.server.MinecraftServer;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public class Server implements DedicatedServerModInitializer {
    private final PlayerTracker playerTracker = new PlayerTracker();
    private static final long PLAYER_LIST_CACHE_DURATION = 50L;
    private final Set<UUID> newPlayers = ConcurrentHashMap.newKeySet(16);
    private final Set<UUID> disconnectedPlayers = ConcurrentHashMap.newKeySet(16);
    private final Map<UUID, Map<UUID, Double>> distanceCache = new ConcurrentHashMap<>(32);
    private final Map<UUID, CachedPlayerData> playerDataCache = new ConcurrentHashMap<>(64);
    private long lastUpdateTime = 0;
    private final ExecutorService executor = Executors.newFixedThreadPool(
            Math.max(1, Runtime.getRuntime().availableProcessors() / 2), r -> {
                Thread t = new Thread(r, "BPLB-Worker");
                t.setDaemon(true);
                return t;
            }
    );
    private List<class_3222> playerListCache = new ArrayList<>();
    private long playerListCacheTime = 0;

    @Override
    public void onInitializeServer() {
        Constants.LOGGER.info("[{}] Initializing server!", Constants.MOD_NAME_SHORT);
        ServerConfig.getInstance();
        ServerPlayConnectionEvents.JOIN.register((handler, sender, server) -> {
            if (handler == null || handler.field_14140 == null) return;
            UUID playerId = handler.field_14140.method_5667();
            if (playerId == null) return;
            playerTracker.updatePlayer(handler.field_14140);
            newPlayers.add(playerId);
            invalidatePlayerListCache();
            if (ServerPlayNetworking.canSend(handler.field_14140, HandshakePayload.ID)) {
                boolean hasOp = server.method_3760().method_14569(handler.field_14140.method_7334());
                ServerPlayNetworking.send(handler.field_14140, new HandshakePayload(hasOp));
            }
        });

        ServerPlayConnectionEvents.DISCONNECT.register((handler, server) -> {
            if (handler == null || handler.field_14140 == null) return;
            UUID playerId = handler.field_14140.method_5667();
            if (playerId == null) return;
            disconnectedPlayers.add(playerId);
            playerTracker.removePlayer(playerId);
            newPlayers.remove(playerId);
            distanceCache.remove(playerId);
            invalidatePlayerListCache();
        });
        ServerTickEvents.END_SERVER_TICK.register(this::tick);
        Constants.LOGGER.info("[{}] Server initialized!", Constants.MOD_NAME_SHORT);
    }

    private void tick(@Nullable MinecraftServer server) {
        if (server == null) return;

        long currentTime = server.method_3780();
        List<class_3222> players = getCachedPlayerList(server);

        if (players.size() > 10) {
            List<CompletableFuture<Void>> futures = new ArrayList<>();
            int chunkSize = Math.max(5, players.size() / 4);

            for (int start = 0; start < players.size(); start += chunkSize) {
                int end = Math.min(start + chunkSize, players.size());
                List<class_3222> chunk = players.subList(start, end);

                futures.add(CompletableFuture.runAsync(() -> {
                    for (class_3222 player : chunk) {
                        if (player != null) {
                            playerTracker.updatePlayer(player);
                        }
                    }
                }, executor));
            }

            CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).join();
        } else {
            for (class_3222 player : players) {
                if (player != null) {
                    playerTracker.updatePlayer(player);
                }
            }
        }

        ServerConfig config = ServerConfig.getInstance();

        if (currentTime - lastUpdateTime >= config.positionUpdateRateTicks()) {
            sendUpdate(server);
            lastUpdateTime = currentTime;
        }
    }

    private List<class_3222> getCachedPlayerList(@Nullable MinecraftServer server) {
        if (server == null) return Collections.emptyList();
        long now = System.currentTimeMillis();
        if (now - playerListCacheTime > PLAYER_LIST_CACHE_DURATION) {
            playerListCache = server.method_3760().method_14571();
            playerListCacheTime = now;
        }
        return playerListCache;
    }

    private void invalidatePlayerListCache() {
        playerListCacheTime = 0;
    }

    private void sendUpdate(MinecraftServer server) {
        if (server == null) return;
        List<class_3222> players = getCachedPlayerList(server);
        if (players.isEmpty()) return;
        List<PositionUpdatePayload.PlayerInfo> newPlayerInfos = new ArrayList<>();
        Map<UUID, PlayerTracker.PlayerPosition> initialPositions = new HashMap<>();

        newPlayers.removeIf(uuid -> {
            if (uuid == null) return true;
            PlayerTracker.PlayerInfo info = playerTracker.getPlayerInfo(uuid);
            if (info != null) {
                newPlayerInfos.add(new PositionUpdatePayload.PlayerInfo(uuid, info.name));

                class_3222 player = server.method_3760().method_14602(uuid);
                if (player != null) {
                    PlayerTracker.PlayerPosition pos = new PlayerTracker.PlayerPosition(
                            player.method_23317(), player.method_23318(), player.method_23321()
                    );
                    initialPositions.put(uuid, pos);
                }
                return true;
            }
            return false;
        });

        Map<UUID, PlayerTracker.PlayerPosition> movedPlayersData = new HashMap<>(playerTracker.getAndClearMovedPlayers());
        movedPlayersData.putAll(initialPositions);

        for (class_3222 player : players) {
            if (player != null && !movedPlayersData.containsKey(player.method_5667())) {
                PlayerTracker.PlayerPosition pos = new PlayerTracker.PlayerPosition(
                        player.method_23317(), player.method_23318(), player.method_23321()
                );
                movedPlayersData.put(player.method_5667(), pos);
            }
        }

        if (movedPlayersData.isEmpty() && newPlayerInfos.isEmpty() && disconnectedPlayers.isEmpty()) {
            return;
        }
        for (class_3222 viewer : players) {
            if (viewer == null) continue;
            UUID viewerId = viewer.method_5667();
            if (viewerId == null) continue;
            List<PositionUpdatePayload.PositionData> positions = new ArrayList<>();
            Map<UUID, Double> viewerDistCache = distanceCache.computeIfAbsent(viewerId, k -> new ConcurrentHashMap<>());
            double vx = viewer.method_23317();
            double vy = viewer.method_23318();
            double vz = viewer.method_23321();

            long now = System.currentTimeMillis();
            for (Map.Entry<UUID, PlayerTracker.PlayerPosition> entry : movedPlayersData.entrySet()) {
                UUID uuid = entry.getKey();
                PlayerTracker.PlayerPosition pos = entry.getValue();
                if (uuid == null || pos == null || uuid.equals(viewerId)) continue;

                CachedPlayerData cached = playerDataCache.get(uuid);
                if (cached != null && cached.isValid(now)) {
                    if (cached.shouldHide) continue;
                    double distance = DistanceUtils.calculateDistance(vx, vy, vz, cached.x, cached.y, cached.z);
                    if (distance > ServerConfig.getInstance().maxRelevantDistance()) {
                        viewerDistCache.remove(uuid);
                        continue;
                    }
                    viewerDistCache.put(uuid, distance);
                    positions.add(new PositionUpdatePayload.PositionData(uuid, pos.x, pos.y, pos.z, distance));
                    continue;
                }

                class_3222 targetPlayer = server.method_3760().method_14602(uuid);
                if (targetPlayer == null) continue;
                if (viewer.method_37908() != targetPlayer.method_37908()) continue;

                boolean shouldHide = shouldHideTarget(targetPlayer);
                String playerName = targetPlayer.method_5477().getString();
                playerDataCache.put(uuid, new CachedPlayerData(pos.x, pos.y, pos.z, shouldHide, playerName));

                if (shouldHide) continue;
                if (targetPlayer.method_7325()) continue;

                double distance = DistanceUtils.calculateDistance(vx, vy, vz, pos.x, pos.y, pos.z);
                if (distance > ServerConfig.getInstance().maxRelevantDistance()) {
                    viewerDistCache.remove(uuid);
                    continue;
                }
                viewerDistCache.put(uuid, distance);
                playerTracker.updatePlayer(targetPlayer);
                positions.add(new PositionUpdatePayload.PositionData(uuid, pos.x, pos.y, pos.z, distance));
            }

            if (!newPlayerInfos.isEmpty() || !positions.isEmpty() || !disconnectedPlayers.isEmpty()) {
                PositionUpdatePayload payload = new PositionUpdatePayload(newPlayerInfos, positions, new ArrayList<>(disconnectedPlayers)
                );

                if (ServerPlayNetworking.canSend(viewer, PositionUpdatePayload.ID)) {
                    ServerPlayNetworking.send(viewer, payload);
                }
            }
        }

        disconnectedPlayers.clear();
    }

    private static class CachedPlayerData {
        final double x, y, z;
        final boolean shouldHide;
        final String playerName;
        final long timestamp;

        CachedPlayerData(double x, double y, double z, boolean shouldHide, String playerName) {
            this.x = x;
            this.y = y;
            this.z = z;
            this.shouldHide = shouldHide;
            this.playerName = playerName;
            this.timestamp = System.currentTimeMillis();
        }

        boolean isValid(long now) {
            return (now - timestamp) < 50L;
        }
    }

    private boolean shouldHideTarget(class_3222 target) {
        class_1799 headStack = target.method_6118(class_1304.field_6169);
        return target.method_5715() || target.method_5767() || (!headStack.method_7960() && !(headStack.method_7909() instanceof class_1738));
    }
}
