package co.secretonline.clientsidepaintingvariants;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import net.minecraft.class_1535;
import net.minecraft.class_2960;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;

public class PaintingsInfo {
	private static Logger LOGGER = ClientSidePaintingVariants.LOGGER;

	private static PaintingsInfo INSTANCE = null;

	public static PaintingsInfo getInstance() {
		if (INSTANCE == null) {
			INSTANCE = new PaintingsInfo();
		}

		return INSTANCE;
	}

	private static String toKey(int a, int b) {
		return a + "x" + b;
	}

	private Map<class_2960, class_1535> registryPaintings;
	private Map<class_2960, class_1535> resourcePaintings;

	private Map<String, PaintingsForSize> resolvedPaintingsMap = new HashMap<>();

	private PaintingsInfo() {
	}

	public void setRegistryPaintings(Map<class_2960, class_1535> paintings) {
		registryPaintings = paintings;
		this.resolvePaintings();
	}

	public void setResourcePaintings(Map<class_2960, class_1535> paintings) {
		resourcePaintings = paintings;
		this.resolvePaintings();
	}

	private boolean resolvePaintings() {
		if (registryPaintings == null && resourcePaintings == null) {
			LOGGER.warn("Waiting for world data and resources to be loaded");
			return false;
		}
		if (registryPaintings != null && resourcePaintings == null) {
			LOGGER.warn("Waiting for resources to be loaded");
			return false;
		}
		if (registryPaintings == null && resourcePaintings != null) {
			LOGGER.warn("Waiting for world data to be loaded");
			return false;
		}

		resolvedPaintingsMap.clear();

		// Add all paintings in the registry
		registryPaintings.forEach((id, variant) -> {
			String key = toKey(variant.comp_2670(), variant.comp_2671());
			if (!resolvedPaintingsMap.containsKey(key)) {
				resolvedPaintingsMap.put(key, new PaintingsForSize());
			}

			PaintingsForSize forSize = resolvedPaintingsMap.get(key);
			forSize.registryPaintings.put(id, variant);
		});

		// Add all paintings defined in resource packs, unless they conflict with a
		// registry entry
		resourcePaintings.forEach((id, variant) -> {
			String key = toKey(variant.comp_2670(), variant.comp_2671());
			if (!resolvedPaintingsMap.containsKey(key)) {
				resolvedPaintingsMap.put(key, new PaintingsForSize());
			}

			PaintingsForSize forSize = resolvedPaintingsMap.get(key);
			if (forSize.registryPaintings.containsKey(id)) {
				LOGGER.warn("Resource painting " + id.toString() + " is already defined in the registry");
				return;
			}

			forSize.resourcePaintings.put(id, variant);
		});

		LOGGER.info(this.getSummaryString());

		return true;
	}

	public String getSummaryString() {
		int numPaintings = 0;
		var sizeSummaries = new ArrayList<String>(resolvedPaintingsMap.size());

		for (var entry : resolvedPaintingsMap.entrySet()) {
			var key = entry.getKey();
			var paintings = entry.getValue();

			numPaintings += paintings.registryPaintings.size() + paintings.resourcePaintings.size();
			sizeSummaries.add(new StringBuilder()
					.append(key)
					.append(" (")
					.append(paintings.registryPaintings.size())
					.append("+")
					.append(paintings.resourcePaintings.size())
					.append(")")
					.toString());
		}

		var sb = new StringBuilder()
				.append(numPaintings)
				.append(" paintings for ")
				.append(resolvedPaintingsMap.size())
				.append(" sizes: ")
				.append(String.join(", ", sizeSummaries));
		return sb.toString();
	}

	@Nullable
	public List<class_1535> getRegistryPaintingsForSize(int width, int height) {
		PaintingsForSize forSize = resolvedPaintingsMap.get(toKey(width, height));
		if (forSize == null) {
			return null;
		}

		return forSize.getRegistryPaintings();
	}

	@Nullable
	public List<class_1535> getResourcePaintingsForSize(int width, int height) {
		PaintingsForSize forSize = resolvedPaintingsMap.get(toKey(width, height));
		if (forSize == null) {
			return null;
		}

		return forSize.getResourcePaintings();
	}

	static public class PaintingsForSize {
		private Map<class_2960, class_1535> registryPaintings = new HashMap<>();
		private Map<class_2960, class_1535> resourcePaintings = new HashMap<>();

		public List<class_1535> getRegistryPaintings() {
			return List.copyOf(registryPaintings.values());
		}

		public List<class_1535> getResourcePaintings() {
			return List.copyOf(resourcePaintings.values());
		}
	}
}
