/*
 * Decompiled with CFR 0.152.
 */
package com.example.chatmoderator.services;

import com.example.chatmoderator.ChatModeratorPlugin;
import com.example.chatmoderator.config.ConfigManager;
import com.example.chatmoderator.libs.gson.Gson;
import com.example.chatmoderator.libs.gson.JsonElement;
import com.example.chatmoderator.libs.gson.JsonObject;
import com.example.chatmoderator.libs.gson.JsonParser;
import com.example.chatmoderator.libs.okhttp3.MediaType;
import com.example.chatmoderator.libs.okhttp3.OkHttpClient;
import com.example.chatmoderator.libs.okhttp3.Request;
import com.example.chatmoderator.libs.okhttp3.RequestBody;
import com.example.chatmoderator.libs.okhttp3.Response;
import com.example.chatmoderator.utils.SchedulerUtil;
import java.io.IOException;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import org.bukkit.Bukkit;
import org.bukkit.entity.Player;

public class ModerationService {
    private final ChatModeratorPlugin plugin;
    private final ConfigManager configManager;
    private final OkHttpClient httpClient;
    private final Gson gson;
    private static final String OPENAI_MODERATION_URL = "https://api.openai.com/v1/moderations";
    private static final int MAX_REQUESTS_PER_MINUTE = 20;
    private final Map<String, RateLimiter> rateLimiters = new ConcurrentHashMap<String, RateLimiter>();

    public ModerationService(ChatModeratorPlugin plugin) {
        this.plugin = plugin;
        this.configManager = plugin.getConfigManager();
        this.httpClient = new OkHttpClient.Builder().retryOnConnectionFailure(true).build();
        this.gson = new Gson();
    }

    public void moderateAfterSend(Player sender, String message) {
        SchedulerUtil.runAsync(() -> {
            block4: {
                try {
                    ModerationResult aiResult;
                    ModerationResult wordFilterResult = this.checkWordFilter(message);
                    if (wordFilterResult.isBlocked()) {
                        this.hideBlockedMessage(sender, message, wordFilterResult);
                        return;
                    }
                    if (this.configManager.isAIModerationEnabled() && (aiResult = this.checkAIModerationWithRetries(message, 0)).isBlocked()) {
                        this.hideBlockedMessage(sender, message, aiResult);
                    }
                }
                catch (Exception e) {
                    this.plugin.getLogger().severe("Error during post-send moderation: " + e.getMessage());
                    if (!this.configManager.isDebugEnabled()) break block4;
                    e.printStackTrace();
                }
            }
        });
    }

    public CompletableFuture<ModerationResult> moderateMessageTest(String message) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                ModerationResult wordResult = this.checkWordFilter(message);
                if (wordResult.isBlocked()) {
                    return wordResult;
                }
                if (this.configManager.isAIModerationEnabled()) {
                    return this.checkAIModerationWithRetries(message, 0);
                }
                return new ModerationResult(false, "No violations detected", ModerationResult.ViolationType.NONE);
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Error during test moderation: " + e.getMessage());
                return new ModerationResult(false, "Moderation error", ModerationResult.ViolationType.ERROR);
            }
        });
    }

    public void hideBlockedMessage(Player sender, String message, ModerationResult result) {
        for (Player player : Bukkit.getOnlinePlayers()) {
            if (!player.isOp() && !player.hasPermission("chatmoderator.bypass")) continue;
            player.sendMessage("[MODERATION] " + sender.getName() + " tried to send: " + message + " | Reason: " + result.getReason());
        }
    }

    public ModerationResult checkWordFilter(String message) {
        List<String> blockedWords = this.configManager.getBlockedWords();
        String lowerMessage = message.toLowerCase();
        for (String word : blockedWords) {
            if (!lowerMessage.contains(word.toLowerCase())) continue;
            return new ModerationResult(true, "Contains blocked word: " + word, ModerationResult.ViolationType.WORD_FILTER);
        }
        return new ModerationResult(false, "No blocked words found", ModerationResult.ViolationType.NONE);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public ModerationResult checkAIModerationWithRetries(String message, int retryCount) throws Exception {
        String apiKey = this.configManager.getOpenAIApiKey();
        if (apiKey == null) return new ModerationResult(false, "AI moderation not configured", ModerationResult.ViolationType.NONE);
        if (apiKey.isBlank()) return new ModerationResult(false, "AI moderation not configured", ModerationResult.ViolationType.NONE);
        if (apiKey.equals("your-openai-api-key-here")) {
            return new ModerationResult(false, "AI moderation not configured", ModerationResult.ViolationType.NONE);
        }
        RateLimiter limiter = this.rateLimiters.computeIfAbsent(apiKey, k -> new RateLimiter(20));
        if (!limiter.tryAcquire()) {
            return new ModerationResult(false, "Rate limit exceeded", ModerationResult.ViolationType.ERROR);
        }
        JsonObject requestBody = new JsonObject();
        requestBody.addProperty("model", this.configManager.getOpenAIModel());
        requestBody.addProperty("input", message);
        RequestBody body = RequestBody.create(requestBody.toString(), MediaType.get("application/json"));
        Request request = new Request.Builder().url(OPENAI_MODERATION_URL).addHeader("Authorization", "Bearer " + apiKey).post(body).build();
        try (Response response = this.httpClient.newCall(request).execute();){
            if (response.code() == 429) {
                if (retryCount >= 5) {
                    ModerationResult backoffMillis = new ModerationResult(false, "AI moderation rate limit exceeded", ModerationResult.ViolationType.ERROR);
                    return backoffMillis;
                }
                long backoffMillis = (long)Math.pow(2.0, retryCount) * 500L;
                if (this.configManager.isDebugEnabled()) {
                    this.plugin.getLogger().info("Rate limited by AI moderation API. Retrying in " + backoffMillis + "ms");
                }
                Thread.sleep(backoffMillis);
                ModerationResult moderationResult = this.checkAIModerationWithRetries(message, retryCount + 1);
                return moderationResult;
            }
            if (!response.isSuccessful()) {
                this.plugin.getLogger().warning("AI moderation API failed: " + response.code() + " " + response.message());
                ModerationResult backoffMillis = new ModerationResult(false, "AI moderation request failed", ModerationResult.ViolationType.ERROR);
                return backoffMillis;
            }
            String responseBody = response.body().string();
            JsonObject jsonResponse = JsonParser.parseString(responseBody).getAsJsonObject();
            JsonObject result = jsonResponse.getAsJsonArray("results").get(0).getAsJsonObject();
            boolean flagged = result.get("flagged").getAsBoolean();
            if (!flagged) {
                ModerationResult moderationResult = new ModerationResult(false, "AI moderation passed", ModerationResult.ViolationType.NONE);
                return moderationResult;
            }
            JsonObject categories = result.getAsJsonObject("categories");
            JsonObject categoryScores = result.getAsJsonObject("category_scores");
            Map<String, Double> thresholds = this.configManager.getModerationThresholds();
            for (Map.Entry<String, JsonElement> entry : categories.entrySet()) {
                double threshold;
                double score;
                String category = entry.getKey();
                boolean categoryFlagged = entry.getValue().getAsBoolean();
                if (!categoryFlagged || !((score = categoryScores.get(category).getAsDouble()) >= (threshold = thresholds.getOrDefault(category, 0.5).doubleValue()))) continue;
                ModerationResult moderationResult = new ModerationResult(true, "AI detected violation: " + category + " (score: " + score + ")", ModerationResult.ViolationType.AI_DETECTED);
                return moderationResult;
            }
            ModerationResult moderationResult = new ModerationResult(false, "AI moderation passed", ModerationResult.ViolationType.NONE);
            return moderationResult;
        }
        catch (IOException e) {
            this.plugin.getLogger().severe("Error parsing AI moderation response: " + e.getMessage());
            if (!this.configManager.isDebugEnabled()) return new ModerationResult(false, "AI moderation error", ModerationResult.ViolationType.ERROR);
            e.printStackTrace();
            return new ModerationResult(false, "AI moderation error", ModerationResult.ViolationType.ERROR);
        }
    }

    public void updateConfiguration() {
        if (this.configManager.isDebugEnabled()) {
            this.plugin.getLogger().info("Moderation service configuration updated");
        }
    }

    public static class ModerationResult {
        private final boolean blocked;
        private final String reason;
        private final ViolationType type;

        public ModerationResult(boolean blocked, String reason, ViolationType type) {
            this.blocked = blocked;
            this.reason = reason;
            this.type = type;
        }

        public boolean isBlocked() {
            return this.blocked;
        }

        public String getReason() {
            return this.reason;
        }

        public ViolationType getType() {
            return this.type;
        }

        public static enum ViolationType {
            NONE,
            WORD_FILTER,
            AI_DETECTED,
            ERROR;

        }
    }

    private static class RateLimiter {
        private final int maxRequestsPerMinute;
        private int availableTokens;
        private long lastRefillTimestamp;

        public RateLimiter(int maxRequestsPerMinute) {
            this.maxRequestsPerMinute = maxRequestsPerMinute;
            this.availableTokens = maxRequestsPerMinute;
            this.lastRefillTimestamp = Instant.now().toEpochMilli();
        }

        public synchronized boolean tryAcquire() {
            this.refill();
            if (this.availableTokens > 0) {
                --this.availableTokens;
                return true;
            }
            return false;
        }

        private void refill() {
            long now = Instant.now().toEpochMilli();
            long millisSinceLast = now - this.lastRefillTimestamp;
            long tokensToAdd = millisSinceLast * (long)this.maxRequestsPerMinute / 60000L;
            if (tokensToAdd > 0L) {
                this.availableTokens = Math.min(this.maxRequestsPerMinute, this.availableTokens + (int)tokensToAdd);
                this.lastRefillTimestamp = now;
            }
        }
    }
}

