/*
 * Decompiled with CFR 0.152.
 */
package com.solegendary.reignofnether.bot.ml;

import com.solegendary.reignofnether.ReignOfNether;
import com.solegendary.reignofnether.bot.ml.AdaptiveDifficultyManager;
import com.solegendary.reignofnether.bot.ml.GameState;
import com.solegendary.reignofnether.bot.ml.GameStateVectorizer;
import com.solegendary.reignofnether.bot.ml.MLDecision;
import com.solegendary.reignofnether.bot.ml.NeuralNetworkManager;
import com.solegendary.reignofnether.bot.ml.TrainingDataManager;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.nd4j.linalg.api.ndarray.INDArray;
import org.nd4j.linalg.dataset.DataSet;
import org.nd4j.linalg.factory.Nd4j;

public class OnlineLearningManager {
    private NeuralNetworkManager networkManager;
    private TrainingDataManager dataManager;
    private AdaptiveDifficultyManager difficultyManager;
    private static final int MAX_ONLINE_BATCH_SIZE = 16;
    private static final double ONLINE_LEARNING_RATE_SCALE = 0.3;
    private static final long ONLINE_UPDATE_INTERVAL_MS = 60000L;
    private static final int MIN_SAMPLES_FOR_UPDATE = 8;
    private final Map<String, ModelVersion> currentModelVersions = new ConcurrentHashMap<String, ModelVersion>();
    private final AtomicInteger globalVersionCounter = new AtomicInteger(1);
    private final Map<String, Queue<OnlineLearningExample>> onlineUpdateQueues = new ConcurrentHashMap<String, Queue<OnlineLearningExample>>();
    private final ExecutorService onlineUpdateExecutor = Executors.newCachedThreadPool(r -> {
        Thread t = new Thread(r, "ML-Online-Learning");
        t.setDaemon(true);
        return t;
    });
    private final AtomicBoolean onlineLearningEnabled = new AtomicBoolean(true);
    private final AtomicBoolean shutdownRequested = new AtomicBoolean(false);
    private final Map<String, OnlineLearningStats> learningStats = new ConcurrentHashMap<String, OnlineLearningStats>();

    public void initialize(NeuralNetworkManager networkManager, TrainingDataManager dataManager, AdaptiveDifficultyManager difficultyManager) {
        this.networkManager = networkManager;
        this.dataManager = dataManager;
        this.difficultyManager = difficultyManager;
        this.startPeriodicUpdates();
        ReignOfNether.LOGGER.info("Online Learning Manager initialized and ready");
    }

    public void queueOnlineLearning(String botName, GameState gameState, MLDecision decision, double actualOutcome, String outcomeContext) {
        if (this.shutdownRequested.get() || !this.onlineLearningEnabled.get()) {
            return;
        }
        OnlineLearningExample example = new OnlineLearningExample(gameState, decision, actualOutcome, outcomeContext, System.currentTimeMillis());
        Queue queue = this.onlineUpdateQueues.computeIfAbsent(botName, k -> new ConcurrentLinkedQueue());
        queue.offer(example);
        OnlineLearningStats stats = this.learningStats.computeIfAbsent(botName, k -> new OnlineLearningStats());
        stats.examplesQueued.incrementAndGet();
        if (queue.size() >= 8) {
            this.triggerOnlineUpdate(botName);
        }
        ReignOfNether.LOGGER.debug("Queued online learning example for bot '{}' (queue size: {})", (Object)botName, (Object)queue.size());
    }

    public void triggerOnlineUpdate(String botName) {
        if (this.shutdownRequested.get() || !this.onlineLearningEnabled.get()) {
            return;
        }
        CompletableFuture.runAsync(() -> this.performOnlineUpdate(botName), this.onlineUpdateExecutor).whenComplete((result, throwable) -> {
            if (throwable != null) {
                ReignOfNether.LOGGER.error("Online learning update failed for bot '{}': {}", (Object)botName, (Object)throwable.getMessage());
                OnlineLearningStats stats = this.learningStats.get(botName);
                if (stats != null) {
                    stats.updateFailures.incrementAndGet();
                }
            }
        });
    }

    private void performOnlineUpdate(String botName) {
        try {
            OnlineLearningExample example;
            Queue<OnlineLearningExample> queue = this.onlineUpdateQueues.get(botName);
            if (queue == null || queue.isEmpty()) {
                return;
            }
            int batchSize = Math.min(queue.size(), 16);
            if (batchSize < 8) {
                return;
            }
            ReignOfNether.LOGGER.debug("Performing online learning update for bot '{}' with {} examples", (Object)botName, (Object)batchSize);
            double[][] inputs = new double[batchSize][];
            double[][] outputs = new double[batchSize][];
            GameStateVectorizer vectorizer = new GameStateVectorizer();
            for (int i = 0; i < batchSize && (example = queue.poll()) != null; ++i) {
                inputs[i] = vectorizer.vectorizeGameState(example.gameState);
                outputs[i] = this.createLearningTarget(example.decision, example.actualOutcome);
            }
            INDArray inputArray = Nd4j.create((double[][])inputs);
            INDArray outputArray = Nd4j.create((double[][])outputs);
            DataSet onlineDataSet = new DataSet(inputArray, outputArray);
            double adaptiveLearningRate = this.difficultyManager.getAdaptiveLearningRate(botName);
            double onlineLearningRate = adaptiveLearningRate * 0.3;
            this.networkManager.performOnlineTraining(botName, onlineDataSet, onlineLearningRate);
            this.updateModelVersion(botName);
            OnlineLearningStats stats = this.learningStats.get(botName);
            if (stats != null) {
                stats.onlineUpdates.incrementAndGet();
                stats.examplesProcessed.addAndGet(batchSize);
                stats.lastUpdateTime.set(System.currentTimeMillis());
            }
            ReignOfNether.LOGGER.debug("Online learning update completed for bot '{}' (learning rate: {:.4f})", (Object)botName, (Object)onlineLearningRate);
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Error during online learning update for bot '{}': {}", (Object)botName, (Object)e.getMessage());
        }
    }

    private double[] createLearningTarget(MLDecision decision, double actualOutcome) {
        int decisionTypeCount = MLDecision.DecisionType.values().length;
        double[] target = new double[decisionTypeCount];
        int decisionIndex = decision.getDecisionType().ordinal();
        double adjustedConfidence = decision.getConfidence() * actualOutcome * 2.0;
        target[decisionIndex] = adjustedConfidence = Math.max(0.0, Math.min(1.0, adjustedConfidence));
        return target;
    }

    private void updateModelVersion(String botName) {
        int newVersion = this.globalVersionCounter.getAndIncrement();
        long timestamp = System.currentTimeMillis();
        ModelVersion version = new ModelVersion(newVersion, timestamp, "Online learning update");
        this.currentModelVersions.put(botName, version);
        ReignOfNether.LOGGER.debug("Updated model version for bot '{}' to version {}", (Object)botName, (Object)newVersion);
    }

    private void startPeriodicUpdates() {
        ScheduledExecutorService scheduler = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = new Thread(r, "ML-Periodic-Online-Learning");
            t.setDaemon(true);
            return t;
        });
        scheduler.scheduleAtFixedRate(this::performPeriodicUpdates, 60000L, 60000L, TimeUnit.MILLISECONDS);
    }

    private void performPeriodicUpdates() {
        if (this.shutdownRequested.get() || !this.onlineLearningEnabled.get()) {
            return;
        }
        for (String botName : this.onlineUpdateQueues.keySet()) {
            Queue<OnlineLearningExample> queue = this.onlineUpdateQueues.get(botName);
            if (queue == null || queue.size() < 8) continue;
            this.triggerOnlineUpdate(botName);
        }
    }

    public ModelVersion getCurrentModelVersion(String botName) {
        return this.currentModelVersions.get(botName);
    }

    public void setOnlineLearningEnabled(boolean enabled) {
        this.onlineLearningEnabled.set(enabled);
        ReignOfNether.LOGGER.info("Online learning {}", (Object)(enabled ? "enabled" : "disabled"));
    }

    public boolean isOnlineLearningEnabled() {
        return this.onlineLearningEnabled.get();
    }

    public OnlineLearningStats getStats(String botName) {
        return this.learningStats.get(botName);
    }

    public String getComprehensiveStats() {
        StringBuilder sb = new StringBuilder("Online Learning Statistics:\n");
        for (Map.Entry<String, OnlineLearningStats> entry : this.learningStats.entrySet()) {
            String botName = entry.getKey();
            OnlineLearningStats stats = entry.getValue();
            ModelVersion version = this.currentModelVersions.get(botName);
            sb.append(String.format("Bot '%s': %d updates, %d examples processed, %d failures, version %d\n", botName, stats.onlineUpdates.get(), stats.examplesProcessed.get(), stats.updateFailures.get(), version != null ? version.version : 0));
        }
        return sb.toString();
    }

    public void resetOnlineLearningData(String botName) {
        Queue<OnlineLearningExample> queue = this.onlineUpdateQueues.get(botName);
        if (queue != null) {
            queue.clear();
        }
        this.learningStats.remove(botName);
        this.currentModelVersions.remove(botName);
        ReignOfNether.LOGGER.info("Reset online learning data for bot '{}'", (Object)botName);
    }

    public void shutdown() {
        if (this.shutdownRequested.compareAndSet(false, true)) {
            ReignOfNether.LOGGER.info("Shutting down Online Learning Manager...");
            this.onlineUpdateExecutor.shutdown();
            try {
                if (!this.onlineUpdateExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.onlineUpdateExecutor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.onlineUpdateExecutor.shutdownNow();
                Thread.currentThread().interrupt();
            }
            ReignOfNether.LOGGER.info("Online Learning Manager shutdown complete");
        }
    }

    private static class OnlineLearningExample {
        final GameState gameState;
        final MLDecision decision;
        final double actualOutcome;
        final String outcomeContext;
        final long timestamp;

        OnlineLearningExample(GameState gameState, MLDecision decision, double actualOutcome, String outcomeContext, long timestamp) {
            this.gameState = gameState;
            this.decision = decision;
            this.actualOutcome = actualOutcome;
            this.outcomeContext = outcomeContext;
            this.timestamp = timestamp;
        }
    }

    public static class OnlineLearningStats {
        public final AtomicInteger examplesQueued = new AtomicInteger(0);
        public final AtomicInteger examplesProcessed = new AtomicInteger(0);
        public final AtomicInteger onlineUpdates = new AtomicInteger(0);
        public final AtomicInteger updateFailures = new AtomicInteger(0);
        public final AtomicLong lastUpdateTime = new AtomicLong(0L);

        public double getProcessingRate() {
            int queued = this.examplesQueued.get();
            int processed = this.examplesProcessed.get();
            return queued > 0 ? (double)processed / (double)queued : 0.0;
        }

        public long getTimeSinceLastUpdate() {
            return System.currentTimeMillis() - this.lastUpdateTime.get();
        }
    }

    public static class ModelVersion {
        public final int version;
        public final long timestamp;
        public final String updateReason;

        public ModelVersion(int version, long timestamp, String updateReason) {
            this.version = version;
            this.timestamp = timestamp;
            this.updateReason = updateReason;
        }

        public String toString() {
            return String.format("ModelVersion{version=%d, timestamp=%d, reason='%s'}", this.version, this.timestamp, this.updateReason);
        }
    }
}

