/*
 * Decompiled with CFR 0.152.
 */
package com.solegendary.reignofnether.bot.ml;

import com.solegendary.reignofnether.ReignOfNether;
import com.solegendary.reignofnether.bot.ml.ActionType;
import com.solegendary.reignofnether.bot.ml.GameState;
import com.solegendary.reignofnether.bot.ml.PlayerAction;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

public class PlayerBehaviorAnalyzer {
    private final Map<String, PlayerProfile> playerProfiles = new ConcurrentHashMap<String, PlayerProfile>();
    private final Map<String, Queue<BehaviorEvent>> recentEvents = new ConcurrentHashMap<String, Queue<BehaviorEvent>>();
    private static final int MAX_EVENTS_PER_PLAYER = 1000;
    private static final long BEHAVIOR_WINDOW_MS = 1800000L;
    private static final int MIN_ACTIONS_FOR_ANALYSIS = 20;

    public void recordPlayerAction(String playerName, GameState gameState, PlayerAction action, long timestamp, ActionOutcome outcome) {
        PlayerProfile profile = this.playerProfiles.computeIfAbsent(playerName, k -> new PlayerProfile(playerName));
        BehaviorEvent event = new BehaviorEvent(gameState, action, timestamp, outcome);
        Queue events = this.recentEvents.computeIfAbsent(playerName, k -> new ConcurrentLinkedQueue());
        events.offer(event);
        while (events.size() > 1000) {
            events.poll();
        }
        this.updatePlayerProfile(profile, event);
        ReignOfNether.LOGGER.debug("Recorded behavior event for player '{}': {}", (Object)playerName, (Object)action.getActionType());
    }

    private void updatePlayerProfile(PlayerProfile profile, BehaviorEvent event) {
        profile.totalActions.incrementAndGet();
        profile.lastActionTime.set(event.timestamp);
        ActionType actionType = event.action.getActionType();
        profile.actionFrequency.merge(actionType, 1, Integer::sum);
        long timeSinceLastAction = event.timestamp - profile.lastActionTime.get();
        if (timeSinceLastAction > 0L) {
            profile.averageActionInterval.set((long)((double)profile.averageActionInterval.get() * 0.9 + (double)timeSinceLastAction * 0.1));
        }
        this.updateStrategicPreferences(profile, event);
        if (event.outcome != null) {
            this.updateOutcomePatterns(profile, event);
        }
    }

    private void updateStrategicPreferences(PlayerProfile profile, BehaviorEvent event) {
        GameState state = event.gameState;
        ActionType action = event.action.getActionType();
        if (action == ActionType.GATHER_RESOURCES || action == ActionType.BUILD_ECONOMIC) {
            profile.economicPreference.set(Math.min(1.0, profile.economicPreference.get() + 0.01));
        }
        if (action == ActionType.PRODUCE_MILITARY || action == ActionType.BUILD_MILITARY || action == ActionType.ATTACK) {
            profile.militaryPreference.set(Math.min(1.0, profile.militaryPreference.get() + 0.01));
        }
        if (action == ActionType.EXPAND) {
            profile.expansionPreference.set(Math.min(1.0, profile.expansionPreference.get() + 0.01));
        }
        if (action == ActionType.ATTACK) {
            profile.aggressionLevel.set(Math.min(1.0, profile.aggressionLevel.get() + 0.02));
        } else if (action == ActionType.DEFEND) {
            profile.aggressionLevel.set(Math.max(0.0, profile.aggressionLevel.get() - 0.01));
        }
        double resourceRisk = this.calculateResourceRisk(state, event.action);
        profile.riskTolerance.set(profile.riskTolerance.get() * 0.95 + resourceRisk * 0.05);
    }

    private double calculateResourceRisk(GameState gameState, PlayerAction action) {
        Map<String, Double> params = action.getParameters();
        double totalCommitment = params.values().stream().mapToDouble(Double::doubleValue).sum();
        double totalResources = gameState.selfResources.food + gameState.selfResources.wood + gameState.selfResources.ore;
        return totalResources > 0.0 ? Math.min(1.0, totalCommitment / totalResources) : 0.5;
    }

    private void updateOutcomePatterns(PlayerProfile profile, BehaviorEvent event) {
        ActionType actionType = event.action.getActionType();
        double outcome = event.outcome.successRate;
        ActionOutcomeStats stats = profile.actionOutcomes.computeIfAbsent(actionType, k -> new ActionOutcomeStats());
        ++stats.totalAttempts;
        stats.totalSuccess += outcome;
        stats.averageOutcome = stats.totalSuccess / (double)stats.totalAttempts;
        profile.actionConfidence.merge(actionType, outcome * 0.1, (old, add) -> Math.max(0.0, Math.min(1.0, old + add - 0.05)));
    }

    public BehaviorAnalysis analyzeBehavior(String playerName) {
        PlayerProfile profile = this.playerProfiles.get(playerName);
        Queue<BehaviorEvent> events = this.recentEvents.get(playerName);
        if (profile == null || events == null || events.size() < 20) {
            return null;
        }
        BehaviorPatterns patterns = this.analyzePatterns(events);
        DecisionTiming timing = this.analyzeTiming(events);
        StrategicStyle style = this.analyzeStrategicStyle(profile);
        AdaptabilityMetrics adaptability = this.analyzeAdaptability(events);
        return new BehaviorAnalysis(playerName, patterns, timing, style, adaptability, profile.totalActions.get(), System.currentTimeMillis());
    }

    private BehaviorPatterns analyzePatterns(Queue<BehaviorEvent> events) {
        ConcurrentHashMap<ActionType, Integer> actionCounts = new ConcurrentHashMap<ActionType, Integer>();
        ConcurrentHashMap<String, Integer> sequencePatterns = new ConcurrentHashMap<String, Integer>();
        ArrayList<BehaviorEvent> eventList = new ArrayList<BehaviorEvent>(events);
        for (BehaviorEvent event : eventList) {
            actionCounts.merge(event.action.getActionType(), 1, Integer::sum);
        }
        for (int i = 0; i < eventList.size() - 1; ++i) {
            String sequence = String.valueOf((Object)((BehaviorEvent)eventList.get((int)i)).action.getActionType()) + "->" + String.valueOf((Object)((BehaviorEvent)eventList.get((int)(i + 1))).action.getActionType());
            sequencePatterns.merge(sequence, 1, Integer::sum);
        }
        ActionType mostCommonAction = actionCounts.entrySet().stream().max(Map.Entry.comparingByValue()).map(Map.Entry::getKey).orElse(ActionType.NONE);
        String mostCommonSequence = sequencePatterns.entrySet().stream().max(Map.Entry.comparingByValue()).map(Map.Entry::getKey).orElse("none");
        return new BehaviorPatterns(actionCounts, sequencePatterns, mostCommonAction, mostCommonSequence);
    }

    private DecisionTiming analyzeTiming(Queue<BehaviorEvent> events) {
        ArrayList<Long> intervals = new ArrayList<Long>();
        BehaviorEvent lastEvent = null;
        for (BehaviorEvent event : events) {
            if (lastEvent != null) {
                intervals.add(event.timestamp - lastEvent.timestamp);
            }
            lastEvent = event;
        }
        if (intervals.isEmpty()) {
            return new DecisionTiming(0.0, 0L, 0L, 0.0);
        }
        double avgInterval = intervals.stream().mapToLong(Long::longValue).average().orElse(0.0);
        long minInterval = intervals.stream().mapToLong(Long::longValue).min().orElse(0L);
        long maxInterval = intervals.stream().mapToLong(Long::longValue).max().orElse(0L);
        double variance = intervals.stream().mapToDouble(i -> Math.pow((double)i.longValue() - avgInterval, 2.0)).average().orElse(0.0);
        double consistency = Math.max(0.0, 1.0 - variance / (avgInterval * avgInterval));
        return new DecisionTiming(avgInterval, minInterval, maxInterval, consistency);
    }

    private StrategicStyle analyzeStrategicStyle(PlayerProfile profile) {
        return new StrategicStyle(profile.economicPreference.get(), profile.militaryPreference.get(), profile.expansionPreference.get(), profile.aggressionLevel.get(), profile.riskTolerance.get());
    }

    private AdaptabilityMetrics analyzeAdaptability(Queue<BehaviorEvent> events) {
        ArrayList<BehaviorEvent> eventList = new ArrayList<BehaviorEvent>(events);
        if (eventList.size() < 10) {
            return new AdaptabilityMetrics(0.5, 0.5, 0.5);
        }
        int splitPoint = eventList.size() / 2;
        List<BehaviorEvent> earlyEvents = eventList.subList(0, splitPoint);
        List<BehaviorEvent> lateEvents = eventList.subList(splitPoint, eventList.size());
        double strategyChange = this.calculateStrategyChange(earlyEvents, lateEvents);
        double learningRate = this.calculateLearningRate(eventList);
        double flexibility = this.calculateFlexibility(eventList);
        return new AdaptabilityMetrics(strategyChange, learningRate, flexibility);
    }

    private double calculateStrategyChange(List<BehaviorEvent> early, List<BehaviorEvent> late) {
        Map<ActionType, Double> earlyFreq = this.calculateActionFrequencies(early);
        Map<ActionType, Double> lateFreq = this.calculateActionFrequencies(late);
        double totalChange = 0.0;
        for (ActionType action : ActionType.values()) {
            double earlyRate = earlyFreq.getOrDefault((Object)action, 0.0);
            double lateRate = lateFreq.getOrDefault((Object)action, 0.0);
            totalChange += Math.abs(earlyRate - lateRate);
        }
        return Math.min(1.0, totalChange / (double)ActionType.values().length);
    }

    private Map<ActionType, Double> calculateActionFrequencies(List<BehaviorEvent> events) {
        ConcurrentHashMap<ActionType, Integer> counts = new ConcurrentHashMap<ActionType, Integer>();
        for (BehaviorEvent event : events) {
            counts.merge(event.action.getActionType(), 1, Integer::sum);
        }
        ConcurrentHashMap<ActionType, Double> frequencies = new ConcurrentHashMap<ActionType, Double>();
        int total = events.size();
        for (Map.Entry entry : counts.entrySet()) {
            frequencies.put((ActionType)((Object)entry.getKey()), (double)((Integer)entry.getValue()).intValue() / (double)total);
        }
        return frequencies;
    }

    private double calculateLearningRate(List<BehaviorEvent> events) {
        if (events.size() < 10) {
            return 0.5;
        }
        double earlyOutcome = events.stream().limit(events.size() / 2).filter(e -> e.outcome != null).mapToDouble(e -> e.outcome.successRate).average().orElse(0.5);
        double lateOutcome = events.stream().skip(events.size() / 2).filter(e -> e.outcome != null).mapToDouble(e -> e.outcome.successRate).average().orElse(0.5);
        return Math.max(0.0, Math.min(1.0, lateOutcome - earlyOutcome + 0.5));
    }

    private double calculateFlexibility(List<BehaviorEvent> events) {
        long uniqueActions = events.stream().map(e -> e.action.getActionType()).distinct().count();
        return (double)uniqueActions / (double)ActionType.values().length;
    }

    public PlayerProfile getPlayerProfile(String playerName) {
        return this.playerProfiles.get(playerName);
    }

    public Map<String, BehaviorAnalysis> getAllBehaviorAnalyses() {
        ConcurrentHashMap<String, BehaviorAnalysis> analyses = new ConcurrentHashMap<String, BehaviorAnalysis>();
        for (String playerName : this.playerProfiles.keySet()) {
            BehaviorAnalysis analysis = this.analyzeBehavior(playerName);
            if (analysis == null) continue;
            analyses.put(playerName, analysis);
        }
        return analyses;
    }

    public void cleanupOldData() {
        long cutoffTime = System.currentTimeMillis() - 1800000L;
        for (Queue<BehaviorEvent> events : this.recentEvents.values()) {
            events.removeIf(event -> event.timestamp < cutoffTime);
        }
        ReignOfNether.LOGGER.debug("Cleaned up old behavior analysis data");
    }

    public static class PlayerProfile {
        final String playerName;
        final AtomicInteger totalActions = new AtomicInteger(0);
        final AtomicLong lastActionTime = new AtomicLong(0L);
        final AtomicLong averageActionInterval = new AtomicLong(5000L);
        final AtomicReference<Double> economicPreference = new AtomicReference<Double>(0.5);
        final AtomicReference<Double> militaryPreference = new AtomicReference<Double>(0.5);
        final AtomicReference<Double> expansionPreference = new AtomicReference<Double>(0.5);
        final AtomicReference<Double> aggressionLevel = new AtomicReference<Double>(0.5);
        final AtomicReference<Double> riskTolerance = new AtomicReference<Double>(0.5);
        final Map<ActionType, Integer> actionFrequency = new ConcurrentHashMap<ActionType, Integer>();
        final Map<ActionType, Double> actionConfidence = new ConcurrentHashMap<ActionType, Double>();
        final Map<ActionType, ActionOutcomeStats> actionOutcomes = new ConcurrentHashMap<ActionType, ActionOutcomeStats>();

        PlayerProfile(String playerName) {
            this.playerName = playerName;
        }
    }

    private static class BehaviorEvent {
        final GameState gameState;
        final PlayerAction action;
        final long timestamp;
        final ActionOutcome outcome;

        BehaviorEvent(GameState gameState, PlayerAction action, long timestamp, ActionOutcome outcome) {
            this.gameState = gameState;
            this.action = action;
            this.timestamp = timestamp;
            this.outcome = outcome;
        }
    }

    public static class ActionOutcome {
        public final double successRate;
        public final String context;
        public final long duration;

        public ActionOutcome(double successRate, String context, long duration) {
            this.successRate = Math.max(0.0, Math.min(1.0, successRate));
            this.context = context;
            this.duration = duration;
        }
    }

    private static class ActionOutcomeStats {
        int totalAttempts = 0;
        double totalSuccess = 0.0;
        double averageOutcome = 0.5;

        private ActionOutcomeStats() {
        }
    }

    public static class BehaviorPatterns {
        public final Map<ActionType, Integer> actionFrequencies;
        public final Map<String, Integer> sequencePatterns;
        public final ActionType dominantAction;
        public final String dominantSequence;

        BehaviorPatterns(Map<ActionType, Integer> actionFrequencies, Map<String, Integer> sequencePatterns, ActionType dominantAction, String dominantSequence) {
            this.actionFrequencies = new ConcurrentHashMap<ActionType, Integer>(actionFrequencies);
            this.sequencePatterns = new ConcurrentHashMap<String, Integer>(sequencePatterns);
            this.dominantAction = dominantAction;
            this.dominantSequence = dominantSequence;
        }
    }

    public static class DecisionTiming {
        public final double averageInterval;
        public final long fastestInterval;
        public final long slowestInterval;
        public final double consistency;

        DecisionTiming(double averageInterval, long fastestInterval, long slowestInterval, double consistency) {
            this.averageInterval = averageInterval;
            this.fastestInterval = fastestInterval;
            this.slowestInterval = slowestInterval;
            this.consistency = consistency;
        }
    }

    public static class StrategicStyle {
        public final double economicPreference;
        public final double militaryPreference;
        public final double expansionPreference;
        public final double aggressionLevel;
        public final double riskTolerance;

        StrategicStyle(double economicPreference, double militaryPreference, double expansionPreference, double aggressionLevel, double riskTolerance) {
            this.economicPreference = economicPreference;
            this.militaryPreference = militaryPreference;
            this.expansionPreference = expansionPreference;
            this.aggressionLevel = aggressionLevel;
            this.riskTolerance = riskTolerance;
        }
    }

    public static class AdaptabilityMetrics {
        public final double strategyChangeRate;
        public final double learningRate;
        public final double flexibility;

        AdaptabilityMetrics(double strategyChangeRate, double learningRate, double flexibility) {
            this.strategyChangeRate = strategyChangeRate;
            this.learningRate = learningRate;
            this.flexibility = flexibility;
        }
    }

    public static class BehaviorAnalysis {
        public final String playerName;
        public final BehaviorPatterns patterns;
        public final DecisionTiming timing;
        public final StrategicStyle style;
        public final AdaptabilityMetrics adaptability;
        public final int totalActions;
        public final long analysisTimestamp;

        BehaviorAnalysis(String playerName, BehaviorPatterns patterns, DecisionTiming timing, StrategicStyle style, AdaptabilityMetrics adaptability, int totalActions, long analysisTimestamp) {
            this.playerName = playerName;
            this.patterns = patterns;
            this.timing = timing;
            this.style = style;
            this.adaptability = adaptability;
            this.totalActions = totalActions;
            this.analysisTimestamp = analysisTimestamp;
        }

        public String toString() {
            return String.format("BehaviorAnalysis{player='%s', actions=%d, style=%.2f/%.2f/%.2f}", this.playerName, this.totalActions, this.style.economicPreference, this.style.militaryPreference, this.style.aggressionLevel);
        }
    }
}

