/*
 * Decompiled with CFR 0.152.
 */
package com.solegendary.reignofnether.bot.ml;

import com.solegendary.reignofnether.ReignOfNether;
import com.solegendary.reignofnether.bot.ml.ActionType;
import com.solegendary.reignofnether.bot.ml.GameMetrics;
import com.solegendary.reignofnether.bot.ml.GameState;
import com.solegendary.reignofnether.bot.ml.GameStateVectorizer;
import com.solegendary.reignofnether.bot.ml.MLDecision;
import com.solegendary.reignofnether.bot.ml.PlayerAction;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import org.nd4j.linalg.api.ndarray.INDArray;
import org.nd4j.linalg.dataset.DataSet;
import org.nd4j.linalg.factory.Nd4j;

public class TrainingDataManager {
    private final Map<String, Queue<TrainingExample>> botTrainingData = new ConcurrentHashMap<String, Queue<TrainingExample>>();
    private final Queue<HumanActionExample> humanActionData = new ConcurrentLinkedQueue<HumanActionExample>();
    private final Map<String, GameOutcome> gameOutcomes = new ConcurrentHashMap<String, GameOutcome>();
    private static final int MAX_EXAMPLES_PER_BOT = 10000;
    private static final int MAX_HUMAN_EXAMPLES = 5000;
    private static final int MIN_EXAMPLES_FOR_TRAINING = 100;
    private static final int TRAINING_TRIGGER_THRESHOLD = 500;
    private final String dataDirectory = "config/reignofnether/ml_data/";
    private boolean initialized = false;
    private int totalDecisionsRecorded = 0;
    private int totalGamesAnalyzed = 0;
    private final Map<String, Integer> botExampleCounts = new ConcurrentHashMap<String, Integer>();

    public void initialize() {
        try {
            ReignOfNether.LOGGER.info("Initializing Training Data Manager...");
            File dataDir = new File("config/reignofnether/ml_data/");
            if (!dataDir.exists()) {
                dataDir.mkdirs();
            }
            this.loadExistingData();
            this.initialized = true;
            ReignOfNether.LOGGER.info("Training Data Manager initialized with {} bot datasets, {} human examples", (Object)this.botTrainingData.size(), (Object)this.humanActionData.size());
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Failed to initialize Training Data Manager: {}", (Object)e.getMessage());
            throw new RuntimeException("Training data manager initialization failed", e);
        }
    }

    public void recordDecision(GameState gameState, MLDecision decision, String botName) {
        if (!this.initialized) {
            return;
        }
        try {
            Queue botData = this.botTrainingData.computeIfAbsent(botName, k -> new ConcurrentLinkedQueue());
            TrainingExample example = new TrainingExample(gameState, decision, System.currentTimeMillis());
            botData.offer(example);
            if (botData.size() > 10000) {
                botData.poll();
            }
            ++this.totalDecisionsRecorded;
            this.botExampleCounts.put(botName, botData.size());
            ReignOfNether.LOGGER.debug("Recorded decision for bot '{}': {} (total: {})", (Object)botName, (Object)decision.getDecisionType(), (Object)botData.size());
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Error recording decision for bot '{}': {}", (Object)botName, (Object)e.getMessage());
        }
    }

    public void recordGameOutcome(String botName, boolean won, GameMetrics metrics) {
        if (!this.initialized) {
            return;
        }
        try {
            GameOutcome outcome = new GameOutcome(botName, won, metrics, System.currentTimeMillis());
            this.gameOutcomes.put(botName + "_" + outcome.timestamp, outcome);
            ++this.totalGamesAnalyzed;
            Queue<TrainingExample> botData = this.botTrainingData.get(botName);
            if (botData != null) {
                this.updateDecisionRewards(botData, won, metrics);
            }
            ReignOfNether.LOGGER.info("Recorded game outcome for bot '{}': {} (score: {})", (Object)botName, (Object)(won ? "WIN" : "LOSS"), (Object)metrics.finalScore);
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Error recording game outcome for bot '{}': {}", (Object)botName, (Object)e.getMessage());
        }
    }

    public void recordHumanAction(GameState gameState, PlayerAction action, String playerName) {
        if (!this.initialized) {
            return;
        }
        try {
            HumanActionExample example = new HumanActionExample(gameState, action, System.currentTimeMillis());
            this.humanActionData.offer(example);
            if (this.humanActionData.size() > 5000) {
                this.humanActionData.poll();
            }
            ReignOfNether.LOGGER.debug("Recorded human action from '{}': {} (total: {})", (Object)playerName, (Object)action.getActionType(), (Object)this.humanActionData.size());
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Error recording human action for player '{}': {}", (Object)playerName, (Object)e.getMessage());
        }
    }

    public DataSet getTrainingDataSet(String botName) {
        Queue<TrainingExample> botData = this.botTrainingData.get(botName);
        if (botData == null || botData.size() < 100) {
            return null;
        }
        try {
            GameStateVectorizer vectorizer = new GameStateVectorizer();
            ArrayList<TrainingExample> examples = new ArrayList<TrainingExample>(botData);
            double[][] inputs = new double[examples.size()][64];
            double[][] outputs = new double[examples.size()][MLDecision.DecisionType.values().length];
            for (int i = 0; i < examples.size(); ++i) {
                TrainingExample example = (TrainingExample)examples.get(i);
                inputs[i] = vectorizer.vectorizeGameState(example.gameState);
                Arrays.fill(outputs[i], 0.0);
                int decisionIndex = example.decision.getDecisionType().ordinal();
                outputs[i][decisionIndex] = 1.0;
                if (example.reward == 0.0) continue;
                double weight = example.reward > 0.0 ? 1.2 : 0.8;
                int j = 0;
                while (j < outputs[i].length) {
                    double[] dArray = outputs[i];
                    int n = j++;
                    dArray[n] = dArray[n] * weight;
                }
            }
            INDArray inputArray = Nd4j.create((double[][])inputs);
            INDArray outputArray = Nd4j.create((double[][])outputs);
            return new DataSet(inputArray, outputArray);
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Error creating training dataset for bot '{}': {}", (Object)botName, (Object)e.getMessage());
            return null;
        }
    }

    public DataSet getImitationDataSet() {
        if (this.humanActionData.size() < 100) {
            return null;
        }
        try {
            GameStateVectorizer vectorizer = new GameStateVectorizer();
            ArrayList<HumanActionExample> examples = new ArrayList<HumanActionExample>(this.humanActionData);
            double[][] inputs = new double[examples.size()][64];
            double[][] outputs = new double[examples.size()][MLDecision.DecisionType.values().length];
            for (int i = 0; i < examples.size(); ++i) {
                HumanActionExample example = (HumanActionExample)examples.get(i);
                inputs[i] = vectorizer.vectorizeGameState(example.gameState);
                MLDecision.DecisionType decisionType = this.convertActionToDecision(example.humanAction.getActionType());
                Arrays.fill(outputs[i], 0.0);
                outputs[i][decisionType.ordinal()] = 1.0;
            }
            INDArray inputArray = Nd4j.create((double[][])inputs);
            INDArray outputArray = Nd4j.create((double[][])outputs);
            return new DataSet(inputArray, outputArray);
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Error creating imitation dataset: {}", (Object)e.getMessage());
            return null;
        }
    }

    public boolean shouldTriggerLearning() {
        for (Queue<TrainingExample> botData : this.botTrainingData.values()) {
            if (botData.size() < 500) continue;
            return true;
        }
        return this.humanActionData.size() >= 500;
    }

    private void updateDecisionRewards(Queue<TrainingExample> botData, boolean won, GameMetrics metrics) {
        long gameEndTime = System.currentTimeMillis();
        double baseReward = won ? 1.0 : -0.5;
        for (TrainingExample example : botData) {
            long decisionAge = gameEndTime - example.timestamp;
            double decayFactor = Math.exp((double)(-decisionAge) / 300000.0);
            example.reward = baseReward * decayFactor;
            if (!won) continue;
            if (metrics.economicEfficiency > 0.8 && example.decision.getDecisionType().name().contains("ECONOMY")) {
                example.reward *= 1.2;
            }
            if (!(metrics.militaryEfficiency > 0.8) || !example.decision.getDecisionType().name().contains("MILITARY")) continue;
            example.reward *= 1.2;
        }
    }

    private MLDecision.DecisionType convertActionToDecision(ActionType actionType) {
        return switch (actionType) {
            case ActionType.GATHER_RESOURCES -> MLDecision.DecisionType.FOCUS_ON_ECONOMY;
            case ActionType.BUILD_ECONOMIC -> MLDecision.DecisionType.BUILD_ECONOMIC_BUILDINGS;
            case ActionType.BUILD_MILITARY -> MLDecision.DecisionType.BUILD_MILITARY_BUILDINGS;
            case ActionType.PRODUCE_WORKERS -> MLDecision.DecisionType.PRODUCE_WORKERS;
            case ActionType.PRODUCE_MILITARY -> MLDecision.DecisionType.PRODUCE_MILITARY_UNITS;
            case ActionType.ATTACK -> MLDecision.DecisionType.ATTACK_ENEMY;
            case ActionType.DEFEND -> MLDecision.DecisionType.DEFEND_BASE;
            case ActionType.EXPAND -> MLDecision.DecisionType.EXPAND_TERRITORY;
            case ActionType.RESEARCH -> MLDecision.DecisionType.RESEARCH_TECHNOLOGY;
            default -> MLDecision.DecisionType.NO_DECISION;
        };
    }

    public void saveData() {
        if (!this.initialized) {
            return;
        }
        try {
            ReignOfNether.LOGGER.info("Saving training data...");
            for (Map.Entry<String, Queue<TrainingExample>> entry : this.botTrainingData.entrySet()) {
                String botName = entry.getKey();
                String filename = "config/reignofnether/ml_data/bot_" + this.sanitizeFilename(botName) + "_training.dat";
                this.saveTrainingExamples(entry.getValue(), filename);
            }
            String humanFilename = "config/reignofnether/ml_data/human_actions.dat";
            this.saveHumanExamples(this.humanActionData, humanFilename);
            ReignOfNether.LOGGER.info("Training data saved successfully");
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Error saving training data: {}", (Object)e.getMessage());
        }
    }

    private void loadExistingData() {
        File humanFile;
        File dataDir = new File("config/reignofnether/ml_data/");
        if (!dataDir.exists() || !dataDir.isDirectory()) {
            return;
        }
        File[] dataFiles = dataDir.listFiles((dir, name) -> name.endsWith("_training.dat"));
        if (dataFiles != null) {
            for (File dataFile : dataFiles) {
                try {
                    String filename = dataFile.getName();
                    String botName = filename.substring(4, filename.length() - 13);
                    botName = botName.replaceAll("_", " ");
                    Queue<TrainingExample> examples = this.loadTrainingExamples(dataFile.getAbsolutePath());
                    if (examples.isEmpty()) continue;
                    this.botTrainingData.put(botName, examples);
                    this.botExampleCounts.put(botName, examples.size());
                    ReignOfNether.LOGGER.info("Loaded {} training examples for bot '{}'", (Object)examples.size(), (Object)botName);
                }
                catch (Exception e) {
                    ReignOfNether.LOGGER.error("Failed to load training data from {}: {}", (Object)dataFile.getName(), (Object)e.getMessage());
                }
            }
        }
        if ((humanFile = new File("config/reignofnether/ml_data/human_actions.dat")).exists()) {
            try {
                Queue<HumanActionExample> examples = this.loadHumanExamples(humanFile.getAbsolutePath());
                this.humanActionData.addAll(examples);
                ReignOfNether.LOGGER.info("Loaded {} human action examples", (Object)examples.size());
            }
            catch (Exception e) {
                ReignOfNether.LOGGER.error("Failed to load human action data: {}", (Object)e.getMessage());
            }
        }
    }

    private void saveTrainingExamples(Queue<TrainingExample> examples, String filename) throws IOException {
        try (ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(filename));){
            oos.writeObject(new ArrayList<TrainingExample>(examples));
        }
    }

    private Queue<TrainingExample> loadTrainingExamples(String filename) throws IOException, ClassNotFoundException {
        try (ObjectInputStream ois = new ObjectInputStream(new FileInputStream(filename));){
            List list = (List)ois.readObject();
            ConcurrentLinkedQueue<TrainingExample> concurrentLinkedQueue = new ConcurrentLinkedQueue<TrainingExample>(list);
            return concurrentLinkedQueue;
        }
    }

    private void saveHumanExamples(Queue<HumanActionExample> examples, String filename) throws IOException {
        try (ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(filename));){
            oos.writeObject(new ArrayList<HumanActionExample>(examples));
        }
    }

    private Queue<HumanActionExample> loadHumanExamples(String filename) throws IOException, ClassNotFoundException {
        try (ObjectInputStream ois = new ObjectInputStream(new FileInputStream(filename));){
            List list = (List)ois.readObject();
            ConcurrentLinkedQueue<HumanActionExample> concurrentLinkedQueue = new ConcurrentLinkedQueue<HumanActionExample>(list);
            return concurrentLinkedQueue;
        }
    }

    private String sanitizeFilename(String name) {
        return name.replaceAll("[^a-zA-Z0-9_-]", "_");
    }

    public int getTrainingExamples() {
        return this.totalDecisionsRecorded;
    }

    public int getGameCount() {
        return this.totalGamesAnalyzed;
    }

    public int getBotExampleCount(String botName) {
        return this.botExampleCounts.getOrDefault(botName, 0);
    }

    public int getHumanExampleCount() {
        return this.humanActionData.size();
    }

    private static class TrainingExample
    implements Serializable {
        public final GameState gameState;
        public final MLDecision decision;
        public final long timestamp;
        public double reward = 0.0;

        public TrainingExample(GameState gameState, MLDecision decision, long timestamp) {
            this.gameState = gameState;
            this.decision = decision;
            this.timestamp = timestamp;
        }
    }

    private static class GameOutcome
    implements Serializable {
        public final String botName;
        public final boolean won;
        public final GameMetrics metrics;
        public final long timestamp;

        public GameOutcome(String botName, boolean won, GameMetrics metrics, long timestamp) {
            this.botName = botName;
            this.won = won;
            this.metrics = metrics;
            this.timestamp = timestamp;
        }
    }

    private static class HumanActionExample
    implements Serializable {
        public final GameState gameState;
        public final PlayerAction humanAction;
        public final long timestamp;

        public HumanActionExample(GameState gameState, PlayerAction humanAction, long timestamp) {
            this.gameState = gameState;
            this.humanAction = humanAction;
            this.timestamp = timestamp;
        }
    }
}

