/*
 * Decompiled with CFR 0.152.
 */
package com.solegendary.reignofnether.unit.pathfinding.async;

import com.solegendary.reignofnether.unit.pathfinding.async.AsyncPathfinder;
import com.solegendary.reignofnether.unit.pathfinding.async.PathfindingTask;
import com.solegendary.reignofnether.unit.pathfinding.async.PathfindingTaskQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import net.minecraft.server.MinecraftServer;
import net.minecraft.world.level.pathfinder.Path;

public class PathfindingThreadPool {
    private static final int DEFAULT_THREAD_COUNT = Math.max(2, Runtime.getRuntime().availableProcessors() - 2);
    private static final long SHUTDOWN_TIMEOUT_MS = 5000L;
    private static final long RESULT_DELIVERY_INTERVAL_MS = 50L;
    private final ExecutorService workerThreads;
    private final PathfindingTaskQueue taskQueue;
    private final ScheduledExecutorService resultDeliveryExecutor;
    private final MinecraftServer server;
    private final PathfindingWorker[] workers;
    private volatile boolean shutdown = false;
    private final AtomicLong totalTasksProcessed = new AtomicLong(0L);
    private final AtomicLong totalTasksFailed = new AtomicLong(0L);
    private final AtomicLong totalProcessingTimeMs = new AtomicLong(0L);
    private final BlockingQueue<PathfindingResult> resultQueue = new LinkedBlockingQueue<PathfindingResult>();

    public PathfindingThreadPool(MinecraftServer server) {
        this(server, DEFAULT_THREAD_COUNT);
    }

    public PathfindingThreadPool(MinecraftServer server, int threadCount) {
        this.server = server;
        this.taskQueue = new PathfindingTaskQueue();
        ThreadFactory threadFactory = new ThreadFactory(){
            private final AtomicInteger threadNumber = new AtomicInteger(1);

            @Override
            public Thread newThread(Runnable r) {
                Thread t = new Thread(r, "PathfindingWorker-" + this.threadNumber.getAndIncrement());
                t.setDaemon(true);
                t.setPriority(4);
                return t;
            }
        };
        this.workerThreads = Executors.newFixedThreadPool(threadCount, threadFactory);
        this.workers = new PathfindingWorker[threadCount];
        for (int i = 0; i < threadCount; ++i) {
            this.workers[i] = new PathfindingWorker(i);
            this.workerThreads.submit(this.workers[i]);
        }
        this.resultDeliveryExecutor = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = new Thread(r, "PathfindingResultDelivery");
            t.setDaemon(true);
            return t;
        });
        this.resultDeliveryExecutor.scheduleAtFixedRate(this::deliverResults, 50L, 50L, TimeUnit.MILLISECONDS);
    }

    public boolean submitTask(PathfindingTask task) {
        if (this.shutdown || task == null) {
            return false;
        }
        return this.taskQueue.enqueue(task);
    }

    public boolean cancelTasksForEntity(int entityId) {
        return this.taskQueue.cancelTasksForEntity(entityId);
    }

    public ThreadPoolStatistics getStatistics() {
        PathfindingTaskQueue.QueueStatistics queueStats = this.taskQueue.getStatistics();
        long activeWorkers = 0L;
        long totalTasksExecuted = 0L;
        long totalExecutionTime = 0L;
        for (PathfindingWorker worker : this.workers) {
            if (worker.isActive()) {
                ++activeWorkers;
            }
            totalTasksExecuted += worker.getTasksExecuted();
            totalExecutionTime += worker.getTotalExecutionTime();
        }
        return new ThreadPoolStatistics(this.workers.length, (int)activeWorkers, queueStats, this.totalTasksProcessed.get(), this.totalTasksFailed.get(), this.totalProcessingTimeMs.get(), this.resultQueue.size());
    }

    public void shutdown() {
        if (this.shutdown) {
            return;
        }
        this.shutdown = true;
        this.taskQueue.cancelAllTasks();
        for (PathfindingWorker worker : this.workers) {
            worker.interrupt();
        }
        this.workerThreads.shutdown();
        this.resultDeliveryExecutor.shutdown();
        try {
            if (!this.workerThreads.awaitTermination(5000L, TimeUnit.MILLISECONDS)) {
                this.workerThreads.shutdownNow();
            }
            if (!this.resultDeliveryExecutor.awaitTermination(1000L, TimeUnit.MILLISECONDS)) {
                this.resultDeliveryExecutor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.workerThreads.shutdownNow();
            this.resultDeliveryExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    private void deliverResults() {
        if (this.server == null || !this.server.m_130010_()) {
            return;
        }
        this.server.execute(() -> {
            PathfindingResult result;
            int maxBatch = 50;
            for (int processed = 0; (result = (PathfindingResult)this.resultQueue.poll()) != null && processed < 50; ++processed) {
                try {
                    result.task.deliverResult(result.path, result.success, result.errorMessage);
                    this.taskQueue.markTaskCompleted();
                    continue;
                }
                catch (Exception e) {
                    System.err.printf("Error delivering pathfinding result: %s%n", e.getMessage());
                }
            }
        });
    }

    private class PathfindingWorker
    implements Runnable {
        private final int workerId;
        private final AsyncPathfinder pathfinder;
        private volatile boolean active = false;
        private volatile PathfindingTask currentTask = null;
        private final AtomicLong tasksExecuted = new AtomicLong(0L);
        private final AtomicLong totalExecutionTime = new AtomicLong(0L);

        public PathfindingWorker(int workerId) {
            this.workerId = workerId;
            this.pathfinder = new AsyncPathfinder();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            Thread.currentThread().setName("PathfindingWorker-" + this.workerId);
            while (!PathfindingThreadPool.this.shutdown && !Thread.currentThread().isInterrupted()) {
                try {
                    PathfindingTask task = PathfindingThreadPool.this.taskQueue.dequeue();
                    if (task == null) continue;
                    this.currentTask = task;
                    this.active = true;
                    long startTime = System.currentTimeMillis();
                    Object errorMessage = null;
                    Path result = null;
                    boolean success = false;
                    try {
                        result = this.pathfinder.findPath(task);
                        boolean bl = success = result != null;
                        if (!success) {
                            errorMessage = "Pathfinding algorithm failed to find path";
                        }
                    }
                    catch (Exception e) {
                        errorMessage = "Pathfinding error: " + e.getMessage();
                    }
                    long executionTime = System.currentTimeMillis() - startTime;
                    this.tasksExecuted.incrementAndGet();
                    this.totalExecutionTime.addAndGet(executionTime);
                    PathfindingThreadPool.this.totalProcessingTimeMs.addAndGet(executionTime);
                    if (success) {
                        PathfindingThreadPool.this.totalTasksProcessed.incrementAndGet();
                    } else {
                        PathfindingThreadPool.this.totalTasksFailed.incrementAndGet();
                    }
                    PathfindingThreadPool.this.resultQueue.offer(new PathfindingResult(task, result, success, (String)errorMessage));
                }
                catch (Exception e) {
                    System.err.printf("PathfindingWorker-%d error: %s%n", this.workerId, e.getMessage());
                }
                finally {
                    this.currentTask = null;
                    this.active = false;
                }
            }
        }

        public boolean isActive() {
            return this.active;
        }

        public PathfindingTask getCurrentTask() {
            return this.currentTask;
        }

        public long getTasksExecuted() {
            return this.tasksExecuted.get();
        }

        public long getTotalExecutionTime() {
            return this.totalExecutionTime.get();
        }

        public void interrupt() {
            if (this.currentTask != null) {
                this.currentTask.cancel();
            }
        }
    }

    public static class ThreadPoolStatistics {
        public final int totalWorkers;
        public final int activeWorkers;
        public final PathfindingTaskQueue.QueueStatistics queueStats;
        public final long totalTasksProcessed;
        public final long totalTasksFailed;
        public final long totalProcessingTimeMs;
        public final int pendingResults;

        public ThreadPoolStatistics(int totalWorkers, int activeWorkers, PathfindingTaskQueue.QueueStatistics queueStats, long totalTasksProcessed, long totalTasksFailed, long totalProcessingTimeMs, int pendingResults) {
            this.totalWorkers = totalWorkers;
            this.activeWorkers = activeWorkers;
            this.queueStats = queueStats;
            this.totalTasksProcessed = totalTasksProcessed;
            this.totalTasksFailed = totalTasksFailed;
            this.totalProcessingTimeMs = totalProcessingTimeMs;
            this.pendingResults = pendingResults;
        }

        public double getAverageProcessingTimeMs() {
            long total = this.totalTasksProcessed + this.totalTasksFailed;
            return total > 0L ? (double)this.totalProcessingTimeMs / (double)total : 0.0;
        }

        public double getSuccessRate() {
            long total = this.totalTasksProcessed + this.totalTasksFailed;
            return total > 0L ? (double)this.totalTasksProcessed / (double)total : 0.0;
        }

        public String toString() {
            return String.format("ThreadPoolStats{workers=%d/%d, processed=%d, failed=%d, avgTime=%.2fms, success=%.1f%%, queue=%s}", this.activeWorkers, this.totalWorkers, this.totalTasksProcessed, this.totalTasksFailed, this.getAverageProcessingTimeMs(), this.getSuccessRate() * 100.0, this.queueStats);
        }
    }

    private static class PathfindingResult {
        final PathfindingTask task;
        final Path path;
        final boolean success;
        final String errorMessage;

        PathfindingResult(PathfindingTask task, Path path, boolean success, String errorMessage) {
            this.task = task;
            this.path = path;
            this.success = success;
            this.errorMessage = errorMessage;
        }
    }
}

