/*
 * Decompiled with CFR 0.152.
 */
package com.solegendary.reignofnether.bot.controllers.managers;

import com.solegendary.reignofnether.ReignOfNether;
import com.solegendary.reignofnether.bot.command.BotCommandInterface;
import com.solegendary.reignofnether.bot.controllers.AIStrategicController;
import com.solegendary.reignofnether.building.BuildingPlacement;
import com.solegendary.reignofnether.resources.ResourceCost;
import com.solegendary.reignofnether.resources.Resources;
import com.solegendary.reignofnether.resources.ResourcesServerEvents;
import java.util.List;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Vec3i;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.entity.LivingEntity;

public class EconomicManager {
    private final AIStrategicController strategicController;
    private int idealWorkerCount = 5;
    private int currentWorkerCount = 0;
    private ResourcePriority resourcePriority = ResourcePriority.BALANCED;
    private int targetFoodIncome = 20;
    private int targetWoodIncome = 30;
    private int targetOreIncome = 15;

    public EconomicManager(AIStrategicController strategicController) {
        this.strategicController = strategicController;
    }

    public void tick(ServerLevel level) {
        this.updateEconomicState(level);
        this.assessResourceNeeds(level);
    }

    public void makeEconomicDecision(ServerLevel level) {
        try {
            Resources resources = this.getPlayerResources();
            if (resources == null) {
                return;
            }
            if (this.shouldProduceWorkers()) {
                this.requestWorkerProduction();
            }
            if (this.shouldBuildEconomicBuildings(resources)) {
                this.planEconomicBuildings(resources, level);
            }
            if (this.shouldReallocateWorkers()) {
                this.optimizeWorkerAllocation(level);
            }
            ReignOfNether.LOGGER.debug("Bot '{}' made economic decision: {} workers, priority: {}", (Object)this.getBotName(), (Object)this.currentWorkerCount, (Object)this.resourcePriority);
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Error in economic decision making for bot '{}': {}", (Object)this.getBotName(), (Object)e.getMessage());
        }
    }

    private void updateEconomicState(ServerLevel level) {
        try {
            BotCommandInterface commandInterface = this.strategicController.getBotPlayer().getCommandInterface();
            List<LivingEntity> workers = commandInterface.getOwnedWorkers(level);
            this.currentWorkerCount = workers.size();
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Error counting workers for bot '{}': {}", (Object)this.strategicController.getBotPlayer().getName(), (Object)e.getMessage());
            this.currentWorkerCount = 3;
        }
        switch (this.strategicController.getCurrentPhase()) {
            case EARLY_GAME: {
                this.idealWorkerCount = 5;
                break;
            }
            case MID_GAME: {
                this.idealWorkerCount = 8;
                break;
            }
            case LATE_GAME: {
                this.idealWorkerCount = 6;
            }
        }
    }

    private void assessResourceNeeds(ServerLevel level) {
        Resources resources = this.getPlayerResources();
        if (resources == null) {
            return;
        }
        int food = resources.food;
        int wood = resources.wood;
        int ore = resources.ore;
        double foodRatio = (double)food / (double)Math.max(this.targetFoodIncome * 10, 1);
        double woodRatio = (double)wood / (double)Math.max(this.targetWoodIncome * 10, 1);
        double oreRatio = (double)ore / (double)Math.max(this.targetOreIncome * 10, 1);
        this.resourcePriority = foodRatio < woodRatio && foodRatio < oreRatio ? ResourcePriority.FOOD : (woodRatio < oreRatio ? ResourcePriority.WOOD : ResourcePriority.ORE);
        if (Math.abs(foodRatio - woodRatio) < 0.3 && Math.abs(woodRatio - oreRatio) < 0.3 && Math.abs(foodRatio - oreRatio) < 0.3) {
            this.resourcePriority = ResourcePriority.BALANCED;
        }
    }

    private boolean shouldProduceWorkers() {
        if (this.currentWorkerCount >= this.idealWorkerCount) {
            return false;
        }
        Resources resources = this.getPlayerResources();
        if (resources == null) {
            return false;
        }
        try {
            String factionName = this.strategicController.getBotPlayer().getFaction().name().toLowerCase();
            ResourceCost workerCost = this.getWorkerCostByFaction(factionName);
            return resources.food >= workerCost.food && resources.wood >= workerCost.wood && resources.ore >= workerCost.ore;
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Error checking worker costs for bot '{}': {}", (Object)this.strategicController.getBotPlayer().getName(), (Object)e.getMessage());
            return resources.food >= 50;
        }
    }

    private boolean shouldBuildEconomicBuildings(Resources resources) {
        return resources.wood >= 100 && resources.ore >= 50;
    }

    private boolean shouldReallocateWorkers() {
        return Math.random() < 0.3;
    }

    private void requestWorkerProduction() {
        ReignOfNether.LOGGER.debug("Bot '{}' requesting worker production", (Object)this.getBotName());
        this.strategicController.getMilitaryManager().requestWorkerProduction();
    }

    private void planEconomicBuildings(Resources resources, ServerLevel level) {
        BlockPos buildLocation;
        ReignOfNether.LOGGER.debug("Bot '{}' planning economic buildings", (Object)this.getBotName());
        BotCommandInterface commandInterface = this.strategicController.getBotPlayer().getCommandInterface();
        List<LivingEntity> workers = commandInterface.getOwnedWorkers(level);
        if (workers.isEmpty()) {
            return;
        }
        int farmCount = commandInterface.countBuildings("farm");
        int stockpileCount = commandInterface.countBuildings("stockpile");
        if (farmCount < 2 && resources.wood >= 100 && (buildLocation = this.findGoodBuildingLocation(workers.get(0).m_20183_())) != null) {
            ReignOfNether.LOGGER.debug("Bot '{}' planning to build farm at {}", (Object)this.getBotName(), (Object)buildLocation);
        }
        if (stockpileCount < 1 && resources.wood >= 150 && (buildLocation = this.findGoodBuildingLocation(workers.get(0).m_20183_())) != null) {
            ReignOfNether.LOGGER.debug("Bot '{}' planning to build stockpile at {}", (Object)this.getBotName(), (Object)buildLocation);
        }
    }

    private BlockPos findGoodBuildingLocation(BlockPos nearPos) {
        try {
            for (int radius = 8; radius <= 32; radius += 8) {
                for (int x = -radius; x <= radius; x += 4) {
                    for (int z = -radius; z <= radius; z += 4) {
                        BlockPos candidatePos = nearPos.m_7918_(x, 0, z);
                        if (!this.isSuitableForBuilding(candidatePos)) continue;
                        return candidatePos;
                    }
                }
            }
            return nearPos.m_7918_(12, 0, 8);
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Error finding building location for bot '{}': {}", (Object)this.getBotName(), (Object)e.getMessage());
            return nearPos.m_7918_(10, 0, 10);
        }
    }

    private boolean isSuitableForBuilding(BlockPos pos) {
        try {
            BotCommandInterface commandInterface = this.strategicController.getBotPlayer().getCommandInterface();
            List<BuildingPlacement> nearbyBuildings = commandInterface.getOwnedBuildings();
            for (BuildingPlacement building : nearbyBuildings) {
                if (!(building.originPos.m_123331_((Vec3i)pos) < 36.0)) continue;
                return false;
            }
            return true;
        }
        catch (Exception e) {
            return true;
        }
    }

    private void optimizeWorkerAllocation(ServerLevel level) {
        ReignOfNether.LOGGER.debug("Bot '{}' optimizing worker allocation for {}", (Object)this.getBotName(), (Object)this.resourcePriority);
        BotCommandInterface commandInterface = this.strategicController.getBotPlayer().getCommandInterface();
        List<LivingEntity> workers = commandInterface.getOwnedWorkers(level);
        if (workers.isEmpty()) {
            return;
        }
        int totalWorkers = workers.size();
        switch (this.resourcePriority) {
            case FOOD: {
                BlockPos foodLocation = commandInterface.findNearestResourceLocation(workers.get(0).m_20183_(), "food");
                if (foodLocation == null) break;
                commandInterface.commandWorkersToGather(workers.subList(0, Math.min(2, totalWorkers)), foodLocation);
                break;
            }
            case WOOD: {
                BlockPos woodLocation = commandInterface.findNearestResourceLocation(workers.get(0).m_20183_(), "wood");
                if (woodLocation == null) break;
                commandInterface.commandWorkersToGather(workers.subList(0, Math.min(3, totalWorkers)), woodLocation);
                break;
            }
            case ORE: {
                BlockPos oreLocation = commandInterface.findNearestResourceLocation(workers.get(0).m_20183_(), "ore");
                if (oreLocation == null) break;
                commandInterface.commandWorkersToGather(workers.subList(0, Math.min(2, totalWorkers)), oreLocation);
                break;
            }
            case BALANCED: {
                int perResource = Math.max(1, totalWorkers / 3);
                if (totalWorkers < 3) break;
                BlockPos foodLoc = commandInterface.findNearestResourceLocation(workers.get(0).m_20183_(), "food");
                BlockPos woodLoc = commandInterface.findNearestResourceLocation(workers.get(0).m_20183_(), "wood");
                BlockPos oreLoc = commandInterface.findNearestResourceLocation(workers.get(0).m_20183_(), "ore");
                if (foodLoc != null) {
                    commandInterface.commandWorkersToGather(workers.subList(0, perResource), foodLoc);
                }
                if (woodLoc != null && totalWorkers > perResource) {
                    commandInterface.commandWorkersToGather(workers.subList(perResource, Math.min(perResource * 2, totalWorkers)), woodLoc);
                }
                if (oreLoc == null || totalWorkers <= perResource * 2) break;
                commandInterface.commandWorkersToGather(workers.subList(perResource * 2, totalWorkers), oreLoc);
            }
        }
    }

    private ResourceCost getWorkerCostByFaction(String faction) {
        return switch (faction) {
            case "villagers" -> ResourceCost.Unit(50, 0, 0, 15, 1);
            case "piglins" -> ResourceCost.Unit(40, 20, 0, 15, 1);
            case "monsters" -> ResourceCost.Unit(60, 0, 10, 15, 1);
            default -> ResourceCost.Unit(50, 0, 0, 15, 1);
        };
    }

    private Resources getPlayerResources() {
        String botName = this.getBotName();
        for (Resources resources : ResourcesServerEvents.resourcesList) {
            if (!resources.ownerName.equals(botName)) continue;
            return resources;
        }
        return null;
    }

    private String getBotName() {
        return this.strategicController.getBotPlayer().getName();
    }

    public void reset() {
        this.currentWorkerCount = 0;
        this.resourcePriority = ResourcePriority.BALANCED;
        float difficultyMultiplier = this.strategicController.getBotPlayer().getDifficulty().getSkillMultiplier();
        this.targetFoodIncome = (int)(20.0f * difficultyMultiplier);
        this.targetWoodIncome = (int)(30.0f * difficultyMultiplier);
        this.targetOreIncome = (int)(15.0f * difficultyMultiplier);
    }

    public int getCurrentWorkerCount() {
        return this.currentWorkerCount;
    }

    public int getIdealWorkerCount() {
        return this.idealWorkerCount;
    }

    public ResourcePriority getResourcePriority() {
        return this.resourcePriority;
    }

    public int getTargetFoodIncome() {
        return this.targetFoodIncome;
    }

    public int getTargetWoodIncome() {
        return this.targetWoodIncome;
    }

    public int getTargetOreIncome() {
        return this.targetOreIncome;
    }

    public static enum ResourcePriority {
        FOOD,
        WOOD,
        ORE,
        BALANCED;

    }
}

