/*
 * Decompiled with CFR 0.152.
 */
package com.solegendary.reignofnether.bot.ml;

import com.solegendary.reignofnether.ReignOfNether;
import com.solegendary.reignofnether.bot.BotDifficulty;
import com.solegendary.reignofnether.bot.ml.GameMetrics;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;

public class AdaptiveDifficultyManager {
    private final Map<String, PlayerPerformanceTracker> playerTrackers = new ConcurrentHashMap<String, PlayerPerformanceTracker>();
    private final Map<String, BotLearningTracker> botTrackers = new ConcurrentHashMap<String, BotLearningTracker>();
    private static final double MIN_LEARNING_RATE = 0.001;
    private static final double MAX_LEARNING_RATE = 0.1;
    private static final double BASE_LEARNING_RATE = 0.01;
    private static final int PERFORMANCE_WINDOW_SIZE = 10;
    private static final double TARGET_WIN_RATE = 0.45;
    private static final double WIN_RATE_TOLERANCE = 0.1;

    public void recordGameOutcome(String playerName, String botName, boolean playerWon, GameMetrics gameMetrics) {
        PlayerPerformanceTracker playerTracker = this.playerTrackers.computeIfAbsent(playerName, k -> new PlayerPerformanceTracker());
        playerTracker.recordGame(playerWon, gameMetrics);
        BotLearningTracker botTracker = this.botTrackers.computeIfAbsent(botName, k -> new BotLearningTracker());
        botTracker.recordGame(!playerWon, gameMetrics);
        this.calculateAdaptiveDifficulty(playerName, botName);
        ReignOfNether.LOGGER.debug("Recorded game outcome: Player '{}' vs Bot '{}', Player won: {}", (Object)playerName, (Object)botName, (Object)playerWon);
    }

    public double getAdaptiveLearningRate(String botName) {
        BotLearningTracker tracker = this.botTrackers.get(botName);
        if (tracker == null) {
            return 0.01;
        }
        return tracker.getCurrentLearningRate();
    }

    public DifficultyAdjustment getDifficultyAdjustment(String playerName, String botName) {
        PlayerPerformanceTracker playerTracker = this.playerTrackers.get(playerName);
        BotLearningTracker botTracker = this.botTrackers.get(botName);
        if (playerTracker == null || botTracker == null) {
            return new DifficultyAdjustment(1.0, 1.0, BotDifficulty.MEDIUM);
        }
        double playerWinRate = playerTracker.getWinRate();
        double skillMultiplier = this.calculateSkillMultiplier(playerWinRate);
        double aggressionMultiplier = this.calculateAggressionMultiplier(playerTracker, botTracker);
        BotDifficulty recommendedDifficulty = this.calculateRecommendedDifficulty(playerWinRate);
        return new DifficultyAdjustment(skillMultiplier, aggressionMultiplier, recommendedDifficulty);
    }

    private void calculateAdaptiveDifficulty(String playerName, String botName) {
        PlayerPerformanceTracker playerTracker = this.playerTrackers.get(playerName);
        BotLearningTracker botTracker = this.botTrackers.get(botName);
        if (playerTracker == null || botTracker == null) {
            return;
        }
        double playerWinRate = playerTracker.getWinRate();
        double learningRateAdjustment = this.calculateLearningRateAdjustment(playerWinRate);
        botTracker.adjustLearningRate(learningRateAdjustment);
        ReignOfNether.LOGGER.debug("Adaptive difficulty update: Player '{}' win rate: {:.2f}, Bot '{}' learning rate: {:.4f}", (Object)playerName, (Object)playerWinRate, (Object)botName, (Object)botTracker.getCurrentLearningRate());
    }

    private double calculateLearningRateAdjustment(double playerWinRate) {
        double deviation = playerWinRate - 0.45;
        if (Math.abs(deviation) <= 0.1) {
            return 1.0;
        }
        if (deviation > 0.1) {
            double adjustment = 1.0 + (deviation - 0.1) * 2.0;
            return Math.min(adjustment, 10.0);
        }
        double adjustment = 1.0 - (0.1 - deviation) * 0.5;
        return Math.max(adjustment, 0.1);
    }

    private double calculateSkillMultiplier(double playerWinRate) {
        if (playerWinRate > 0.55) {
            return 1.0 + (playerWinRate - 0.45) * 0.8;
        }
        if (playerWinRate < 0.35) {
            return Math.max(0.5, 1.0 - (0.45 - playerWinRate) * 0.6);
        }
        return 1.0;
    }

    private double calculateAggressionMultiplier(PlayerPerformanceTracker playerTracker, BotLearningTracker botTracker) {
        double playerAvgGameLength = playerTracker.getAverageGameLength();
        double botAvgGameLength = botTracker.getAverageGameLength();
        if (playerAvgGameLength < 600000.0) {
            return 0.7;
        }
        if (playerAvgGameLength > 2400000.0) {
            return 1.3;
        }
        return 1.0;
    }

    private BotDifficulty calculateRecommendedDifficulty(double playerWinRate) {
        if (playerWinRate > 0.7) {
            return BotDifficulty.HARD;
        }
        if (playerWinRate > 0.55) {
            return BotDifficulty.MEDIUM;
        }
        if (playerWinRate > 0.35) {
            return BotDifficulty.EASY;
        }
        return BotDifficulty.EASY;
    }

    public String getPerformanceStats(String playerName, String botName) {
        PlayerPerformanceTracker playerTracker = this.playerTrackers.get(playerName);
        BotLearningTracker botTracker = this.botTrackers.get(botName);
        if (playerTracker == null || botTracker == null) {
            return "No performance data available";
        }
        return String.format("Player '%s': %.2f%% win rate, %.1fm avg game | Bot '%s': %.4f learning rate", playerName, playerTracker.getWinRate() * 100.0, playerTracker.getAverageGameLength() / 60000.0, botName, botTracker.getCurrentLearningRate());
    }

    public void resetPerformanceData(String playerName, String botName) {
        if (playerName != null) {
            this.playerTrackers.remove(playerName);
        }
        if (botName != null) {
            this.botTrackers.remove(botName);
        }
        ReignOfNether.LOGGER.info("Reset adaptive difficulty data for player: '{}', bot: '{}'", (Object)playerName, (Object)botName);
    }

    private static class PlayerPerformanceTracker {
        private final Queue<GameResult> recentGames = new ConcurrentLinkedQueue<GameResult>();
        private double totalWins = 0.0;
        private double totalGames = 0.0;
        private double totalGameLength = 0.0;

        private PlayerPerformanceTracker() {
        }

        void recordGame(boolean won, GameMetrics metrics) {
            GameResult result = new GameResult(won, metrics.gameLength, System.currentTimeMillis());
            this.recentGames.offer(result);
            this.totalGames += 1.0;
            this.totalGameLength += (double)metrics.gameLength;
            if (won) {
                this.totalWins += 1.0;
            }
            while (this.recentGames.size() > 10) {
                GameResult old = this.recentGames.poll();
                if (old == null) continue;
                this.totalGames -= 1.0;
                this.totalGameLength -= (double)old.gameLength;
                if (!old.won) continue;
                this.totalWins -= 1.0;
            }
        }

        double getWinRate() {
            return this.totalGames > 0.0 ? this.totalWins / this.totalGames : 0.5;
        }

        double getAverageGameLength() {
            return this.totalGames > 0.0 ? this.totalGameLength / this.totalGames : 1200000.0;
        }
    }

    private static class BotLearningTracker {
        private final Queue<GameResult> recentGames = new ConcurrentLinkedQueue<GameResult>();
        private double currentLearningRate = 0.01;
        private double totalWins = 0.0;
        private double totalGames = 0.0;
        private double totalGameLength = 0.0;

        private BotLearningTracker() {
        }

        void recordGame(boolean won, GameMetrics metrics) {
            GameResult result = new GameResult(won, metrics.gameLength, System.currentTimeMillis());
            this.recentGames.offer(result);
            this.totalGames += 1.0;
            this.totalGameLength += (double)metrics.gameLength;
            if (won) {
                this.totalWins += 1.0;
            }
            while (this.recentGames.size() > 10) {
                GameResult old = this.recentGames.poll();
                if (old == null) continue;
                this.totalGames -= 1.0;
                this.totalGameLength -= (double)old.gameLength;
                if (!old.won) continue;
                this.totalWins -= 1.0;
            }
        }

        void adjustLearningRate(double adjustment) {
            this.currentLearningRate = Math.max(0.001, Math.min(0.1, 0.01 * adjustment));
        }

        double getCurrentLearningRate() {
            return this.currentLearningRate;
        }

        double getWinRate() {
            return this.totalGames > 0.0 ? this.totalWins / this.totalGames : 0.5;
        }

        double getAverageGameLength() {
            return this.totalGames > 0.0 ? this.totalGameLength / this.totalGames : 1200000.0;
        }
    }

    public static class DifficultyAdjustment {
        public final double skillMultiplier;
        public final double aggressionMultiplier;
        public final BotDifficulty recommendedDifficulty;

        public DifficultyAdjustment(double skillMultiplier, double aggressionMultiplier, BotDifficulty recommendedDifficulty) {
            this.skillMultiplier = skillMultiplier;
            this.aggressionMultiplier = aggressionMultiplier;
            this.recommendedDifficulty = recommendedDifficulty;
        }

        public String toString() {
            return String.format("DifficultyAdjustment{skill: %.2f, aggression: %.2f, difficulty: %s}", new Object[]{this.skillMultiplier, this.aggressionMultiplier, this.recommendedDifficulty});
        }
    }

    private static class GameResult {
        final boolean won;
        final long gameLength;
        final long timestamp;

        GameResult(boolean won, long gameLength, long timestamp) {
            this.won = won;
            this.gameLength = gameLength;
            this.timestamp = timestamp;
        }
    }
}

