/*
 * Decompiled with CFR 0.152.
 */
package com.solegendary.reignofnether.bot.ml;

import com.solegendary.reignofnether.ReignOfNether;
import com.solegendary.reignofnether.bot.ml.GameMetrics;
import com.solegendary.reignofnether.bot.ml.GameState;
import com.solegendary.reignofnether.bot.ml.GameStateVectorizer;
import com.solegendary.reignofnether.bot.ml.MLDecision;
import com.solegendary.reignofnether.bot.ml.PlayerAction;
import com.solegendary.reignofnether.bot.ml.PlayerBehaviorAnalyzer;
import com.solegendary.reignofnether.bot.ml.TrainingDataManager;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

public class DataCollectionManager {
    private PlayerBehaviorAnalyzer behaviorAnalyzer;
    private TrainingDataManager trainingDataManager;
    private GameStateVectorizer vectorizer;
    private final AtomicBoolean collectionEnabled = new AtomicBoolean(true);
    private final AtomicBoolean initialized = new AtomicBoolean(false);
    private final Map<String, GameSessionData> activeSessions = new ConcurrentHashMap<String, GameSessionData>();
    private final AtomicInteger totalSessionsTracked = new AtomicInteger(0);
    private ScheduledExecutorService cleanupExecutor;
    private final AtomicInteger playerActionsRecorded = new AtomicInteger(0);
    private final AtomicInteger botDecisionsRecorded = new AtomicInteger(0);
    private final AtomicInteger gameStatesVectorized = new AtomicInteger(0);

    public void initialize(PlayerBehaviorAnalyzer behaviorAnalyzer, TrainingDataManager trainingDataManager) {
        if (this.initialized.get()) {
            return;
        }
        this.behaviorAnalyzer = behaviorAnalyzer;
        this.trainingDataManager = trainingDataManager;
        this.vectorizer = new GameStateVectorizer();
        this.startPeriodicCleanup();
        this.initialized.set(true);
        ReignOfNether.LOGGER.info("Data Collection Manager initialized and ready");
    }

    public void startGameSession(String sessionId, List<String> playerNames, List<String> botNames, String gameMode, Map<String, String> gameParameters) {
        if (!this.isEnabled()) {
            return;
        }
        GameSessionData session = new GameSessionData(sessionId, playerNames, botNames, gameMode, gameParameters, System.currentTimeMillis());
        this.activeSessions.put(sessionId, session);
        this.totalSessionsTracked.incrementAndGet();
        ReignOfNether.LOGGER.info("Started ML data collection for game session '{}' with {} players and {} bots", (Object)sessionId, (Object)playerNames.size(), (Object)botNames.size());
    }

    public void recordPlayerAction(String sessionId, String playerName, GameState gameState, PlayerAction action, PlayerBehaviorAnalyzer.ActionOutcome outcome) {
        if (!this.isEnabled()) {
            return;
        }
        try {
            this.behaviorAnalyzer.recordPlayerAction(playerName, gameState, action, System.currentTimeMillis(), outcome);
            this.trainingDataManager.recordHumanAction(gameState, action, playerName);
            GameSessionData session = this.activeSessions.get(sessionId);
            if (session != null) {
                session.recordPlayerAction(playerName, action);
            }
            this.playerActionsRecorded.incrementAndGet();
            ReignOfNether.LOGGER.debug("Recorded player action for ML: {} -> {}", (Object)playerName, (Object)action.getActionType());
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Error recording player action: {}", (Object)e.getMessage());
        }
    }

    public void recordBotDecision(String sessionId, String botName, GameState gameState, MLDecision decision, double actualOutcome, String outcomeContext) {
        if (!this.isEnabled()) {
            return;
        }
        try {
            this.trainingDataManager.recordDecision(gameState, decision, botName);
            GameSessionData session = this.activeSessions.get(sessionId);
            if (session != null) {
                session.recordBotDecision(botName, decision);
            }
            this.botDecisionsRecorded.incrementAndGet();
            ReignOfNether.LOGGER.debug("Recorded bot decision for ML: {} -> {} (confidence: {:.2f})", (Object)botName, (Object)decision.getDecisionType(), (Object)decision.getConfidence());
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Error recording bot decision: {}", (Object)e.getMessage());
        }
    }

    public double[] recordGameStateVector(String sessionId, GameState gameState, String context) {
        if (!this.isEnabled()) {
            return new double[0];
        }
        try {
            double[] vector = this.vectorizer.vectorizeGameState(gameState);
            GameSessionData session = this.activeSessions.get(sessionId);
            if (session != null) {
                session.recordGameStateComplexity(this.calculateComplexity(vector));
            }
            this.gameStatesVectorized.incrementAndGet();
            return vector;
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Error vectorizing game state: {}", (Object)e.getMessage());
            return new double[0];
        }
    }

    private double calculateComplexity(double[] vector) {
        int nonZeroFeatures = 0;
        double sum = 0.0;
        double sumSquares = 0.0;
        for (double value : vector) {
            if (Math.abs(value) > 0.001) {
                ++nonZeroFeatures;
            }
            sum += value;
            sumSquares += value * value;
        }
        double mean = sum / (double)vector.length;
        double variance = sumSquares / (double)vector.length - mean * mean;
        double featureDensity = (double)nonZeroFeatures / (double)vector.length;
        return (featureDensity + Math.sqrt(variance)) / 2.0;
    }

    public void endGameSession(String sessionId, Map<String, Boolean> playerResults, GameMetrics gameMetrics, String endReason) {
        GameSessionData session = this.activeSessions.remove(sessionId);
        if (session == null) {
            return;
        }
        try {
            session.endSession(playerResults, gameMetrics, endReason);
            SessionAnalysisReport report = this.generateSessionReport(session);
            this.storeCompletedSession(session, report);
            ReignOfNether.LOGGER.info("Ended ML data collection for session '{}': {} player actions, {} bot decisions", (Object)sessionId, (Object)session.getTotalPlayerActions(), (Object)session.getTotalBotDecisions());
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Error ending game session '{}': {}", (Object)sessionId, (Object)e.getMessage());
        }
    }

    private SessionAnalysisReport generateSessionReport(GameSessionData session) {
        ConcurrentHashMap<String, PlayerBehaviorAnalyzer.BehaviorAnalysis> playerAnalyses = new ConcurrentHashMap<String, PlayerBehaviorAnalyzer.BehaviorAnalysis>();
        for (String playerName : session.playerNames) {
            PlayerBehaviorAnalyzer.BehaviorAnalysis analysis = this.behaviorAnalyzer.analyzeBehavior(playerName);
            if (analysis == null) continue;
            playerAnalyses.put(playerName, analysis);
        }
        return new SessionAnalysisReport(session.sessionId, playerAnalyses, session.getTotalPlayerActions(), session.getTotalBotDecisions(), session.getAverageStateComplexity(), session.getDuration());
    }

    private void storeCompletedSession(GameSessionData session, SessionAnalysisReport report) {
        ReignOfNether.LOGGER.debug("Storing completed session analysis: {}", (Object)report);
    }

    public DataCollectionStatistics getStatistics() {
        return new DataCollectionStatistics(this.collectionEnabled.get(), this.totalSessionsTracked.get(), this.activeSessions.size(), this.playerActionsRecorded.get(), this.botDecisionsRecorded.get(), this.gameStatesVectorized.get(), this.behaviorAnalyzer.getAllBehaviorAnalyses().size());
    }

    public void setCollectionEnabled(boolean enabled) {
        this.collectionEnabled.set(enabled);
        ReignOfNether.LOGGER.info("ML data collection {}", (Object)(enabled ? "enabled" : "disabled"));
    }

    public boolean isEnabled() {
        return this.initialized.get() && this.collectionEnabled.get();
    }

    private void startPeriodicCleanup() {
        this.cleanupExecutor = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = new Thread(r, "ML-Data-Cleanup");
            t.setDaemon(true);
            return t;
        });
        this.cleanupExecutor.scheduleAtFixedRate(this::performCleanup, 30L, 30L, TimeUnit.MINUTES);
    }

    private void performCleanup() {
        try {
            this.behaviorAnalyzer.cleanupOldData();
            long cutoffTime = System.currentTimeMillis() - 14400000L;
            this.activeSessions.entrySet().removeIf(entry -> ((GameSessionData)entry.getValue()).startTime < cutoffTime);
            ReignOfNether.LOGGER.debug("Performed ML data cleanup");
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Error during ML data cleanup: {}", (Object)e.getMessage());
        }
    }

    public void shutdown() {
        if (!this.initialized.get()) {
            return;
        }
        try {
            ReignOfNether.LOGGER.info("Shutting down Data Collection Manager...");
            if (this.cleanupExecutor != null) {
                this.cleanupExecutor.shutdown();
                this.cleanupExecutor.awaitTermination(5L, TimeUnit.SECONDS);
            }
            for (String sessionId : new ArrayList<String>(this.activeSessions.keySet())) {
                this.endGameSession(sessionId, new ConcurrentHashMap<String, Boolean>(), new GameMetrics(0L, 0, 0.0, 0.0, 0, 0, 0, 0, 0), "System shutdown");
            }
            this.initialized.set(false);
            ReignOfNether.LOGGER.info("Data Collection Manager shutdown complete");
        }
        catch (Exception e) {
            ReignOfNether.LOGGER.error("Error during Data Collection Manager shutdown: {}", (Object)e.getMessage());
        }
    }

    private static class GameSessionData {
        final String sessionId;
        final List<String> playerNames;
        final List<String> botNames;
        final String gameMode;
        final Map<String, String> gameParameters;
        final long startTime;
        private final AtomicInteger totalPlayerActions = new AtomicInteger(0);
        private final AtomicInteger totalBotDecisions = new AtomicInteger(0);
        private final Map<String, Integer> playerActionCounts = new ConcurrentHashMap<String, Integer>();
        private final Map<String, Integer> botDecisionCounts = new ConcurrentHashMap<String, Integer>();
        private double totalStateComplexity = 0.0;
        private int stateComplexityCount = 0;
        private long endTime = 0L;
        private Map<String, Boolean> finalResults;
        private GameMetrics finalMetrics;
        private String endReason;

        GameSessionData(String sessionId, List<String> playerNames, List<String> botNames, String gameMode, Map<String, String> gameParameters, long startTime) {
            this.sessionId = sessionId;
            this.playerNames = new ArrayList<String>(playerNames);
            this.botNames = new ArrayList<String>(botNames);
            this.gameMode = gameMode;
            this.gameParameters = new ConcurrentHashMap<String, String>(gameParameters);
            this.startTime = startTime;
        }

        void recordPlayerAction(String playerName, PlayerAction action) {
            this.totalPlayerActions.incrementAndGet();
            this.playerActionCounts.merge(playerName, 1, Integer::sum);
        }

        void recordBotDecision(String botName, MLDecision decision) {
            this.totalBotDecisions.incrementAndGet();
            this.botDecisionCounts.merge(botName, 1, Integer::sum);
        }

        void recordGameStateComplexity(double complexity) {
            this.totalStateComplexity += complexity;
            ++this.stateComplexityCount;
        }

        void endSession(Map<String, Boolean> results, GameMetrics metrics, String reason) {
            this.endTime = System.currentTimeMillis();
            this.finalResults = new ConcurrentHashMap<String, Boolean>(results);
            this.finalMetrics = metrics;
            this.endReason = reason;
        }

        int getTotalPlayerActions() {
            return this.totalPlayerActions.get();
        }

        int getTotalBotDecisions() {
            return this.totalBotDecisions.get();
        }

        long getDuration() {
            return this.endTime > 0L ? this.endTime - this.startTime : System.currentTimeMillis() - this.startTime;
        }

        double getAverageStateComplexity() {
            return this.stateComplexityCount > 0 ? this.totalStateComplexity / (double)this.stateComplexityCount : 0.0;
        }
    }

    public static class SessionAnalysisReport {
        public final String sessionId;
        public final Map<String, PlayerBehaviorAnalyzer.BehaviorAnalysis> playerAnalyses;
        public final int totalPlayerActions;
        public final int totalBotDecisions;
        public final double averageStateComplexity;
        public final long sessionDuration;

        SessionAnalysisReport(String sessionId, Map<String, PlayerBehaviorAnalyzer.BehaviorAnalysis> playerAnalyses, int totalPlayerActions, int totalBotDecisions, double averageStateComplexity, long sessionDuration) {
            this.sessionId = sessionId;
            this.playerAnalyses = new ConcurrentHashMap<String, PlayerBehaviorAnalyzer.BehaviorAnalysis>(playerAnalyses);
            this.totalPlayerActions = totalPlayerActions;
            this.totalBotDecisions = totalBotDecisions;
            this.averageStateComplexity = averageStateComplexity;
            this.sessionDuration = sessionDuration;
        }

        public String toString() {
            return String.format("SessionReport{id='%s', playerActions=%d, botDecisions=%d, duration=%dms}", this.sessionId, this.totalPlayerActions, this.totalBotDecisions, this.sessionDuration);
        }
    }

    public static class DataCollectionStatistics {
        public final boolean enabled;
        public final int totalSessions;
        public final int activeSessions;
        public final int playerActionsRecorded;
        public final int botDecisionsRecorded;
        public final int gameStatesVectorized;
        public final int playersAnalyzed;

        DataCollectionStatistics(boolean enabled, int totalSessions, int activeSessions, int playerActionsRecorded, int botDecisionsRecorded, int gameStatesVectorized, int playersAnalyzed) {
            this.enabled = enabled;
            this.totalSessions = totalSessions;
            this.activeSessions = activeSessions;
            this.playerActionsRecorded = playerActionsRecorded;
            this.botDecisionsRecorded = botDecisionsRecorded;
            this.gameStatesVectorized = gameStatesVectorized;
            this.playersAnalyzed = playersAnalyzed;
        }

        public String toString() {
            return String.format("DataCollection{enabled=%s, sessions=%d/%d, actions=%d, decisions=%d, states=%d, players=%d}", this.enabled, this.activeSessions, this.totalSessions, this.playerActionsRecorded, this.botDecisionsRecorded, this.gameStatesVectorized, this.playersAnalyzed);
        }
    }
}

