/*
 * Decompiled with CFR 0.152.
 */
package com.solegendary.reignofnether.orthoview;

import com.solegendary.reignofnether.orthoview.CameraManager;
import com.solegendary.reignofnether.orthoview.InputManager;
import com.solegendary.reignofnether.orthoview.OrthoviewClientEvents;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import net.minecraft.client.Minecraft;
import net.minecraft.network.chat.Component;
import net.minecraft.world.level.GameType;

public class OrthoviewErrorHandler {
    private static final Minecraft MC = Minecraft.m_91087_();
    private static final ConcurrentHashMap<ErrorType, AtomicInteger> errorCounts = new ConcurrentHashMap();
    private static final ConcurrentHashMap<ErrorType, Long> lastErrorTime = new ConcurrentHashMap();
    private static final int MAX_ERRORS_PER_TYPE = 5;
    private static final long ERROR_RESET_TIME_MS = 30000L;
    private static final long MIN_ERROR_INTERVAL_MS = 1000L;
    private static boolean systemInRecoveryMode = false;
    private static long recoveryStartTime = 0L;
    private static final long RECOVERY_DURATION_MS = 10000L;

    public static boolean handleError(ErrorType errorType, Exception exception, String context) {
        try {
            if (OrthoviewErrorHandler.shouldThrottleError(errorType)) {
                return false;
            }
            errorCounts.computeIfAbsent(errorType, k -> new AtomicInteger(0)).incrementAndGet();
            lastErrorTime.put(errorType, System.currentTimeMillis());
            OrthoviewErrorHandler.logError(errorType, exception, context);
            if (OrthoviewErrorHandler.shouldEnterRecoveryMode(errorType)) {
                OrthoviewErrorHandler.enterRecoveryMode();
            }
            RecoveryStrategy strategy = OrthoviewErrorHandler.getRecoveryStrategy(errorType);
            return OrthoviewErrorHandler.applyRecoveryStrategy(strategy, errorType, context);
        }
        catch (Exception handlerException) {
            OrthoviewErrorHandler.emergencyShutdown("Error handler failed: " + handlerException.getMessage());
            return false;
        }
    }

    public static boolean handleError(ErrorType errorType, String message, String context) {
        return OrthoviewErrorHandler.handleError(errorType, new RuntimeException(message), context);
    }

    private static boolean shouldThrottleError(ErrorType errorType) {
        Long lastTime = lastErrorTime.get((Object)errorType);
        if (lastTime != null) {
            long timeSinceLastError = System.currentTimeMillis() - lastTime;
            return timeSinceLastError < 1000L;
        }
        return false;
    }

    private static boolean shouldEnterRecoveryMode(ErrorType errorType) {
        AtomicInteger count = errorCounts.get((Object)errorType);
        return count != null && count.get() >= 5;
    }

    private static void enterRecoveryMode() {
        if (systemInRecoveryMode) {
            return;
        }
        systemInRecoveryMode = true;
        recoveryStartTime = System.currentTimeMillis();
        if (OrthoviewErrorHandler.MC.f_91074_ != null) {
            OrthoviewErrorHandler.MC.f_91074_.m_213846_((Component)Component.m_237113_((String)"[Orthoview] System entering recovery mode due to errors. Some features may be temporarily disabled."));
        }
        CameraManager.reset();
        errorCounts.clear();
    }

    public static void checkRecoveryMode() {
        long timeSinceRecovery;
        if (systemInRecoveryMode && (timeSinceRecovery = System.currentTimeMillis() - recoveryStartTime) >= 10000L) {
            OrthoviewErrorHandler.exitRecoveryMode();
        }
        OrthoviewErrorHandler.resetOldErrors();
    }

    private static void exitRecoveryMode() {
        systemInRecoveryMode = false;
        if (OrthoviewErrorHandler.MC.f_91074_ != null) {
            OrthoviewErrorHandler.MC.f_91074_.m_213846_((Component)Component.m_237113_((String)"[Orthoview] System recovery complete. Full functionality restored."));
        }
    }

    private static void resetOldErrors() {
        long currentTime = System.currentTimeMillis();
        lastErrorTime.entrySet().removeIf(entry -> currentTime - (Long)entry.getValue() > 30000L);
        errorCounts.entrySet().removeIf(entry -> !lastErrorTime.containsKey(entry.getKey()));
    }

    private static RecoveryStrategy getRecoveryStrategy(ErrorType errorType) {
        switch (errorType) {
            case CAMERA_MOVEMENT_FAILED: 
            case COORDINATE_CALCULATION_ERROR: {
                return RecoveryStrategy.RESET_CAMERA;
            }
            case SMOOTH_TRANSITION_ERROR: {
                return RecoveryStrategy.DISABLE_SMOOTH_TRANSITIONS;
            }
            case INVALID_PLAYER_STATE: 
            case WORLD_ACCESS_ERROR: {
                return RecoveryStrategy.FALLBACK_TO_BASIC_MODE;
            }
            case INPUT_PROCESSING_ERROR: 
            case RENDERING_ERROR: {
                return RecoveryStrategy.RESTART_ORTHOVIEW;
            }
            case FOG_INTEGRATION_ERROR: 
            case LEAVES_HIDING_ERROR: {
                return RecoveryStrategy.FALLBACK_TO_BASIC_MODE;
            }
        }
        return RecoveryStrategy.RESET_CAMERA;
    }

    private static boolean applyRecoveryStrategy(RecoveryStrategy strategy, ErrorType errorType, String context) {
        try {
            switch (strategy) {
                case RESET_CAMERA: {
                    CameraManager.reset();
                    OrthoviewErrorHandler.showUserMessage("Camera reset due to " + errorType.getDescription().toLowerCase());
                    return true;
                }
                case DISABLE_SMOOTH_TRANSITIONS: {
                    CameraManager.reset();
                    OrthoviewErrorHandler.showUserMessage("Smooth camera transitions temporarily disabled");
                    return true;
                }
                case FALLBACK_TO_BASIC_MODE: {
                    CameraManager.reset();
                    InputManager.reset();
                    OrthoviewErrorHandler.showUserMessage("Orthoview switched to basic mode");
                    return true;
                }
                case RESTART_ORTHOVIEW: {
                    if (OrthoviewClientEvents.isEnabled()) {
                        OrthoviewClientEvents.toggleEnable();
                        OrthoviewErrorHandler.showUserMessage("Orthoview restarting...");
                    }
                    return true;
                }
                case GRACEFUL_SHUTDOWN: {
                    OrthoviewErrorHandler.emergencyShutdown("Critical error: " + errorType.getDescription());
                    return false;
                }
            }
            return false;
        }
        catch (Exception recoveryException) {
            OrthoviewErrorHandler.logError(ErrorType.RENDERING_ERROR, recoveryException, "Recovery strategy failed: " + strategy);
            return false;
        }
    }

    private static void emergencyShutdown(String reason) {
        try {
            if (OrthoviewClientEvents.isEnabled()) {
                OrthoviewClientEvents.toggleEnable();
            }
            if (OrthoviewErrorHandler.MC.f_91074_ != null) {
                OrthoviewErrorHandler.MC.f_91074_.m_213846_((Component)Component.m_237113_((String)("[Orthoview] Emergency shutdown: " + reason)));
            }
        }
        catch (Exception e) {
            System.err.println("[Orthoview] Emergency shutdown failed: " + e.getMessage());
        }
    }

    private static void logError(ErrorType errorType, Exception exception, String context) {
        String message = String.format("[Orthoview Error] %s in %s: %s", errorType.getDescription(), context, exception != null ? exception.getMessage() : "Unknown error");
        System.err.println(message);
        if (OrthoviewErrorHandler.MC.f_91066_.f_92063_ && exception != null) {
            exception.printStackTrace();
        }
    }

    private static void showUserMessage(String message) {
        if (OrthoviewErrorHandler.MC.f_91074_ != null) {
            OrthoviewErrorHandler.MC.f_91074_.m_213846_((Component)Component.m_237113_((String)("[Orthoview] " + message)));
        }
    }

    public static boolean isInRecoveryMode() {
        return systemInRecoveryMode;
    }

    public static String getErrorStatistics() {
        StringBuilder stats = new StringBuilder();
        stats.append("Orthoview Error Statistics:\n");
        if (errorCounts.isEmpty()) {
            stats.append("No errors recorded");
        } else {
            for (ErrorType errorType : ErrorType.values()) {
                AtomicInteger count = errorCounts.get((Object)errorType);
                if (count == null || count.get() <= 0) continue;
                stats.append(String.format("- %s: %d errors\n", errorType.getDescription(), count.get()));
            }
        }
        if (systemInRecoveryMode) {
            long timeInRecovery = System.currentTimeMillis() - recoveryStartTime;
            stats.append(String.format("System in recovery mode for %d seconds\n", timeInRecovery / 1000L));
        }
        return stats.toString();
    }

    public static boolean safeExecute(Runnable operation, ErrorType errorType, String context) {
        try {
            operation.run();
            return true;
        }
        catch (Exception e) {
            return OrthoviewErrorHandler.handleError(errorType, e, context);
        }
    }

    public static <T> T safeExecute(Supplier<T> operation, T defaultValue, ErrorType errorType, String context) {
        try {
            return operation.get();
        }
        catch (Exception e) {
            OrthoviewErrorHandler.handleError(errorType, e, context);
            return defaultValue;
        }
    }

    public static boolean validateOrthoviewState() {
        if (OrthoviewErrorHandler.MC.f_91074_ == null) {
            OrthoviewErrorHandler.handleError(ErrorType.INVALID_PLAYER_STATE, "Player is null", "validateOrthoviewState");
            return false;
        }
        if (OrthoviewErrorHandler.MC.f_91073_ == null) {
            OrthoviewErrorHandler.handleError(ErrorType.WORLD_ACCESS_ERROR, "Level is null", "validateOrthoviewState");
            return false;
        }
        if (OrthoviewErrorHandler.MC.f_91072_ != null && (OrthoviewErrorHandler.MC.f_91072_.m_105295_() == GameType.ADVENTURE || OrthoviewErrorHandler.MC.f_91072_.m_105295_() == GameType.SURVIVAL)) {
            OrthoviewErrorHandler.handleError(ErrorType.INVALID_PLAYER_STATE, "Orthoview not supported in Adventure/Survival mode", "validateOrthoviewState");
            return false;
        }
        return true;
    }

    public static enum ErrorType {
        CAMERA_MOVEMENT_FAILED("Camera movement failed"),
        INVALID_PLAYER_STATE("Invalid player state"),
        WORLD_ACCESS_ERROR("World access error"),
        COORDINATE_CALCULATION_ERROR("Coordinate calculation error"),
        SMOOTH_TRANSITION_ERROR("Smooth transition error"),
        INPUT_PROCESSING_ERROR("Input processing error"),
        RENDERING_ERROR("Rendering error"),
        FOG_INTEGRATION_ERROR("Fog integration error"),
        LEAVES_HIDING_ERROR("Leaves hiding error"),
        COLLISION_DETECTION_ERROR("Collision detection error");

        private final String description;

        private ErrorType(String description) {
            this.description = description;
        }

        public String getDescription() {
            return this.description;
        }
    }

    public static enum RecoveryStrategy {
        RESET_CAMERA,
        DISABLE_SMOOTH_TRANSITIONS,
        FALLBACK_TO_BASIC_MODE,
        RESTART_ORTHOVIEW,
        GRACEFUL_SHUTDOWN;

    }
}

