#version 120

// Bloom settings
#define BLOOM_ENABLED 1 // Enable bloom effect [0 1]
#define BLOOM_RADIUS 10.0 // Bloom blur radius [2.5 5.0 10.0 15.0 20.0]
#define BLOOM_DOWNSAMPLE 2 // Downsample factor for better connectivity [1 2 4]
#define PIXEL_BLOOM 1 //pixlated bloom [0 1]

uniform sampler2D colortex2; // Bright areas from composite1
uniform float viewWidth;
uniform float viewHeight;

/* RENDERTARGETS:3 */

varying vec2 texcoord;

// Downsample with box filter for better connectivity
vec3 downsampleBrightAreas(sampler2D tex, vec2 texcoord) {
    #if BLOOM_DOWNSAMPLE <= 1
		return texture2D(tex, texcoord).rgb;
    #endif
    
    vec2 pixelSize = vec2(1.0 / viewWidth, 1.0 / viewHeight) * BLOOM_DOWNSAMPLE;
    vec3 result = vec3(0.0);
    int samples = 0;
    
    // Box filter with the downsample size
    int radius = int(BLOOM_DOWNSAMPLE);
    for (int x = -radius; x <= radius; x++) {
        for (int y = -radius; y <= radius; y++) {
            vec2 offset = vec2(float(x), float(y)) * pixelSize;
            result += texture2D(tex, texcoord + offset).rgb;
            samples++;
        }
    }
    
    return result / float(samples);
}

// Horizontal gaussian blur
vec3 gaussianBlurHorizontal(sampler2D tex, vec2 texcoord) {
    #if BLOOM_ENABLED == 0
    return vec3(0.0);
    #endif
    #if PIXEL_BLOOM == 1
		vec2 pixelSize = vec2(1.0 / viewWidth, 1.0 / viewHeight);
		vec3 result = vec3(0.0);
		float totalWeight = 0.0;

		// Gaussian weights
		float weights[5] = float[](0.227027, 0.1945946, 0.1216216, 0.054054, 0.016216);

		// Center sample
		result += texture2D(tex, texcoord).rgb * weights[0];
		totalWeight += weights[0];

		// Horizontal samples
		for (int i = 1; i < 5; i++) {
			vec2 offset = vec2(1.0, 0.0) * pixelSize * float(i) * BLOOM_RADIUS;
			result += texture2D(tex, texcoord + offset).rgb * weights[i];
			result += texture2D(tex, texcoord - offset).rgb * weights[i];
			totalWeight += weights[i] * 2.0;
		}
	#else
		// First downsample for better connectivity
		vec3 downsampled = downsampleBrightAreas(tex, texcoord);
		
		vec2 pixelSize = vec2(1.0 / viewWidth, 1.0 / viewHeight);
		vec3 result = vec3(0.0);
		float totalWeight = 0.0;
		
		// Gaussian weights
		float weights[5] = float[](0.227027, 0.1945946, 0.1216216, 0.054054, 0.016216);
		
		// Center sample
		result += downsampled * weights[0];
		totalWeight += weights[0];
		
		// Horizontal samples
		for (int i = 1; i < 5; i++) {
			vec2 offset = vec2(1.0, 0.0) * pixelSize * float(i) * BLOOM_RADIUS;
			vec3 leftSample = downsampleBrightAreas(tex, texcoord - offset);
			vec3 rightSample = downsampleBrightAreas(tex, texcoord + offset);
			result += leftSample * weights[i];
			result += rightSample * weights[i];
			totalWeight += weights[i] * 2.0;
		}
	#endif
    
    return result / totalWeight;
}

void main() {
    // Horizontal blur with improved connectivity
    vec3 bloomBlur = gaussianBlurHorizontal(colortex2, texcoord);
    
    // Store horizontally blurred bloom in colortex3
    gl_FragData[0] = vec4(bloomBlur, 1.0);
}