#version 120

// DRAWING STYLE SYSTEM
#define DRAWING_STYLE 1 // Drawing Style [1 2]

// Style-dependent settings resolution
#if DRAWING_STYLE == 1
    // Pencil on Paper Style
    #define STYLE_GRAYSCALE_STRENGTH 1.0
#elif DRAWING_STYLE == 2
    // Colorful Sketch Style
    #define STYLE_GRAYSCALE_STRENGTH 0.0
#else
    // Fallback to Pencil style
    #define STYLE_GRAYSCALE_STRENGTH 1.0
#endif

// Paper Settings
#define PAPER_STYLE 1 // Enable paper style [0 1]
#define DISABLE_TEXTURES 0 // Permanently disable all texture rendering [0 1]
#define PAPER_WHITENESS 1.0 // How white the paper is [0.5 0.6 0.7 0.8 0.9 1.0]
#define PAPER_MAX_BRIGHTNESS 0.75 // Maximum paper brightness multiplier [0.7 0.75 0.8 0.85 0.9 0.95 1.0]
#define PAPER_MIN_BRIGHTNESS 0.05 // Minimum paper brightness [0.05 0.1 0.15 0.2 0.25 0.3]
#define PAPER_BRIGHTNESS_SOFTNESS 0.05 // How soft the minimum clamp is [0.02 0.05 0.1 0.15]
#define PAPER_WARMTH 0.2 // How warm/sepia the paper tone is [0.0 0.1 0.15 0.2 0.3 0.4 0.5 0.6 0.7 0.8 0.9 1.0]
#define PAPER_TEXTURE_STRENGTH 0.05 // Paper texture intensity [0.0 0.05 0.08 0.15 0.3]
#define PAPER_TEXTURE_SCALE 64.0 // Paper texture scale [8.0 16.0 32.0 64.0]
#define PAPER_TEXTURE_SPEED 2.0 // Paper texture update rate [0.1 0.5 1.0 2.0 5.0 10.0]
#define PAPER_CRINKLE_STRENGTH 0.03 // Paper crinkle/ridge intensity [0.0 0.02 0.03 0.08 0.15]
#define PAPER_CRINKLE_SCALE 8.0 // Paper crinkle scale [2.0 4.0 8.0 16.0]
#define DARKNESS_THRESHOLD 0.1 // Below this brightness, preserve darkness [0.1 0.15 0.2 0.3 0.4]
#define DARKNESS_SMOOTHNESS 0.05 // How smoothly darkness transitions [0.02 0.05 0.1 0.15]
#define GRAYSCALE_STRENGTH STYLE_GRAYSCALE_STRENGTH // How much to convert to grayscale [0.0 0.125 0.25 0.375 0.5 0.625 0.75 0.875 1.0]	

// MASTER GRAIN SYSTEM TOGGLE
#define ADVANCED_GRAIN_SYSTEM 1 // Enable advanced grain interaction system [0 1] - Set to 0 for original basic paper texture

// Advanced Pencil Grain Interaction Settings (only used when ADVANCED_GRAIN_SYSTEM = 1)
#define PENCIL_GRAIN_ENABLED 1 // Enable pencil-paper grain interaction [0 1]
#define GRAIN_HEIGHT_SCALE 128.0 // Scale of grain height map [64.0 96.0 128.0 192.0 256.0]
#define GRAIN_INTENSITY 1.2 // Overall grain effect strength [0.4 0.6 0.8 1.0 1.2]
#define GRAIN_DIRECTIONAL_BIAS 0.5 // Diagonal bias strength [0.0 0.2 0.3 0.4 0.5]
#define GRAIN_FINE_DETAIL 0.6 // Fine speckle detail strength [0.2 0.4 0.6 0.8 1.0]
#define PRESSURE_SOFTENING 0.4 // How much pressure softens grain effect [0.2 0.3 0.4 0.5 0.6]

// Graphite-on-paper (multi-layer) controls (only used when ADVANCED_GRAIN_SYSTEM = 1)
#define GRAIN_L1_SCALE        192.0   // fine light speckle (higher -> finer) [32.0 64.0 96.0 128.0 192.0 256.0]
#define GRAIN_L2_SCALE         96.0   // darker, sparser flecks (lower -> larger blobs) [32.0 64.0 96.0 128.0 192.0 256.0]

// Pencil Line Settings
#define OUTLINE_STRENGTH 1.5 // Overall outline intensity [0.0 0.5 0.8 1.0 1.5]
#define FINAL_THICKNESS_BOOST 2.0 // Overall thickness multiplier [1.0 2.0 4.0 6.0]
#define PENCIL_CORE_STRENGTH 0.7 // How dark the line center is [0.3 0.5 0.7 0.9]
#define THIN_LINE_FADE_FACTOR 2.0 // How much thinner lines fade [1.0 1.5 2.0 3.0]
#define ENABLE_SOFT_PENCIL_LINE 1 // Soft or solid pencil line [0 1]
#define PENCIL_R 0.25 //Pencil color red [0.00 0.05 0.10 0.15 0.20 0.25 0.30 0.35 0.40 0.45 0.50 0.55 0.60 0.65 0.70 0.75 0.80 0.85 0.90 0.95 1.00]
#define PENCIL_G 0.25 //Pencil color blue [0.00 0.05 0.10 0.15 0.20 0.25 0.30 0.35 0.40 0.45 0.50 0.55 0.60 0.65 0.70 0.75 0.80 0.85 0.90 0.95 1.00]
#define PENCIL_B 0.30 //Pencil color green [0.00 0.05 0.10 0.15 0.20 0.25 0.30 0.35 0.40 0.45 0.50 0.55 0.60 0.65 0.70 0.75 0.80 0.85 0.90 0.95 1.00]

// Pencil Line Variation Effects
#define PENCIL_LINE_VARIATION 1 // Enable pencil line variation [0 1]
#define PENCIL_GRAIN_INTENSITY 0.4 // Pencil grain/texture intensity [0.0 0.2 0.4 0.6 0.8]
#define PENCIL_PRESSURE_VARIATION 0.6 // Simulated pressure variation [0.0 0.3 0.6 0.9]
#define LINE_BLUR_RADIUS 1.5 // Edge blur radius for pencil lines [0.5 1.0 1.5 2.0]

// Bloom Settings
#define BLOOM_ENABLED 1 // Enable bloom effect [0 1]
#define BLOOM_INTENSITY 0.9 // Bloom intensity [0.1 0.3 0.6 0.9 1.2]

// Draw-On Animation Settings
#define DRAW_ON_ENABLED 1 // Enable draw-on effect [0 1]
#define TEXTURE_START_TIME 1.7 // When textures start appearing [5.0 6.0 7.0]
#define TEXTURE_DURATION 0.5// Duration for textures to fully appear [1.0 2.0 3.0]

// Color Grading Controls
#define COLOR_GRADING_ENABLED 1 // Enable color grading system [0 1]
#define VIBRANCE_AMOUNT 0.0 // Vibrance adjustment [-1.0 -0.75 -0.5 -0.25 0.0 0.25 0.5 0.75 1.0]
#define SATURATION_AMOUNT 1.0 // Saturation multiplier [0.0 0.25 0.5 0.75 1.0 1.25 1.5 1.75 2.0]
#define CONTRAST_AMOUNT 1.0 // Contrast adjustment [0.1 0.5 0.75 1.0 1.25 1.5 2.0 2.5 3.0]
#define BRIGHTNESS_FACTOR 0.0 // Brightness adjustment [-1.0 -0.5 -0.1 0.0 0.1 0.5 0.75 1.0 1.25 1.5 2.0 2.5 3.0]
#define LIFT_AMOUNT 0.0 // Lift (shadows)  [-1.0 -0.75 -0.5 -0.25 0.0 0.25 0.5 0.75 1.0]
#define GAMMA_AMOUNT 1.0 // Gamma (midtones) [0.1 0.5 0.75 1.0 1.25 1.5 2.0 2.5 3.0]
#define GAIN_AMOUNT 1.0 // Gain (highlights) [0.0 0.25 0.5 0.75 1.0 1.25 1.5 1.75 2.0]

// Fun/experiemental effect
#define COLOR_GRID_ENABLED 0 // Enable color assimilation grid effect [0 1]
#define GRID_COLOR_SATURATION 6.5 // Oversaturation of grid lines [1.5 2.0 2.5 3.0 3.5]
#define GRID_COLOR_INTENSITY 8.0 // How much color to apply to grid lines [0.5 0.6 0.7 0.8 0.9 1.0]
#define BASE_DESATURATION 1.0 // How much to desaturate the base image [0.7 0.8 0.85 0.9 0.95 1.0]
#define COLOR_BLEEDING 0.0 // How much grid color bleeds into surrounding areas [0.1 0.2 0.3 0.4 0.5]
#define ADAPTIVE_COLOR_STRENGTH 1 // Make color strength adapt to original image colors [0 1]


uniform sampler2D colortex0; // Color with outline data in alpha
uniform sampler2D colortex1; // Now contains edge direction in RG and hatching in B
uniform sampler2D colortex2; // Bloom buffer
uniform float frameTimeCounter;
uniform float viewWidth;
uniform float viewHeight;
varying vec2 texcoord;

// Simple hash function for noise generation
float hash(vec2 p) {
    return fract(sin(dot(p, vec2(127.1, 311.7))) * 43758.5453);
}

// 2D noise function
float noise(vec2 p) {
    vec2 i = floor(p);
    vec2 f = fract(p);
    f = f * f * (3.0 - 2.0 * f);
    
    return mix(mix(hash(i), hash(i + vec2(1,0)), f.x),
               mix(hash(i + vec2(0,1)), hash(i + vec2(1,1)), f.x), f.y);
}

// Ridged noise for paper crinkles
float ridgedNoise(vec2 p) {
    return abs(noise(p) * 2.0 - 1.0);
}

// Generate paper-like texture with crinkles and fine grain (ORIGINAL SYSTEM)
float paperTexture(vec2 uv) {
    // Discrete time steps for random texture changes
    float timeStep = floor(frameTimeCounter * PAPER_TEXTURE_SPEED);
    
    // Create different random seeds for each time step
    float seed1 = hash(vec2(timeStep, 1.0)) * 1000.0;
    float seed2 = hash(vec2(timeStep, 2.0)) * 1000.0;
    float seed3 = hash(vec2(timeStep, 3.0)) * 1000.0;
    float seed4 = hash(vec2(timeStep, 4.0)) * 1000.0;
    float seed5 = hash(vec2(timeStep, 5.0)) * 1000.0;
    
    // Large-scale paper crinkles
    float crinkles = 0.0;
    crinkles += ridgedNoise(uv * PAPER_CRINKLE_SCALE + seed5) * 0.6;
    crinkles += ridgedNoise(uv * PAPER_CRINKLE_SCALE * 2.0 + seed5 * 1.3) * 0.4;
    crinkles *= PAPER_CRINKLE_STRENGTH;
    
    // Fine paper grain texture
    float fineGrain = 0.0;
    fineGrain += noise(uv * PAPER_TEXTURE_SCALE * 2.0 + seed1) * 0.5;
    fineGrain += noise(uv * PAPER_TEXTURE_SCALE + seed2) * 0.3;
    fineGrain += noise(uv * PAPER_TEXTURE_SCALE * 0.5 + seed3) * 0.2;
    fineGrain += noise(uv * vec2(PAPER_TEXTURE_SCALE * 4.0, PAPER_TEXTURE_SCALE * 0.5) + seed4) * 0.1;
    fineGrain = (fineGrain - 0.5) * PAPER_TEXTURE_STRENGTH;
    
    return crinkles + fineGrain;
}

// ADVANCED GRAIN SYSTEM FUNCTIONS (only compiled when ADVANCED_GRAIN_SYSTEM = 1)
#if ADVANCED_GRAIN_SYSTEM == 1

	// Generate grain height map for pencil interaction
	float generateGrainHeightMap(vec2 uv) {
		float timeStep = floor(frameTimeCounter * PAPER_TEXTURE_SPEED);
		float grainSeed = hash(vec2(timeStep, 10.0)) * 1000.0;
		
		// Multi-octave noise for realistic grain peaks and valleys
		float height = 0.0;
		
		// Primary grain structure (medium frequency)
		height += noise(uv * GRAIN_HEIGHT_SCALE + grainSeed) * 0.5;
		height += noise(uv * GRAIN_HEIGHT_SCALE * 2.0 + grainSeed * 1.3) * 0.25;
		
		// Fine detail (high frequency speckles)
		height += noise(uv * GRAIN_HEIGHT_SCALE * 4.0 + grainSeed * 1.7) * 0.15 * GRAIN_FINE_DETAIL;
		height += noise(uv * GRAIN_HEIGHT_SCALE * 8.0 + grainSeed * 2.1) * 0.1 * GRAIN_FINE_DETAIL;
		
		// Organic directional variation (no screen-space gradients)
		if (GRAIN_DIRECTIONAL_BIAS > 0.0) {
			// Per-pixel directional noise instead of screen-space bias
			float dirNoise1 = noise(uv * GRAIN_HEIGHT_SCALE * 0.3 + grainSeed * 0.7);
			float dirNoise2 = noise(uv * GRAIN_HEIGHT_SCALE * 0.6 + grainSeed * 1.1);
			
			// Create organic directional variation
			float directionalHeight = (dirNoise1 - 0.5) * (dirNoise2 - 0.5) * 4.0; // -1 to 1 range
			height += directionalHeight * GRAIN_DIRECTIONAL_BIAS * 0.1;
		}
		
		return clamp(height, 0.0, 1.0);
	}

	// Apply pencil grain interaction to darken colors realistically
	vec3 applyPencilGrainInteraction(vec3 inputColor, vec2 uv) {
		#if PENCIL_GRAIN_ENABLED == 0
		return inputColor;
		#endif

		// Base luminance
		float L = dot(inputColor, vec3(0.299, 0.587, 0.114));
		float darkness = 1.0 - L;
		if (darkness < 0.05) return inputColor;

		// Seeds
		float timeStep = floor(frameTimeCounter * PAPER_TEXTURE_SPEED);
		float seedA = hash(vec2(timeStep, 11.0)) * 1000.0;
		float seedB = hash(vec2(timeStep, 17.0)) * 1000.0;

		// Paper peaks/valleys
		float Hbase = generateGrainHeightMap(uv);

		// -----------------------------
		// Layer 1: fine gray speckle
		float H1 = noise(uv * GRAIN_L1_SCALE + seedA);
		float Hmix1 = mix(H1, Hbase, 0.5);
		float cov1 = smoothstep(0.8, 0.3, Hmix1) * pow(darkness, 1.1);
		float lay1Dark = 0.14 * cov1;

		// Layer 2: sparse dark flecks
		float H2 = noise(uv * GRAIN_L2_SCALE + seedB);
		float Hmix2 = mix(H2, Hbase, 0.7);
		float cov2 = smoothstep(0.9, 0.2, Hmix2) * pow(darkness, 9.0); //to make lighter increase pow
		float lay2Dark = 0.5 * cov2; //to make lighter decrease multiplier

		// -----------------------------
		// Stroke directionality
		vec2 dir = normalize(vec2(0.7, 0.5)); // ~diagonal, tune this
		float Dp = noise(uv * GRAIN_L1_SCALE + dir * 3.1 + seedA);
		float Dm = noise(uv * GRAIN_L1_SCALE - dir * 3.1 + seedA);
		float Ddir = clamp(Dp - Dm, -1.0, 1.0); // -1..1 ridge side test

		// Bias graphite layers: one side of hill gets more, one side less
		lay1Dark *= (1.0 + 0.3 * Ddir);
		lay2Dark *= (1.0 + 0.4 * Ddir);

		// -----------------------------
		// Layer 3: forced white specks (negative graphite)
		// strongest on trough side, fades out as darkness rises
		float whiteMask = smoothstep(0.4, 0.8, Hbase); // valleys = high
		float lay3Bright = whiteMask * (1.0 - darkness) * (0.3 + 0.3 * (-Ddir));

		// -----------------------------
		// Combine
		float graphite = clamp(lay1Dark + lay2Dark - lay3Bright, 0.0, 0.9);
		float Lout = max(L - graphite * 0.7, 0.0); // Apply with 70% strength

		// Add salt specks *after* darkening
		float salt = noise(uv * 256.0 + seedB * 3.3);
		float saltMask = step(0.9, salt); // rare specks
		float lay4Bright = saltMask * (0.4 * (1.0 - darkness));

		// Specks brighten back toward paper (1.0 = white)
		Lout = mix(Lout, 1.0, lay4Bright);

		float safeL = max(L, 1e-4);
		vec3 outColor = inputColor * (Lout / safeL);

		return outColor;
	}

#endif // End of ADVANCED_GRAIN_SYSTEM functions

// Pencil line rendering
float renderPencilLine(float thickness, vec2 uv) {
    if (thickness <= 0.0) return 0.0;
    
    // Use thickness directly from composite stage
    float lineStrength = thickness;
    
    // Time-consistent grain system
    float timeStep = floor(frameTimeCounter * PAPER_TEXTURE_SPEED);
    float grainSeed = hash(vec2(timeStep, 15.0)) * 1000.0;
    
    // Single unified grain system
    vec2 grainUV = uv * viewWidth * 0.3;
    float pencilGrain = 0.0;
    pencilGrain += noise(grainUV * 6.0 + grainSeed) * 0.5;
    pencilGrain += noise(grainUV * 12.0 + grainSeed * 1.3) * 0.3;
    pencilGrain += noise(grainUV * 24.0 + grainSeed * 1.7) * 0.2;
    pencilGrain = (pencilGrain - 0.5) * 2.0; // -1 to 1 range
    
    // Thickness-based opacity: thinner lines are more transparent
    float thicknessOpacity = pow(lineStrength, 1.0 / (1.0 + THIN_LINE_FADE_FACTOR));
    
    // Edge fading: simulate pencil pressure variation
    float pressureNoise = noise(uv * viewWidth * 0.1 + grainSeed);
    float pressureVariation = mix(0.6, 1.0, (pressureNoise + 1.0) * 0.5);
    pressureVariation = mix(1.0 - PENCIL_PRESSURE_VARIATION * 0.5, 1.0, pressureVariation);
    
    // Core line strength varies with thickness
    float coreStrength = mix(0.2, PENCIL_CORE_STRENGTH, pow(lineStrength, 0.8));
    
    // Combine effects
    float baseIntensity = coreStrength * thicknessOpacity * pressureVariation;
    
    // Add grain (more visible in thicker lines)
    float grainContribution = pencilGrain * PENCIL_GRAIN_INTENSITY * baseIntensity * lineStrength;
    
    // Final intensity with pressure variation
    float finalIntensity = baseIntensity + grainContribution;
    finalIntensity *= mix(1.0 - PENCIL_PRESSURE_VARIATION * 0.3, 1.0, pressureVariation);
    
    return clamp(finalIntensity, 0.0, 1.0);
}

float renderPencilLineWithDirectionalEdges(float thickness, vec2 uv, vec2 edgeDirection) {
    if (thickness <= 0.0) return 0.0;
    
    // Decode edge direction from 0,1 range back to -1,1 range
    vec2 decodedDirection = edgeDirection * 2.0 - 1.0;
    
    // If we don't have a valid edge direction, fall back to regular rendering
    if (length(decodedDirection) < 0.1) {
        return renderPencilLine(thickness, uv);
    }
    
    // Calculate perpendicular direction (rotate 90 degrees)
    vec2 perpendicular = vec2(-decodedDirection.y, decodedDirection.x);
    
    // Sample along the perpendicular direction to find line boundaries
    vec2 pixelSize = vec2(1.0 / viewWidth, 1.0 / viewHeight);
    float maxSearchDistance = 20.0; // Maximum pixels to search in each direction
    
    // Find line boundaries by sampling outward in both perpendicular directions
    float leftBoundary = 0.0;
    float rightBoundary = 0.0;
    
    // Search left (negative perpendicular direction)
    for (float i = 1.0; i <= maxSearchDistance; i += 1.0) {
        vec2 samplePos = uv + perpendicular * pixelSize * -i;
        float sampleThickness = texture2D(colortex0, samplePos).a;
        if (sampleThickness <= 0.1) {
            leftBoundary = i;
            break;
        }
    }
    if (leftBoundary == 0.0) leftBoundary = maxSearchDistance;
    
    // Search right (positive perpendicular direction)
    for (float i = 1.0; i <= maxSearchDistance; i += 1.0) {
        vec2 samplePos = uv + perpendicular * pixelSize * i;
        float sampleThickness = texture2D(colortex0, samplePos).a;
        if (sampleThickness <= 0.1) {
            rightBoundary = i;
            break;
        }
    }
    if (rightBoundary == 0.0) rightBoundary = maxSearchDistance;
    
    // Calculate line width and center position
    float totalLineWidth = leftBoundary + rightBoundary;
    float distanceFromCenter = min(leftBoundary, rightBoundary);
    
    // Calculate edge falloff - closer to edge = more transparent
    float edgeFalloff = 1.0;
    if (totalLineWidth > 2.0) { // Only apply falloff for lines wide enough
        float edgeDistance = distanceFromCenter / (totalLineWidth * 0.5);
        // Smooth falloff from center (1.0) to edges (0.2)
        edgeFalloff = mix(0.2, 1.0, smoothstep(0.0, 1.0, edgeDistance));
    }
    
    // Time-consistent grain system (same as working renderPencilLine)
    float timeStep = floor(frameTimeCounter * PAPER_TEXTURE_SPEED);
    float grainSeed = hash(vec2(timeStep, 15.0)) * 1000.0;
    
    // Single unified grain system
    vec2 grainUV = uv * viewWidth * 0.3;
    float pencilGrain = 0.0;
    pencilGrain += noise(grainUV * 6.0 + grainSeed) * 0.5;
    pencilGrain += noise(grainUV * 12.0 + grainSeed * 1.3) * 0.3;
    pencilGrain += noise(grainUV * 24.0 + grainSeed * 1.7) * 0.2;
    pencilGrain = (pencilGrain - 0.5) * 2.0; // -1 to 1 range
    
    // Thickness-based opacity: thinner lines are more transparent
    float thicknessOpacity = pow(thickness, 1.0 / (1.0 + THIN_LINE_FADE_FACTOR));
    
    // Edge fading: simulate pencil pressure variation
    float pressureNoise = noise(uv * viewWidth * 0.1 + grainSeed);
    float pressureVariation = mix(0.6, 1.0, (pressureNoise + 1.0) * 0.5);
    pressureVariation = mix(1.0 - PENCIL_PRESSURE_VARIATION * 0.5, 1.0, pressureVariation);
    
    // Core line strength varies with thickness and edge falloff
    float coreStrength = mix(0.2, PENCIL_CORE_STRENGTH, pow(thickness, 0.8));
    
    // Apply edge falloff to core strength
    coreStrength *= edgeFalloff;
    
    // Combine effects
    float baseIntensity = coreStrength * thicknessOpacity * pressureVariation;
    
    // Add grain (more visible in thicker lines)
    float grainContribution = pencilGrain * PENCIL_GRAIN_INTENSITY * baseIntensity * thickness;
    
    // Final intensity with pressure variation
    float finalIntensity = baseIntensity + grainContribution;
    finalIntensity *= mix(1.0 - PENCIL_PRESSURE_VARIATION * 0.3, 1.0, pressureVariation);
    
    return clamp(finalIntensity, 0.0, 1.0);
}

// Function to convert RGB to HSV
vec3 rgb2hsv(vec3 c) {
    vec4 K = vec4(0.0, -1.0 / 3.0, 2.0 / 3.0, -1.0);
    vec4 p = mix(vec4(c.bg, K.wz), vec4(c.gb, K.xy), step(c.b, c.g));
    vec4 q = mix(vec4(p.xyw, c.r), vec4(c.r, p.yzx), step(p.x, c.r));
    
    float d = q.x - min(q.w, q.y);
    float e = 1.0e-10;
    return vec3(abs(q.z + (q.w - q.y) / (6.0 * d + e)), d / (q.x + e), q.x);
}

// Function to convert HSV to RGB
vec3 hsv2rgb(vec3 c) {
    vec4 K = vec4(1.0, 2.0 / 3.0, 1.0 / 3.0, 3.0);
    vec3 p = abs(fract(c.xxx + K.xyz) * 6.0 - K.www);
    return c.z * mix(K.xxx, clamp(p - K.xxx, 0.0, 1.0), c.y);
}

// Enhanced desaturation that preserves some color information
vec3 smartDesaturate(vec3 color, float amount) {
    vec3 hsv = rgb2hsv(color);
    float originalSaturation = hsv.y;
    
    // Reduce saturation but keep a hint of the original color
    hsv.y = mix(hsv.y, hsv.y * (1.0 - amount), amount);
    
    return hsv2rgb(hsv);
}

// Generate oversaturated color based on original image color
vec3 generateGridColor(vec3 originalColor, vec2 texcoord) {
    vec3 hsv = rgb2hsv(originalColor);
    
    // If the original color has very low saturation, create a synthetic color
    if (hsv.y < 0.1) {
        // Use position-based hue for neutral areas
        float syntheticHue = fract(dot(texcoord, vec2(127.1, 311.7)) * 0.1);
        hsv.x = syntheticHue;
        hsv.y = 0.6; // Moderate base saturation
    }
    
    // Oversaturate the color
    hsv.y = min(hsv.y * GRID_COLOR_SATURATION, 1.0);
    
    // Optionally adjust brightness for better visibility
    hsv.z = clamp(hsv.z * 1.2, 0.3, 0.9);
    
    return hsv2rgb(hsv);
}

// Calculate draw-on progress for different elements
float getDrawOnProgress() {
    #if DRAW_ON_ENABLED == 0 ||  PAPER_STYLE == 0
    return float(1.0); // All elements fully visible
    #endif
    
    float time = frameTimeCounter;
	
    // Texture fade-in (0 to 1)
    float textureProgress = smoothstep(TEXTURE_START_TIME, TEXTURE_START_TIME + TEXTURE_DURATION, time);
    
    return textureProgress;
}

// Replace your getEnhancedBrushMask function with this corrected version:
float getEnhancedBrushMask(vec2 texcoord, float progress) {
    return progress;
}

float minOf3(vec3 v) {
    return min(min(v.r, v.g), v.b);
}

float maxOf3(vec3 v) {
    return max(max(v.r, v.g), v.b);
}

// Get index of maximum component (0=r, 1=g, 2=b)
int maxIdx(vec3 color, out float maxVal) {
    maxVal = maxOf3(color);
    if (color.r == maxVal) return 0;
    else if (color.g == maxVal) return 1;
    else return 2;
}

// Get index of minimum component (0=r, 1=g, 2=b)
int minIdx(vec3 color, out float minVal) {
    minVal = minOf3(color);
    if (color.r == minVal) return 0;
    else if (color.g == minVal) return 1;
    else return 2;
}

float luminance(vec3 color) {
    return dot(color, vec3(0.299, 0.587, 0.114));
}

float clamp01(float x) {
    return clamp(x, 0.0, 1.0);
}

float pow2(float x) {
    return x * x;
}

// =============================================================================
// VIBRANCE FUNCTION Based on snippit from belmu (shaderlabs discord)
// =============================================================================
void applyVibrance(inout vec3 color, float intensity) {
    if (abs(intensity) < 0.001) return; // Skip if no effect
    
    float mn = minOf3(color);
    float mx = maxOf3(color);
    float sat = (1.0 - clamp01(mx - mn)) * clamp01(1.0 - mx) * luminance(color) * 5.0;
    vec3 lightness = vec3((mn + mx) * 0.5);
    
    // Positive vibrance
    color = mix(color, mix(lightness, color, intensity), sat);
    // Negative vibrance
    color = mix(color, lightness, (1.0 - lightness) * (1.0 - intensity) * 0.5 * abs(intensity));
}

// =============================================================================
// SATURATION Based on snippit from belmu (shaderlabs discord)
// =============================================================================
void applySaturation(inout vec3 color, float intensity) {
    if (abs(intensity - 1.0) < 0.001) return; // Skip if no change
    color = mix(vec3(luminance(color)), color, intensity);
}

// =============================================================================
// NATURAL CONTRAST FUNCTION based on snippit from RedstoneCake (shaderlabs discord)
// =============================================================================
void applyContrastChannel(inout float channel, float a, float b) {
    if (channel < 0.5) {
        channel = b * pow(channel, a);
    } else {
        channel = 1.0 - b * pow(abs(channel - 1.0), a);
    }
}

void applyContrast(inout vec3 color, float amount) {
    if (abs(amount - 1.0) < 0.001) return; // Skip if no change
    
    float b = exp2(amount - 1.0);
    applyContrastChannel(color.r, amount, b);
    applyContrastChannel(color.g, amount, b);
    applyContrastChannel(color.b, amount, b);
}

// =============================================================================
// BRIGHTNESS FUNCTION based on snippit form Builderb0y (shaderlabs discord)
// =============================================================================
void applyBrightness(inout vec3 color, float factor) {
    if (abs(factor) < 0.001) return; // Skip if no change
    
    float coefficient = exp(factor) - 1.0;
    color = (coefficient * color + color) / (coefficient * color + 1.0);
    color = clamp(color, vec3(0.0), vec3(1.0));
}

// =============================================================================
// LIFT GAMMA GAIN FUNCTION Based on snippit from belmu (shaderlabs discord)
// =============================================================================
void applyLiftGammaGain(inout vec3 color, float lift, float gamma, float gain) {
    if (abs(lift) < 0.001 && abs(gamma - 1.0) < 0.001 && abs(gain - 1.0) < 0.001) return;
    
    vec3 lerpV = clamp(pow(color, vec3(gamma)), vec3(0.0), vec3(1.0));
    color = gain * lerpV + lift * (1.0 - lerpV);
    color = clamp(color, vec3(0.0), vec3(1.0));
}

// MASTER COLOR GRADING FUNCTION
vec3 applyColorGrading(vec3 color) {
    #if COLOR_GRADING_ENABLED == 0
    return color;
    #endif
    
    vec3 gradedColor = color;
    
    // Pre-clamp the input to prevent issues with out-of-range values
    gradedColor = clamp(gradedColor, vec3(0.0), vec3(1.0));
    
    // Apply color adjustments in proper order
    applyLiftGammaGain(gradedColor, LIFT_AMOUNT, GAMMA_AMOUNT, GAIN_AMOUNT);
    applyContrast(gradedColor, CONTRAST_AMOUNT);
    applyBrightness(gradedColor, BRIGHTNESS_FACTOR);
    applySaturation(gradedColor, SATURATION_AMOUNT);
    applyVibrance(gradedColor, VIBRANCE_AMOUNT);
    
    return gradedColor;
}

float softMinClamp(float value, float minVal, float softness) {
    float diff = value - minVal;
    if (diff < 0.0) {
        // Use exponential curve to approach but never quite reach minimum
        return minVal + diff * exp(diff / softness);
    }
    return value;
}

vec3 applySoftBrightnessLimits(vec3 color, vec3 paperBase) {
    vec3 minColor = paperBase * PAPER_MIN_BRIGHTNESS;
    vec3 maxColor = paperBase;
    
    // Apply soft minimum and hard maximum
    color.r = softMinClamp(color.r, minColor.r, PAPER_BRIGHTNESS_SOFTNESS);
    color.g = softMinClamp(color.g, minColor.g, PAPER_BRIGHTNESS_SOFTNESS);
    color.b = softMinClamp(color.b, minColor.b, PAPER_BRIGHTNESS_SOFTNESS);
    
    // Keep hard maximum clamp
    return min(color, maxColor);
}

// Pre-adjust colors before clamping
vec3 prepareDarkRegionsForGrain(vec3 color, vec3 paperBase) {
    float brightness = dot(color, vec3(0.299, 0.587, 0.114));
    vec3 minColor = paperBase * PAPER_MIN_BRIGHTNESS;
    
    if (brightness < DARKNESS_THRESHOLD * 1.5) {
        // In dark regions, lift the color slightly to give room for grain darkening
        float liftAmount = smoothstep(DARKNESS_THRESHOLD * 1.5, 0.0, brightness) * 0.15;
        color = mix(color, color + liftAmount, 0.5);
    }
    
    return color;
}

void main() {
    vec4 colorData = texture2D(colortex0, texcoord);
    vec3 color = colorData.rgb;
    float rawOutline = colorData.a;
    
    vec4 edgeData = texture2D(colortex1, texcoord);
    vec2 edgeDirection = edgeData.rg;
    float hatchingOutline = edgeData.b;
    
    // Get draw-on progress
    float textureProgress = getDrawOnProgress();
    
    #if BLOOM_ENABLED == 1
        vec3 bloomColor = texture2D(colortex2, texcoord).rgb;
    #else
        vec3 bloomColor = vec3(0.0);
    #endif
    
    #if PAPER_STYLE == 1
        // Start with paper base color
        vec3 paperBase = vec3(
            PAPER_WHITENESS * (1.0 + PAPER_WARMTH * 0.1),
            PAPER_WHITENESS * (1.0 + PAPER_WARMTH * 0.05),
            PAPER_WHITENESS * (1.0 - PAPER_WARMTH * 0.1)
        );
        
        // Initialize with pure paper
        vec3 finalColor = paperBase;
        

		// Only add content based on draw-on progress
		if (textureProgress > 0.0 && DISABLE_TEXTURES == 0) {
			// Apply your existing color processing
			vec3 processedColor = color;
			
			#if ADVANCED_GRAIN_SYSTEM == 1
				vec2 grainUV = texcoord * vec2(viewWidth, viewHeight) / 512.0;
				
				// Pre-adjust dark regions to give room for grain
				processedColor = prepareDarkRegionsForGrain(processedColor, paperBase);
				
				// Apply advanced grain system
				processedColor = applyPencilGrainInteraction(processedColor, grainUV);
			#endif
			
			float brightness = dot(processedColor, vec3(0.299, 0.587, 0.114));
			float darknessFactor = smoothstep(DARKNESS_THRESHOLD - DARKNESS_SMOOTHNESS, DARKNESS_THRESHOLD, brightness);
			darknessFactor = darknessFactor * darknessFactor;
			
			vec3 darkColor = processedColor * (1.0 + vec3(PAPER_WARMTH * 0.1, PAPER_WARMTH * 0.05, -PAPER_WARMTH * 0.1));
			darkColor *= 0.3;
			
			vec3 coloredPaper = processedColor * paperBase;
			vec3 grayscalePaper = paperBase * brightness;
			vec3 paperColor = mix(grayscalePaper, coloredPaper, 1.0 - GRAYSCALE_STRENGTH);
			
			vec3 textureColor = mix(darkColor, paperColor, darknessFactor);
			
			float brushMask;
			if (textureProgress >= 1.0) {
				// Effect is complete, show everything normally
				brushMask = 1.0;
			} else {
				brushMask = getEnhancedBrushMask(texcoord, textureProgress);
			}

			// Blend texture based on brush mask
			finalColor = mix(paperBase, textureColor, brushMask);
			
			// Clamp to paper brightness
			finalColor = clamp(finalColor, 
                  paperBase * PAPER_MIN_BRIGHTNESS*0.9, 
                  paperBase * PAPER_MAX_BRIGHTNESS);
		}
	
        
        // Add bloom with texture progress
        #if BLOOM_ENABLED == 1
            vec3 processedBloom = bloomColor;
            if (GRAYSCALE_STRENGTH > 0.0) {
                float bloomBrightness = dot(bloomColor, vec3(0.299, 0.587, 0.114));
                vec3 grayscaleBloom = vec3(bloomBrightness);
                processedBloom = mix(bloomColor, grayscaleBloom, GRAYSCALE_STRENGTH);
            }
            
            vec3 paperMaxBright = paperBase * PAPER_MAX_BRIGHTNESS;
            vec3 headroom = max(paperMaxBright - finalColor, vec3(0.0));
            float avgHeadroom = (headroom.r + headroom.g + headroom.b) / 3.0;
            float maxHeadroom = (paperMaxBright.r + paperMaxBright.g + paperMaxBright.b) / 3.0;
            
            float headroomFactor = avgHeadroom / max(maxHeadroom, 0.001);
            float adaptiveBloomIntensity = BLOOM_INTENSITY * mix(0.2, 1.0, headroomFactor) * textureProgress;
            
            vec3 bloomContribution = processedBloom * adaptiveBloomIntensity;
            finalColor += bloomContribution;
            finalColor = min(finalColor, paperMaxBright);
        #endif
        
		vec2 paperUV = texcoord * vec2(viewWidth, viewHeight) / 512.0;
		float paperNoise = paperTexture(paperUV);
		float brightness = dot(finalColor, vec3(0.299, 0.587, 0.114));
		float darknessFactor = smoothstep(DARKNESS_THRESHOLD - DARKNESS_SMOOTHNESS, DARKNESS_THRESHOLD, brightness);
		float textureStrength = mix(0.1, 1.0, darknessFactor);
		finalColor += paperNoise * textureStrength;
		
		// Render regular pencil lines with standard blending
		vec2 decodedEdgeDir = edgeDirection * 2.0 - 1.0;
		#if ENABLE_SOFT_PENCIL_LINE == 1
			float pencilLine = renderPencilLineWithDirectionalEdges(rawOutline, texcoord, edgeDirection);
		#else
			float pencilLine = renderPencilLine(rawOutline, texcoord);
		#endif
		float finalOutline = pencilLine * OUTLINE_STRENGTH;
		
        // Pencil color - dark graphite with slight blue tint
        vec3 pencilTone = vec3(PENCIL_R, PENCIL_G, PENCIL_B);
        
        // Apply regular outlines with standard blending
        float sceneBrightness = dot(finalColor.rgb, vec3(0.299, 0.587, 0.114));
        float adjustedOutline = finalOutline * smoothstep(0.1, 0.6, sceneBrightness);
        finalColor *= mix(vec3(1.0), pencilTone, adjustedOutline);
        
        // Add hatching based on progress
        if (hatchingOutline > 0.0) {
            float hatchingLine = renderPencilLine(hatchingOutline, texcoord);
            float finalHatching = hatchingLine * OUTLINE_STRENGTH;
            
            #if COLOR_GRID_ENABLED == 1
				// Get the original color before any processing for grid color generation
				vec3 originalSceneColor = texture2D(colortex0, texcoord).rgb;
				
				// Generate oversaturated grid color based on original image
				vec3 gridColor = generateGridColor(originalSceneColor, texcoord);
				
				#if ADAPTIVE_COLOR_STRENGTH == 1
				// Make color strength adaptive to original image properties
				float originalBrightness = dot(originalSceneColor, vec3(0.299, 0.587, 0.114));
				float originalSaturation = rgb2hsv(originalSceneColor).y;
				float adaptiveStrength = GRID_COLOR_INTENSITY * (0.5 + originalSaturation * 0.5) * (0.7 + originalBrightness * 0.3);
				#else
				float adaptiveStrength = GRID_COLOR_INTENSITY;
				#endif
				
				// Apply colored hatching lines
				vec3 coloredHatching = mix(vec3(1.0), gridColor, adaptiveStrength);
				finalColor *= mix(vec3(1.0), coloredHatching, finalHatching);
				
				// Color bleeding effect - subtle color influence in surrounding areas
				#if COLOR_BLEEDING > 0.0
					if (finalHatching > 0.1) {
						// Sample neighboring pixels to create bleeding effect
						vec2 pixelSize = vec2(1.0 / viewWidth, 1.0 / viewHeight);
						float bleedRadius = 3.0;
						vec3 bleedColor = vec3(0.0);
						float bleedWeight = 0.0;
						
						for (float x = -bleedRadius; x <= bleedRadius; x += 1.0) {
							for (float y = -bleedRadius; y <= bleedRadius; y += 1.0) {
								if (x == 0.0 && y == 0.0) continue;
								
								vec2 sampleUV = texcoord + vec2(x, y) * pixelSize;
								float sampleHatching = texture2D(colortex1, sampleUV).b;
								
								if (sampleHatching > 0.1) {
									float dist = sqrt(x*x + y*y);
									float weight = 1.0 / (1.0 + dist);
									bleedColor += gridColor * weight * sampleHatching;
									bleedWeight += weight;
								}
							}
						}
						
						if (bleedWeight > 0.0) {
							bleedColor /= bleedWeight;
							// Apply subtle bleeding
							finalColor = mix(finalColor, finalColor * (1.0 + bleedColor * COLOR_BLEEDING * 0.2), 0.3);
						}
					}
				#endif
            #else
                vec3 pencilTone = vec3(PENCIL_R, PENCIL_G, PENCIL_B);
                finalColor = finalColor - finalHatching * (1.0 - pencilTone);
                finalColor = max(finalColor, vec3(0.0));
            #endif
        }
		
		// Option 1: Hard Clamp (matches existing PAPER_MAX_BRIGHTNESS approach)
		//finalColor = clamp(finalColor, paperBase * PAPER_MIN_BRIGHTNESS, paperBase * PAPER_MAX_BRIGHTNESS);
		
		// OR Option 2: Soft minimum with hard maximum
		finalColor = applySoftBrightnessLimits(finalColor, paperBase);
		
		//Darkness paper grain effect
		if (brightness < DARKNESS_THRESHOLD * 2.0) { // Affect areas up to 20% brightness
			float darkGrain = paperTexture(paperUV * 1.5); // Different scale/seed
			float darkStrength = smoothstep(DARKNESS_THRESHOLD * 2.0, 0.0, brightness);
			
			// Apply darkening grain to dark areas
			finalColor += darkGrain * darkStrength * -0.3; // Negative = darkening
			finalColor = max(finalColor, vec3(0.0)); // Prevent negative colors
		}
		
        // Apply color grading to final color before output
        finalColor = applyColorGrading(finalColor);
		
		
        
        gl_FragData[0] = vec4(finalColor, 1.0);
    #else
        // Non-paper mode - still apply progress
        vec3 finalColor = mix(vec3(1.0), color, textureProgress);
        
        #if BLOOM_ENABLED == 1
            finalColor += bloomColor * BLOOM_INTENSITY * textureProgress;
        #endif
		
        finalColor = applyColorGrading(finalColor);
        
        gl_FragData[0] = vec4(finalColor, 1.0);
    #endif
}