/*
 * Decompiled with CFR 0.152.
 */
package me.demstudios.database;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.logging.Level;
import me.demstudios.Auction;
import me.demstudios.BLAuction;
import me.demstudios.stats.PlayerStats;
import org.bukkit.Bukkit;
import org.bukkit.inventory.ItemStack;
import org.bukkit.inventory.meta.ItemMeta;
import org.bukkit.util.io.BukkitObjectInputStream;
import org.bukkit.util.io.BukkitObjectOutputStream;
import org.yaml.snakeyaml.external.biz.base64Coder.Base64Coder;

public class AuctionDatabase {
    private final BLAuction plugin;
    private Connection connection;
    private String tablePrefix;
    private boolean isMySQL;

    public AuctionDatabase(BLAuction plugin) {
        this.plugin = plugin;
    }

    public void connect() throws SQLException {
        String dbType = this.plugin.getConfig().getString("database.type", "sqlite").toLowerCase();
        try {
            if (dbType.equals("mysql")) {
                String host = this.plugin.getConfig().getString("database.mysql.host", "localhost");
                int port = this.plugin.getConfig().getInt("database.mysql.port", 3306);
                String dbName = this.plugin.getConfig().getString("database.mysql.database", "minecraft");
                String user = this.plugin.getConfig().getString("database.mysql.username", "root");
                String pass = this.plugin.getConfig().getString("database.mysql.password", "");
                this.tablePrefix = this.plugin.getConfig().getString("database.mysql.table_prefix", "");
                this.isMySQL = true;
                String url = "jdbc:mysql://" + host + ":" + port + "/" + dbName + "?useSSL=false&autoReconnect=true&characterEncoding=utf8";
                this.connection = DriverManager.getConnection(url, user, pass);
                this.plugin.getLogger().info("Successfully connected to MySQL database");
            } else {
                String filename = this.plugin.getConfig().getString("database.sqlite.filename", "auctions.db");
                this.tablePrefix = "";
                this.isMySQL = false;
                this.connection = DriverManager.getConnection("jdbc:sqlite:" + this.plugin.getDataFolder().getAbsolutePath() + "/" + filename);
                this.plugin.getLogger().info("Successfully connected to SQLite database");
            }
            this.createTables();
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Database connection error: " + e.getMessage());
            throw e;
        }
    }

    private void createTables() throws SQLException {
        try (Statement stmt = this.connection.createStatement();){
            String autoIncrement = this.isMySQL ? "AUTO_INCREMENT" : "AUTOINCREMENT";
            stmt.execute("CREATE TABLE IF NOT EXISTS " + this.tablePrefix + "active_auctions (id VARCHAR(36) PRIMARY KEY,seller_uuid VARCHAR(36) NOT NULL,seller_name VARCHAR(16) NOT NULL,item_stack TEXT NOT NULL,price DOUBLE NOT NULL,time_remaining INTEGER NOT NULL,amount INTEGER DEFAULT 1,created_at BIGINT NOT NULL,buyer_uuid VARCHAR(36),buyer_name VARCHAR(16))");
            stmt.execute("CREATE TABLE IF NOT EXISTS " + this.tablePrefix + "pending_items (id INTEGER PRIMARY KEY " + autoIncrement + ",player_uuid VARCHAR(36) NOT NULL,item_stack TEXT NOT NULL,price DOUBLE DEFAULT 0)");
            stmt.execute("CREATE TABLE IF NOT EXISTS " + this.tablePrefix + "player_stats (player_uuid VARCHAR(36) PRIMARY KEY,player_name VARCHAR(16) NOT NULL,auctions_created INTEGER DEFAULT 0,auctions_won INTEGER DEFAULT 0,bids_placed INTEGER DEFAULT 0,total_revenue DOUBLE DEFAULT 0,total_spent DOUBLE DEFAULT 0,first_join_time BIGINT,last_auction_time BIGINT,cooldown_until BIGINT DEFAULT 0)");
            stmt.execute("CREATE INDEX IF NOT EXISTS idx_seller_uuid ON " + this.tablePrefix + "active_auctions (seller_uuid)");
            stmt.execute("CREATE INDEX IF NOT EXISTS idx_player_uuid ON " + this.tablePrefix + "pending_items (player_uuid)");
            if (this.isMySQL) {
                stmt.execute("CREATE INDEX IF NOT EXISTS idx_time_remaining ON " + this.tablePrefix + "active_auctions (time_remaining)");
                stmt.execute("CREATE INDEX IF NOT EXISTS idx_created_at ON " + this.tablePrefix + "active_auctions (created_at)");
            }
        }
    }

    public void saveAuction(Auction auction) {
        String sql = "INSERT INTO " + this.tablePrefix + "active_auctions (id, seller_uuid, seller_name, item_stack, price, time_remaining, amount, created_at, buyer_uuid, buyer_name) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, auction.getId().toString());
            stmt.setString(2, auction.getSellerId().toString());
            stmt.setString(3, auction.getSellerName());
            stmt.setString(4, this.serializeItemStack(auction.getItemStack()));
            stmt.setDouble(5, auction.getPrice());
            stmt.setInt(6, auction.getTimeRemaining());
            stmt.setInt(7, auction.getAmount());
            stmt.setLong(8, auction.getCreatedAt());
            if (auction.getBuyerId() != null && auction.getBuyerName() != null) {
                stmt.setString(9, auction.getBuyerId().toString());
                stmt.setString(10, auction.getBuyerName());
            } else {
                stmt.setNull(9, 12);
                stmt.setNull(10, 12);
            }
            stmt.executeUpdate();
        }
        catch (IOException | SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Error saving auction", e);
        }
    }

    public void removeAuction(UUID auctionId) {
        String sql = "DELETE FROM " + this.tablePrefix + "active_auctions WHERE id = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, auctionId.toString());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Error removing auction", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void savePlayerStorage(UUID playerId, List<ItemStack> items) {
        try (Statement stmt = this.connection.createStatement();){
            stmt.execute("DELETE FROM " + this.tablePrefix + "pending_items WHERE player_uuid = '" + playerId.toString() + "'");
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Error clearing player storage", e);
            return;
        }
        String sql = "INSERT INTO " + this.tablePrefix + "pending_items (player_uuid, item_stack, price) VALUES (?, ?, ?)";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            this.connection.setAutoCommit(false);
            for (ItemStack item : items) {
                if (item == null) continue;
                try {
                    double price = 0.0;
                    ItemMeta meta = item.getItemMeta();
                    if (meta != null && meta.hasLore()) {
                        for (String line : meta.getLore()) {
                            if (!line.startsWith("\u00a77AuctionPrice:")) continue;
                            try {
                                price = Double.parseDouble(line.replace("\u00a77AuctionPrice:", "").trim());
                            }
                            catch (NumberFormatException numberFormatException) {}
                        }
                    }
                    stmt.setString(1, playerId.toString());
                    stmt.setString(2, this.serializeItemStack(item));
                    stmt.setDouble(3, price);
                    stmt.addBatch();
                }
                catch (IOException e) {
                    this.plugin.getLogger().log(Level.WARNING, "Error serializing item", e);
                }
            }
            stmt.executeBatch();
            this.connection.commit();
        }
        catch (SQLException e) {
            try {
                this.connection.rollback();
            }
            catch (SQLException ex) {
                this.plugin.getLogger().log(Level.SEVERE, "Error rolling back transaction", ex);
            }
            this.plugin.getLogger().log(Level.SEVERE, "Error saving player storage", e);
        }
        finally {
            try {
                this.connection.setAutoCommit(true);
            }
            catch (SQLException e) {
                this.plugin.getLogger().log(Level.WARNING, "Error resetting auto-commit", e);
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public Map<UUID, List<ItemStack>> loadAllStorage() {
        HashMap<UUID, List<ItemStack>> storage = new HashMap<UUID, List<ItemStack>>();
        String sql = "SELECT player_uuid, item_stack FROM " + this.tablePrefix + "pending_items";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);
             ResultSet rs = stmt.executeQuery();){
            while (rs.next()) {
                try {
                    UUID playerId = UUID.fromString(rs.getString("player_uuid"));
                    ItemStack item = this.deserializeItemStack(rs.getString("item_stack"));
                    if (item == null) continue;
                    storage.computeIfAbsent(playerId, k -> new ArrayList()).add(item);
                }
                catch (IllegalArgumentException e) {
                    this.plugin.getLogger().warning("Invalid UUID in storage: " + rs.getString("player_uuid"));
                }
            }
            return storage;
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Error loading storage", e);
        }
        return storage;
    }

    public List<ItemStack> loadPlayerStorage(UUID playerId) {
        ArrayList<ItemStack> items = new ArrayList<ItemStack>();
        String sql = "SELECT item_stack FROM " + this.tablePrefix + "pending_items WHERE player_uuid = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, playerId.toString());
            ResultSet rs = stmt.executeQuery();
            while (rs.next()) {
                try {
                    ItemStack item = this.deserializeItemStack(rs.getString("item_stack"));
                    if (item == null) continue;
                    items.add(item);
                }
                catch (IllegalArgumentException e) {
                    this.plugin.getLogger().warning("Invalid item data in storage for player: " + playerId);
                }
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Error loading player storage", e);
        }
        return items;
    }

    public void savePlayerStats(PlayerStats stats) {
        String sql = this.isMySQL ? "INSERT INTO " + this.tablePrefix + "player_stats (player_uuid, player_name, auctions_created, auctions_won, bids_placed, total_revenue, total_spent, first_join_time, last_auction_time, cooldown_until) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE player_name = VALUES(player_name), auctions_created = VALUES(auctions_created), auctions_won = VALUES(auctions_won), bids_placed = VALUES(bids_placed), total_revenue = VALUES(total_revenue), total_spent = VALUES(total_spent), first_join_time = VALUES(first_join_time), last_auction_time = VALUES(last_auction_time), cooldown_until = VALUES(cooldown_until)" : "REPLACE INTO " + this.tablePrefix + "player_stats (player_uuid, player_name, auctions_created, auctions_won, bids_placed, total_revenue, total_spent, first_join_time, last_auction_time, cooldown_until) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, stats.getPlayerId().toString());
            stmt.setString(2, stats.getPlayerName());
            stmt.setInt(3, stats.getAuctionsCreated());
            stmt.setInt(4, stats.getAuctionsWon());
            stmt.setInt(5, stats.getBidsPlaced());
            stmt.setDouble(6, stats.getTotalRevenue());
            stmt.setDouble(7, stats.getTotalSpent());
            stmt.setLong(8, stats.getFirstJoinTime());
            stmt.setLong(9, stats.getLastAuctionTime());
            stmt.setLong(10, this.getCooldownUntil(stats.getPlayerId()));
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Error saving player stats", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public PlayerStats loadPlayerStats(UUID playerId) {
        String sql = "SELECT * FROM " + this.tablePrefix + "player_stats WHERE player_uuid = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, playerId.toString());
            ResultSet rs = stmt.executeQuery();
            if (!rs.next()) return new PlayerStats(playerId, Bukkit.getOfflinePlayer((UUID)playerId).getName());
            PlayerStats stats = new PlayerStats(playerId, rs.getString("player_name"));
            stats.setAuctionsCreated(rs.getInt("auctions_created"));
            stats.setAuctionsWon(rs.getInt("auctions_won"));
            stats.setBidsPlaced(rs.getInt("bids_placed"));
            stats.setTotalRevenue(rs.getDouble("total_revenue"));
            stats.setTotalSpent(rs.getDouble("total_spent"));
            stats.setFirstJoinTime(rs.getLong("first_join_time"));
            stats.setLastAuctionTime(rs.getLong("last_auction_time"));
            this.setCooldownUntil(playerId, rs.getLong("cooldown_until"));
            PlayerStats playerStats = stats;
            return playerStats;
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Error loading player stats", e);
        }
        return new PlayerStats(playerId, Bukkit.getOfflinePlayer((UUID)playerId).getName());
    }

    public boolean isOnCooldown(UUID playerId) {
        return this.getCooldownUntil(playerId) > System.currentTimeMillis();
    }

    public void setCooldown(UUID playerId, int seconds) {
        this.setCooldownUntil(playerId, System.currentTimeMillis() + (long)seconds * 1000L);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private long getCooldownUntil(UUID playerId) {
        String sql = "SELECT cooldown_until FROM " + this.tablePrefix + "player_stats WHERE player_uuid = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, playerId.toString());
            ResultSet rs = stmt.executeQuery();
            if (!rs.next()) return 0L;
            long l = rs.getLong("cooldown_until");
            return l;
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Error getting cooldown", e);
        }
        return 0L;
    }

    private void setCooldownUntil(UUID playerId, long cooldownUntil) {
        String sql = "UPDATE " + this.tablePrefix + "player_stats SET cooldown_until = ? WHERE player_uuid = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setLong(1, cooldownUntil);
            stmt.setString(2, playerId.toString());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Error setting cooldown", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public List<Auction> loadActiveAuctions() {
        ArrayList<Auction> auctions = new ArrayList<Auction>();
        String sql = "SELECT * FROM " + this.tablePrefix + "active_auctions WHERE time_remaining > 0";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);
             ResultSet rs = stmt.executeQuery();){
            while (rs.next()) {
                try {
                    ItemStack item;
                    String itemData = rs.getString("item_stack");
                    if (itemData == null || itemData.isEmpty() || (item = this.deserializeItemStack(itemData)) == null) continue;
                    UUID sellerId = UUID.fromString(rs.getString("seller_uuid"));
                    String sellerName = rs.getString("seller_name");
                    Auction auction = new Auction(UUID.fromString(rs.getString("id")), sellerId, sellerName, item, rs.getDouble("price"), rs.getInt("amount"), this.plugin.getEconomy(), this.plugin);
                    auction.setTimeRemaining(rs.getInt("time_remaining"));
                    String buyerUuidStr = rs.getString("buyer_uuid");
                    String buyerName = rs.getString("buyer_name");
                    if (buyerUuidStr != null && buyerName != null) {
                        auction.setBuyer(UUID.fromString(buyerUuidStr), buyerName);
                    }
                    auctions.add(auction);
                }
                catch (Exception e) {
                    this.plugin.getLogger().log(Level.WARNING, "Error loading auction", e);
                }
            }
            return auctions;
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Error loading auctions", e);
        }
        return auctions;
    }

    public void saveActiveAuctions(List<Auction> auctions) {
        try (Statement stmt = this.connection.createStatement();){
            stmt.execute("DELETE FROM " + this.tablePrefix + "active_auctions");
            String sql = "INSERT INTO " + this.tablePrefix + "active_auctions (id, seller_uuid, seller_name, item_stack, price, time_remaining, amount, created_at, buyer_uuid, buyer_name) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
                this.connection.setAutoCommit(false);
                for (Auction auction : auctions) {
                    if (!auction.isActive() || auction.getTimeRemaining() <= 0) continue;
                    try {
                        pstmt.setString(1, auction.getId().toString());
                        pstmt.setString(2, auction.getSellerId().toString());
                        pstmt.setString(3, auction.getSellerName());
                        pstmt.setString(4, this.serializeItemStack(auction.getItemStack()));
                        pstmt.setDouble(5, auction.getPrice());
                        pstmt.setInt(6, auction.getTimeRemaining());
                        pstmt.setInt(7, auction.getAmount());
                        pstmt.setLong(8, auction.getCreatedAt());
                        if (auction.getBuyerId() != null && auction.getBuyerName() != null) {
                            pstmt.setString(9, auction.getBuyerId().toString());
                            pstmt.setString(10, auction.getBuyerName());
                        } else {
                            pstmt.setNull(9, 12);
                            pstmt.setNull(10, 12);
                        }
                        pstmt.addBatch();
                    }
                    catch (IOException e) {
                        this.plugin.getLogger().log(Level.WARNING, "Error serializing item", e);
                    }
                }
                pstmt.executeBatch();
                this.connection.commit();
            }
            catch (SQLException e) {
                this.connection.rollback();
                throw e;
            }
            finally {
                this.connection.setAutoCommit(true);
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Error saving auctions", e);
        }
    }

    private String serializeItemStack(ItemStack item) throws IOException {
        if (item == null) {
            throw new IOException("ItemStack is null");
        }
        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream();){
            String string;
            try (BukkitObjectOutputStream dataOutput = new BukkitObjectOutputStream((OutputStream)outputStream);){
                dataOutput.writeObject((Object)item);
                string = Base64Coder.encodeLines((byte[])outputStream.toByteArray());
            }
            return string;
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private ItemStack deserializeItemStack(String data) {
        if (data == null || data.isEmpty()) {
            return null;
        }
        try (ByteArrayInputStream inputStream = new ByteArrayInputStream(Base64Coder.decodeLines((String)data));){
            ItemStack itemStack;
            try (BukkitObjectInputStream dataInput = new BukkitObjectInputStream((InputStream)inputStream);){
                itemStack = (ItemStack)dataInput.readObject();
            }
            return itemStack;
        }
        catch (IOException | ClassNotFoundException e) {
            this.plugin.getLogger().log(Level.WARNING, "Error deserializing item", e);
            return null;
        }
    }

    public void close() {
        try {
            if (this.connection != null && !this.connection.isClosed()) {
                this.connection.close();
                this.plugin.getLogger().info("Database connection closed");
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Error closing connection", e);
        }
    }
}

